# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###


import gtk
import gobject
import pango

import stock
import config
from widget.view import MultiDragTreeview

from widget.song_columns import *
from widget.song_model import SongModel
from widget.song_menu import SongMenu
from helper import helper

from library import library_wrapper

        
class SongView(MultiDragTreeview):
    show_all_columns = False
    #def __init__(self,pl,*model_types):
    def __init__(self,pl,**kargs):
        super(SongView,self).__init__()
        if not kargs: #Default column display
            kargs = {
                        "#track":(0,gobject.TYPE_STRING),
                        "title":(1,gobject.TYPE_STRING),
                        "genre":(2,gobject.TYPE_STRING),
                        "artist":(3,gobject.TYPE_STRING),
                        "album":(4,gobject.TYPE_STRING),
                        "date":(5,gobject.TYPE_STRING),
                        "#duration":(6,gobject.TYPE_STRING),
                        "#bitrate":(7,gobject.TYPE_STRING),
                        "#playcount":(8,gobject.TYPE_STRING),
                        "#lastplayed":(9,gobject.TYPE_STRING)
                      }
        if kargs:
            model_types = tuple()
            col_tag = tuple()
            i = 0
            kargs = [ (pos,tag,col_type) for tag,(pos,col_type) in kargs.iteritems()]
            kargs.sort()
            for pos,tag,col_type in kargs:
                i+=1
                model_types += col_type,
                col_tag += tag,
                self.append_column_by_tag(tag,i)

        self.model = SongModel(self,*model_types)
        if kargs:
            self.model.set_column_tags(*col_tag)
        self.set_model(self.model)
        
        self.pl = pl
    
        targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1),("text/uri-list", 0, 2)]
        self.drag_source_set( gtk.gdk.BUTTON1_MASK, targets, gtk.gdk.ACTION_COPY)
        
        self.connect("row-activated", self.on_activated)
        self.connect("drag-data-get", self.on_drag_data_get)
        self.connect("popup-menu",self.__popup_menu)
        self.connect("popup-empty-menu",self.__popup_empty_menu)
        
        self.set_rules_hint(True)
        
        self.menu = None
        self.empty_menu = None

        self.set_search_equal_func(self.on_song_search)
        self.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        
        if not self.show_all_columns: 
            helper.connect("config-changed",self.on_config_change)
            gobject.idle_add(self.set_visible_column)
        
    def on_drag_data_get(self,treeview, drag_context, selection, info, timestamp):
        model,rows = treeview.get_selection().get_selected_rows()
        if len(rows)<0: return
        list_uri = list([ model[row[0]][0].get("uri") for row in rows])
        selection.set("text/listen-songs", 8, "\n".join(list_uri))
        selection.set_uris(list_uri)
    
    def on_config_change(self,helper,section,option,value):
        if section=="song_view":
            self.set_visible_column()
        
    def set_visible_column(self):
        for col in self.get_columns():
            col.set_visible(config.get("song_view",col.tag)=="true")
        
        
    def on_activated(self,treeview, path, view_column):
        helper.cur_playlist_play([treeview.get_model()[path][0]])
        
    def append_column_by_tag(self,tag,*ncol):
        self.append_column(TAG_COLUMNS[tag](*ncol))
        
    def insert_column_by_tag(self,pos,tag,*ncol):
        self.insert_column(TAG_COLUMNS[tag](*ncol),pos)
        
    def set_column_types(self,*param):
        self.get_model().set_column_types(object,*param)
        
    def set_column_tags(self,*tags):
        self.get_model().set_column_tags(*tags)
        
    def populate(self):
        pass
    
    def set_menu(self,menu,empty_menu = None):
        self.menu = menu
        self.empty_menu = empty_menu
        
    def get_select_songs(self):
        model, rows = self.get_selection().get_selected_rows()
        if len(rows) <= 0:
            return None
        else:
            return [model[row][0] for row in rows]
    
    """""""""""""""""""""
        FUNC SEARCH
    """""""""""""""""""""
    def on_song_search(self,model, column, key, iterator):
        title = model.get_value(iterator, 0).get_str("title")
        if title.lower().find(key.lower()) == 0:
          return False
        return True
    
    """""""""""""""""""""""""""
        FUNC SELECTION
    """""""""""""""""""""""""""
    def on_row_song_activated(self, treeview, path, view_column):
        self.play_selected()
        
    def __popup_empty_menu(self,widget):
        if self.empty_menu:
            self.empty_menu.popup([])

    def __popup_menu(self,widget):
        songs = self.get_select_songs()
        if self.menu and songs:
            self.menu.popup(songs)


        
        
class SortableSongView(SongView):
    def __init__(self,pl,conf_prefix,**kargs):
        self.conf_prefix = conf_prefix
        
        super(SortableSongView,self).__init__(pl,**kargs)
        self.get_model().set_get_sort_by_func(self.get_sort_by)
        
        self.set_sort_order()
        
    def set_sort_order(self):
        col_tag = [ col.tag for col in self.get_columns() ]
        if self.conf_prefix and config.get("browser",self.conf_prefix+"_sort_tag") in col_tag and config.get("browser",self.conf_prefix+"_sort_order") in ["ASC","DESC"]:
            if config.get("browser",self.conf_prefix+"_sort_order") == "ASC" : 
                sens = gtk.SORT_ASCENDING 
            else: 
                sens = gtk.SORT_DESCENDING
            self.set_sort_by(None,config.get("browser",self.conf_prefix+"_sort_tag"),sens, False)
        
    def set_model(self,model):
        super(SortableSongView,self).set_model(model)
        if model:
            self.set_sort_order()
        
    def set_sort_by(self, header, tag=None, order=None, refresh=True):
        #print "set_sort_by", header, tag, order, refresh
        if header!=None and tag==None: tag = header.tag
        for h in self.get_columns():
            if h.tag == tag:
                if self.conf_prefix:    
                    config.set("browser",self.conf_prefix+"_sort_tag",tag)
                if order!=None:
                     h.set_sort_order(order)
                     h.set_sort_indicator(True)
                elif h.get_sort_indicator() :
                    if h.get_sort_order() == gtk.SORT_ASCENDING:
                        if self.conf_prefix: config.set("browser",self.conf_prefix+"_sort_order","DESC")    
                        h.set_sort_order(gtk.SORT_DESCENDING)
                        h.set_sort_indicator(True)
                    else:
                        if self.conf_prefix: config.set("browser",self.conf_prefix+"_sort_order","")
                        h.set_sort_indicator(False)
                else: 
                    if self.conf_prefix: config.set("browser",self.conf_prefix+"_sort_order","ASC")
                    h.set_sort_order(gtk.SORT_ASCENDING)
                    h.set_sort_indicator(True)
                #print tag,h.get_sort_indicator(), h.get_sort_order()
            else: 
                h.set_sort_indicator(False)
                
        if refresh:
            self.get_model().set_songs()
        
    def get_sort_by(self):
        for header in self.get_columns():
            if header.get_sort_indicator():
                return (header.tag,
                        header.get_sort_order() == gtk.SORT_DESCENDING)
        #else: return "artist", None
        else: return "album", None
    
    def append_column_by_tag(self,tag,*ncol):
        super(SortableSongView,self).append_column_by_tag(tag,*ncol)
        col = self.get_column(len(self.get_columns())-1)
        col.__id_header_clicked = col.connect('clicked', self.set_sort_by)
        col.set_clickable(True)
        col.set_reorderable(True)
        
    def insert_column_by_tag(self,pos,tag,*ncol):
        super(SortableSongView,self).insert_column_by_tag(pos,tag,*ncol)  
        col = self.get_column(pos)
        col.__id_header_clicked = col.connect('clicked', self.set_sort_by)
        col.set_clickable(True)
        col.set_reorderable(True)
 
class PlaylistSongView(SongView):
    def __init__(self,pl,**kargs):
        super(PlaylistSongView,self).__init__(pl,**kargs)

        self.pl = pl
        
        menu = SongMenu(pl,True,True,False,False,True,True)
        if not pl.is_smart(): menu.insert_item(self.delete,stock.DELETE,2)
        menu.show_all()
        self.set_menu(menu)
        
        if pl.is_smart():
            helper.connect("pl-smart-populate",self.smart_populate_songs)
        else:
            targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1),("text/uri-list", 0, 2)]
            self.enable_model_drag_dest(targets, gtk.gdk.ACTION_COPY)
            self.connect("drag-data-received", self.on_drag_data_reveived)    
            helper.connect("pl-add-songs",self.append_songs)
            helper.connect("pl-insert-songs",self.insert_songs)
            helper.connect("pl-remove-songs",self.remove_songs)
        
        helper.connect("changed",self.get_model().change_songs)
        helper.connect("deleted",self.get_model().remove_songs)
        
    def smart_populate_songs(self,helper,pl,songs):
        self.get_model().clear()
        self.get_model().fill(songs)
        
    def append_songs(self,helper,pl,songs):
        if pl == self.pl:
            self.get_model().append_songs(songs)
            
    def remove_songs(self,helper,pl,songs,pos):
        if pl == self.pl:
            model = self.get_model()    
            if pos==None:
                model.remove_songs(songs)
            else:
                for i,s in enumerate(songs):
                    del model[pos[i]]
        
    def insert_songs(self,helper,pl,songs,pos):
        if pl == self.pl:
            self.get_model().insert_songs(pos,songs)
            
    def delete(self,w):
        model, rows = self.get_selection().get_selected_rows()
        positions = [row[0] for row in rows]
        self.pl.remove_pos(positions)
        
    def on_drag_data_reveived(self,treeview, context, x, y, selection, info, timestamp):
        drop_info = treeview.get_dest_row_at_pos(x, y)
        model = treeview.get_model()
        #print selection.data
        if selection.target in ["text/uri-list","text/listen-songs"]:
            if context.get_source_widget()==self:
                position = gtk.TREE_VIEW_DROP_AFTER
                model, rows = treeview.get_selection().get_selected_rows()
                if len(rows) < 1:
                    return
                rows = [row[0] for row in rows]

                if drop_info:
                    new_pos, position = drop_info
                    new_pos = new_pos[0]
                else:
                    new_pos = len(model)-1

                if new_pos < rows[0]:
                    rows.reverse()

                i=0
                for row in rows:
                    row += i

                    iter = model.get_iter(row)
                    new_iter = model.get_iter(new_pos)
                    if position == gtk.TREE_VIEW_DROP_BEFORE or position == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE:
                        model.move_before(iter, new_iter);
                    if position == gtk.TREE_VIEW_DROP_AFTER or position == gtk.TREE_VIEW_DROP_INTO_OR_AFTER:
                        model.move_after(iter, new_iter);
                        
                    if new_pos < rows[0]:
                        i += 1
                    else:
                        i -= 1 
                self.save_playlist()
                
            elif selection.target in ["text/listen-songs"]:#"text/uri-list", #DISABLE FOR NOW
                if selection.data:
                    if drop_info:
                        new_pos, position = drop_info
                        new_pos = new_pos[0]
                        if position == gtk.TREE_VIEW_DROP_AFTER:
                            new_pos += 1
                    else:
                        new_pos = len(model)
    
                    songs = []
                    for uri in  selection.data.splitlines():
                        song = library_wrapper.get_song(uri)
                        if song :
                            songs.append(song)
                    #self.get_model().insert_songs(new_pos,songs)
                    self.pl.insert(songs,new_pos)
                
                
    def save_playlist(self):
        model = self.get_model()
        self.pl.remove_songs(self.pl.get_songs())
        new_songs = []
        for i in range(0,len(model)):
            new_songs.append(model[i][0])
        model.clear()
        self.pl.append(new_songs)


