/*
 * auth.c - deal with authentication.
 *
 * This file implements authentication when setting up an RFB connection.
 *
 * Modified for XFree86 4.x by Alan Hourihane <alanh@fairlite.demon.co.uk>
 * Modified for MetaVNC by UCHINO Satoshi <utinos@users.sourceforge.net>
 */

/*
 *  Copyright (C) 2004 UCHINO Satoshi. All Rights Reserved.
 *  Copyright (C) 2003 Constantin Kaplinsky.  All Rights Reserved.
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "rfb.h"
#include "windowstr.h"

static void rfbAuthSendCaps(rfbClientPtr cl);
static void rfbAuthSendChallenge(rfbClientPtr cl);
static void rfbClientConnFailed(rfbClientPtr cl, char *reason);

/*
 * rfbAuthNewClient is called when we reach the point of authenticating
 * a new client.  If authentication isn't being used then we simply send
 * rfbNoAuth.  Otherwise we send rfbVncAuth plus the challenge.
 */

void
rfbAuthNewClient(cl)
    rfbClientPtr cl;
{
    VNCSCREENPTR(cl->pScreen);
    int secType;
    char *errReason = NULL;

    if ((!pVNC->rfbAuthPasswdFile && !pVNC->loginAuthEnabled) || cl->reverseConnection) {
	cl->authRequired = FALSE;
	secType = rfbSecTypeNone;
    } else {
	cl->authRequired = TRUE;
	if (pVNC->rfbAuthPasswdFile)
	    secType = rfbSecTypeVncAuth;
        else {
	    secType = rfbSecTypeInvalid;
            errReason = "VNC authentication disabled, please use an RFB3.7t compatible viewer";
        }
    }

    if (cl->protocol_minor_ver == rfbProtocolMinorVersion) {    /* RFB3.7 */
        CARD8 msg[3];
        int msgLen = 1;

        if (rfbAuthIsBlocked(cl)) {
	    errReason = "Too many authentication failures";
        } else {
            if (secType != rfbSecTypeInvalid)
                msg[msgLen++] = secType;
            msg[msgLen++] = rfbSecTypeTight;
        }
        msg[0] = msgLen - 1; 

	if (WriteExact(cl->sock, (char *)&msg, msgLen) < 0) {
	    rfbLogPerror("rfbAuthNewClient: write");
	    rfbCloseSock(cl->pScreen, cl->sock);
	    return;
	}

        if (msgLen == 1) {
            rfbClientConnFailed(cl, errReason);
            return;
        }

        /* Dispatch client input to rfbSecurityType. */
        cl->state = RFB_SECURITY_TYPE;

    } else {    /* RFB3.3 */
        CARD32 msg;

        if (rfbAuthIsBlocked(cl)) {
	    secType = rfbSecTypeInvalid;
	    errReason = "Too many authentication failures";
        }

        msg = Swap32IfLE(secType);
	if (WriteExact(cl->sock, (char *)&msg, sizeof(msg)) < 0) {
	    rfbLogPerror("rfbAuthNewClient: write");
	    rfbCloseSock(cl->pScreen, cl->sock);
	    return;
	}

        if (secType == rfbSecTypeInvalid) {
            rfbClientConnFailed(cl, errReason);
            return;
        }

	if (cl->authRequired) {
	    rfbAuthSendChallenge(cl);
	} else {
	    /* Dispatch client input to rfbProcessClientInitMessage. */
	    cl->state = RFB_INITIALISATION;
	}
    }
}

/*
 * rfbClientConnFailed is called when a client connection has failed
 */

static void
rfbClientConnFailed(cl, reason)
    rfbClientPtr cl;
    char *reason;
{
    int len = strlen(reason);
    len = Swap32IfLE(len);
    if (WriteExact(cl->sock, &len, sizeof(len)) < 0)
        rfbLogPerror("rfbClientConnFailed: write");
    if (WriteExact(cl->sock, reason, len) < 0)
	rfbLogPerror("rfbClientConnFailed: write");

    rfbCloseSock(cl->pScreen, cl->sock);
}

/*
 * rfbSendTunnelingCaps is called after deciding the protocol version,
 * and only if the protocol version is 3.7t.  In this function, we send
 * the list of our tunneling capabilities.
 */

static void
rfbSendTunnelingCaps(cl)
    rfbClientPtr cl;
{
    rfbTunnelingCapsMsg caps;
    int nTypes = 0;             /* we don't support tunneling yet */

    caps.nTunnelTypes = Swap32IfLE(nTypes);
    if (WriteExact(cl->sock, (char *)&caps, sz_rfbTunnelingCapsMsg) < 0) {
	rfbLogPerror("rfbSendTunnelingCaps: write");
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    if (nTypes) {
	/* Dispatch client input to rfbProcessClientTunnelingType(). */
	cl->state = RFB_TUNNELING_TYPE;
    } else {
	rfbAuthSendCaps(cl);
    }
}


/*
 * In rfbAuthSendCaps, we send the list of our authentication capabilities
 * to the client (protocol 3.7t).
 */

static void
rfbAuthSendCaps(cl)
    rfbClientPtr cl;
{
    VNCSCREENPTR(cl->pScreen);
    rfbAuthenticationCapsMsg caps;
    rfbCapabilityInfo caplist[2];
    int count = 0;

    if (cl->authRequired) {
	if (pVNC->loginAuthEnabled)
	    SetCapInfo(&caplist[count++], rfbAuthUnixLogin, rfbTightVncVendor);
	if (pVNC->rfbAuthPasswdFile != NULL)
	    SetCapInfo(&caplist[count++], rfbAuthVNC, rfbStandardVendor);
	if (count == 0) {
	    /* Should never happen. */
	    rfbLog("rfbAuthSendCaps: assertion failed\n");
	    rfbCloseSock(cl->pScreen, cl->sock);
	    return;
	}
    }

    caps.nAuthTypes = Swap32IfLE(count);
    if (WriteExact(cl->sock, (char *)&caps, sz_rfbAuthenticationCapsMsg) < 0) {
	rfbLogPerror("rfbAuthSendCaps: write");
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    if (count) {
	if (WriteExact(cl->sock, (char *)&caplist[0],
		       count * sz_rfbCapabilityInfo) < 0) {
	    rfbLogPerror("rfbAuthSendCaps: write");
	    rfbCloseSock(cl->pScreen, cl->sock);
	    return;
	}
	/* Dispatch client input to rfbProcessClientAuthType. */
	cl->state = RFB_AUTH_TYPE;
    } else {
	/* Dispatch client input to rfbProcessClientInitMessage. */
	cl->state = RFB_INITIALISATION;
    }
}


/*
 * Read client's preferred authentication type (protocol 3.7t).
 */

void
rfbAuthProcessType(cl)
    rfbClientPtr cl;
{
    CARD32 auth_type;
    int n;

    n = ReadExact(cl->sock, (char *)&auth_type, sizeof(auth_type));
    if (n <= 0) {
	if (n == 0)
	    rfbLog("rfbAuthProcessType: client gone\n");
	else
	    rfbLogPerror("rfbAuthProcessType: read");
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }
    auth_type = Swap32IfLE(auth_type);

    switch (auth_type) {
    case rfbAuthVNC:
	rfbAuthSendChallenge(cl);
	break;
    case rfbAuthUnixLogin:
	/* FIXME: Do (cl->state = RFB_LOGIN_AUTH) instead? */
	rfbLoginAuthProcessClientMessage(cl);
	break;
    default:
	rfbLog("rfbAuthProcessType: unknown authentication scheme\n");
	rfbCloseSock(cl->pScreen, cl->sock);
    }
}


/*
 * Read client's preferred security type (protocol 3.7).
 */

void
rfbSecurityType(rfbClientPtr cl)
{
    CARD8 sec_type;
    int n;

    n = ReadExact(cl->sock, (char *)&sec_type, sizeof(sec_type));
    if (n <= 0) {
	if (n == 0)
	    rfbLog("rfbSecurityType: client gone\n");
	else
	    rfbLogPerror("rfbSecurityType: read");
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    switch (sec_type) {
    case rfbSecTypeNone:
        /* Dispatch client input to rfbProcessClientInitMessage. */
        cl->state = RFB_INITIALISATION;
	break;
    case rfbSecTypeVncAuth:
	rfbAuthSendChallenge(cl);
	break;
    case rfbSecTypeTight:
        cl->protocol_tightvnc = TRUE;
	rfbSendTunnelingCaps(cl); /* protocol 3.7t */
	break;
    default:
	rfbLog("rfbSecurityType: unknown security type: %d\n", sec_type);
	rfbCloseSock(cl->pScreen, cl->sock);
    }
}


/*
 * Send the authentication challenge.
 */

static void
rfbAuthSendChallenge(cl)
    rfbClientPtr cl;
{
    vncRandomBytes(cl->authChallenge);
    if (WriteExact(cl->sock, (char *)cl->authChallenge, CHALLENGESIZE) < 0) {
	rfbLogPerror("rfbAuthSendChallenge: write");
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    /* Dispatch client input to rfbAuthProcessResponse. */
    cl->state = RFB_AUTHENTICATION;
}


/*
 * rfbAuthProcessResponse is called when the client sends its
 * authentication response.
 */

void
rfbAuthProcessResponse(cl)
    rfbClientPtr cl;
{
    VNCSCREENPTR(cl->pScreen);
    char passwdFullControl[9];
    char passwdViewOnly[9];
    int numPasswords;
    Bool ok;
    int n;
    CARD8 encryptedChallenge1[CHALLENGESIZE];
    CARD8 encryptedChallenge2[CHALLENGESIZE];
    CARD8 response[CHALLENGESIZE];
    CARD32 authResult;

    n = ReadExact(cl->sock, (char *)response, CHALLENGESIZE);
    if (n <= 0) {
	if (n != 0)
	    rfbLogPerror("rfbAuthProcessResponse: read");
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    numPasswords = vncDecryptPasswdFromFile2(pVNC->rfbAuthPasswdFile,
					     passwdFullControl,
					     passwdViewOnly);
    if (numPasswords == 0) {
	rfbLog("rfbAuthProcessResponse: could not get password from %s\n",
	       pVNC->rfbAuthPasswdFile);

	authResult = Swap32IfLE(rfbVncAuthFailed);

	if (WriteExact(cl->sock, (char *)&authResult, 4) < 0) {
	    rfbLogPerror("rfbAuthProcessResponse: write");
	}
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    memcpy(encryptedChallenge1, cl->authChallenge, CHALLENGESIZE);
    vncEncryptBytes(encryptedChallenge1, passwdFullControl);
    memcpy(encryptedChallenge2, cl->authChallenge, CHALLENGESIZE);
    vncEncryptBytes(encryptedChallenge2,
		    (numPasswords == 2) ? passwdViewOnly : passwdFullControl);

    /* Lose the passwords from memory */
    memset(passwdFullControl, 0, 9);
    memset(passwdViewOnly, 0, 9);

    ok = FALSE;
    if (memcmp(encryptedChallenge1, response, CHALLENGESIZE) == 0) {
	rfbLog("Full-control authentication passed by %s\n", cl->host);
	ok = TRUE;
	cl->viewOnly = FALSE;
    } else if (memcmp(encryptedChallenge2, response, CHALLENGESIZE) == 0) {
	rfbLog("View-only authentication passed by %s\n", cl->host);
	ok = TRUE;
	cl->viewOnly = TRUE;
    }

    if (!ok) {
	rfbLog("rfbAuthProcessResponse: authentication failed from %s\n",
	       cl->host);

	if (rfbAuthConsiderBlocking(cl)) {
	    authResult = Swap32IfLE(rfbVncAuthTooMany);
	} else {
	    authResult = Swap32IfLE(rfbVncAuthFailed);
	}

	if (WriteExact(cl->sock, (char *)&authResult, 4) < 0) {
	    rfbLogPerror("rfbAuthProcessResponse: write");
	}
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    rfbAuthUnblock(cl);

    authResult = Swap32IfLE(rfbVncAuthOK);

    if (WriteExact(cl->sock, (char *)&authResult, 4) < 0) {
	rfbLogPerror("rfbAuthProcessResponse: write");
	rfbCloseSock(cl->pScreen, cl->sock);
	return;
    }

    /* Dispatch client input to rfbProcessClientInitMessage(). */
    cl->state = RFB_INITIALISATION;
}


/*
 * Functions to prevent too many successive authentication failures.
 * FIXME: This should be performed separately per each client IP.
 */

/* Maximum authentication failures before blocking connections */
#define MAX_AUTH_TRIES 5

/* Delay in ms, doubles for each failure over MAX_AUTH_TRIES */
#define AUTH_TOO_MANY_BASE_DELAY 10 * 1000

/*
 * This function should not be called directly, it is called by
 * setting a timer in rfbAuthConsiderBlocking().
 */

static CARD32
rfbAuthReenable(OsTimerPtr timer, CARD32 now, pointer arg)
{
    rfbClientPtr cl = (rfbClientPtr) arg;
    VNCSCREENPTR(cl->pScreen);
    pVNC->rfbAuthTooManyTries = FALSE;
    return 0;
}

/*
 * This function should be called after each authentication failure.
 * The return value will be true if there was too many failures.
 */

Bool
rfbAuthConsiderBlocking(rfbClientPtr cl)
{
    VNCSCREENPTR(cl->pScreen);
    int i;

    pVNC->rfbAuthTries++;

    if (pVNC->rfbAuthTries >= MAX_AUTH_TRIES) {
	CARD32 delay = AUTH_TOO_MANY_BASE_DELAY;
	for (i = MAX_AUTH_TRIES; i < pVNC->rfbAuthTries; i++)
	    delay *= 2;
	pVNC->timer = TimerSet(pVNC->timer, 0, delay, rfbAuthReenable, NULL);
	pVNC->rfbAuthTooManyTries = TRUE;
	return TRUE;
    }

    return FALSE;
}

/*
 * This function should be called after successful authentication.
 * It resets the counter of authentication failures. Note that it's
 * not necessary to clear the rfbAuthTooManyTries flag as it will be
 * reset by the timer function.
 */

void
rfbAuthUnblock(rfbClientPtr cl)
{
    VNCSCREENPTR(cl->pScreen);
    pVNC->rfbAuthTries = 0;
}

/*
 * This function should be called before authentication process.
 * The return value will be true if there was too many authentication
 * failures, and the server should not allow another try.
 */

Bool
rfbAuthIsBlocked(rfbClientPtr cl)
{
    VNCSCREENPTR(cl->pScreen);
    return pVNC->rfbAuthTooManyTries;
}

