/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include <apr_strings.h>
#include <ruby.h>

#undef PACKAGE_NAME
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#undef PACKAGE_STRING

#include "rast/filter.h"
#include "rast/ruby.h"

#define SUPPORTED_FRAMEWORK_VERSION 1
#define RUBY_SUPPORTED_FRAMEWORK_VERION 1

typedef struct {
    VALUE class;
} ruby_wrapper_data_t;

typedef struct {
    VALUE instance;
} ruby_wrapper_context_t;

/* todo: rename AprHash */
static VALUE mFilterModule;
static VALUE cMimeFilter, cTextFilter;
static VALUE cAprHash;

static rast_filter_t *get_filter(VALUE self);

static VALUE
filter_new(VALUE klass, rast_filter_t *filter)
{
    VALUE vfilter;

    vfilter = Data_Wrap_Struct(klass, NULL, NULL, NULL);
    DATA_PTR(vfilter) = filter;

    return vfilter;
}

static rast_filter_t *
get_filter(VALUE self)
{
    if (TYPE(self) != T_DATA) {
        rb_raise(rb_eTypeError,
                 "wrong argument type %s (expected Rast::Framework::Filter)",
                 rb_obj_classname(self));
    }
    return (rast_filter_t *) DATA_PTR(self);
}

static VALUE
mime_filter_pass(int argc, VALUE *argv, VALUE self)
{
    rast_filter_t *filter = get_filter(self);
    apr_bucket_brigade *brigade;
    const char *mime_type, *filename;
    VALUE vbrigade, vmime_type, vfilename;

    rb_scan_args(argc, argv, "21", &vbrigade, &vmime_type, &vfilename);

    brigade = rast_rb_get_brigade(vbrigade);
    mime_type = rast_rb_get_safe_string_ptr(vmime_type);
    filename = rast_rb_get_safe_string_ptr(vfilename);
    rast_rb_raise_error(rast_mime_filter_pass(filter, brigade, mime_type,
                                              filename));

    return Qnil;
}

static VALUE
text_filter_pass(VALUE self, VALUE vbrigade, VALUE vmime_type)
{
    rast_filter_t *filter = get_filter(self);
    apr_bucket_brigade *brigade;
    const char *mime_type;

    SafeStringValue(vmime_type);
    mime_type = StringValuePtr(vmime_type);
    brigade = rast_rb_get_brigade(vbrigade);
    rast_rb_raise_error(rast_text_filter_pass(filter, brigade, mime_type));

    return Qnil;
}

static VALUE
filter_set_property(VALUE self, VALUE vname, VALUE vvalue)
{
    rast_filter_t *filter = get_filter(self);
    const char *name;
    rast_value_t *value;
    apr_pool_t *pool;

    apr_pool_create(&pool, filter->pool);
    value = (rast_value_t *) apr_palloc(pool, sizeof(rast_value_t));
    SafeStringValue(vname);
    name = StringValuePtr(vname);
    switch (TYPE(vvalue)) {
    case T_STRING:
        rast_value_set_string(value, StringValuePtr(vvalue));
        rast_value_set_type(value, RAST_TYPE_STRING);
        break;
    case T_FIXNUM:
        rast_value_set_uint(value, NUM2INT(vvalue));
        rast_value_set_type(value, RAST_TYPE_UINT);
        break;
    default:
        if (RTEST(rb_obj_is_kind_of(vvalue, rast_rb_cDateTime))) {
            VALUE str_value;

            str_value = rb_funcall(vvalue, rb_intern("strftime"), 1,
                                   rb_str_new2("%Y-%m-%dT%H:%M:%S"));
            rast_value_set_date(value,
                                StringValuePtr(str_value));
            rast_value_set_type(value, RAST_TYPE_DATE);
        }
        else {
            rb_raise(rast_rb_eError, "unknown property type");
        }
    }
    rast_rb_raise_error(rast_filter_set_property(filter, name, value));

    return Qnil;
}

static VALUE
filter_db_encoding(VALUE self)
{
    rast_filter_t *filter = get_filter(self);
    const char *db_encoding;

    db_encoding = rast_filter_db_encoding(filter);

    return rb_str_new2(db_encoding);
}

typedef struct {
    rast_encoding_converter_t *converter;
    VALUE vpool;
} encoding_converter_data_t;

static rast_encoding_converter_t *
get_converter(VALUE vconverter)
{
    encoding_converter_data_t *data;

    data = (encoding_converter_data_t *) DATA_PTR(vconverter);
    return data->converter;
}

static void
encoding_converter_mark(encoding_converter_data_t *data)
{
    if (data == NULL) {
        return;
    }
    rb_gc_mark(data->vpool);
}

static VALUE
encoding_converter_s_guess(VALUE self, VALUE s, VALUE vcandidate_encodings)
{
    const char *result, **candidate_encodings;
    rast_error_t *error;
    int i, len;
    VALUE vpool;
    apr_pool_t *pool;

    SafeStringValue(s);
    Check_Type(vcandidate_encodings, T_ARRAY);

    pool = rast_rb_pool_new(&vpool);

    len = RARRAY(vcandidate_encodings)->len;
    candidate_encodings = (const char **)
        apr_palloc(pool, sizeof(const char *) * (len + 1));
    for (i = 0; i < len; i++) {
        VALUE encoding = RARRAY(vcandidate_encodings)->ptr[i];

        SafeStringValue(encoding);
        candidate_encodings[i] = StringValuePtr(encoding);
    }
    candidate_encodings[i] = NULL;

    error = rast_encoding_converter_guess(candidate_encodings,
                                          RSTRING(s)->ptr, RSTRING(s)->len,
                                          &result);
    rast_rb_raise_error(error);

    if (result == NULL) {
        return Qnil;
    }
    return rb_str_new2(result);
}

static VALUE
encoding_converter_s_convert_encoding(VALUE self, VALUE vfrom_encoding,
                                      VALUE vto_encoding, VALUE vtext)
{
    const char *from_encoding, *to_encoding, *text;
    size_t nbytes, out_buf_nbytes;
    char *out_buf;
    VALUE vpool;
    apr_pool_t *pool;
    rast_error_t *error;

    pool = rast_rb_pool_new(&vpool);
    SafeStringValue(vfrom_encoding);
    from_encoding = StringValuePtr(vfrom_encoding);
    SafeStringValue(vto_encoding);
    to_encoding = StringValuePtr(vto_encoding);
    SafeStringValue(vtext);
    text = RSTRING(vtext)->ptr;
    nbytes = RSTRING(vtext)->len;

    error = rast_convert_encoding(from_encoding, to_encoding, text, nbytes,
                                  &out_buf, &out_buf_nbytes, pool);
    rast_rb_raise_error(error);

    return rb_str_new(out_buf, out_buf_nbytes);
}

static VALUE
encoding_converter_alloc(VALUE klass)
{
    return Data_Wrap_Struct(klass, encoding_converter_mark, NULL, NULL);
}

static VALUE
encoding_converter_initialize(VALUE self, VALUE vfrom_encoding,
                              VALUE vto_encodings)
{
    const char *from_encoding, **to_encodings;
    int i, len;
    apr_pool_t *pool;
    VALUE vpool;
    rast_encoding_converter_t *converter;
    encoding_converter_data_t *data;
    rast_error_t *error;

    pool = rast_rb_pool_new(&vpool);
    
    SafeStringValue(vfrom_encoding);
    from_encoding = StringValuePtr(vfrom_encoding);

    len = RARRAY(vto_encodings)->len;
    to_encodings = (const char **)
        apr_palloc(pool, sizeof(const char *) * (len + 1));
    for (i = 0; i < len; i++) {
        VALUE to_encoding = RARRAY(vto_encodings)->ptr[i];

        SafeStringValue(to_encoding);
        to_encodings[i] = StringValuePtr(to_encoding);
    }
    to_encodings[len] = NULL;

    error = rast_encoding_converter_create(&converter, from_encoding,
                                           to_encodings, pool);
    rast_rb_raise_error(error);

    data = (encoding_converter_data_t *)
        apr_palloc(pool, sizeof(encoding_converter_data_t));
    data->converter = converter;
    data->vpool = vpool;

    DATA_PTR(self) = data;
    return Qnil;
}

static VALUE
encoding_converter_convert(VALUE self, VALUE s)
{
    rast_encoding_converter_t *converter = get_converter(self);
    VALUE result = rb_str_new2("");
    char out_buf[1024];
    int out_buf_nbytes;
    rast_error_t *error;

    SafeStringValue(s);
    error = rast_encoding_converter_add_text(converter, StringValuePtr(s),
                                             RSTRING(s)->len);
    rast_rb_raise_error(error);
    do {
        out_buf_nbytes = sizeof(out_buf);
        error = rast_encoding_converter_get_next(converter, out_buf,
                                                 &out_buf_nbytes);
        rast_rb_raise_error(error);
        rb_str_concat(result, rb_str_new(out_buf, out_buf_nbytes));
    } while (!rast_encoding_converter_is_done(converter));

    return result;
}

static void
define_rast_language_encodings(const char *const_name,
                               const char * const *encodings)
{
    int i;
    VALUE vencodings;

    vencodings = rb_ary_new();
    for (i = 0; encodings[i] != NULL; i++) {
        rb_ary_push(vencodings, rb_str_new2(encodings[i]));
    }
    rb_define_const(rast_rb_mRast, const_name, vencodings);
}

static void
ruby_initialize(rast_filter_map_t *map)
{
    /* todo: rename Framework */
    VALUE mFramework;
    VALUE cFilter, cEncodingConverter;

    ruby_init();
    ruby_init_loadpath();

    rast_rb_initialize();

    define_rast_language_encodings("JAPANESE_ENCODINGS",
                                   RAST_JAPANESE_ENCODINGS);

    mFilterModule = rb_define_module_under(rast_rb_mRast, "FilterModule");
    mFramework = rb_define_module_under(rast_rb_mRast, "Framework");

    cAprHash = rb_define_class_under(mFramework, "AprHash", rb_cObject);

    cFilter = rb_define_class_under(mFramework, "Filter", rb_cObject);
    rb_define_method(cFilter, "set_property", filter_set_property, 2);
    rb_define_method(cFilter, "db_encoding", filter_db_encoding, 0);

    cMimeFilter = rb_define_class_under(mFramework, "MimeFilter", cFilter);
    rb_define_method(cMimeFilter, "pass", mime_filter_pass, -1);

    cTextFilter = rb_define_class_under(mFramework, "TextFilter", cFilter);
    rb_define_method(cTextFilter, "pass", text_filter_pass, 2);

    cEncodingConverter = rb_define_class_under(rast_rb_mRast,
                                               "EncodingConverter",
                                               rb_cObject);
    rb_define_singleton_method(cEncodingConverter, "guess",
                               encoding_converter_s_guess, 2);
    rb_define_singleton_method(cEncodingConverter, "convert_encoding",
                               encoding_converter_s_convert_encoding, 3);
    rb_define_alloc_func(cEncodingConverter, encoding_converter_alloc);
    rb_define_method(cEncodingConverter, "initialize",
                     encoding_converter_initialize, 2);
    rb_define_method(cEncodingConverter, "convert",
                     encoding_converter_convert, 1);
}

typedef struct {
    VALUE receiver;
    ID mid;
    int argc;
    VALUE *argv;
} protect_arg_t;

static VALUE
protect_funcall0(VALUE data)
{
    protect_arg_t *arg = (protect_arg_t *) data;

    return rb_funcall2(arg->receiver, arg->mid, arg->argc, arg->argv);
}

static rast_error_t *
protect_funcall(VALUE *result, VALUE receiver, ID mid, int argc, ...)
{
    va_list ap;
    VALUE args, res;
    protect_arg_t arg;
    int state;

    arg.receiver = receiver;
    arg.mid = mid;
    arg.argc = argc;
    args = rb_ary_new();
    if (argc > 0) {
        int i;

        va_start(ap, argc);
        for (i = 0; i < argc; i++) {
            VALUE arg;

            arg = va_arg(ap, VALUE);
            rb_ary_push(args, arg);
        }
        va_end(ap);

        arg.argv = RARRAY(args)->ptr;
    }
    else {
        arg.argv = NULL;
    }

    res = rb_protect(protect_funcall0, (VALUE) &arg, &state);

    if (state != 0) {
        if (NIL_P(ruby_errinfo)) {
            /* todo: classify error type */
            return rast_error_create(RAST_ERROR_TYPE_RUBY, RAST_ERROR_GENERAL,
                                     "error occured");
        }
        else {
            return rast_rb_exception_to_rast_error(ruby_errinfo);
        }
    }

    if (result != NULL) {
        *result = res;
    }
    return RAST_OK;
}

static rast_error_t *
ruby_wrapper_filter_invoke(VALUE filter_class,
                           rast_filter_t *filter, apr_bucket_brigade *brigade,
                           const char *mime_type)
{
    ruby_wrapper_context_t *context;
    VALUE vfilter, vbrigade, vmime_type;
    rast_error_t *error;

    if (filter->context == NULL) {
        ruby_wrapper_data_t *user_data;

        context = (ruby_wrapper_context_t *)
            apr_palloc(filter->pool, sizeof(ruby_wrapper_context_t));

        user_data = (ruby_wrapper_data_t *) filter->filter_module->user_data;
        error = protect_funcall(&context->instance, user_data->class,
                                rb_intern("new"), 0);
        if (error != RAST_OK) {
            return error;
        }
        filter->context = context;
    }
    else {
        context = (ruby_wrapper_context_t *) filter->context;
    }

    vfilter = filter_new(filter_class, filter);
    vbrigade = rast_rb_brigade_new(brigade);
    vmime_type = rb_str_new2(mime_type);
    return protect_funcall(NULL, context->instance, rb_intern("invoke"), 3,
                           vfilter, vbrigade, vmime_type);
}

static rast_error_t *
ruby_wrapper_mime_filter_invoke(rast_filter_t *filter,
                                apr_bucket_brigade *brigade,
                                const char *mime_type)
{
    return ruby_wrapper_filter_invoke(cMimeFilter, filter, brigade, mime_type);
}

static rast_error_t *
ruby_wrapper_text_filter_invoke(rast_filter_t *filter,
                                apr_bucket_brigade *brigade,
                                const char *mime_type)
{
    return ruby_wrapper_filter_invoke(cTextFilter, filter, brigade, mime_type);
}

static rast_filter_module_t *
ruby_filter_module_create(VALUE class,
                          rast_error_t *(*invoke_func)(rast_filter_t *,
                                                       apr_bucket_brigade *,
                                                       const char *),
                          apr_pool_t *pool)
{
    rast_filter_module_t *filter_module;
    ruby_wrapper_data_t *data;

    data = (ruby_wrapper_data_t *) apr_palloc(pool,
                                              sizeof(ruby_wrapper_data_t));
    data->class = class;
    filter_module = (rast_filter_module_t *)
        apr_palloc(pool, sizeof(rast_filter_module_t));
    filter_module->supported_version = SUPPORTED_FRAMEWORK_VERSION;
    filter_module->user_data = data;
    filter_module->invoke = invoke_func;
    return filter_module;
}

static VALUE
call_module_eval(VALUE data)
{
    VALUE *eval_args = (VALUE *) data;

    return rb_mod_module_eval(2, eval_args, mFilterModule);
}

static rast_error_t *
load_ruby_filter_modules(rast_filter_map_t *map, const char *dirname,
                         apr_pool_t *pool)
{
    apr_status_t status;
    int name_len, rbext_len = strlen(".rb");
    apr_dir_t *dir;
    rast_error_t *error;

    status = apr_dir_open(&dir, dirname, pool);
    if (status != APR_SUCCESS) {
        return apr_status_to_rast_error(status);
    }

    while (1) {
        VALUE eval_args[2], vbuf, constants, classes;
        apr_finfo_t finfo;
        const char *path;
        int i, state;

        status = apr_dir_read(&finfo, APR_FINFO_TYPE | APR_FINFO_NAME, dir);
        if (status == APR_ENOENT) {
            break;
        }
        if (status != APR_SUCCESS || finfo.filetype != APR_REG) {
            continue;
        }
        name_len = strlen(finfo.name);
        if (name_len <= rbext_len ||
            strcmp(finfo.name + name_len - rbext_len, ".rb") != 0) {
            continue;
        }
        path = apr_pstrcat(pool, dirname, "/", finfo.name, NULL);
        eval_args[1] = rb_str_new2(path);
        vbuf = rb_funcall(rb_cFile, rb_intern("read"), 1, eval_args[1]);
        eval_args[0] = vbuf;
        (void) rb_protect(call_module_eval, (VALUE) eval_args, &state);
        if (state != 0) {
            continue;
        }
        constants = rb_funcall(mFilterModule, rb_intern("constants"), 0);
        classes = rb_ary_new();
        for (i = 0; i < RARRAY(constants)->len; i++) {
            VALUE obj;

            obj = rb_funcall(mFilterModule, rb_intern("const_get"), 1,
                             RARRAY(constants)->ptr[i]);

            if (RTEST(rb_funcall(obj, rb_intern("is_a?"), 1, rb_cClass))) {
                VALUE vversion;
                int version;

                vversion = rb_funcall(obj, rb_intern("const_get"), 1,
                                      rb_str_new2("SUPPORTED_VERSION"));
                version = NUM2INT(vversion);

                if (version <= RUBY_SUPPORTED_FRAMEWORK_VERION) {
                    rb_ary_push(classes, obj);
                }
            }
        }

        for (i = 0; i < RARRAY(classes)->len; i++) {
            VALUE class;
            rast_filter_module_t *filter_module;

            class = RARRAY(classes)->ptr[i];
            if (rb_const_defined(class, rb_intern("MIME_TYPE"))) {
                const char *mime_type;
                VALUE vmime_type;

                vmime_type = rb_const_get(class, rb_intern("MIME_TYPE"));
                SafeStringValue(vmime_type);
                mime_type = StringValuePtr(vmime_type);

                filter_module =
                    ruby_filter_module_create(class,
                                              ruby_wrapper_mime_filter_invoke,
                                              map->pool);
                error = rast_filter_map_add_mime_filter(map, mime_type,
                                                        filter_module);
                if (error != RAST_OK) {
                    /* todo: error handling */
                    rast_error_destroy(error);
                }

                if (rb_const_defined(class, rb_intern("EXTENSIONS"))) {
                    VALUE exts;
                    int j;

                    exts = rb_const_get(class, rb_intern("EXTENSIONS"));
                    Check_Type(exts, T_ARRAY);

                    for (j = 0; j < RARRAY(exts)->len; j++) {
                        VALUE vext;
                        const char *ext;

                        vext = RARRAY(exts)->ptr[j];
                        SafeStringValue(vext);
                        ext = StringValuePtr(vext);
                        error = rast_filter_map_add_extension(map, ext,
                                                              mime_type);
                        if (error != RAST_OK) {
                            /* todo: error handling */
                            rast_error_destroy(error);
                        }
                    }
                }
            }
            if (rb_const_defined(class, rb_intern("NAME"))) {
                VALUE name;

                name = rb_const_get(class, rb_intern("NAME"));
                SafeStringValue(name);
                filter_module =
                    ruby_filter_module_create(class,
                                              ruby_wrapper_text_filter_invoke,
                                              map->pool);
                error = rast_filter_map_add_text_filter(map,
                                                        StringValuePtr(name),
                                                        filter_module);
                if (error != RAST_OK) {
                    /* todo: error handling */
                    rast_error_destroy(error);
                }
            }
        }
    }
    apr_dir_close(dir);
    return RAST_OK;
}

rast_error_t *
rast_ruby_wrapper_filter_module_initialize(rast_filter_map_t *map)
{
    apr_pool_t *pool;
    const char *load_pathes, *dirname, *p, *q;
    rast_error_t *error;

    apr_pool_create(&pool, map->pool);
    ruby_initialize(map);

    load_pathes = getenv("RAST_RUBY_FILTER_MODULEDIR");
    if (load_pathes == NULL) {
        load_pathes = RAST_RUBY_FILTER_MODULEDIR;
    }

    p = load_pathes;
    while ((q = strchr(p, ':')) != NULL) {
        dirname = apr_pstrndup(pool, p, q - p);
        error = load_ruby_filter_modules(map, dirname, pool);
        if (error != RAST_OK) {
            apr_pool_destroy(pool);
            return error;
        }
        p = q + 1;
        apr_pool_clear(pool);
    }

    error = load_ruby_filter_modules(map, p, pool);
    apr_pool_destroy(pool);
    return error;
}
