# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "fileutils"
require "bdb"

require "rast_test"
require "test-utility"
require "rast_xmlrpc_client"
require "rast/xmlrpc-server-testable"

module Rast
  class DBTest < Test::Unit::TestCase
    include XMLRPCServerTestable

    def test_create
      options = {
        "byte_order" => Rast::LITTLE_ENDIAN,
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => true,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      DB.create(db_name, options)
      DB.open(db_name, Rast::DB::RDWR) do |db|
        assert_equal(1, db.register("本日は晴天なり", {"title" => "hare"}))
      end
      DB.open(db_name, Rast::DB::RDONLY) do |db|
        result = db.search("晴天")
        assert_equal(1, result.items.length)
        assert_equal(1, result.items[0].doc_id)
      end
    end

    def test_open
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      DB.open(db_name, Rast::DB::RDWR) do |db|
        assert_nothing_raised do
          db.register("", {})
        end
      end
      DB.open(db_name, Rast::DB::RDONLY) do |db|
        assert_raise(RastError) do
          db.register("", {})
        end
      end

      DB.open("file://" + db_name, Rast::DB::RDWR) do |db|
        assert_nothing_raised do
          db.register("", {})
        end
      end
    end

    def test_register
      register_test_simple
      register_test_only_property
    end

    def register_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)

      DB.open(db_name) do |db|
        assert_equal(1, db.register("本日は晴天なり", {}))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)
      end

      kick_xmlrpc_server("--read-write", db_name) do |db|
        assert_equal(2, db.register("明日は雨っす", {}))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("雨っ")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("雨っ", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)
      end
    end

    def register_test_only_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)

      time = DateTime.new(2005, 2, 17, 19, 4, 38)
      DB.open(db_name) do |db|
        properties = {
          "title" =>"test",
          "modified_date" => time.strftime("%F"),
          "modified" => time.strftime("%FT%T"),
          "serial" => 10,
        }
        assert_equal(1, db.register("", properties))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 10",
                           {
                             "properties" => [
                               "title",
                               "modified_date",
                               "modified",
                               "serial",
                             ]
                           })
        assert_equal(1, result.items[0].doc_id)
        assert_equal("test", result.items[0].properties[0])
        assert_equal(time.strftime("%F"), result.items[0].properties[1])
        assert_equal(time.strftime("%FT%T"), result.items[0].properties[2])
        assert_equal(10, result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
      end

      time = DateTime.new(2005, 4, 14, 15, 8, 33)
      kick_xmlrpc_server("--read-write", db_name) do |db|
        properties = {
          "title" =>"test2",
          "modified_date" => time.strftime("%F"),
          "modified" => time.strftime("%FT%T"),
          "serial" => 11,
        }
        assert_equal(2, db.register("", properties))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 11",
                           {
                             "properties" => [
                               "title",
                               "modified_date",
                               "modified",
                               "serial",
                             ]
                           })
        assert_equal(2, result.items[0].doc_id)
        assert_equal("test2", result.items[0].properties[0])
        assert_equal(time.strftime("%F"), result.items[0].properties[1])
        assert_equal(time.strftime("%FT%T"), result.items[0].properties[2])
        assert_equal(11, result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
      end
    end

    def kick_xmlrpc_server(*args)
      kick_rast_xmlrpc_server_process(*args) do |port|
        DB.open("xmlrpc.http://localhost:#{port}/RPC2") do |db|
          yield(db)
        end
      end
    end
  end
end
