/* Copyright(C) 2004-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "senna_in.h"
#include "str.h"
#include "inv.h"
#include "sym.h"
#include "str.h"
#include "store.h"
#include <string.h>
#include <stdio.h>

/* rectangular arrays */

#define SEN_RA_IDSTR "SENNA:RA:01.000"
#define SEN_RA_SEGMENT_SIZE (1 << 22)

#define SEN_RA_MAX_CACHE (4294967295U)

struct sen_ra_header {
  char idstr[16];
  unsigned element_size;
  sen_id curr_max;
  uint32_t reserved[10];
};

sen_ra *
sen_ra_create(const char *path, unsigned int element_size)
{
  sen_io *io;
  int max_segments, n_elm, w_elm;
  sen_ra *ra = NULL;
  struct sen_ra_header *header;
  unsigned actual_size;
  if (element_size > SEN_RA_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "element_size too large (%d)", element_size);
    return NULL;
  }
  for (actual_size = 1; actual_size < element_size; actual_size *= 2) ;
  max_segments = ((SEN_ID_MAX + 1) / SEN_RA_SEGMENT_SIZE) * actual_size;
  io = sen_io_create(path, sizeof(struct sen_ra_header),
                     SEN_RA_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_RA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_RA_IDSTR, 16);
  header->element_size = actual_size;
  header->curr_max = 0;
  if (!(ra = SEN_MALLOC(sizeof(sen_ra)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_RA_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  ra->io = io;
  ra->header = header;
  ra->element_mask =  n_elm - 1;
  ra->element_width = w_elm;
  return ra;
}

sen_ra *
sen_ra_open(const char *path)
{
  sen_io *io;
  int n_elm, w_elm;
  sen_ra *ra = NULL;
  struct sen_ra_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_RA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_RA_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "ra_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(ra = SEN_MALLOC(sizeof(sen_ra)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_RA_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  ra->io = io;
  ra->header = header;
  ra->element_mask =  n_elm - 1;
  ra->element_width = w_elm;
  return ra;
}

sen_rc
sen_ra_info(sen_ra *ra, unsigned int *element_size, sen_id *curr_max)
{
  if (!ra) { return sen_invalid_argument; }
  if (element_size) { *element_size = ra->header->element_size; }
  if (curr_max) { *curr_max = ra->header->curr_max; }
  return sen_success;
}

sen_rc
sen_ra_close(sen_ra *ra)
{
  sen_rc rc;
  if (!ra) { return sen_invalid_argument; }
  rc = sen_io_close(ra->io);
  SEN_FREE(ra);
  return rc;
}

sen_rc
sen_ra_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  sen_io_remove(path);
  return sen_success;
}

void *
sen_ra_get(sen_ra *ra, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > SEN_ID_MAX) { return NULL; }
  seg = id >> ra->element_width;
  SEN_IO_SEG_MAP(ra->io, seg, p);
  if (!p) { return NULL; }
  if (id > ra->header->curr_max) { ra->header->curr_max = id; }
  return (void *)(((byte *)p) + ((id & ra->element_mask) * ra->header->element_size));
}

void *
sen_ra_at(sen_ra *ra, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > ra->header->curr_max) { return NULL; }
  seg = id >> ra->element_width;
  SEN_IO_SEG_MAP(ra->io, seg, p);
  if (!p) { return NULL; }
  return (void *)(((byte *)p) + ((id & ra->element_mask) * ra->header->element_size));
}

/**** jagged arrays ****/

#define SEN_JA_IDSTR "SENNA:JA:01.000"

#define W_OF_JA_MAX 38
#define W_OF_JA_SEGMENT 22
#define W_OF_JA_MAX_SEGMENTS (W_OF_JA_MAX - W_OF_JA_SEGMENT)

#define W_OF_JA_EINFO 3
#define W_OF_JA_EINFO_IN_A_SEGMENT (W_OF_JA_SEGMENT - W_OF_JA_EINFO)
#define N_OF_JA_EINFO_IN_A_SEGMENT (1U << W_OF_JA_EINFO_IN_A_SEGMENT)
#define JA_EINFO_MASK (N_OF_JA_EINFO_IN_A_SEGMENT - 1)

#define JA_SEGMENT_SIZE (1U << W_OF_JA_SEGMENT)
#define JA_MAX_SEGMENTS (1U << W_OF_JA_MAX_SEGMENTS)

#define JA_BSA_SIZE (1U << (W_OF_JA_SEGMENT - 7))
#define JA_N_BSEGMENTS (1U << (W_OF_JA_MAX_SEGMENTS - 7))

#define JA_N_ESEGMENTS (1U << 9)

#define SEN_JA_MAX_CACHE (4294967295U)

struct _sen_ja_einfo {
  union {
    uint64_t ll;
    struct {
      uint16_t seg;
      uint16_t pos;
      uint16_t size;
      uint8_t tail[2];
    } s;
  } u;
};

#define EINFO_SET(e,_seg,_pos,_size) {\
  (e)->u.s.seg = _seg;\
  (e)->u.s.pos = (_pos) >> 4;\
  (e)->u.s.size = _size;\
  (e)->u.s.tail[0] = (((_pos) >> 14) & 0xc0) + ((_size) >> 16);\
  (e)->u.s.tail[1] = 0;\
}

#define EINFO_GET(e,_seg,_pos,_size) {\
  _seg = (e)->u.s.seg;\
  _pos = ((e)->u.s.pos + (((e)->u.s.tail[0] & 0xc0) << 10)) << 4;\
  _size = (e)->u.s.size + (((e)->u.s.tail[0] & 0x3f) << 16);\
}

typedef struct {
  uint32_t seg;
  uint32_t pos;
} ja_pos;

struct sen_ja_header {
  char idstr[16];
  unsigned max_element_size;
  unsigned max_segments;
  ja_pos free_elements[24];
  uint8_t segments[JA_MAX_SEGMENTS];
  uint32_t esegs[JA_N_ESEGMENTS];
  uint32_t bsegs[JA_N_BSEGMENTS];
};


#define JA_SEG_ESEG 1;
#define JA_SEG_BSEG 2;
#define SEG_NOT_ASSIGNED 0xffffffff

sen_ja *
sen_ja_create(const char *path, unsigned int max_element_size)
{
  int i;
  sen_io *io;
  int max_segments;
  sen_ja *ja = NULL;
  struct sen_ja_header *header;
  if (max_element_size > JA_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "max_element_size too large (%d)", max_element_size);
    return NULL;
  }
  max_segments = max_element_size * 128;
  if (max_segments > JA_MAX_SEGMENTS) { max_segments = JA_MAX_SEGMENTS; }
  io = sen_io_create(path, sizeof(struct sen_ja_header),
                     JA_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_JA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_JA_IDSTR, 16);
  for (i = 0; i < JA_N_ESEGMENTS; i++) { header->esegs[i] = SEG_NOT_ASSIGNED; }
  for (i = 0; i < JA_N_BSEGMENTS; i++) { header->bsegs[i] = SEG_NOT_ASSIGNED; }
  header->max_element_size = max_element_size;
  header->max_segments = max_segments;
  header->segments[0] = JA_SEG_ESEG;
  header->esegs[0] = 0;
  if (!(ja = SEN_MALLOC(sizeof(sen_ja)))) {
    sen_io_close(io);
    return NULL;
  }
  ja->io = io;
  ja->header = header;
  return ja;
}

sen_ja *
sen_ja_open(const char *path)
{
  sen_io *io;
  sen_ja *ja = NULL;
  struct sen_ja_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_JA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_JA_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "ja_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(ja = SEN_MALLOC(sizeof(sen_ja)))) {
    sen_io_close(io);
    return NULL;
  }
  ja->io = io;
  ja->header = header;
  return ja;
}

sen_rc
sen_ja_info(sen_ja *ja, unsigned int *max_element_size)
{
  if (!ja) { return sen_invalid_argument; }
  return sen_success;
}

sen_rc
sen_ja_close(sen_ja *ja)
{
  sen_rc rc;
  if (!ja) { return sen_invalid_argument; }
  rc = sen_io_close(ja->io);
  SEN_FREE(ja);
  return rc;
}

sen_rc
sen_ja_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  sen_io_remove(path);
  return sen_success;
}

sen_rc
sen_ja_put(sen_ja *ja, sen_id id, const void *value, int value_len, int flags)
{
  int rc;
  void *buf;
  sen_ja_einfo einfo;
  sen_log("value='%s'", (char *) value);
  if ((flags & SEN_ST_APPEND)) {
    uint32_t old_len;
    const void *oldvalue = sen_ja_ref(ja, id, &old_len);
    if (oldvalue) {
      if ((rc = sen_ja_alloc(ja, value_len + old_len, &einfo, &buf))) { return rc; }
      memcpy(buf, oldvalue, old_len);
      memcpy((byte *)buf + old_len, value, value_len);
      sen_ja_unref(ja, id);
    } else {
      if ((rc = sen_ja_alloc(ja, value_len, &einfo, &buf))) { return rc; }
      memcpy(buf, value, value_len);
    }
  } else {
    if ((rc = sen_ja_alloc(ja, value_len, &einfo, &buf))) { return rc; }
    // printf("put id=%d, value_len=%d value=%p ei=%p(%d:%d)\n", id, value_len, buf, &einfo, einfo.u.s.pos, einfo.u.s.tail[0]);
    memcpy(buf, value, value_len);
  }
  return sen_ja_replace(ja, id, &einfo);
}

int
sen_ja_at(sen_ja *ja, sen_id id, void *valbuf, int buf_size)
{
  uint32_t len;
  const void *value = sen_ja_ref(ja, id, &len);
  if (!value) { return -1; }
  if (buf_size >= len) { memcpy(valbuf, value, len); }
  sen_ja_unref(ja, id);
  return (int) len;
}

const void *
sen_ja_ref(sen_ja *ja, sen_id id, uint32_t *value_len)
{
  sen_ja_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { *value_len = 0; return NULL; }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { *value_len = 0; return NULL; }
  if (einfo[pos].u.s.tail[1] & 1) {
    *value_len = einfo[pos].u.s.tail[1] >> 1;
    return (void *) &einfo[pos];
  }
  {
    void *value;
    uint32_t jag, vpos, vsize;
    EINFO_GET(&einfo[pos], jag, vpos, vsize);
    SEN_IO_SEG_MAP(ja->io, jag, value);
    // printf("at id=%d value=%p jag=%d vpos=%d ei=%p(%d:%d)\n", id, value, jag, vpos, &einfo[pos], einfo[pos].u.s.pos, einfo[pos].u.s.tail[0]);
    if (!value) { *value_len = 0; return NULL; }
    *value_len = vsize;
    return (byte *)value + vpos;
  }
}

sen_rc
sen_ja_unref(sen_ja *ja, sen_id id)
{
  // todo
  return sen_success;
}

int
sen_ja_size(sen_ja *ja, sen_id id)
{
  sen_ja_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { return -1; }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { return -1; }
  if (einfo[pos].u.s.tail[1] & 1) {
    return einfo[pos].u.s.tail[1] >> 1;
  } else {
    return einfo[pos].u.s.size + ((einfo[pos].u.s.tail[0] & 0x3f) << 16);
  }
}

sen_rc
sen_ja_alloc(sen_ja *ja, int element_size, sen_ja_einfo *einfo, void **value)
{
  int m, size;
  void *addr;
  ja_pos *vp;
  if (element_size < 8) {
    einfo->u.s.tail[1] = element_size * 2 + 1;
    *value = (void *)einfo;
    return sen_success;
  }
  if (element_size >= ja->header->max_element_size) {
    return sen_invalid_argument;
  }
  for (m = 4, size = 16; size < element_size; m++, size *= 2);
  vp = &ja->header->free_elements[m];
  if (!vp->seg) {
    int i = 0;
    while (ja->header->segments[i]) {
      if (++i >= ja->header->max_segments) { return sen_memory_exhausted; }
    }
    ja->header->segments[i] = m;
    vp->seg = i;
    vp->pos = 0;
  }
  EINFO_SET(einfo, vp->seg, vp->pos, element_size);
  SEN_IO_SEG_MAP(ja->io, vp->seg, addr);
  // printf("addr=%p seg=%d pos=%d\n", addr, vp->seg, vp->pos);
  if (!addr) { return sen_memory_exhausted; }
  *value = (byte *)addr + vp->pos;
  if ((vp->pos += size) == JA_SEGMENT_SIZE) {
    vp->seg = 0;
    vp->pos = 0;
  }
  return sen_success;
}

sen_rc
sen_ja_free(sen_ja *ja, sen_ja_einfo *einfo)
{
  uint32_t seg, pos, size;
  if (einfo->u.s.tail[1] & 1) { return sen_success; }
  EINFO_GET(einfo, seg, pos, size);
  // free
  return sen_success;
}

sen_rc
sen_ja_replace(sen_ja *ja, sen_id id, sen_ja_einfo *ei)
{
  uint32_t lseg, *pseg, pos;
  sen_ja_einfo *einfo, eback;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) {
    int i = 0;
    while (ja->header->segments[i]) {
      if (++i >= ja->header->max_segments) { return sen_memory_exhausted; }
    }
    ja->header->segments[i] = 1;
    *pseg = i;
  }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { return sen_memory_exhausted; }
  eback = einfo[pos];
  einfo[pos] = *ei;
  // todo: SEN_ATOMIC_SET64
  sen_ja_free(ja, &eback);
  return sen_success;
}

/**** db ****/

typedef struct _sen_db_relation sen_db_relation;

struct _sen_db {
  sen_sym *keys;
  sen_ja *values;
  sen_set *stores;
};

struct _sen_db_relation {
  sen_db_relation *next;
  sen_db_rel_type type;
  sen_db_store *target;
};

struct _sen_db_store {
  sen_db_type type;
  sen_db *db;
  sen_id id;
  sen_db_relation *relations;
  union {
    struct {
      unsigned int element_size;
    } bc;
    struct {
      sen_sym *keys;
    } c;
    struct {
      sen_id class;
      sen_ra *ra;
    } o;
    struct {
      sen_id class;
      sen_ra *ra;
    } f;
    struct {
      sen_id class;
      sen_ja *ja;
    } v;
    struct {
      sen_id class;
      sen_index *index;
    } i;
  } u;
};

inline static void
gen_pathname(const char *path, char *buffer, int fno)
{
  size_t len = strlen(path);
  memcpy(buffer, path, len);
  if (fno >= 0) {
    buffer[len] = '.';
    sen_str_itoh(fno, buffer + len + 1, 7);
  } else {
    buffer[len] = '\0';
  }
}

sen_db_store *
sen_db_store_by_id(sen_db *s, sen_id id)
{
  sen_db_store *slot;
  const char *name;
  if (sen_set_at(s->stores, &id, (void **) &slot)) { return slot; }
  if (!(name = _sen_sym_key(s->keys, id))) { return NULL; }
  return sen_db_store_open(s, name);
}

sen_db_store *
sen_db_slot_class(sen_db *s, const char *slot)
{
  int i = SEN_SYM_MAX_KEY_SIZE;
  char buf[SEN_SYM_MAX_KEY_SIZE], *dst = buf;
  while (*slot != '.') {
    if (!*slot || !--i) { return NULL; }
    *dst++ = *slot++;
  }
  *dst = '\0';
  return sen_db_store_open(s, buf);
}

sen_db_store *
sen_db_store_open(sen_db *s, const char *name)
{
  sen_id id;
  sen_set_eh *ep;
  uint32_t spec_len;
  sen_db_store *e;
  char buffer[PATH_MAX];
  const sen_db_store_spec *spec;
  if (!(id = sen_sym_at(s->keys, name))) { return NULL; }
  if (sen_set_at(s->stores, &id, (void **) &e)) { return e; }
  if (!(spec = sen_ja_ref(s->values, id, &spec_len))) { return NULL; }
  ep = sen_set_get(s->stores, &id, (void **) &e);
  e->type = spec->type;
  e->db = s;
  e->id = id;
  e->relations = NULL;
  gen_pathname(s->keys->io->path, buffer, id);
  switch (spec->type) {
  case sen_db_raw_class :
    e->u.bc.element_size = spec->u.c.size;
    break;
  case sen_db_structured_class :
    if (!(e->u.c.keys = sen_sym_open(buffer))) { goto exit; }
    break;
  case sen_db_obj_slot :
    e->u.o.class = spec->u.s.class;
    if (!(e->u.o.ra = sen_ra_open(buffer))) { goto exit; }
    break;
  case sen_db_ra_slot :
    e->u.f.class = spec->u.s.class;
    if (!(e->u.f.ra = sen_ra_open(buffer))) { goto exit; }
    break;
  case sen_db_ja_slot :
    e->u.v.class = spec->u.s.class;
    if (!(e->u.v.ja = sen_ja_open(buffer))) { goto exit; }
    break;
  case sen_db_idx_slot :
    e->u.i.class = spec->u.s.class;
    {
      sen_db_store *l, *k;
      if (!(k = sen_db_store_by_id(s, spec->u.s.class))) { goto exit; }
      if (!(l = sen_db_slot_class(s, name))) { goto exit; }
      if (!(e->u.i.index =
            sen_index_open_with_keys_lexicon(buffer, k->u.c.keys, l->u.c.keys))) {
        goto exit;
      }
    }
    break;
  default :
    goto exit;
  }
  {
    int i;
    for (i = 0; i < spec->n_relations; i++) {
      sen_db_store *target = sen_db_store_by_id(s, spec->relations[i].target);
      if (target) {
        sen_db_relation *r = SEN_MALLOC(sizeof(sen_db_relation));
        if (!r) { goto exit; }
        r->next = e->relations;
        r->type = spec->relations[i].type;
        r->target = target;
        e->relations = r;
      }
    }
  }
  sen_ja_unref(s->values, id);
  return e;
exit :
  sen_set_del(s->stores, ep);
  sen_ja_unref(s->values, id);
  return NULL;
}

sen_db_store *
sen_db_store_create(sen_db *s, const char *name, sen_db_store_spec *spec)
{
  sen_id id;
  sen_set_eh *ep;
  sen_db_store *e;
  char buffer[PATH_MAX];
  if (strlen(name) >= SEN_SYM_MAX_KEY_SIZE) {
    SEN_LOG(sen_log_error, "too long store name (%s)", name);
    return NULL;
  }
  if (strchr(name, '.') &&
      ((spec->type == sen_db_raw_class) ||
       (spec->type == sen_db_structured_class))) {
    SEN_LOG(sen_log_error, "class name must not include '.' (%s)", name);
    return NULL;
  }
  if (!(id = sen_sym_get(s->keys, name))) { return NULL; }
  if (!(ep = sen_set_get(s->stores, &id, (void **) &e))) { goto exit; }
  spec->n_relations = 0;
  if (sen_ja_put(s->values, id, spec, SEN_DB_STORE_SPEC_SIZE(0), 0)) { goto exit; }
  e->type = spec->type;
  e->db = s;
  e->id = id;
  e->relations = NULL;
  gen_pathname(s->keys->io->path, buffer, id);
  switch (spec->type) {
  case sen_db_raw_class :
    e->u.bc.element_size = spec->u.c.size;
    break;
  case sen_db_structured_class :
    if (!(e->u.c.keys = sen_sym_create(buffer,
                                       spec->u.c.size,
                                       spec->u.c.flags,
                                       spec->u.c.encoding))) { goto exit; }
    break;
  case sen_db_obj_slot :
    e->u.o.class = spec->u.s.class;
    // todo : handle collection_type
    if (!(e->u.o.ra = sen_ra_create(buffer, sizeof(sen_id)))) { goto exit; }
    break;
  case sen_db_ra_slot :
    e->u.f.class = spec->u.s.class;
    if (!(e->u.f.ra = sen_ra_create(buffer, spec->u.s.size))) { goto exit; }
    break;
  case sen_db_ja_slot :
    e->u.v.class = spec->u.s.class;
    if (!(e->u.v.ja = sen_ja_create(buffer, spec->u.s.size))) { goto exit; }
    break;
  case sen_db_idx_slot :
    e->u.i.class = spec->u.s.class;
    {
      sen_db_store *l, *k;
      if (!(k = sen_db_store_by_id(s, spec->u.s.class))) { goto exit; }
      if (!(l = sen_db_slot_class(s, name))) { goto exit; }
      if (!(e->u.i.index =
            sen_index_create_with_keys_lexicon(buffer, k->u.c.keys, l->u.c.keys,
                                               spec->u.s.size))) {
        goto exit;
      }
    }
    break;
  default :
    goto exit;
  }
  return e;
exit :
  if (ep) { sen_set_del(s->stores, ep); }
  sen_sym_del(s->keys, name);
  // todo : sen_ja_put(s->values, id, NULL, 0, 0);
  return NULL;
}

sen_rc
sen_db_store_add_relation(sen_db_store *e, sen_db_store_rel_spec *t)
{
  sen_rc rc;
  sen_db *s = e->db;
  uint32_t spec_len, newspec_len;
  const sen_db_store_spec *spec;
  sen_db_store_spec *newspec;
  if (!(spec = sen_ja_ref(s->values, e->id, &spec_len))) {
    return sen_invalid_argument;
  }
  newspec_len = SEN_DB_STORE_SPEC_SIZE(spec->n_relations + 1);
  if (!(newspec = SEN_MALLOC(newspec_len))) { return sen_memory_exhausted; }
  memcpy(newspec, spec, spec_len);
  memcpy(&newspec->relations[spec->n_relations], t, sizeof(sen_db_store_rel_spec));
  newspec->n_relations++;
  sen_ja_unref(s->values, e->id);
  if ((rc = sen_ja_put(s->values, e->id, newspec, newspec_len, 0))) {
    goto exit;
  }
  {
    sen_db_store *target = sen_db_store_by_id(s, t->target);
    if (target) {
      sen_db_relation *r = SEN_MALLOC(sizeof(sen_db_relation));
      if (!r) { rc = sen_memory_exhausted; goto exit; }
      r->next = e->relations;
      r->type = t->type;
      r->target = target;
      e->relations = r;
      if (t->type == sen_db_index_target) {
        sen_db_store_rel_spec invrs;
        invrs.type = sen_db_before_update_trigger;
        invrs.target = e->id;
        rc = sen_db_store_add_relation(target, &invrs);
      }
    }
  }
exit :
  SEN_FREE(newspec);
  return rc;
}

sen_db_store *
sen_db_slot_class_by_id(sen_db *s, sen_id slot)
{
  return sen_db_slot_class(s, _sen_sym_key(s->keys, slot));
}

sen_db_store *
sen_db_class_slot(sen_db *s, sen_id class, const char *name)
{
  char buf[SEN_SYM_MAX_KEY_SIZE], *dst;
  const char *src = _sen_sym_key(s->keys, class);
  if (!src) { return NULL; }
  strcpy(buf, src);
  dst = buf + strlen(src);
  *dst++ = '.';
  strcpy(dst, name);
  return sen_db_store_open(s, buf);
}

sen_rc
sen_db_store_close(sen_db_store *slot, int all)
{
  sen_db *s = slot->db;
  sen_db_relation *t, *t_;
  switch (slot->type) {
  case sen_db_obj_slot :
    // sen_db_class_close(slot->u.o.class);
    sen_ra_close(slot->u.o.ra);
    break;
  case sen_db_ra_slot :
    sen_ra_close(slot->u.f.ra);
    break;
  case sen_db_ja_slot :
    sen_ja_close(slot->u.v.ja);
    break;
  case sen_db_idx_slot :
    sen_index_close(slot->u.i.index);
    break;
  case sen_db_structured_class :
    sen_sym_close(slot->u.c.keys);
    break;
  default :
    return sen_invalid_argument;
  }
  for (t = slot->relations; t; t = t_) {
    t_ = t->next;
    SEN_FREE(t);
  }
  if (!all) {
    sen_set_eh *ep;
    if ((ep = sen_set_at(s->stores, &slot->id, NULL))) {
      sen_set_del(s->stores, ep);
    }
  }
  return sen_success;
}

sen_rc
sen_db_prepare_builtin_class(sen_db *db)
{
  sen_db_store_spec spec;
  spec.type = sen_db_raw_class;
  spec.u.c.size = sizeof(int);
  if (!sen_db_store_create(db, "int", &spec)) { return sen_memory_exhausted; }
  spec.u.c.size = 1 << 16;
  if (!sen_db_store_create(db, "text", &spec)) { return sen_memory_exhausted; }
  spec.u.c.size = JA_SEGMENT_SIZE;
  if (!sen_db_store_create(db, "longtext", &spec)) { return sen_memory_exhausted; }
  return sen_success;
}

sen_db *
sen_db_create(const char *path, int flags, sen_encoding encoding)
{
  sen_db *s;
  char buffer[PATH_MAX];
  if (strlen(path) > PATH_MAX - 14) { return NULL; }
  if (!(s = SEN_MALLOC(sizeof(sen_db)))) { return NULL; }
  if ((s->stores = sen_set_open(sizeof(sen_id), sizeof(sen_db_store), 0))) {
    if ((s->keys = sen_sym_create(path, 0, flags, encoding))) {
      gen_pathname(path, buffer, 0);
      if ((s->values = sen_ja_create(buffer, JA_SEGMENT_SIZE))) {
        sen_db_prepare_builtin_class(s);
        SEN_LOG(sen_log_notice, "db created (%s) flags=%x", path, s->keys->flags);
        return s;
      }
      sen_sym_close(s->keys);
    }
    sen_set_close(s->stores);
  }
  SEN_FREE(s);
  return NULL;
}

sen_db *
sen_db_open(const char *path)
{
  sen_db *s;
  char buffer[PATH_MAX];
  if (strlen(path) > PATH_MAX - 14) { return NULL; }
  if (!(s = SEN_MALLOC(sizeof(sen_db)))) { return NULL; }
  if ((s->stores = sen_set_open(sizeof(sen_id), sizeof(sen_db_store), 0))) {
    if ((s->keys = sen_sym_open(path))) {
      gen_pathname(path, buffer, 0);
      if ((s->values = sen_ja_open(buffer))) {
        SEN_LOG(sen_log_notice, "db opened (%s) flags=%x", path, s->keys->flags);
        return s;
      }
      sen_sym_close(s->keys);
    }
    sen_set_close(s->stores);
  }
  SEN_FREE(s);
  return NULL;
}

sen_rc
sen_db_close(sen_db *s)
{
  sen_db_store *e;
  sen_set_cursor *c;
  sen_sym_close(s->keys);
  sen_ja_close(s->values);
  if ((c = sen_set_cursor_open(s->stores))) {
    while (sen_set_cursor_next(c, NULL, (void **) &e)) {
      sen_db_store_close(e, 1);
    }
    sen_set_cursor_close(c);
  }
  sen_set_close(s->stores);
  SEN_FREE(s);
  return sen_success;
}

/* sen_db_obj */

sen_rc
sen_db_class_at(sen_db_store *c, const void *key, int flags, sen_db_obj *res)
{
  sen_id id = flags ? sen_sym_get(c->u.c.keys, key) : sen_sym_at(c->u.c.keys, key);
  if (id) {
    res->u.o.self = id;
    res->type = sen_db_object;
    res->class = c->id;
    return sen_success;
  } else {
    SETNIL(res);
    return sen_other_error;
  }
}

inline static sen_id
slot_class_at(sen_db *s, sen_id slot, const char *key, const sen_db_obj *args)
{
  sen_db_store *c;
  if (!(c = sen_db_slot_class_by_id(s, slot))) { return 0; }
  return args ? sen_sym_get(c->u.c.keys, key) : sen_sym_at(c->u.c.keys, key);
}

inline static sen_rc
slot_value_obj(sen_db_store *slot, sen_id id, const sen_db_obj *args, sen_db_obj *res)
{
  sen_rc rc = sen_success;
  sen_id *ip = args ? sen_ra_get(slot->u.o.ra, id) : sen_ra_at(slot->u.o.ra, id);
  if (!ip) {
    SETNIL(res);
    goto exit;
  }
  if (args) {
    switch (args->type) {
    case sen_db_object :
      if (args->class != slot->u.o.class) {
        rc = sen_invalid_argument;
        SETNIL(res);
        goto exit;
      }
      *ip = args->u.o.self;
      break;
    case sen_db_bulk :
      {
        sen_db_store *c;
        if (!(c = sen_db_store_by_id(slot->db, slot->u.o.class)) ||
            !(*ip = sen_sym_get(c->u.c.keys, args->u.b.value))) {
          rc = sen_invalid_argument;
          SETNIL(res);
          goto exit;
        }
      }
      break;
    default :
      rc = sen_invalid_argument;
      SETNIL(res);
      goto exit;
      break;
    }

  }
  // todo : trigger
  res->type = (*ip) ? sen_db_object : sen_db_nil;
  res->class = slot->u.o.class;
  res->u.o.self = *ip;
exit :
  return rc;
}

inline static sen_rc
slot_value_ra(sen_db_store *slot, sen_id id, const sen_db_obj *args, sen_db_obj *res)
{
  sen_rc rc = sen_success;
  void *vp = args ? sen_ra_get(slot->u.f.ra, id) : sen_ra_at(slot->u.f.ra, id);
  if (!vp) {
    SETNIL(res);
    rc = sen_invalid_argument;
    goto exit;
  }
  if (args) {
    switch (args->type) {
    case sen_db_bulk :
      if (sizeof(int32_t) == slot->u.f.ra->header->element_size) {
        int32_t i = sen_atoi(args->u.b.value,
                             (char *)args->u.b.value + args->u.b.size, NULL);
        memcpy(vp, &i, sizeof(int32_t));
      } else {
        if (args->u.b.size != slot->u.f.ra->header->element_size) {
          SETNIL(res);
          rc = sen_invalid_argument;
          goto exit;
        }
        memcpy(vp, args->u.b.value, args->u.b.size);
      }
      break;
    case sen_db_int :
      if (sizeof(int32_t) != slot->u.f.ra->header->element_size) {
        SETNIL(res);
        rc = sen_invalid_argument;
        goto exit;
      }
      memcpy(vp, &args->u.i.i, sizeof(int32_t));
      break;
    default :
      SETNIL(res);
      rc = sen_invalid_argument;
      goto exit;
    }
  // todo : trigger
  }
  if (slot->u.f.ra->header->element_size == sizeof(int32_t)) {
    res->type = sen_db_int;
    memcpy(&res->u.i.i, vp, sizeof(int32_t));
  } else {
    res->type = sen_db_bulk;
    res->u.b.size = slot->u.f.ra->header->element_size;
    res->u.b.value = vp;
  }
exit :
  return rc;
}

inline static sen_rc
slot_value_ja(sen_db_store *slot, sen_id id, const sen_db_obj *args, sen_db_obj *res)
{
  sen_rc rc = sen_success;
  uint32_t value_len;
  void *vp = (void *)sen_ja_ref(slot->u.v.ja, id, &value_len);
  // todo : unref
  if (args) {
    sen_db_relation *t;
    // todo : support append and so on..
    if (args->type != sen_db_bulk) {
      rc = sen_invalid_argument;
      goto exit;
    }
    for (t = slot->relations; t; t = t->next) {
      if (t->type == sen_db_before_update_trigger) {
        sen_db_store *index = t->target;
        sen_log("updating %d => %d", value_len, args->u.b.size);
        sen_index_upd(index->u.i.index, _sen_sym_key(index->u.i.index->keys, id),
                      vp, value_len, args->u.b.value, args->u.b.size);
      }
    }
    SETNIL(res);
    rc = sen_ja_put(slot->u.v.ja, id, args->u.b.value, args->u.b.size, 0);
  } else {
    if (vp) {
      res->type = sen_db_bulk;
      res->u.b.size = value_len;
      res->u.b.value = vp;
    } else {
      SETNIL(res);
    }
  }
exit :
  return rc;
}

#define SEN_DB_OBJ_ALLOCED 1
#define SEN_DB_OBJ_CONST 2

inline static sen_db_obj *
obj_new(sen_db_ctx *c)
{
  sen_db_obj *o;
  sen_set_get(c->objects, &c->seqno, (void **) &o);
  c->seqno++;
  o->flags = 0;
  return o;
}

inline static sen_db_obj *
obj_alloc(sen_db_ctx *c, uint32_t size)
{
  sen_db_obj *o;
  void *value = SEN_MALLOC(size);
  if (!value) { return NULL; }
  o = obj_new(c);
  o->flags = SEN_DB_OBJ_ALLOCED;
  o->type = sen_db_bulk;
  o->u.b.size = size;
  o->u.b.value = value;
  return o;
}

inline static sen_db_obj *
obj_clear(sen_db_obj *o)
{
  if (o->flags & SEN_DB_OBJ_ALLOCED) {
    switch (o->type) {
    case sen_db_records :
      if (o->u.r.records) { sen_records_close(o->u.r.records); }
      break;
    case sen_db_bulk :
      if (o->u.b.value) { SEN_FREE(o->u.b.value); }
      break;
    case sen_db_query :
      if (o->u.q.query) { sen_query_close(o->u.q.query); }
      break;
    default :
      break;
    }
  }
  o->flags = 0;
  return o;
}

#define obj_res(c,r,s) ((r) ? (*(r) ? obj_clear(*(r)) : (*(r) = obj_new(c))) : &(s))

inline static sen_db_obj *
rec_obj_new(sen_db_ctx *c, sen_db_store *cls, sen_rec_unit record_unit,
            sen_rec_unit subrec_unit, unsigned int max_n_subrecs)
{
  sen_records *r;
  sen_db_obj *res;
  if (!(r = sen_records_open(record_unit, subrec_unit, max_n_subrecs))) { return NULL; }
  r->keys = cls->u.c.keys;
  res = obj_new(c);
  res->type = sen_db_records;
  res->flags = SEN_DB_OBJ_ALLOCED;
  res->class = cls->id;
  res->u.r.records = r;
  return res;
}

inline static sen_db_obj *
query_obj_new(sen_db_ctx *c, const char *str, unsigned int str_len,
              sen_sel_operator default_op, int max_exprs, sen_encoding encoding)
{
  sen_query *q;
  sen_db_obj *res;
  if (!(q = sen_query_open(str, str_len, default_op, max_exprs, encoding))) {
    return NULL;
  }
  res = obj_new(c);
  res->type = sen_db_query;
  res->flags = SEN_DB_OBJ_ALLOCED;
  res->u.q.query = q;
  return res;
}

sen_rc
sen_db_at(sen_db_ctx *c, sen_db_obj *obj,
             const char *key, const sen_db_obj *args, sen_db_obj **res)
{
  sen_rc rc = sen_success;
  sen_db *s = c->db;
  sen_db_obj rs;
  if (obj) {
    sen_id id;
    sen_db_store *slot;
    switch (obj->type) {
    case sen_db_structured_class :
      if (!(slot = sen_db_store_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
      } else {
        rc = sen_db_class_at(slot, key, args ? 1 : 0, obj_res(c, res, rs));
      }
      break;
    case sen_db_obj_slot :
      if (!(id = slot_class_at(s, obj->u.o.self, key, args))) {
        rc = sen_other_error;
        break;
      }
      if (!(slot = sen_db_store_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      rc = slot_value_obj(slot, id, args, obj_res(c, res, rs));
      break;
    case sen_db_ra_slot :
      if (!(id = slot_class_at(s, obj->u.o.self, key, args))) {
        rc = sen_other_error;
        break;
      }
      if (!(slot = sen_db_store_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      rc = slot_value_ra(slot, id, args, obj_res(c, res, rs));
      break;
    case sen_db_ja_slot :
      if (!(id = slot_class_at(s, obj->u.o.self, key, args))) {
        rc = sen_other_error;
        break;
      }
      if (!(slot = sen_db_store_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      rc = slot_value_ja(slot, id, args, obj_res(c, res, rs));
      break;
    case sen_db_idx_slot :
      // todo : simply pickup a term
      if (!(slot = sen_db_store_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      if (res) {
        sen_db_obj *rp = obj_res(c, res, rs);
        if ((rp->u.r.records = sen_index_sel(slot->u.i.index, key, strlen(key)))) {
          rp->type = sen_db_records;
          rp->flags = SEN_DB_OBJ_ALLOCED;
        } else {
          SETNIL(rp);
          rp->flags = 0;
        }
        rp->class = slot->u.i.class;
      }
      break;
    case sen_db_records :
      // todo : break;
    default :
      rc = sen_invalid_argument;
      break;
    }
  } else {
    sen_db_store *slot;
    if (BULKP(args)) {
      // todo : slot = sen_db_store_create(s, key, args);
    } else {
      slot = sen_db_store_open(s, key);
    }
    if (slot) {
      if (res) {
        sen_db_obj *rp = obj_res(c, res, rs);
        rp->type = slot->type;
        rp->class = 0;
        rp->u.o.self = slot->id;
      }
      rc = sen_success;
    } else {
      if (res) {
        sen_db_obj *rp = obj_res(c, res, rs);
        SETNIL(rp);
      }
      rc = sen_other_error;
    }
  }
  return rc;
}

sen_rc
sen_db_send(sen_db_ctx *c, sen_db_obj *obj,
               const char *message, const sen_db_obj *args, sen_db_obj **res)
{
  sen_rc rc = sen_success;
  sen_db *s = c->db;
  sen_db_obj rs, *rp = obj_res(c, res, rs);
  if (obj) {
    sen_db_store *slot;
    switch (obj->type) {
    case sen_db_structured_class :
      slot = sen_db_class_slot(s, obj->u.o.self, message);
      if (slot) {
        rp->type = slot->type;
        rp->u.o.self = slot->id;
      } else {
        SETNIL(rp);
        rc = sen_invalid_argument;
      }
      break;
    case sen_db_object :
      slot = sen_db_class_slot(s, obj->class, message);
      switch (slot->type) {
      case sen_db_obj_slot :
        rc = slot_value_obj(slot, obj->u.o.self, args, rp);
        break;
      case sen_db_ra_slot :
        rc = slot_value_ra(slot, obj->u.o.self, args, rp);
        break;
      case sen_db_ja_slot :
        rc = slot_value_ja(slot, obj->u.o.self, args, rp);
        break;
      case sen_db_idx_slot :
        {
          const char *key = _sen_sym_key(slot->u.i.index->lexicon, obj->u.o.self);
          if (!key) {
            SETNIL(rp);
            rc = sen_invalid_argument;
            break;
          }
          if ((rp->u.r.records = sen_index_sel(slot->u.i.index, key, strlen(key)))) {
            rp->type = sen_db_records;
            rp->flags = SEN_DB_OBJ_ALLOCED;
          } else {
            SETNIL(rp);
            rp->flags = 0;
          }
          rp->class = slot->u.i.class;
        }
        break;
      default :
        SETNIL(rp);
        rc = sen_invalid_argument;
        break;
      }
      break;
    default :
      SETNIL(rp);
      rc = sen_invalid_argument;
      break;
    }
  } else {
    SETNIL(rp);
    rc = sen_invalid_argument;
  }
  return rc;
}

sen_db_obj *sen_db_ctx_at(sen_db_ctx *c, const char *key, sen_db_obj *value);

/* methods */

/* TODO: First, eval all and get nrecords of each. Apply function in it's order */
static sen_db_obj *
_native_method_and(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *base = sen_db_ctx_eval(c, CAR(args));
  if (RECORDSP(base)) {
    sen_db_obj *rest;
    for (rest = CDR(args); rest; rest = CDR(rest)) {
      sen_db_obj *obj = sen_db_ctx_eval(c, CAR(rest));
      if (RECORDSP(obj)) {
        sen_records_intersect(base->u.r.records, obj->u.r.records);
        SETNIL(obj);
      }
    }
    return base;
  } else {
    return NULL;
  }
}

static sen_db_obj *
_native_method_or(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *base = sen_db_ctx_eval(c, CAR(args));
  if (RECORDSP(base)) {
    sen_db_obj *rest;
    for (rest = CDR(args); rest; rest = CDR(rest)) {
      sen_db_obj *obj = sen_db_ctx_eval(c, CAR(rest));
      if (RECORDSP(obj)) {
        sen_records_union(base->u.r.records, obj->u.r.records);
        SETNIL(obj);
      }
    }
    return base;
  } else {
    return NULL;
  }
}

static sen_db_obj *
_native_method_but(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *base = sen_db_ctx_eval(c, CAR(args));
  if (RECORDSP(base)) {
    sen_db_obj *rest;
    for (rest = CDR(args); rest; rest = CDR(rest)) {
      sen_db_obj *obj = sen_db_ctx_eval(c, CAR(rest));
      if (RECORDSP(obj)) {
        sen_records_subtract(base->u.r.records, obj->u.r.records);
        SETNIL(obj);
      }
    }
    return base;
  } else {
    return NULL;
  }
}

static sen_db_obj *
_native_method_set(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *res = NULL;
  sen_db_obj *key = CAR(args);
  sen_db_obj *val = CADR(args);
  if (BULKP(key)) {
    if (val) { val = sen_db_ctx_eval(c, val); }
    res = sen_db_ctx_at(c, key->u.b.value, val);
  }
  return res;
}

static sen_db_obj *
_native_method_defclass(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *res = NULL;
  sen_db_obj *car = CAR(args);
  sen_db_store *cls;
  if (BULKP(car)) {
    sen_db_obj *cdr;
    sen_db_store_spec spec;
    spec.type = sen_db_structured_class;
    spec.u.c.size = 0;
    spec.u.c.flags = SEN_INDEX_NORMALIZE;
    spec.u.c.encoding = sen_enc_default;
    for (cdr = CDR(args); LISTP(cdr); cdr = CDR(cdr)) {
      if (INTP(CAR(cdr))) {
        spec.u.c.size = CAR(cdr)->u.i.i;
      }
      if (BULKP(CAR(cdr))) {
        switch (CAR(cdr)->u.b.value[0]) {
        case 'd' :
        case 'D' :
          spec.u.c.flags |= SEN_INDEX_DELIMITED;
          break;
        case 'e' :
        case 'E' :
          spec.u.c.encoding = sen_enc_euc_jp;
          break;
        case 'k' :
        case 'K' :
          spec.u.c.encoding = sen_enc_koi8r;
          break;
        case 'l' :
        case 'L' :
          spec.u.c.encoding = sen_enc_latin1;
          break;
        case 'n' :
        case 'N' :
          spec.u.c.flags |= SEN_INDEX_NGRAM;
          break;
        case 'r' :
        case 'R' :
          spec.type = sen_db_raw_class;
          break;
        case 's' :
        case 'S' :
          spec.u.c.encoding = sen_enc_sjis;
          break;
        case 'u' :
        case 'U' :
          spec.u.c.encoding = sen_enc_utf8;
          break;
        }
      }
    }
    if ((cls = sen_db_store_create(c->db, car->u.b.value, &spec))) {
      res = obj_new(c);
      res->type = cls->type;
      res->class = 0;
      res->u.o.self = cls->id;
    }
  }
  return res;
}

static sen_db_obj *
_native_method_defslot(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *res = NULL;
  sen_db_obj *car = CAR(args);
  sen_db_obj *cdr = CDR(args);
  if (BULKP(car) && BULKP(CAR(cdr))) {
    sen_db_store *slot;
    sen_db_store_spec spec;
    sen_db_obj *targets = NULL;
    sen_db_store *cls = sen_db_store_open(c->db, CAR(cdr)->u.b.value);
    if (!cls) { goto exit; }
    spec.u.s.class = cls->id;
    spec.u.s.size = 0;
    spec.u.s.collection_type = 0;
    switch (cls->type) {
    case sen_db_raw_class :
      spec.type = (cls->u.bc.element_size > 8) ? sen_db_ja_slot : sen_db_ra_slot;
      spec.u.s.size = cls->u.bc.element_size;
      break;
    case sen_db_structured_class :
      spec.type = sen_db_obj_slot;
      break;
    case sen_db_obj_slot :
    case sen_db_ra_slot :
    case sen_db_ja_slot :
      spec.type = sen_db_idx_slot;
      break;
    default :
      goto exit;
    }
    for (cdr = CDR(cdr); LISTP(cdr); cdr = CDR(cdr)) {
      if (INTP(CAR(cdr))) {
        spec.u.s.size = CAR(cdr)->u.i.i;
      }
      if (BULKP(CAR(cdr))) {
        char ctype = CAR(cdr)->u.b.value[0];
        switch (ctype) {
        case 'a' :
          spec.u.s.collection_type = 1; // todo : array
          break;
        case 's' :
          spec.u.s.collection_type = 2; // todo : set
          break;
        case 'i' :
          spec.type = sen_db_idx_slot;
          spec.u.s.collection_type = 3;
        }
      }
      if (LISTP(CAR(cdr))) { targets = CAR(cdr); }
    }
    if ((slot = sen_db_store_create(c->db, car->u.b.value, &spec))) {
      res = obj_new(c);
      res->type = slot->type;
      res->class = 0;
      res->u.o.self = slot->id;
      if (spec.type == sen_db_idx_slot) {
        sen_db_store *ts;
        sen_db_store_rel_spec rs;
        rs.type = sen_db_index_target;
        while (targets) {
          if (BULKP(CAR(targets))) {
            if ((ts = sen_db_class_slot(c->db, cls->id, CAR(targets)->u.b.value))) {
              rs.target = ts->id;
              sen_db_store_add_relation(slot, &rs);
            }
          }
          targets = CDR(targets);
        }
      }
    }
  }
exit :
  return res;
}

static sen_db_obj *
_native_method_at(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *res = NULL;
  sen_db_obj *arg = CDR(args);
  if (LISTP(arg)) {
    sen_db_obj *msg = CAR(arg);
    if (BULKP(msg)) {
      sen_db_at(c, sen_db_ctx_eval(c, CAR(args)),
                   msg->u.b.value, CADR(arg), &res);
    }
  }
  return res;
}

static sen_db_obj *
_native_method_send(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_obj *res = NULL;
  sen_db_obj *arg = CDR(args);
  if (LISTP(arg)) {
    sen_db_obj *msg = CAR(arg);
    if (BULKP(msg)) {
      sen_db_send(c, sen_db_ctx_eval(c, CAR(args)),
                     msg->u.b.value, CDR(arg), &res);
    }
  }
  return res;
}

const char *
_sen_db_obj_key(sen_db *db, sen_db_obj *obj)
{
  sen_db_store *cls;
  switch (obj->type) {
  case sen_db_object :
    if (!(cls = sen_db_store_by_id(db, obj->class))) { return NULL; }
    return _sen_sym_key(cls->u.c.keys, obj->u.o.self);
  case sen_db_raw_class :
  case sen_db_structured_class :
  case sen_db_obj_slot :
  case sen_db_ra_slot :
  case sen_db_ja_slot :
  case sen_db_idx_slot :
    return _sen_sym_key(db->keys, obj->u.o.self);
  default :
    return NULL;
  }
}

static sen_db_obj *
_native_method__put(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  if (BULKP(args)) {
    fwrite(args->u.b.value, 1, args->u.b.size, stdout);
    putchar('\n');
  }
  return NULL;
}

// from index.c
typedef struct {
  int score;
  int n_subrecs;
  byte subrecs[1];
} recinfo;

inline static sen_db_method_func *
name2method(sen_db_ctx *c, const char *name)
{
  sen_db_obj *obj = sen_db_ctx_at(c, name, NULL);
  if (!NATIVE_METHODP(obj)) { return NULL; }
  return obj->u.n.func;
}

inline static sen_db_obj *
rbuf2obj(sen_rbuf *buf, sen_db_obj *obj)
{
  obj->type = sen_db_bulk;
  obj->u.b.value = buf->head;
  obj->u.b.size = SEN_RBUF_VSIZE(buf);
  return obj;
}

static sen_db_obj *
_native_method_put(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_id *rp;
  recinfo *ri;
  sen_rbuf buf;
  sen_records *r;
  const sen_recordh *rh;
  int i, ofrat = 0, limit = 10;
  sen_db_obj obj, bobj, *rec, *s, *slots, *res = NULL;
  sen_db_method_func *_put;
  if (!(_put = name2method(c, "_put"))) { return NULL; }
  if (sen_rbuf_init(&buf, 1)) { return NULL; }
  rec = sen_db_ctx_eval(c, CAR(args));
  if (!rec || rec->type != sen_db_records) { goto exit; }
  r = rec->u.r.records;
  sen_rbuf_itoa(&buf, sen_records_nhits(r));
  _put(c, rbuf2obj(&buf, &bobj), NULL);
  args = CDR(args);
  if (!args || args->type != sen_db_list) { goto exit; }
  slots = CAR(args);
  if (!slots || slots->type != sen_db_list) { goto exit; }
  // check && compile slots
  args = CDR(args);
  if (INTP(CAR(args))) {
    ofrat = CAR(args)->u.i.i;
  }
  args = CDR(args);
  if (INTP(CAR(args))) {
    limit = CAR(args)->u.i.i;
  }
  sen_records_rewind(r);
  for (i = 0; i < ofrat; i++) {
    if (!sen_records_next(r, NULL, 0, NULL)) { goto exit; }
  }
  obj.type = sen_db_object;
  obj.class = rec->class;
  for (i = 0; i < limit; i++) {
    if (!sen_records_next(r, NULL, 0, NULL) ||
        !(rh = sen_records_curr_rec(r)) ||
        sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) { goto exit; }
    obj.u.o.self = *rp;
    SEN_RBUF_REWIND(&buf);
    for (s = slots;;) {
      if (BULKP(CAR(s))) {
        char *value = (char *)CAR(s)->u.b.value;
        if (value[0] == ':') {
          switch (value[1]) {
          case 'k' :
            {
              const char *key = _sen_db_obj_key(c->db, &obj);
              if (key) { SEN_RBUF_PUTS(&buf, key); }
            }
            break;
          case 's' :
            sen_rbuf_itoa(&buf, ri->score);
            break;
          default :
            break;
          }
        } else {
          sen_db_send(c, &obj, value, NULL, &res);
          switch(res->type) {
          case sen_db_object :
            {
              const char *key = _sen_db_obj_key(c->db, res);
              if (key) { SEN_RBUF_PUTS(&buf, key); }
            }
            break;
          case sen_db_bulk :
            sen_rbuf_write(&buf, res->u.b.value, res->u.b.size);
            break;
          case sen_db_int :
            sen_rbuf_itoa(&buf, res->u.i.i);
            break;
          default :
            break;
          }
        }
      }
      s = CDR(s);
      if (!s || s->type != sen_db_list) { break; }
      SEN_RBUF_PUTC(&buf, '\t');
    }
    _put(c, rbuf2obj(&buf, &bobj), NULL);
  }
exit :
  sen_rbuf_fin(&buf);
  return NULL;
}

static sen_db_obj *
_native_method_quote(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  return CAR(args);
}

#define MAXSLOTS 0x100
#define BUFSIZE 0x100000

struct _ins_stat {
  sen_db_obj *slots;
  int nslots;
  int nrecs;
};

static sen_db_obj *
_native_method_ins(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  int i, n;
  sen_db_obj *res = NULL;
  sen_db_obj *s, *car;
  struct _ins_stat *stat;
  SEN_DB_CONT_BEGIN(cont);
  // sen_db_store *class;
  if (!args || args->type != sen_db_list) { goto exit; }
  car = sen_db_ctx_eval(c, CAR(args));
  if (!car || car->type != sen_db_structured_class) { goto exit; }
  // if (!(class = sen_db_store_by_id(c->db, car->u.o.self))) { goto exit; }
  args = CDR(args);
  for (s = args, n = 0; LISTP(s); s = CDR(s), n++) {
    if (CAR(s)->type != sen_db_bulk) { goto exit; }
    sen_db_send(c, car, CAR(s)->u.b.value, NULL, &s->u.l.car);
    if (NILP(CAR(s))) { goto exit; }
  }
  if (!(s = obj_alloc(c, sizeof(struct _ins_stat)))) { goto exit; }
  stat = (struct _ins_stat *)s->u.b.value;
  stat->slots = args;
  stat->nslots = n + 1;
  stat->nrecs = 0;
  do {
    SEN_DB_CONT_WAIT(cont, stat);
    if (BULKP(args) && args->u.b.size) {
      sen_db_obj val;
      char *tokbuf[MAXSLOTS];
      val.type = sen_db_bulk;
      if (sen_str_tok(args->u.b.value, args->u.b.size, '\t', tokbuf, MAXSLOTS, NULL) == stat->nslots) {
        *tokbuf[0] = '\0';
        // if (sen_db_class_at(class, tokbuf[0], 1, &res)) { continue; }
        for (s = stat->slots, i = 1; i < stat->nslots; s = CDR(s), i++) {
          val.u.b.value = tokbuf[i - 1] + 1;
          val.u.b.size = tokbuf[i] - val.u.b.value;
          sen_db_at(c, CAR(s), args->u.b.value, &val, NULL);
        }
        stat->nrecs++;
      }
    } else {
      cont->mode |= SEN_DB_CTX_TAIL;
    }
  } while (!(cont->mode & SEN_DB_CTX_TAIL));
  res = obj_new(c);
  res->type = sen_db_int;
  res->u.i.i = stat->nrecs;
  SEN_DB_CONT_END(cont);
exit :
  return res;
}

static int
compar_ra(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  void *va, *vb;
  sen_id *pa, *pb;
  sen_ra *raa = (sen_ra *)ra->userdata, *rab = (sen_ra *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = sen_ra_at(raa, *pa);
  vb = sen_ra_at(rab, *pb);
  if (va) {
    // must be compared as int?
    return vb ? memcmp(va, vb, raa->header->element_size) : 1;
  } else {
    return vb ? -1 : 0;
  }
}

static int
compar_ja(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  int r;
  const void *va, *vb;
  uint32_t la, lb;
  sen_id *pa, *pb;
  sen_ja *jaa = (sen_ja *)ra->userdata, *jab = (sen_ja *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = sen_ja_ref(jaa, *pa, &la);
  vb = sen_ja_ref(jab, *pb, &lb);
  if (va) {
    if (vb) {
      if (la > lb) {
        if ((r = memcmp(va, vb, lb))) {
          return r;
        } else {
          return 1;
        }
      } else {
        if ((r = memcmp(va, vb, la))) {
          return r;
        } else {
          return la == lb ? 0 : -1;
        }
      }
    } else {
      return 1;
    }
  } else {
    return vb ? -1 : 0;
  }
}

static int
compar_key(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  const char *va, *vb;
  sen_id *pa, *pb;
  sen_sym *ka = ra->userdata, *kb = rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = _sen_sym_key(ka, *pa);
  vb = _sen_sym_key(kb, *pb);
  // todo : if (key_size)..
  if (va) {
    return vb ? strcmp(va, vb) : 1;
  } else {
    return vb ? -1 : 0;
  }
}

static int
compar_obj(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  const char *va, *vb;
  sen_id *pa, *pb, *oa, *ob;
  sen_sym *key = (sen_sym *)arg;
  // todo : target class may not be identical
  sen_ra *raa = (sen_ra *)ra->userdata, *rab = (sen_ra *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = (oa = sen_ra_at(raa, *pa)) ? _sen_sym_key(key, *oa) : NULL;
  vb = (ob = sen_ra_at(rab, *pb)) ? _sen_sym_key(key, *ob) : NULL;
  // todo : if (key_size)..
  if (va) {
    return vb ? strcmp(va, vb) : 1;
  } else {
    return vb ? -1 : 0;
  }
}

static sen_db_obj *
_native_method_sort(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  int limit = 10;
  sen_sort_optarg arg;
  sen_db_obj *res = NULL, *rec = sen_db_ctx_eval(c, CAR(args));
  if (rec->type != sen_db_records) { goto exit; }
  args = CDR(args);
  arg.compar = NULL;
  arg.compar_arg = (void *)(intptr_t)rec->u.r.records->record_size;
  arg.mode = sen_sort_descending;
  if (BULKP(CAR(args))) {
    const char *str = CAR(args)->u.b.value;
    if (CAR(args)->u.b.size > 1 && *str == ':') {
      switch (str[1]) {
      case 's' : /* :score */
        break;
      case 'k' : /* :key */
        {
          sen_db_store *cls = sen_db_store_by_id(c->db, rec->class);
          if (!cls) { goto exit; }
          rec->u.r.records->userdata = cls->u.c.keys;
          arg.compar = compar_key;
        }
        break;
      default :
        goto exit;
      }
    } else {
      sen_db_store *slot = sen_db_class_slot(c->db, rec->class, str);
      if (!slot) { goto exit; }
      switch (slot->type) {
      case sen_db_ra_slot :
        rec->u.r.records->userdata = slot->u.f.ra;
        arg.compar = compar_ra;
        break;
      case sen_db_ja_slot :
        rec->u.r.records->userdata = slot->u.v.ja;
        arg.compar = compar_ja;
        break;
      case sen_db_obj_slot :
        {
          sen_db_store *cls = sen_db_store_by_id(c->db, slot->u.o.class);
          if (!cls) { goto exit; }
          rec->u.r.records->userdata = slot->u.o.ra;
          arg.compar = compar_obj;
          arg.compar_arg = cls->u.c.keys;
        }
        break;
      default :
        goto exit;
      }
    }
  }
  args = CDR(args);
  if (INTP(CAR(args))) {
    limit = CAR(args)->u.i.i;
  }
  args = CDR(args);
  if (BULKP(CAR(args))) {
    const char *str = CAR(args)->u.b.value;
    if (*str == 'a') {
      arg.mode = sen_sort_ascending;
    }
  }
  if (!sen_records_sort(rec->u.r.records, limit, &arg)) { res = rec; }
exit :
  return res;
}

static sen_db_obj *
_native_method_sel(sen_db_ctx *c, sen_db_obj *args, sen_db_cont *cont)
{
  sen_db_store *cls, *slot;
  sen_id id = SEN_SYM_NIL;
  sen_db_obj o, *res = NULL, *cl = sen_db_ctx_eval(c, CAR(args));
  if (cl->type != sen_db_structured_class ||
      !(cls = sen_db_store_by_id(c->db, cl->u.o.self))) { goto exit; }
  args = CADR(args);
  if (!args || args->type != sen_db_list) { goto exit; }
  if (CAR(args)->type != sen_db_bulk) { goto exit; } // must be "eq"
  args = CDR(args);
  if (!args || CAR(args)->type != sen_db_bulk) { goto exit; } // must be slot name
  if (!(slot = sen_db_class_slot(c->db, cls->id, CAR(args)->u.b.value)) ||
      slot->type != sen_db_ra_slot) { // todo : support other types
    goto exit;
  }
  args = CADR(args);
  if (!args || args->type != sen_db_int) { goto exit; } // todo : support other types
  if (!(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) { goto exit; }
  // while ((id = sen_sym_next(cls->u.c.keys, id)))
  // todo : delete check
  for (id = 1; id <= slot->u.f.ra->header->curr_max; id++) {
    if (slot_value_ra(slot, id, NULL, &o)) { break; }
    if (o.u.i.i == args->u.i.i) {
      sen_set_get(res->u.r.records->records, &id, NULL);
    }
  }
exit :
  return res;
}

void
sen_db_ctx_def_native_method(sen_db_ctx *c, const char *name,
                                sen_db_method_func *func)
{
  sen_db_obj *o = obj_new(c);
  o->type = sen_db_native_method;
  o->u.n.func = func;
  sen_db_ctx_at(c, name, o);
}

sen_db_ctx *
sen_db_ctx_open(sen_db *s, sen_db_parser *parser)
{
  sen_db_ctx *c = SEN_MALLOC(sizeof(sen_db_ctx));
  if (c) {
    c->parser = parser;
    c->doing = NULL;
    c->seqno = 0;
    c->db = s;
    if (!(c->objects = sen_set_open(sizeof(int), sizeof(sen_db_obj), 0))) {
      SEN_FREE(c);
      c = NULL;
      goto exit;
    }
    if (!(c->bindings = sen_set_open(0, sizeof(sen_db_obj *), 0))) {
      sen_set_close(c->objects);
      SEN_FREE(c);
      c = NULL;
      goto exit;
    }
    sen_db_ctx_def_native_method(c, "set", _native_method_set);
    sen_db_ctx_def_native_method(c, "defclass", _native_method_defclass);
    sen_db_ctx_def_native_method(c, "defslot", _native_method_defslot);
    sen_db_ctx_def_native_method(c, "at", _native_method_at);
    sen_db_ctx_def_native_method(c, "send", _native_method_send);
    sen_db_ctx_def_native_method(c, "put", _native_method_put);
    sen_db_ctx_def_native_method(c, "quote", _native_method_quote);
    sen_db_ctx_def_native_method(c, "ins", _native_method_ins);
    sen_db_ctx_def_native_method(c, "sort", _native_method_sort);
    sen_db_ctx_def_native_method(c, "sel", _native_method_sel);
    sen_db_ctx_def_native_method(c, "_put", _native_method__put);
    sen_db_ctx_def_native_method(c, "and", _native_method_and);
    sen_db_ctx_def_native_method(c, "or", _native_method_or);
    sen_db_ctx_def_native_method(c, "but", _native_method_but);
  }
exit :
  return c;
}

sen_rc
sen_db_ctx_close(sen_db_ctx *c)
{
  sen_db_obj *o;
  sen_set_cursor *sc;
  if ((sc = sen_set_cursor_open(c->objects))) {
    while (sen_set_cursor_next(sc, NULL, (void **) &o)) { obj_clear(o); }
    sen_set_cursor_close(sc);
  }
  sen_set_close(c->objects);
  sen_set_close(c->bindings);
  SEN_FREE(c);
  return sen_success;
}

sen_db_obj *
sen_db_ctx_at(sen_db_ctx *c, const char *key, sen_db_obj *value)
{
  sen_db_obj **res;
  // todo : delete
  if (value) {
    sen_set_get(c->bindings, key, (void **) &res);
    *res = value;
  } else {
    if (!sen_set_at(c->bindings, key, (void **) &res)) {
      return NULL;
    }
  }
  return *res;
}

static sen_db_obj *
sen_db_ctx_eval_(sen_db_ctx *c, sen_db_obj *expr, sen_db_cont *cont)
{
  sen_db_obj *res = expr;
  if (!expr) { return NULL; }
  switch (expr->type) {
  case sen_db_nil :
    break;
  case sen_db_raw_class :
    break;
  case sen_db_structured_class :
    break;
  case sen_db_obj_slot :
    break;
  case sen_db_ra_slot :
    break;
  case sen_db_ja_slot :
    break;
  case sen_db_idx_slot :
    break;
  case sen_db_object :
    break;
  case sen_db_records :
    break;
  case sen_db_bulk :
    res = sen_db_ctx_at(c, expr->u.b.value, NULL);
    if (!res) {
      sen_db_at(c, NULL, expr->u.b.value, NULL, &res);
    }
    break;
  case sen_db_list :
    {
      sen_db_obj *car = sen_db_ctx_eval_(c, CAR(expr), NULL);
      sen_db_obj *cdr = CDR(expr);
      if (NATIVE_METHODP(car)) {
        sen_db_method_func *doing = car->u.n.func;
        if (cont) { c->doing = doing; }
        res = doing(c, cdr, cont);
      }
    }
    break;
  case sen_db_native_method :
    break;
  case sen_db_method :
    break;
  case sen_db_int :
    break;
  }
  return res;
}

sen_db_obj *
sen_db_ctx_eval(sen_db_ctx *c, sen_db_obj *expr)
{
  return sen_db_ctx_eval_(c, expr, NULL);
}

sen_db_obj *
sen_db_ctx_feed(sen_db_ctx *c, char *str, uint32_t str_size, int mode)
{
  sen_db_obj *r;
  if ((mode & SEN_DB_CTX_HEAD) && c->doing) {
    c->cont.mode = SEN_DB_CTX_TAIL;
    r = c->doing(c, NULL, &c->cont);
    c->doing = NULL;
  }
  if (c->doing) {
    sen_db_obj arg;
    c->cont.mode = mode & ~SEN_DB_CTX_HEAD;
    arg.type = sen_db_bulk;
    arg.u.b.value = str;
    arg.u.b.size = str_size;
    r = c->doing(c, &arg, &c->cont);
  } else {
    sen_db_obj *expr;
    c->cont.last = 0;
    c->cont.mode = mode | SEN_DB_CTX_HEAD;
    expr = c->parser(c, str, str_size);
    // todo : substitute placeholder
    r = sen_db_ctx_eval_(c, expr, &c->cont);
  }
  if (!c->cont.last || (c->cont.mode & SEN_DB_CTX_TAIL)) { c->doing = NULL; }
  return r;
}

/**** sexp parser ****/

typedef struct {
  char *cur;
  char *str_end;
  char *last;
  sen_encoding encoding;
} parseinfo;

typedef sen_db_obj cons;

inline static cons *
cons_new(sen_db_ctx *c)
{
  cons *o = obj_new(c);
  o->type = sen_db_list;
  o->u.l.car = NULL;
  o->u.l.cdr = NULL;
  return o;
}

inline static cons *
token_new(sen_db_ctx *c)
{
  cons *o = obj_new(c);
  o->type = sen_db_bulk;
  return o;
}

static cons *get_expr(sen_db_ctx *q, parseinfo *i);

inline static void
skip_space(sen_db_ctx *c, parseinfo *i)
{
  unsigned int len;
  while (i->cur < i->str_end && sen_isspace(i->cur, i->encoding)) {
    /* null check and length check */
    if (!(len = sen_str_charlen_nonnull(i->cur, i->str_end, i->encoding))) {
      i->cur = i->str_end;
      break;
    }
    i->cur += len;
  }
}

inline static cons *
get_phrase(sen_db_ctx *c, parseinfo *i)
{
  cons *token;
  char *start = i->cur, *end;
  for (end = i->cur;; end++) {
    if (end >= i->str_end) {
      i->cur = end;
      break;
    }
    if (*end == SEN_QUERY_QUOTER) {
      i->cur = end + 1;
      break;
    }
  }
  if (start < end && (token = token_new(c))) {
    token->u.b.value = start;
    token->u.b.size = end - start;
    i->last = end;
    return token;
  }
  return NULL;
}

inline static cons *
get_word(sen_db_ctx *c, parseinfo *i)
{
  cons *token;
  char *start = i->cur, *end;
  unsigned int len;
  for (end = i->cur;; ) {
    /* null check and length check */
    if (!(len = sen_str_charlen_nonnull(end, i->str_end, i->encoding))) {
      i->cur = i->str_end;
      break;
    }
    if (sen_isspace(end, i->encoding) ||
        *end == SEN_QUERY_PARENR) {
      i->cur = end;
      break;
    }
    end += len;
  }
  if (start < end && (token = token_new(c))) {
    token->u.b.value = start;
    token->u.b.size = end - start;
    i->last = end;
    return token;
  }
  return NULL;
}

inline static cons *
get_int(sen_db_ctx *c, parseinfo *i)
{
  cons *token = obj_new(c);
  token->type = sen_db_int;
  token->u.i.i = sen_atoi(i->cur, i->str_end, (const char **) &i->cur);
  return token;
}

inline static cons *
get_token(sen_db_ctx *c, parseinfo *i)
{
  cons *token = NULL;
  char cur;
  skip_space(c, i);
  if (i->cur >= i->str_end) { return NULL; }
  cur = *i->cur;
  if (i->last) { *i->last = '\0'; }
  switch (cur) {
  case '\0' :
    return NULL;
  case SEN_QUERY_PARENR :
    i->cur++;
    return NULL;
  case SEN_QUERY_QUOTEL :
    i->cur++;
    token = cons_new(c);
    token->u.l.car = get_phrase(c, i);
    break;
  case SEN_QUERY_PARENL :
    i->cur++;
    token = cons_new(c);
    token->u.l.car = get_expr(c, i);
    break;
  case '0' :
  case '1' :
  case '2' :
  case '3' :
  case '4' :
  case '5' :
  case '6' :
  case '7' :
  case '8' :
  case '9' :
    token = cons_new(c);
    token->u.l.car = get_int(c, i);
    break;
  case '-' :
    token = cons_new(c);
    if ('0' <= *(i->cur + 1) && *(i->cur + 1) <= '9') {
      token->u.l.car = get_int(c, i);
    } else {
      token->u.l.car = get_word(c, i);
    }
    break;
  default :
    token = cons_new(c);
    token->u.l.car = get_word(c, i);
    break;
  }
  return token;
}

static cons *
get_expr(sen_db_ctx *c, parseinfo *i)
{
  cons *head, *expr, *last;
  if (!(head = get_token(c, i))) {
    return NULL;
  }
  for (last = head; (expr = get_token(c, i)); last = expr) {
    last->u.l.cdr = expr;
  }
  return head;
}

sen_db_obj *
sen_db_sexp_parser(sen_db_ctx *c, const char *str, uint32_t str_size)
{
  char *buf;
  parseinfo info;
  sen_db_obj *o = obj_alloc(c, str_size + 1);
  if (!o) { return NULL; }
  buf = o->u.b.value;
  memcpy(buf, str, str_size);
  buf[str_size] = '\0';
  info.cur = buf;
  info.str_end = info.cur + str_size;
  info.last = NULL;
  info.encoding = c->db->keys->encoding;
  o = get_expr(c, &info);
  if (info.last) { *info.last = '\0'; }
  return o;
}

void
sen_db_obj_inspect(sen_db *db, sen_db_obj *obj, sen_rbuf *buf)
{
  if (obj) {
    switch (obj->type) {
    case sen_db_nil :
      SEN_RBUF_PUTS(buf, "nil object");
      obj = NULL;
      break;
    case sen_db_raw_class :
      SEN_RBUF_PUTS(buf, "Raw Class (");
      SEN_RBUF_PUTS(buf, _sen_db_obj_key(db, obj));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_structured_class :
      SEN_RBUF_PUTS(buf, "Structured Class (");
      SEN_RBUF_PUTS(buf, _sen_db_obj_key(db, obj));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_obj_slot :
      SEN_RBUF_PUTS(buf, "Object Slot (");
      SEN_RBUF_PUTS(buf, _sen_db_obj_key(db, obj));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_ra_slot :
      SEN_RBUF_PUTS(buf, "RA Slot (");
      SEN_RBUF_PUTS(buf, _sen_db_obj_key(db, obj));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_ja_slot :
      SEN_RBUF_PUTS(buf, "JA Slot (");
      SEN_RBUF_PUTS(buf, _sen_db_obj_key(db, obj));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_idx_slot :
      SEN_RBUF_PUTS(buf, "Index Slot (");
      SEN_RBUF_PUTS(buf, _sen_db_obj_key(db, obj));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_object :
      SEN_RBUF_PUTS(buf, "Object (");
      SEN_RBUF_PUTS(buf, _sen_db_obj_key(db, obj));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_records :
      SEN_RBUF_PUTS(buf, "Records (");
      sen_rbuf_itoa(buf, sen_records_nhits(obj->u.r.records));
      SEN_RBUF_PUTS(buf, ")");
      break;
    case sen_db_bulk :
      sen_rbuf_write(buf, obj->u.b.value, obj->u.b.size);
      break;
    case sen_db_list :
      SEN_RBUF_PUTC(buf, '(');
      for (;;) {
        sen_db_obj_inspect(db, obj->u.l.car, buf);
        if ((obj = obj->u.l.cdr)) {
          if (LISTP(obj)) {
            SEN_RBUF_PUTC(buf, ' ');
          } else {
            SEN_RBUF_PUTS(buf, " . ");
            sen_db_obj_inspect(db, obj->u.l.cdr, buf);
          }
        } else {
          SEN_RBUF_PUTC(buf, ')');
          break;
        }
      }
      break;
    case sen_db_native_method :
      SEN_RBUF_PUTS(buf, "Native method");
      break;
    case sen_db_method :
      SEN_RBUF_PUTS(buf, "Method");
      break;
    case sen_db_int :
      sen_rbuf_itoa(buf, obj->u.i.i);
      break;
    }
  } else {
    SEN_RBUF_PUTS(buf, "nil");
  }
}

/**** vgram ****/

static int len_sum = 0;
static int img_sum = 0;
static int simple_sum = 0;
static int skip_sum = 0;

sen_vgram *
sen_vgram_create(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_create(path, sizeof(sen_id) * 2, 0, sen_enc_none);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram *
sen_vgram_open(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_open(path);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram_buf *
sen_vgram_buf_open(size_t len)
{
  sen_vgram_buf *b;
  if (!(b = SEN_MALLOC(sizeof(sen_vgram_buf)))) { return NULL; }
  b->len = len;
  b->tvs = b->tvp = SEN_MALLOC(sizeof(sen_id) * len);
  if (!b->tvp) { SEN_FREE(b); return NULL; }
  b->tve = b->tvs + len;
  b->vps = b->vpp = SEN_MALLOC(sizeof(sen_vgram_vnode) * len * 2);
  if (!b->vpp) { SEN_FREE(b->tvp); SEN_FREE(b); return NULL; }
  b->vpe = b->vps + len;
  return b;
}

sen_rc
sen_vgram_buf_add(sen_vgram_buf *b, sen_id tid)
{
  uint8_t dummybuf[8], *dummyp;
  if (b->tvp < b->tve) { *b->tvp++ = tid; }
  dummyp = dummybuf;
  SEN_B_ENC(tid, dummyp);
  simple_sum += dummyp - dummybuf;
  return sen_success;
}

typedef struct {
  sen_id vid;
  sen_id tid;
} vgram_key;

sen_rc
sen_vgram_update(sen_vgram *vgram, sen_id rid, sen_vgram_buf *b, sen_set *terms)
{
  sen_inv_updspec **u;
  if (b && b->tvs < b->tvp) {
    sen_id *t0, *tn;
    for (t0 = b->tvs; t0 < b->tvp - 1; t0++) {
      sen_vgram_vnode *v, **vp;
      sen_set_at(terms, t0, (void **) &u);
      vp = &(*u)->vnodes;
      for (tn = t0 + 1; tn < b->tvp; tn++) {
        for (v = *vp; v && v->tid != *tn; v = v->cdr) ;
        if (!v) {
          if (b->vpp < b->vpe) {
            v = b->vpp++;
          } else {
            // todo;
            break;
          }
          v->car = NULL;
          v->cdr = *vp;
          *vp = v;
          v->tid = *tn;
          v->vid = 0;
          v->freq = 0;
          v->len = tn - t0;
        }
        v->freq++;
        if (v->vid) {
          vp = &v->car;
        } else {
          break;
        }
      }
    }
    {
      sen_set *th = sen_set_open(sizeof(sen_id), sizeof(int), 0);
      if (!th) { return sen_memory_exhausted; }
      if (t0 == b->tvp) { SEN_LOG(sen_log_debug, "t0 == tvp"); }
      for (t0 = b->tvs; t0 < b->tvp; t0++) {
        sen_id vid, vid0 = *t0, vid1 = 0;
        sen_vgram_vnode *v, *v2 = NULL, **vp;
        sen_set_at(terms, t0, (void **) &u);
        vp = &(*u)->vnodes;
        for (tn = t0 + 1; tn < b->tvp; tn++) {
          for (v = *vp; v; v = v->cdr) {
            if (!v->vid && (v->freq < 2 || v->freq * v->len < 4)) {
              *vp = v->cdr;
              v->freq = 0;
            }
            if (v->tid == *tn) { break; }
            vp = &v->cdr;
          }
          if (v) {
            if (v->freq) {
              v2 = v;
              vid1 = vid0;
              vid0 = v->vid;
            }
            if (v->vid) {
              vp = &v->car;
              continue;
            }
          }
          break;
        }
        if (v2) {
          if (!v2->vid) {
            vgram_key key;
            key.vid = vid1;
            key.tid = v2->tid;
            v2->vid = sen_sym_get(vgram->vgram, (char *)&key);
          }
          vid = *t0 = v2->vid * 2 + 1;
          memset(t0 + 1, 0, sizeof(sen_id) * v2->len);
          t0 += v2->len;
        } else {
          vid = *t0 *= 2;
        }
        {
          int *tf;
          sen_set_get(th, &vid, (void **) &tf);
          (*tf)++;
        }
      }
      if (!th->n_entries) { SEN_LOG(sen_log_debug, "th->n_entries == 0"); }
      {
        int j = 0;
        int skip = 0;
        sen_set_eh *ehs, *ehp, *ehe;
        sen_set_sort_optarg arg;
        uint8_t *ps = SEN_MALLOC(b->len * 2), *pp, *pe;
        if (!ps) {
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        pp = ps;
        pe = ps + b->len * 2;
        arg.mode = sen_sort_descending;
        arg.compar = NULL;
        arg.compar_arg = (void *)(intptr_t)sizeof(sen_id);
        arg.compar_arg0 = NULL;
        ehs = sen_set_sort(th, 0, &arg);
        if (!ehs) {
          SEN_FREE(ps);
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        SEN_B_ENC(th->n_entries, pp);
        for (ehp = ehs, ehe = ehs + th->n_entries; ehp < ehe; ehp++, j++) {
          int *id = (int *)SEN_SET_INTVAL(*ehp);
          SEN_B_ENC(*SEN_SET_INTKEY(*ehp), pp);
          *id = j;
        }
        for (t0 = b->tvs; t0 < b->tvp; t0++) {
          if (*t0) {
            int *id;
            if (!sen_set_at(th, t0, (void **) &id)) {
              SEN_LOG(sen_log_error, "lookup error (%d)", *t0);
            }
            SEN_B_ENC(*id, pp);
          } else {
            skip++;
          }
        }
        len_sum += b->len;
        img_sum += pp - ps;
        skip_sum += skip;
        SEN_FREE(ehs);
        SEN_FREE(ps);
      }
      sen_set_close(th);
    }
  }
  return sen_success;
}

sen_rc
sen_vgram_buf_close(sen_vgram_buf *b)
{
  if (!b) { return sen_invalid_argument; }
  if (b->tvs) { SEN_FREE(b->tvs); }
  if (b->vps) { SEN_FREE(b->vps); }
  SEN_FREE(b);
  return sen_success;
}

sen_rc
sen_vgram_close(sen_vgram *vgram)
{
  if (!vgram) { return sen_invalid_argument; }
  SEN_LOG(sen_log_debug, "len=%d img=%d skip=%d simple=%d", len_sum, img_sum, skip_sum, simple_sum);
  sen_sym_close(vgram->vgram);
  SEN_FREE(vgram);
  return sen_success;
}
