/* Sylph-Searcher - full-text search program for Sylpheed
 * Copyright (C) 2007 Sylpheed Development Team
 */

#include <sylph/sylmain.h>
#include <sylph/prefs_common.h>
#include <sylph/account.h>
#include <sylph/folder.h>
#include <sylph/procmsg.h>
#include <sylph/procheader.h>
#include <sylph/codeconv.h>
#include <sylph/utils.h>

#include <time.h>
#include <libpq-fe.h>

#include "common.h"

#define PG_CONN(dbinfo) ((PGconn *)dbinfo->dbdata)

static DBInfo *conn;

static gint verbose = 0;
static gint stat_ins_msginfo = 0;
static gint stat_ins_folderinfo = 0;
static gint stat_skip_msginfo = 0;
static gint stat_skip_folderinfo = 0;
static gint stat_error = 0;

gchar *get_text_content(MsgInfo *msginfo)
{
	FILE *fp;
	gchar *text;

	fp = procmime_get_first_text_content(msginfo, NULL);
	if (!fp) {
		g_warning("failed to get text content in msg %u",
			  msginfo->msgnum);
		return g_strdup("");
	}

	text = file_read_stream_to_str(fp);
	fclose(fp);

	if (!text) {
		g_warning("failed to get text content from fp (msg %u)",
			  msginfo->msgnum);
		return g_strdup("");
	}

	return text;
}

gint db_insert_msg(MsgInfo *msginfo)
{
	PGresult *res;
	gchar *query;
	gchar mtime_str[256], date_str[256];
	gchar *msgid;
	struct tm *lt;
	gchar *folder_id, *esc_folder_id;
	gulong sid = 0, fsid = 0;
	gint ret = 0;

	g_return_val_if_fail(msginfo != NULL, -1);

	if (!msginfo->msgid || *msginfo->msgid == '\0') {
		g_warning("message %u doesn't have Message-Id. skipping.",
			  msginfo->msgnum);
		++stat_error;
		return -1;
	}

	lt = localtime(&msginfo->mtime);
	strftime(mtime_str, sizeof(mtime_str), "%Y-%m-%d %H:%M:%S", lt);
	lt = localtime(&msginfo->date_t);
	strftime(date_str, sizeof(date_str), "%Y-%m-%d %H:%M:%S", lt);

	if (msginfo->folder)
		folder_id = folder_item_get_identifier(msginfo->folder);
	else {
		gchar *utf8path;
		utf8path = conv_filename_to_utf8(msginfo->file_path);
		folder_id = g_path_get_dirname(utf8path);
		g_free(utf8path);
	}
	if (!folder_id) {
		g_warning("can't get folder id for message %u", msginfo->msgnum);
		++stat_error;
		return -1;
	}
	esc_folder_id = sql_escape_str(conn, folder_id);

	if (db_start_transaction(conn) < 0) {
		exit(1);
	}

	msgid = sql_escape_str(conn, msginfo->msgid);
	if (db_get_sid_from_msgid(conn, msgid, &sid) < 0) {
		ret = -1;
		++stat_error;
		goto finish;
	}

	if (sid > 0) {
		if (verbose)
			g_print("message '%s' (%s/%u) already exists in msginfo. skipping.\n",
				msginfo->msgid, folder_id, msginfo->msgnum);
		++stat_skip_msginfo;
	} else {
		MsgInfo *fullinfo;
		gchar *from, *to, *cc, *newsgroups, *subject, *inreplyto;
		gchar *body_text, *esc_body_text;
		gchar *body_index_text, *esc_body_index_text;

		if (msginfo->folder)
			fullinfo = procmsg_msginfo_get_full_info(msginfo);
		else
			fullinfo = msginfo;
		if (!fullinfo) {
			ret = -1;
			++stat_error;
			goto finish;
		}
		from = sql_escape_str(conn, fullinfo->from);
		to = sql_escape_str(conn, fullinfo->to);
		cc = sql_escape_str(conn, fullinfo->cc);
		newsgroups = sql_escape_str(conn, fullinfo->newsgroups);
		subject = sql_escape_str(conn, fullinfo->subject);
		inreplyto = sql_escape_str(conn, fullinfo->inreplyto);
		body_text = get_text_content(fullinfo);
		esc_body_text = sql_escape_str(conn, body_text);
		body_index_text = get_wakachi_text(body_text);
		esc_body_index_text = sql_escape_str(conn, body_index_text);

		g_print("inserting %s/%u (%s)\n",
			folder_id, msginfo->msgnum, subject);

		query = g_strdup_printf
			("INSERT INTO msginfo(file_size, file_mtime, msg_date, flags, hdr_from, hdr_to, hdr_cc, hdr_newsgroups, hdr_subject, hdr_msgid, hdr_inreplyto, hdr_references, body_text, body_index)"
		 	" VALUES(%u, E'%s', E'%s', %u, E'%s', E'%s', E'%s', E'%s', E'%s', E'%s', E'%s', E'%s', E'%s', to_tsvector(E'%s'))",
		 	msginfo->size, mtime_str, date_str, msginfo->flags.perm_flags, from, to, cc, newsgroups, subject, msgid, inreplyto, "", esc_body_text, esc_body_index_text);

		res = PQexec(PG_CONN(conn), query);
		g_free(query);
		g_free(esc_body_index_text);
		g_free(body_index_text);
		g_free(esc_body_text);
		g_free(body_text);
		g_free(inreplyto);
		g_free(subject);
		g_free(newsgroups);
		g_free(cc);
		g_free(to);
		g_free(from);
		if (msginfo->folder)
			procmsg_msginfo_free(fullinfo);

		if (PQresultStatus(res) != PGRES_COMMAND_OK) {
			g_warning("message: %u %s %s %s", msginfo->msgnum, msginfo->subject, msginfo->from, msginfo->date);
			db_error_message(conn, "INSERT INTO msginfo failed");
			PQclear(res);
			ret = -1;
			++stat_error;
			goto finish;
		}

		PQclear(res);
		++stat_ins_msginfo;
	}

	if (db_get_sid_from_folderinfo(conn, esc_folder_id, msginfo->msgnum,
				       &fsid) < 0) {
		ret = -1;
		++stat_error;
		goto finish;
	}

	if (fsid > 0) {
		if (fsid == sid) {
			if (verbose)
				g_print("message %s/%u already exists in msg_folderinfo. skipping.\n",
					folder_id, msginfo->msgnum);
			++stat_skip_folderinfo;
			goto finish;
		}

		g_print("message %s/%u has been changed. updating.\n",
			folder_id, msginfo->msgnum);
		if (sid == 0) {
			query = g_strdup_printf("UPDATE msg_folderinfo SET msg_sid = currval('msginfo_msg_sid_seq') WHERE folder_id = E'%s' AND msgnum = %u", esc_folder_id, msginfo->msgnum);
		} else {
			query = g_strdup_printf("UPDATE msg_folderinfo SET msg_sid = %u WHERE folder_id = E'%s' AND msgnum = %u", sid, esc_folder_id, msginfo->msgnum);
		}
	} else {
		if (sid == 0) {
			query = g_strdup_printf("INSERT INTO msg_folderinfo(msg_sid, folder_id, msgnum) VALUES(currval('msginfo_msg_sid_seq'), E'%s', %u)",
						esc_folder_id, msginfo->msgnum);
		} else {
			query = g_strdup_printf("INSERT INTO msg_folderinfo(msg_sid, folder_id, msgnum) VALUES(%u, E'%s', %u)",
						sid, esc_folder_id, msginfo->msgnum);
		}
	}

	res = PQexec(PG_CONN(conn), query);
	g_free(query);
	if (PQresultStatus(res) != PGRES_COMMAND_OK) {
		g_warning("message: %u %s %s %s", msginfo->msgnum, msginfo->subject, msginfo->from, msginfo->date);
		db_error_message(conn, "INSERT INTO msg_folderinfo failed");
		PQclear(res);
		ret = -1;
		++stat_error;
		goto finish;
	}

	PQclear(res);
	++stat_ins_folderinfo;

finish:
	if (db_end_transaction(conn) < 0) {
		ret = -1;
		++stat_error;
	}

	g_free(msgid);
	g_free(esc_folder_id);
	g_free(folder_id);

	return ret;
}

gint db_cleanup_removed_msgs(const gchar *folder_id, GSList *mlist,
			     gboolean delete_msginfo)
{
	GHashTable *table;
	gchar *esc_folder_id;
	gchar *query;
	gint i;
	PGresult *res;
	gint ret = 0;
	gint stat_del_msginfo = 0;
	gint stat_del_folderinfo = 0;

	g_return_val_if_fail(folder_id != NULL, -1);

	g_print("\ncleanup removed messages in %s ...\n", folder_id);

	table = procmsg_msg_hash_table_create(mlist);

	esc_folder_id = sql_escape_str(conn, folder_id);

	query = g_strdup_printf("SELECT msg_sid, msgnum FROM msg_folderinfo WHERE folder_id = E'%s'", esc_folder_id);
	res = PQexec(PG_CONN(conn), query);
	g_free(query);

	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		db_error_message(conn, "SELECT from msg_folderinfo failed");
		PQclear(res);
		g_free(esc_folder_id);
		if (table)
			g_hash_table_destroy(table);
		return -1;
	}

	for (i = 0; i < PQntuples(res); i++) {
		gchar *val;
		gulong sid;
		guint msgnum;

		val = PQgetvalue(res, i, 0);
		sid = atol(val);
		val = PQgetvalue(res, i, 1);
		msgnum = atoi(val);

		if (!table ||
		    !g_hash_table_lookup(table, GUINT_TO_POINTER(msgnum))) {
			g_print("message %s/%u does not exist. removing.\n",
				folder_id, msgnum);
			if (db_delete_msg_from_folderinfo(conn, esc_folder_id,
							  msgnum) < 0) {
				ret = -1;
				break;
			}
			++stat_del_folderinfo;
			if (delete_msginfo &&
			    db_is_sid_exist_in_folderinfo(conn, sid) == 0) {
				if (db_delete_msg(conn, sid) < 0) {
					ret = -1;
					break;
				}
				++stat_del_msginfo;
			}
		}
	}

	PQclear(res);
	g_free(esc_folder_id);
	if (table)
		g_hash_table_destroy(table);

	g_print("%d deleted from msginfo, and %d deleted from msg_folderinfo\n",
		stat_del_msginfo, stat_del_folderinfo);

	return ret;
}

gint db_import_mh_folder(const gchar *path, gboolean del_nonexist,
			 gboolean recursive)
{
	gchar *path_, *p;
	const gchar *filename;
	GDir *dir;
	MsgInfo *msginfo;
	GSList *mlist = NULL, *cur;
	GSList *dirlist = NULL;
	gchar *folder_id;
	gint total = 0;
	gint ret = 0;

	g_return_val_if_fail(path != NULL, -1);

	if (*path == '\0')
		return -1;
	path_ = g_strdup(path);
	p = path_ + strlen(path_) - 1;
	while (p >= path_ && *p == G_DIR_SEPARATOR) {
		*p = '\0';
		--p;
	}
	if (*path_ == '\0') {
		g_free(path_);
		return -1;
	}

	folder_id = conv_filename_to_utf8(path_);

	g_print("\nimporting %s ...\n", folder_id);

	if ((dir = g_dir_open(path_, 0, NULL)) == NULL) {
		g_warning("failed to open directory: %s", folder_id);
		g_free(folder_id);
		g_free(path_);
		return -1;
	}

	while ((filename = g_dir_read_name(dir)) != NULL) {
		gchar *file;
		gint msgnum;

		file = g_strconcat(path_, G_DIR_SEPARATOR_S, filename, NULL);

		if (is_dir_exist(file)) {
			dirlist = g_slist_prepend(dirlist, file);
			continue;
		}

		msgnum = to_number(filename);
		if (msgnum > 0) {
			MsgFlags flags = {0, 0};

			msginfo = procheader_parse_file(file, flags, TRUE);
			if (msginfo) {
				msginfo->msgnum = msgnum;
				msginfo->file_path = file;
				mlist = g_slist_prepend(mlist, msginfo);
				++total;
				continue;
			}
		}

		g_free(file);
	}

	g_dir_close(dir);

	mlist = g_slist_reverse(mlist);
	mlist = procmsg_sort_msg_list(mlist, SORT_BY_NUMBER, SORT_ASCENDING);
	dirlist = g_slist_reverse(dirlist);

	stat_ins_msginfo = stat_ins_folderinfo =
		stat_skip_msginfo = stat_skip_folderinfo = stat_error = 0;

	for (cur = mlist; cur != NULL; cur = cur->next) {
		MsgInfo *msginfo = (MsgInfo *)cur->data;
		db_insert_msg(msginfo);
	}

	g_print("\n%s: %d total messages\n", folder_id, total);
	g_print("%d inserted into msginfo (%d skipped)\n",
		stat_ins_msginfo, stat_skip_msginfo);
	g_print("%d inserted into msg_folderinfo (%d skipped)\n",
		stat_ins_folderinfo, stat_skip_folderinfo);
	g_print("%d error\n", stat_error);

	db_cleanup_removed_msgs(folder_id, mlist, del_nonexist);

	procmsg_msg_list_free(mlist);
	g_free(folder_id);

	if (recursive && dirlist) {
		cur = dirlist;
		while (cur) {
			ret = db_import_mh_folder((gchar *)cur->data,
						  del_nonexist, recursive);
			if (ret < 0)
				break;
			cur = cur->next;
		}
	}

	slist_free_strings(dirlist);
	g_slist_free(dirlist);

	g_free(path_);

	return ret;
}

gint db_import_folder_item(FolderItem *item, gboolean del_nonexist,
			   gboolean recursive)
{
	GSList *mlist, *cur;
	gchar *full_id;
	gint ret = 0;

	g_return_val_if_fail(item != NULL, -1);

	if (item->stype == F_VIRTUAL || item->stype == F_QUEUE ||
	    item->stype == F_TRASH)
		return 0;

	if (item->path) {
		full_id = folder_item_get_identifier(item);

		g_print("\nimporting %s ...\n", full_id);

		mlist = folder_item_get_msg_list(item, TRUE);
		mlist = procmsg_sort_msg_list
			(mlist, SORT_BY_NUMBER, SORT_ASCENDING);

		stat_ins_msginfo = stat_ins_folderinfo =
			stat_skip_msginfo = stat_skip_folderinfo =
			stat_error = 0;

		for (cur = mlist; cur != NULL; cur = cur->next) {
			MsgInfo *msginfo = (MsgInfo *)cur->data;
			db_insert_msg(msginfo);
		}

		g_print("\n%s: %d total messages\n", full_id, item->total);
		g_print("%d inserted into msginfo (%d skipped)\n",
			stat_ins_msginfo, stat_skip_msginfo);
		g_print("%d inserted into msg_folderinfo (%d skipped)\n",
			stat_ins_folderinfo, stat_skip_folderinfo);
		g_print("%d error\n", stat_error);

		db_cleanup_removed_msgs(full_id, mlist, del_nonexist);

		procmsg_msg_list_free(mlist);
		g_free(full_id);
	}

	if (recursive && item->node && item->node->children) {
		GNode *child = item->node->children;

		while (child) {
			ret = db_import_folder_item(FOLDER_ITEM(child->data),
						    del_nonexist, recursive);
			if (ret < 0)
				return ret;
			child = child->next;
		}
	}

	return ret;
}

gint db_import_folder(const gchar *folder_id, gboolean del_nonexist,
		      gboolean recursive)
{
	gint ret;

	g_return_val_if_fail(folder_id != NULL, -1);

	if (g_path_is_absolute(folder_id)) {
		gchar *path;

		path = conv_filename_from_utf8(folder_id);
		ret = db_import_mh_folder(path, del_nonexist, recursive);
		g_free(path);
	} else {
		FolderItem *item;

		item = folder_find_item_from_identifier(folder_id);
		if (!item) {
			g_warning("folder item '%s' not found.\n", folder_id);
			return -1;
		}
		ret = db_import_folder_item(item, del_nonexist, recursive);
	}

	return ret;
}

int main(int argc, char *argv[])
{
	gchar *folder_id = NULL;
	gchar *dbname, *host, *user, *pass;
	gushort port;
	gint n;
	gboolean del_nonexist = TRUE;
	gboolean recursive = FALSE;

	syl_init();

	n = parse_cmdline(argc, argv, &dbname, &host, &port, &user, &pass);
	if (argc > n)
		folder_id = conv_localetodisp(argv[n], NULL);
	else {
		g_warning("specify target folder.");
		exit(1);
	}
	if (cmdline_has_option(argc, argv, "-n"))
		del_nonexist = FALSE;
	if (cmdline_has_option(argc, argv, "-r"))
		recursive = TRUE;
	if (cmdline_has_option(argc, argv, "-v"))
		verbose = 1;

	prefs_common_read_config();
	account_read_config_all();

	if (folder_read_list() < 0) {
		g_warning("can't read folder list");
	}

	conn = db_connect(dbname, host, port, user, pass);
	if (!conn)
		exit(1);

	db_import_folder(folder_id, del_nonexist, recursive);

	db_disconnect(conn);

	g_free(folder_id);
	//syl_cleanup();

	return 0;
}
