/*
 * Copyright 2004-2005 Luc Verhaegen.
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*************************************************************************
 *
 *  File:       via_cursor.c
 *  Content:    Hardware cursor support for VIA/S3G UniChrome
 *
 ************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#endif

/*
 *
 */
static void
ViaCursorShow(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    CARD32 dwCursorMode;

    dwCursorMode = VIAGETREG(VIA_REG_CURSOR_MODE);

    /* Turn on Hardware Cursor */
    VIASETREG(VIA_REG_CURSOR_MODE, dwCursorMode | 0x01);
}

/*
 *
 */
static void
ViaCursorHide(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    CARD32 dwCursorMode;

    dwCursorMode = VIAGETREG(VIA_REG_CURSOR_MODE);

    /* Turn cursor off. */
    VIASETREG(VIA_REG_CURSOR_MODE, dwCursorMode & 0xFFFFFFFE);
}

/*
 *
 */
static void
ViaCursorLoadImage(ScrnInfoPtr pScrn, unsigned char* src)
{
    VIAPtr pVia = VIAPTR(pScrn);
    CARD32 dwCursorMode;

    VIAAccelSync(pScrn);

    dwCursorMode = VIAGETREG(VIA_REG_CURSOR_MODE);

    /* Turn cursor off. */
    VIASETREG(VIA_REG_CURSOR_MODE, dwCursorMode & 0xFFFFFFFE);

    /* Upload the cursor image to the frame buffer. */
    memcpy(pVia->FBBase + pVia->CursorAddress, src, pVia->CursorSize);

    /* Restore cursor status */
    VIASETREG(VIA_REG_CURSOR_MODE, dwCursorMode);
}

/*
 *
 */
static void
ViaCursorSetPosition(ScrnInfoPtr pScrn, int x, int y)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    CARD8   xoff, yoff;
    CARD32  dwCursorMode;

    if (x < 0) {
        xoff = ((-x) & 0xFE);
        x = 0;
    } else
        xoff = 0;

    if (y < 0) {
        yoff = ((-y) & 0xFE);
        y = 0;
    } else
        yoff = 0;

    /* Hide cursor before set cursor position in order to avoid ghost cursor
     * image when directly set cursor position. It should be a HW bug but
     * we can use patch by SW. */
    dwCursorMode = VIAGETREG(VIA_REG_CURSOR_MODE);

    /* Turn cursor off. */
    VIASETREG(VIA_REG_CURSOR_MODE, dwCursorMode & 0xFFFFFFFE);

    VIASETREG(VIA_REG_CURSOR_ORG, ((xoff << 16) | (yoff & 0x003f)));
    VIASETREG(VIA_REG_CURSOR_POS, ((x << 16) | (y & 0x07ff)));

    /* Restore cursor status */
    VIASETREG(VIA_REG_CURSOR_MODE, dwCursorMode);
}

/*
 *
 */
static void
ViaCursorSetColors(ScrnInfoPtr pScrn, int bg, int fg)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIASETREG(VIA_REG_CURSOR_FG, fg);
    VIASETREG(VIA_REG_CURSOR_BG, bg);

}

/*
 *
 */
Bool
ViaCursorInit(ScrnInfoPtr pScrn, ScreenPtr pScreen)
{
    VIAPtr pVia = VIAPTR(pScrn);
    xf86CursorInfoPtr infoPtr;

    VIAFUNC(pScrn->scrnIndex);

    infoPtr = xf86CreateCursorInfoRec();
    if (!infoPtr)
        return FALSE;

    infoPtr->Flags = HARDWARE_CURSOR_AND_SOURCE_WITH_MASK |
                     HARDWARE_CURSOR_TRUECOLOR_AT_8BPP |
                     HARDWARE_CURSOR_INVERT_MASK |
                     HARDWARE_CURSOR_BIT_ORDER_MSBFIRST ;

    infoPtr->SetCursorColors = ViaCursorSetColors;
    infoPtr->SetCursorPosition = ViaCursorSetPosition;
    infoPtr->LoadCursorImage = ViaCursorLoadImage;
    infoPtr->HideCursor = ViaCursorHide;
    infoPtr->ShowCursor = ViaCursorShow;
    infoPtr->UseHWCursor = NULL;

    /* cursor is 2bit plane; 64*64*2/8 = 1024 and requires this alignment */
    if (((pVia->FBFreeStart + 1024 + 0x3FF) & ~0x3FF) <= pVia->FBFreeEnd) {
        infoPtr->MaxWidth = 64;
        infoPtr->MaxHeight = 64;
        infoPtr->Flags |= HARDWARE_CURSOR_SOURCE_MASK_INTERLEAVE_64;

        pVia->CursorSize = 1024;

        pVia->CursorAddress = (pVia->FBFreeStart + 0x3FF) & ~0x3FF;
        VIASETREG(VIA_REG_CURSOR_MODE, pVia->CursorAddress);

        ViaDebug(pScrn->scrnIndex, "%s: Using 64x64 HW Cursor.\n", __func__);
    } else {
        if (((pVia->FBFreeStart + 256 + 0xFF) & ~0xFF) > pVia->FBFreeEnd) {
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Not enough FB room "
                       "available for HW Cursor.\n", __func__);
            xf86DestroyCursorInfoRec(infoPtr);
            return FALSE;
        }

        infoPtr->MaxWidth = 32;
        infoPtr->MaxHeight = 32;
        infoPtr->Flags |= HARDWARE_CURSOR_SOURCE_MASK_INTERLEAVE_32;

        pVia->CursorSize = 256;

        pVia->CursorAddress = (pVia->FBFreeStart + 0xFF) & ~0xFF;
        VIASETREG(VIA_REG_CURSOR_MODE, pVia->CursorAddress | 0x2);

        ViaDebug(pScrn->scrnIndex, "%s: Using 32x32 HW Cursor.\n", __func__);
    }

    pVia->FBFreeStart = pVia->CursorAddress + pVia->CursorSize;

    if (!xf86InitCursor(pScreen, infoPtr)) {
        xf86DestroyCursorInfoRec(infoPtr);
        return FALSE;
    } else {
        pVia->Cursor = infoPtr;
        return TRUE;
    }
}

/*
 *
 */
void
ViaCursorStore(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    
    VIAFUNC(pScrn->scrnIndex);

    if (!pVia->Cursor)
        return;
    
    if (pVia->CursorImage) {
	xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: stale image left.\n", __func__);
	xfree(pVia->CursorImage);
    }

    pVia->CursorImage = xcalloc(1, pVia->CursorSize);
    memcpy(pVia->CursorImage, pVia->FBBase + pVia->CursorAddress, pVia->CursorSize);
    pVia->CursorFG = (CARD32)VIAGETREG(VIA_REG_CURSOR_FG);
    pVia->CursorBG = (CARD32)VIAGETREG(VIA_REG_CURSOR_BG);
    pVia->CursorMC = (CARD32)VIAGETREG(VIA_REG_CURSOR_MODE);
}

/*
 *
 */
void
ViaCursorRestore(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    if (!pVia->Cursor)
        return;

    if (pVia->CursorImage) {
	memcpy(pVia->FBBase + pVia->CursorAddress, pVia->CursorImage, pVia->CursorSize);
	VIASETREG(VIA_REG_CURSOR_FG, pVia->CursorFG);
	VIASETREG(VIA_REG_CURSOR_BG, pVia->CursorBG);
	VIASETREG(VIA_REG_CURSOR_MODE, pVia->CursorMC);
	xfree(pVia->CursorImage);
	pVia->CursorImage = NULL;
    } else
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: No cursor image stored.\n", __func__);
}
