/* Automatic generation of BuildRequires dependencies for rpmbuild.
 * Copyright (C) 2008 Red Hat Inc.
 * Written by Richard W.M. Jones <rjones@redhat.com>
 * Patches from Rajeesh K Nambiar <rajeeshknambiar@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This is an LD_PRELOAD extension which logs accesses to any files by
 * trapping access to any system calls which open files.  Details of
 * opened files are written to the file named in
 * the AUTO_BUILDREQUIRES_LOGFILE environment variable.
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <dlfcn.h>

#define ALIAS(ret,syscall,params,brfunc)				\
  extern ret syscall params __attribute__((alias (brfunc)))

static char *br_path (const char *path);
static void br_init (void) __attribute__((constructor));
static void br_log (const char *fs, ...) __attribute__((format (printf,1,2)));

/* These are the real glibc symbols, initialized in br_init. */
static int (*glibc_open) (const char *pathname, int flags, mode_t mode);
static int (*glibc_execve) (const char *filename, char *const argv[],
			    char *const envp[]);

/* Canonicalize a relative or absolute path.  If that isn't possible,
 * return an absolute path.
 *
 * The caller must free the result.
 */
static char *
br_path (const char *path)
{
  char *dir, *cat, *rp;
  int len;

  if (path == NULL) return NULL;

  if (path[0] == '/') {
    rp = realpath (path, NULL);
    if (!rp) rp = strdup (path);
    return rp;
  }

  dir = get_current_dir_name ();
  if (dir == NULL) return NULL;

  len = strlen (dir) + 1 + strlen (path) + 1;
  cat = malloc (len);
  if (cat == NULL) {
    perror ("malloc");
    abort ();
  }

  strcpy (cat, dir);
  strcat (cat, "/");
  strcat (cat, path);

  rp = realpath (cat, NULL);
  free (dir);
  if (!rp) rp = cat; else free (cat);
  return rp;
}

int
abr_open (const char *pathname, int flags, mode_t mode)
{
  int fd;
  char *rp;

  rp = br_path (pathname);
  if (rp)
    br_log ("open %s\n", rp);
  else
    perror (pathname);
  free (rp);

  fd = glibc_open (pathname, flags, mode);
  return fd;
}

ALIAS (int, open, (const char *, int, ...), "abr_open");

int
abr_execve (const char *filename, char *const argv[], char *const envp[])
{
  int r;
  char *rp;

  rp = br_path (filename);
  if (rp)
    br_log ("execve %s\n", rp);
  else
    perror (filename);
  free (rp);

  r = glibc_execve (filename, argv, envp);
  return r;
}

ALIAS (int, execve, (const char *filename, char *const argv[],
		     char *const envp[]), "abr_execve");

/* Other syscalls to consider:
 * access
 * statfs
 * stat, lstat
 * chdir
 * mkdir
 * unlink
 * readlink
 * 'connect' params can contain a path in some rare circumstances.
 */

/* This is the logging function.  We have to be very cautious in this
 * function and not disturb any part of the process (so far as that is
 * possible).  eg. When we open a file descriptor we must close it
 * afterwards.
 *
 * Any errors should either call abort() if they are serious, or
 * return without harm if not.
 *
 * Also remember to call glibc_* for any syscalls that we are
 * intercepting, otherwise you'll get an infinite loop.
 */
static void
br_log (const char *fs, ...)
{
  const char *filename;
  int fd, len;
  va_list args;
  char *msg;

  /* Got the logging environment variable?  Should we cleanse this
   * variable for security reasons, or is it not a problem because RPM
   * specfiles can do arbitrary Bad Stuff already? (XXX)
   */
  filename = getenv ("AUTO_BUILDREQUIRES_LOGFILE");
  if (filename == NULL) return;

  fd = glibc_open (filename, O_WRONLY | O_APPEND, 0);
  if (fd == -1) { perror ("open logfile"); abort (); }

  /* Create the log string. */
  va_start (args, fs);
  len = vasprintf (&msg, fs, args);
  if (len == -1) {
    perror ("vasprintf");
    abort ();
  }
  va_end (args);

  /* Write it in a single operation.  Should be atomic, right?? (XXX) */
  if (write (fd, msg, len) != len) {
    perror ("write");
    abort ();
  }

  close (fd);
  free (msg);
}

static void
br_init (void)
{
  void *dl;

  dl = dlopen ("/lib64/libc.so.6", RTLD_LAZY|RTLD_LOCAL);
  if (dl == NULL)	// Try '/lib/' also
	  dl = dlopen("/lib/libc.so.6", RTLD_LAZY|RTLD_LOCAL);
  if (dl == NULL) {
    fprintf (stderr, "%s\n", dlerror ());
    abort ();
  }
  glibc_open = dlsym (dl, "open");
  glibc_execve = dlsym (dl, "execve");
}
