/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <qregexp.h>
#include <qstringlist.h>
#include <klocale.h>

#include "datinfo.h"
#include "datmanager.h"
#include "access.h"
#include "thread.h"
#include "kitaconfig.h"
#include "kita-utf8.h"
#include "kita-utf16.h"
#include "kita_misc.h"
#include "account.h"
#include "threadindex.h"
#include "cache.h"


using namespace Kita;

#define RESDAT_DEFAULTSIZE 10
#define RESDAT_DELTA       1000


/*------------------------------------------------------*/
/* DatInfo stores & handles all information about *.dat */

DatInfo::DatInfo( const KURL& url ) : m_access ( 0 ), m_access2( 0 )
{
    QString refstr;
    m_datURL = Kita::getDatURL( url, refstr );

    /* get the pointer of Thread class */
    m_thread = Kita::Thread::getByURLNew( m_datURL );
    if ( m_thread == NULL ) {

        /* create Thread */
        m_thread = Kita::Thread::getByURL( m_datURL );
        if ( m_thread == NULL ) return ;

        /* read idx file */
        ThreadIndex::loadIndex( m_thread, m_datURL );
    }

    m_thread = Thread::getByURL( m_datURL );

    /* japanese strings */
#if KDE_IS_VERSION( 3, 3, 0 )
    m_spacestr = Kita::utf8ToUnicode( KITAUTF8_ZENSPACE );
#else
    m_spacestr = ". ";
#endif
    m_framestr1 = Kita::utf8ToUnicode( KITAUTF8_FRAME1 ); /* |  */
    m_framestr2 = Kita::utf8ToUnicode( KITAUTF8_FRAME2 ); /* |- */
    m_framestr3 = Kita::utf8ToUnicode( KITAUTF8_FRAME3 ); /* L  */

    /* make directory */
    QString cacheDir = Cache::baseDir() + Cache::serverDir( m_datURL ) + Cache::boardDir( m_datURL );
    if ( !Kita::mkdir( cacheDir ) ) return ;

    initPrivate(
        TRUE /* load cache */
    );
}

DatInfo::~DatInfo()
{
    initPrivate( FALSE );
}


/* Init  */
/* Usually, don't call this. */ /* public */
void DatInfo::init()
{
    QMutexLocker locker( &m_mutex );

    return initPrivate( TRUE );
}

/* public */
void DatInfo::wait()
{
    QMutexLocker locker( &m_mutex );
}

/* Init. If loadCache = TRUE, load data from cache. */ /* private */
void DatInfo::initPrivate( bool loadCache )
{
    /* stop & delete dat loader */
    deleteAccessJob();

    /* init variables */
    m_broken = FALSE;
    m_lock = 0;
    m_nowLoading = FALSE;
    m_lastLine = QString::null;

    /* clear ResDatVec */
    m_resDatVec.clear();
    increaseResDatVec( RESDAT_DEFAULTSIZE );

    /* reset Abone */
    resetAbonePrivate();

    /* create dat loader */
    m_access = new Kita::Access( m_datURL );

    connect( m_access, SIGNAL( receiveData( const QStringList& ) ),
             SLOT( slotReceiveData( const QStringList& ) ) );
    connect( m_access, SIGNAL( finishLoad() ), SLOT( slotFinishLoad() ) );

    if ( !loadCache ) return ;

    /* reset ReadNum before loading cache. */
    /* ReadNum & subject are updated by Access::getcache() */
    m_thread->setReadNum( 0 );

    /* get dat from cahce  */
    /* slotReceiveData() is called from Access::getcache() */
    m_access->getcache();

    /* save up-to-date thread information */
    ThreadIndex::saveIndex( m_thread, m_datURL );
}


/* private */
void DatInfo::resetResDat( RESDAT& resdat )
{
    resdat.num = 0;
    resdat.parsed = FALSE;
    resdat.broken = FALSE;
    resdat.anclist.clear();
    resdat.checkAbone = FALSE;
    resdat.abone = FALSE;
    resdat.isResponsed = FALSE;
}


/* private */
void DatInfo::increaseResDatVec( int delta )
{
    int size = m_resDatVec.size();
    RESDAT resdat;
    resetResDat( resdat );
    m_resDatVec.resize( size + delta, resdat );
}


/* delete dat loader */ /* private */
void DatInfo::deleteAccessJob()
{
    if ( m_access ) {
        m_access->killJob();
        delete m_access;
        m_access = NULL;
    }
    if ( m_access2 ) {
        m_access2->killJob();
        delete m_access2;
        m_access2 = NULL;
    }
}

/* public */
const KURL& DatInfo::url()
{
    QMutexLocker locker( &m_mutex );

    return m_datURL;
}



/*--------------------------------------*/
/* cache handling functions             */

/* Update cache  */

/* When Kita::Access received new data,
   slotReceiveData is called.           */

/* When Kita::Access fineshed loading,
   slotFinishLoad is called, and
   DatInfo emits the finishLoad signal to the parent object  */ /* public */
bool DatInfo::updateCache( const QObject* parent )
{
    QMutexLocker locker( &m_mutex );
    if ( m_access == NULL ) return FALSE;
    if ( m_nowLoading ) return FALSE;

    m_nowLoading = TRUE;
    m_lock++; /* By locking, DatManager can't delete this while loading. */

    connect( this, SIGNAL( receiveData() ),
             parent, SLOT( slotReceiveData() ) );

    connect( this, SIGNAL( finishLoad() ),
             parent, SLOT( slotFinishLoad() ) );

    m_access->getupdate( m_thread->readNum() );

    return TRUE;
}


/* slot called when Kita::Access
   received new data              */      /* private  slot */
void DatInfo::slotReceiveData( const QStringList& lineList )
{
    int rescode = m_access->responseCode();
    if ( m_access2 ) {
        rescode = m_access2->responseCode();
    }

    if ( rescode != 200 && rescode != 206 ) return ;

    /* copy lines to buffer */
    int count = lineList.count();
    for ( int i = 0; i < count ; ++i ) copyOneLineToResDat( lineList[ i ] );

    emit receiveData();
}


/* copy one line to resdat.
   See also DatInfo::slotReceiveData()   */ /* private */
bool DatInfo::copyOneLineToResDat( const QString& line )
{
    if ( line == QString::null ) return FALSE;

    /* update ReadNum */
    const int num = m_thread->readNum() + 1;
    m_thread->setReadNum( num );

    /* If resdat vector is short, then resize the vector. */
    while ( ( int ) m_resDatVec.size() <= num ) increaseResDatVec( RESDAT_DELTA );

    /* reset ResDat */
    RESDAT& resdat = m_resDatVec[ num ];
    resetResDat( resdat );

    resdat.num = num;
    resdat.linestr = line;

    /* get subject */
    if ( num == 1 ) parseDat( num );

    /* search all responses which are responsed by this line. */
    if ( KitaConfig::checkResponsed() ) {

        if ( parseDat( num ) && !checkAbonePrivate( num ) ) { /* parse line here to get AncList */

            const int maxRange = 10;

            AncList& anclist = m_resDatVec[ num ].anclist;
            for ( AncList::iterator it = anclist.begin(); it != anclist.end(); ++it ) {

                int fromNum = ( *it ).from;
                int toNum = QMIN( num - 1, ( *it ).to );
                if ( toNum - fromNum + 1 > maxRange ) continue;

                for ( int i = fromNum; i <= toNum; ++i ) {

                    if ( !checkAbonePrivate( i ) ) m_resDatVec[ i ].isResponsed = TRUE;
                }
            }
        }
    }

    return TRUE;
}


/* slot called when Kita::Access
   finished loading new dat */      /* private  slot */
void DatInfo::slotFinishLoad()
{
    /* save thread information */
    ThreadIndex::saveIndex( m_thread, m_datURL );

    /* re-try by offlaw.cgi */
    if ( m_thread->readNum() == 0 && m_access2 == NULL && DatManager::is2chThread( m_datURL ) ) {
        if ( Account::isLogged() ) {
            initPrivate( TRUE );
            m_access2 = new OfflawAccess( m_datURL );
            connect( m_access2, SIGNAL( receiveData( const QStringList& ) ),
                     SLOT( slotReceiveData( const QStringList& ) ) );
            connect( m_access2, SIGNAL( finishLoad() ), SLOT( slotFinishLoad() ) );
            m_access2->get();
            return ;
        }
    }
    /* finish loading session & emit signal to the parent object */
    m_nowLoading = FALSE;
    emit finishLoad();

    /* disconnect signals */
    disconnect( SIGNAL( receiveData() ) );
    disconnect( SIGNAL( finishLoad() ) );

    if ( m_lock ) m_lock--;
}


/* public */
int DatInfo::getResponseCode()
{
    QMutexLocker locker( &m_mutex );
    if ( m_access == NULL ) return 0;

    return m_access->responseCode();
}


/* public */
int DatInfo::getServerTime()
{
    QMutexLocker locker( &m_mutex );
    if ( m_access == NULL ) return 0;

    return m_access->serverTime();
}


/* public */
bool DatInfo::deleteCache()
{
    QMutexLocker locker( &m_mutex );
    if ( m_nowLoading ) return FALSE;

    initPrivate( FALSE );

    return TRUE;
}


/* public */
bool DatInfo::isLoadingNow()
{

    QMutexLocker locker( &m_mutex );

    return m_nowLoading;
}



/* public */
void DatInfo::stopLoading()
{

    /* Don't lock the mutex here !!!
       It will cause deadlock , because
       Kita::Access::stopJob() calls KitaHTMLPart::slotFinishLoad() back,
       then KitaHTMLPart::slotFinishLoad() calls another functions in DatInfo. */
    if ( m_access == NULL ) return ;
    if ( ! m_nowLoading ) return ;

    m_access->stopJob();
}




/*---------------------------------------------------*/
/* locking , unlocking functions                     */
/*                                                   */
/* If m_lock is not 0, DatManager can't delete this. */
/* Don't forget to call unlock() after locking this. */

/* They are public */

void DatInfo::lock ()
{
    QMutexLocker locker( &m_mutex );

    m_lock++;
}

void DatInfo::unlock()
{
    QMutexLocker locker( &m_mutex );

    if ( m_lock ) m_lock--;
}

int DatInfo::isLocked()
{
    QMutexLocker locker( &m_mutex );

    return m_lock;
}


/*------------------------------------------------------*/
/* get subject, linedata,  id, body, name, HTML, etc.   */

/* They are public */

const QString& DatInfo::getDat( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    return m_resDatVec[ num ].linestr;
}

const QString& DatInfo::getId( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    return m_resDatVec[ num ].id;
}

/* plain strings of name */
QString DatInfo::getPlainName( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    return m_resDatVec[ num ].name;
}


/* plain strings of title */
QString DatInfo::getPlainTitle( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    QString titleHTML;
    Kita::createTitleHTML( m_resDatVec[ num ], titleHTML );

    QString retStr;
    Kita::DatToText( titleHTML, retStr );

    return retStr;
}


/* plain strings of body  */
QString DatInfo::getPlainBody( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    QString retStr;
    Kita::DatToText( m_resDatVec[ num ].bodyHTML, retStr );

    return retStr;
}


/*-----------------------------------------*/
/* HTML data                               */


/* get HTML strings of title & body.
 
   return values are defined in datinfo.h.  */ /* public */

int DatInfo::getHTML( int num, bool checkAbone, QString& titleHTML, QString& bodyHTML )
{
    QMutexLocker locker( &m_mutex );
    return getHTMLPrivate( num, checkAbone, titleHTML, bodyHTML );
}

/**
 * @param[in] num
 * @param[in] checkAbone
 *
 * @param[out] titleHTML
 * @param[out] bodyHTML
 *
 * @retval KITA_HTML_NOTPARSED The dat is not parsed.
 * @retval KITA_HTML_ABONE The res dat is marked as abone.
 * @retval KITA_HTML_BROKEN The res dat is marked as broken.
 * @retval KITA_HTML_NORMAL The res dat is normal.
 *
 */
int DatInfo::getHTMLPrivate( int num, bool checkAbone, QString& titleHTML, QString& bodyHTML )
{
    if ( !parseDat( num ) ) return KITA_HTML_NOTPARSED;

    bool abone = checkAbone & checkAbonePrivate( num );
    RESDAT& resdat = m_resDatVec[ num ];

    if ( abone ) {
        titleHTML = QString().setNum( num ) + " " + i18n( "Abone" );
        bodyHTML = "<a href=\"#abone" + QString().setNum( num ) + "\">";
        bodyHTML += i18n( "Abone" ) + "</a>";

        return KITA_HTML_ABONE;
    } else if ( resdat.broken ) {
        titleHTML = QString().setNum( num ) + " " + i18n( "Broken" );
        bodyHTML = i18n( "Broken" );

        return KITA_HTML_BROKEN;
    } else {
        createTitleHTML( resdat, titleHTML );
        bodyHTML = resdat.bodyHTML;
        
        return KITA_HTML_NORMAL;
    }
}

/* get HTML strings from startnum to endnum.
 
   return value is HTML strings               */  /* public */
QString DatInfo::getHTMLString( int startnum, int endnum, bool checkAbone )
{
    QMutexLocker locker( &m_mutex );

    QString retHTML = QString::null;

    for ( int num = startnum; num <= endnum; num++ ) {

        QString html;
        getHTMLofOneRes( num, checkAbone, html );
        retHTML += html;
    }

    return retHTML;
}


/* return HTML strings that have ID = strid. */ /* public */
QString DatInfo::getHtmlByID( const QString& strid, int &count )
{
    QMutexLocker locker( &m_mutex );

    QString retHTML = QString::null;
    count = 0;

    for ( int i = 1; i <= m_thread->readNum(); i++ ) {

        if ( !parseDat( i ) ) continue;

        if ( m_resDatVec[ i ].id == strid ) {
            count ++;

            QString html;
            getHTMLofOneRes( i, TRUE, html );
            retHTML += html;
        }
    }

    return retHTML;
}

/**
 *
 * convert res dat to html.
 *
 * @param[in] num
 * @param[in] checkAbone
 *
 * @param[out] html
 *
 */
void DatInfo::getHTMLofOneRes( int num, bool checkAbone, QString& html )
{
    html = QString::null;
    QString titleHTML, bodyHTML;
    if ( getHTMLPrivate( num, checkAbone, titleHTML, bodyHTML ) == KITA_HTML_NOTPARSED ) return ;

    if ( m_resDatVec[ num ].isResponsed ) titleHTML.replace( "<a href=\"#write", "<a class=\"coloredLink\" href=\"#write" );
    html += "<div class=\"pop_res_title\">" + titleHTML + "</div>";
    html += "<div class=\"pop_res_body\">" + bodyHTML + "</div>";
}



/*-------------------------------*/
/* Get HTML document of res tree.*/
/* For example, when rootnum = 1,
 
>>1 
|-->>4
|  |--->>10
|
|-->>20, and return count = 3.  */

/* Note that this function checks Abone internally. */ /* public */
QString DatInfo::getTreeByRes( const int rootnum, int& count )
{
    QMutexLocker locker( &m_mutex );
    return getTreeByResPrivate( rootnum, FALSE, count );
}

/*---------------------------------------*/
/* Get HTML document of reverse res tree.*/
/* For example, when rootnum = 10,
 
>>10 
|-->>5
|  |--->>2
|
|-->>6, and returns count = 3.  */

/* Note that this function checks Abone internally. */ /* public */
QString DatInfo::getTreeByResReverse( const int rootnum, int& count )
{
    QMutexLocker locker( &m_mutex );
    return getTreeByResPrivate( rootnum, TRUE, count );
}


/* private */
QString DatInfo::getTreeByResPrivate(
    const int rootnum,
    bool reverse,    /* reverse search */
    int& count )
{

    QString tmp = QString().setNum( rootnum );
    QString retstr = "<a href=\"#" + tmp + "\">&gt;&gt;" + tmp + "</a><br>";

    retstr += getTreeByResCore( rootnum, reverse, count, "" );

    return retstr;
}

/* private */
QString DatInfo::getTreeByResCore(
    const int rootnum,
    bool reverse,    /* reverse search */
    int& count, QString prestr )
{
    if ( !parseDat( rootnum ) ) return QString::null;
    if ( checkAbonePrivate( rootnum ) ) return QString::null;

    QString retstr = QString::null ;
    count = 0;
    QStringList strlists;

    if ( !reverse ) {

        /* collect responses that have anchor to rootnum */
        for ( int i = rootnum + 1; i <= m_thread->readNum(); i++ ) {
            if ( checkAbonePrivate( i ) ) continue;
            if ( checkRes( i, rootnum ) ) {
                count ++;
                strlists += QString().setNum( i );
            }
        }

    } else { /* collect responses for which rootnum has anchors */

        AncList& anclist = m_resDatVec[ rootnum ].anclist;
        for ( AncList::iterator it = anclist.begin(); it != anclist.end(); ++it ) {
            for ( int i = ( *it ).from; i <= QMIN( rootnum - 1, ( *it ).to ) ; i++ ) {
                if ( checkAbonePrivate( i ) ) continue;
                count ++;
                strlists += QString().setNum( i );
            }
        }
    }

    /* make HTML document */
    if ( count ) {

        for ( QStringList::iterator it = strlists.begin(); it != strlists.end(); ++it ) {
            QString tmpstr;
            if ( ( *it ) == strlists.last() ) tmpstr = m_framestr3;  /* 'L' */
            else tmpstr = m_framestr2;  /* '|-' */

            retstr += prestr + tmpstr + "<a href=\"#" + ( *it ) + "\">&gt;&gt;" + ( *it ) + "</a><br>";

            /* call myself recursively */
            int tmpnum;
            tmpstr = prestr;
            if ( ( *it ) == strlists.last() ) tmpstr += m_spacestr + m_spacestr + m_spacestr; /* "   " */
            else tmpstr += m_framestr1 + m_spacestr; /* "| " */
            retstr += getTreeByResCore( ( *it ).toInt(), reverse, tmpnum, tmpstr );
            count += tmpnum;
        }
    }

    return retstr;
}



/*----------------------------------------------*/
/* Check if No.num has anchors to No.target     */
/* For exsample, if target = 4, and No.num have
   an anchor >>4, or >>2-6, etc.,
   then return TRUE.                            */  /* private */
bool DatInfo::checkRes( const int num, const int target )
{
    const int range = 20;
    if ( !parseDat( num ) ) return FALSE;

    AncList& anclist = m_resDatVec[ num ].anclist;

    for ( AncList::iterator it = anclist.begin(); it != anclist.end(); ++it ) {
        if ( ( *it ).to - ( *it ).from > range ) continue;
        if ( target >= ( *it ).from && target <= ( *it ).to ) return TRUE;
    }

    return FALSE;

}


/*-----------------------*/
/* several information */

int DatInfo::getResNum()
{
    return m_thread->resNum();
}

/* public */
int DatInfo::getReadNum()
{
    return m_thread->readNum();
}

/* public */
int DatInfo::getViewPos()
{
    return m_thread->viewPos();
}


/* return number of responses that have ID = strid. */
/* Note that this function checks Abone internally. */ /* public */
int DatInfo::getNumByID( const QString& strid )
{
    QMutexLocker locker( &m_mutex );

    int count = 0;

    for ( int i = 1; i <= m_thread->readNum(); i++ ) {

        if ( !parseDat( i ) ) continue;
        if ( checkAbonePrivate( i ) ) continue;

        if ( m_resDatVec[ i ].id == strid ) count++;
    }

    return count;
}


/* public */
int DatInfo::getDatSize()
{
    QMutexLocker locker( &m_mutex );
    if ( m_access == NULL ) return 0;

    return m_access->dataSize();
}


/* public */
const bool DatInfo::isResponsed( int num ) const
{
    return m_resDatVec[ num ].isResponsed;
}


/* public */
bool DatInfo::isResValid( int num )
{
    QMutexLocker locker( &m_mutex );

    return parseDat( num );
}

/* public */
bool DatInfo::isBroken()
{
    QMutexLocker locker( &m_mutex );

    if ( m_broken ) return m_broken;

    if ( m_access == NULL ) return FALSE;

    int rescode = m_access->responseCode();
    bool invalid = m_access->invalidDataReceived();

    /* see also Access::slotReceiveThreadData() */
    if ( invalid && ( rescode == 200 || rescode == 206 ) ) return TRUE;

    /* maybe "Dat Ochi" */
    return FALSE;
}

/* public */
bool DatInfo::isResBroken( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return FALSE;

    return m_resDatVec[ num ].broken;
}

/* ID = strid ? */ /* public */
bool DatInfo::checkID( const QString& strid, int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return FALSE;

    if ( m_resDatVec[ num ].id == strid ) return TRUE;

    return FALSE;
}


/* Are keywords included ? */ /* public */
bool DatInfo::checkWord( QStringList& stlist,   /* list of keywords */
                         int num,
                         bool checkOR /* AND or OR search */
                       )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return FALSE;

    QString str_text = m_resDatVec[ num ].bodyHTML;

    for ( QStringList::iterator it = stlist.begin(); it != stlist.end(); ++it ) {

        QRegExp regexp( ( *it ) );
        regexp.setCaseSensitive( FALSE );

        if ( checkOR ) { /* OR */
            if ( str_text.find( regexp, 0 ) != -1 ) {
                return TRUE;
            }
        } else { /* AND */
            if ( str_text.find( regexp, 0 ) == -1 ) return FALSE;
        }
    }

    if ( checkOR ) return FALSE;

    return TRUE;
}



/*--------------------------------*/
/* abone functions                */


/*-----------------------*/
/* reset abone.          */

/* call this when config
   of abone changed.     */  /* public */

void DatInfo::resetAbone()
{
    QMutexLocker locker( &m_mutex );

    return resetAbonePrivate();
}

/* private */
void DatInfo::resetAbonePrivate()
{
    for ( int i = 1; i < ( int ) m_resDatVec.size(); i++ ) m_resDatVec[ i ].checkAbone = FALSE;

    m_aboneByID = ( !KitaConfig::aboneIDList().empty() );
    m_aboneByName = ( !KitaConfig::aboneNameList().empty() );
    m_aboneByBody = ( !KitaConfig::aboneWordList().empty() );
    m_aboneChain = ( m_aboneByID | m_aboneByName | m_aboneByBody ) & KitaConfig::aboneChain() ;
}


/*--------------*/
/* check abone  */ /* public */

bool DatInfo::checkAbone( int num )
{
    QMutexLocker locker( &m_mutex );

    return checkAbonePrivate( num );
}


/* private */
bool DatInfo::checkAbonePrivate( int num )
{
    if ( !parseDat( num ) ) return FALSE;

    if ( m_resDatVec[ num ].checkAbone ) return m_resDatVec[ num ].abone;

    m_resDatVec[ num ].checkAbone = TRUE;
    bool checktmp = FALSE;

    if ( m_aboneByID )
        checktmp = checkAboneCore( m_resDatVec[ num ].id, KitaConfig::aboneIDList() );

    if ( !checktmp && m_aboneByName )
        checktmp = checkAboneCore( m_resDatVec[ num ].name, KitaConfig::aboneNameList() );

    if ( !checktmp && m_aboneByBody )
        checktmp = checkAboneCore( m_resDatVec[ num ].bodyHTML, KitaConfig::aboneWordList() );

    if ( !checktmp && m_aboneChain ) {
        AncList & anclist = m_resDatVec[ num ].anclist;

        for ( AncList::iterator it = anclist.begin();
                it != anclist.end() && !checktmp ; ++it ) {

            int refNum = ( *it ).from;
            int refNum2 = ( *it ).to;

            /* I don't want to enter loop... */
            if ( refNum >= num ) continue;
            if ( refNum2 >= num ) refNum2 = num - 1;

            for ( int i = refNum; i <= refNum2; i++ ) {
                if ( checkAbonePrivate( i ) ) {
                    checktmp = TRUE;
                    break;
                }
            }
        }
    }

    m_resDatVec[ num ].abone = checktmp;

    return m_resDatVec[ num ].abone;
}

/* private */
bool DatInfo::checkAboneCore( const QString& str, QStringList& strlist )
{
    if ( strlist.count() ) {

        int i;
        for ( QStringList::iterator it = strlist.begin();
                it != strlist.end(); ++it ) {
            i = str.find( ( *it ) );
            if ( i != -1 ) {
                return TRUE;
            }
        }
    }

    return FALSE;
}



/* parsing function for ResDat */
/* This function parses the raw data by Kita::parseResDat() */ /* private */
bool DatInfo::parseDat( int num )
{
    if ( num <= 0 || m_thread->readNum() < num ) return FALSE;
    if ( m_resDatVec[ num ].parsed ) return TRUE;

    //   qDebug("parseDat %d",num);

    QString subject = QString::null;
    Kita::parseResDat( m_resDatVec[ num ], subject );
    if ( num == 1 && subject != QString::null ) m_thread->setThreadName( subject );
    if ( m_resDatVec[ num ].broken ) m_broken = TRUE;

    return TRUE;
}
