/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "writeview.h"

#include "libkita/qcp932codec.h"
#include "libkita/kitaconfig.h"
#include "libkita/datmanager.h"
#include "libkita/boardmanager.h"
#include "libkita/signalcollection.h"
#include "libkita/account.h"
#include "libkita/kita-utf8.h"
#include "libkita/kita_misc.h"

#include "thread/htmlpart.h"

#include "kitaui/htmlview.h"

#include <ksqueezedtextlabel.h>
#include <klocale.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <klineedit.h>
#include <kmessagebox.h>
#include <kio/job.h>

#include <qapplication.h>
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qeucjpcodec.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtabwidget.h>
#include <qcheckbox.h>
#include <qlayout.h>

#include <cstring>
#include <cstdlib>

/* result code */
enum{
    K2ch_Unknown,
    K2ch_True,
    K2ch_False,
    K2ch_Error,
    K2ch_Check,
    K2ch_Cookie,
    K2ch_NewThread
};


/*--------------------------------------------------------------------*/

QCp932Codec* KitaWriteView::m_cp932Codec = NULL;


/*
 
If mode == WRITEMODE_NORMAL, then write response.
If mode == WRITEMODE_NEWTHREAD, then create the new thread. "newThreadName" is the name of the new thread.
 
Call setMessage() to set message later.
See also KitaWriteTabWidget::slotShowWriteView().
 
Call slotPostMessage() to post the message.
 
*/
KitaWriteView::KitaWriteView( QWidget* parent, const char* name, const KURL& url, int mode, const QString& newThreadName )
        : KitaWriteDialogBase( parent, name, 0 )
{
    if ( !m_cp932Codec ) m_cp932Codec = new QCp932Codec();

    m_mode = mode;

    if ( m_mode == WRITEMODE_NORMAL ) m_datURL = Kita::getDatURL( url );
    else {
        /* create new thread */

        /* set dummy URL  */
        m_datURL = Kita::BoardManager::boardURL( url );
        QString dummyPath = "dat/dummy" + Kita::BoardManager::ext( url );
        m_datURL.addPath( dummyPath );
    }

    m_bbstype = Kita::BoardManager::type( m_datURL );

    /* set path of bbs.cgi */
    switch ( m_bbstype ) {

    case Kita::Board_JBBS: {
            QString cgipath = Kita::BoardManager::boardRoot( m_datURL )
                              + "/bbs/write.cgi/" + Kita::BoardManager::boardID( m_datURL ) + "/";

            if ( m_mode == WRITEMODE_NORMAL ) cgipath += Kita::DatManager::threadID( m_datURL ) + "/";
            else cgipath += "new/";

            m_bbscgi = cgipath;
        }

        break;

    default:

        m_bbscgi = Kita::BoardManager::boardRoot( m_datURL ) + "/test/bbs.cgi";
    }

    /* connect signals */
    connect( buttonOk, SIGNAL( clicked() ),
             SLOT( slotPostMessage() ) );

    connect( buttonCancel, SIGNAL( clicked() ),
             SLOT( slotCancel() ) );

    connect( this, SIGNAL( openURLRequestExt(
                               const KURL&, const KParts::URLArgs&, QString, int,
                               const KURL& ) ),
             Kita::SignalCollection::getInstance(), SIGNAL( openURLRequestExt(
                         const KURL& , const KParts::URLArgs&, QString, int,
                         const KURL& ) ) );

    connect( faceCombo, SIGNAL( activated( const QString& ) ),
             bodyText, SLOT( insert( const QString& ) ) );

    connect( qtw, SIGNAL( currentChanged ( QWidget * ) ),
             this, SLOT( slotCurrentChanged ( QWidget * ) ) );

    connect( bodyText, SIGNAL( textChanged() ),
             SLOT( slotBodyTextChanged() ) );

    /* setup preview view */
    m_preview = new KitaHTMLPart( NULL );
    qtw->addTab( m_preview->view(), i18n( "preview" ) );
    m_preview->setup( HTMLPART_MODE_KHTML, m_datURL );
    qtw->setCurrentPage( 0 );

    /* setup labels and edit lines */
    QFont font = KitaConfig::threadFont();
    bodyText->setFont( font );
    bodyText->setTabChangesFocus( TRUE );

    boardNameLabel->setText( Kita::BoardManager::boardName( m_datURL ) );
    if ( m_mode == WRITEMODE_NORMAL ) {
        threadName->setText( Kita::DatManager::threadName( m_datURL ) );
        threadName->setReadOnly( TRUE );
        threadName->setFrame( FALSE );
        threadName->setFocusPolicy( NoFocus );
    } else {
        threadName->setText( newThreadName );
    }

    if ( KitaConfig::DefaultNameUseAlways() ) {
        nameLine->setText( KitaConfig::DefaultName() );
    } else {
        QString defaultName = Kita::BoardManager::getBBSDefaultName( m_datURL );
        if ( defaultName == "fusianasan" || defaultName == "(default name)" ) {
            nameLine->setText( KitaConfig::DefaultName() );
        } else {
            nameLine->setText( defaultName );
        }
    }
    KCompletion* comp = KitaConfig::nameCompletion();
    nameLine->setCompletionObject( comp );

    if ( KitaConfig::DefaultSage() ) {
        mailLine->setText( "sage" );
        sageBox->setChecked( true );
    } else {
        mailLine->setText( KitaConfig::DefaultMail() );
    }
    m_mailswap = "";

    QRegExp host_2ch( ".+\\.2ch\\.net" );
    if ( host_2ch.search( m_bbscgi.host() ) != -1
        && KitaConfig::BeMailAddress().length() > 0
        && KitaConfig::BeAuthCode().length() > 0 ) {
        beBox->setChecked( true );
    }

    /* setup AA */
    faceCombo->clear();
    faceCombo->setFont( KitaConfig::threadFont() );
    faceCombo->insertItem( "" );
    QStringList list = KitaConfig::asciiArtList();
    QStringList::iterator it;
    for ( it = list.begin(); it != list.end(); ++it ) {
        faceCombo->insertItem( *it );
    }

    cookie_type = NULL;
    cookie_value = NULL;

}


KitaWriteView::~KitaWriteView()
{
    if ( m_preview ) {
        delete m_preview;
    }
}


/* public */
void KitaWriteView::setMessage( const QString& bodyStr )
{
    bodyText->clear();
    bodyText->insert( bodyStr );
    bodyText->setFocus();
}


/* public */
void KitaWriteView::insertMessage( const QString& str )
{
    bodyText->insert( str );
    bodyText->setFocus();
}


/* public information */

const QString KitaWriteView::name() const
{
    return nameLine->text();
}

const QString KitaWriteView::mail() const
{
    return mailLine->text();
}

const QString KitaWriteView::body() const
{
    return bodyText->text();
}

const KURL KitaWriteView::datURL() const
{
    return m_datURL;
}


/* public slot */ /* virtual */
void KitaWriteView::setFocus()
{
    bodyText->setFocus();
}


/* call this slot to post the message. */ /* public slot */
void KitaWriteView::slotPostMessage()
{
    if ( bodyText->text().length() == 0 ) return ;
    if ( m_mode == WRITEMODE_NEWTHREAD && threadName->text().length() == 0 ) {
        QMessageBox::warning( this, "Kita", i18n( "Enter the thread name." ) );
        return ;
    }

    /* fusianasan */
    if ( nameLine->text().length() == 0
            && Kita::BoardManager::getBBSDefaultName( m_datURL ) == "fusianasan" ) {
        QMessageBox::warning( this, "Kita", i18n( "fusianasan." ) );
        return ;
    }

    if ( !slotBodyTextChanged() ) {
        QMessageBox::warning( this, "Kita", i18n( "Body text is too long." ) );
        return ;
    }

    QString name = nameLine->text();
    KitaConfig::addNameCompletionItem( name );

    /* set post message */
    QString postStr;
    switch ( m_bbstype ) {

    case Kita::Board_JBBS: postStr = setJBBSPostStr(); break;

    case Kita::Board_FlashCGI: postStr = setFlashCGIPostStr(); break;

    default: postStr = setPostStr(); break;
    }

    /* referrer */
    QString refStr = Kita::BoardManager::boardURL( m_datURL );

    m_array.resize( 0 );

    KIO::TransferJob* job = KIO::http_post( m_bbscgi, postStr.utf8(), true );
    job->addMetaData( "content-type", "Content-type: application/x-www-form-urlencoded" );
    job->addMetaData( "referrer", refStr );

    /* 2ch.net cookie modify */
    if ( m_bbstype == Kita::Board_2ch ) {
        set2chCookieStr( job );
    }

    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             this, SLOT( slotRecieveData( KIO::Job*, const QByteArray& ) ) );

    /* slotPostFinished() is called when done. */
    connect( job, SIGNAL( result( KIO::Job* ) ),
             this, SLOT( slotPostFinished( KIO::Job* ) ) );

}


/* public slot */
void KitaWriteView::slotCancel()
{
    if ( bodyText->text().length() == 0 ) {
        emit closeCurrentTab(); /* to KitaWriteTabWidget */
        return ;
    }

    switch ( QMessageBox::warning( this, "Kita",
                                   i18n( "If you close this dialog, you will lose text.\n"
                                         "Do you want to close?" ),
                                   QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default ) ) {
    case QMessageBox::Ok:
        setMessage( QString::null );
        emit closeCurrentTab(); /* to KitaWriteTabWidget */
        break;
    case QMessageBox::Cancel:
        // do nothing
        break;
    }
}


/* public slot */
void KitaWriteView::slotEnableWriting( bool enable )
{
    buttonOk->setEnabled( enable );
}


/* see also slotPostMessage() */ /* private slot */
void KitaWriteView::slotRecieveData( KIO::Job*, const QByteArray& data )
{
    m_array.append( data.data() );
}

void KitaWriteView::find_cookie_typevalue( const char *response ){
	char *str_cmp, *str;
	static char *cookie_type_static = NULL;
	static char *cookie_value_static = NULL;
	cookie_type = cookie_value = NULL;

	str_cmp = "<input type=hidden name=\"";
	str = strstr(response, str_cmp);
	if (str){
		char *str2 = strstr(str, "\"");
		char *str3 = NULL;
		str2++;
		if ((!cookie_type_static) || (!(strstr(str2, cookie_type_static)))){
			free(cookie_type_static);
			free(cookie_value_static);

			cookie_type_static = strdup(str2);
			str3 = strstr(cookie_type_static, "\"");
			*str3 = 0;

			str = str2;
			str2 = strstr(str, "value=\"");
			str = str2;
			str2 = strstr(str, "\"");
			str2++;
			cookie_value_static = strdup(str2);
			str3 = strstr(cookie_value_static, "\"");
			*str3 = 0;
		}
	}
	cookie_type = cookie_type_static;
	cookie_value = cookie_value_static;
}



/* This slot is called when posting is done. */
/* see also slotPostMessage()                */ /* private slot */
void KitaWriteView::slotPostFinished( KIO::Job* )
{
    QString response;
    const char *response_ascii;

    // x-euc-jp & euc-jp
    if ( m_array.contains( "euc-jp" ) ) {
        response = QTextCodec::codecForName( "eucJP" ) ->toUnicode( m_array );
    } else {
        response = QTextCodec::codecForName( "sjis" ) ->toUnicode( m_array );
    }

    int retcode = resultCode( response );

    switch ( retcode ) {

    case K2ch_True:

        /* save log */
        logPostMessage();

        /* clear message */
        setMessage( QString::null );

        /* reload thread */
        if ( m_mode == WRITEMODE_NORMAL ) {

            emit openURLRequestExt( m_datURL, KParts::URLArgs(), "kita_open_2chthread", 0 );

            emit closeCurrentTab(); /* to KitaWriteTabWidget */
        }

        /* reload board */
        else {

            QString board = Kita::BoardManager::boardURL( m_datURL );
            emit openURLRequestExt( board, KParts::URLArgs(), "kita_open_2chboard", 0 );

            emit closeCurrentTab(); /* to KitaWriteTabWidget */
        }

        break;

    case K2ch_Unknown:
    case K2ch_False:
    case K2ch_Check:
    case K2ch_Error:

        KMessageBox::error( 0, resultMessage( response ), resultTitle( response ) );

        break;

        /* eat cookie, then re-post message */
    case K2ch_Cookie:

	response_ascii = (const char *) QString::fromLocal8Bit(response).utf8();
	find_cookie_typevalue(response_ascii);

        if ( KMessageBox::questionYesNo( 0,
                                         QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_WRITECOOKIEMSG ),
                                         resultTitle( response ) )
                == KMessageBox::Yes ) {
            slotPostMessage();
        }

        break;

        /* re-post message to create the new thread */
    case K2ch_NewThread:

        if ( KMessageBox::questionYesNo( 0,
                                         QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_WRITENEWTHREAD ),
                                         resultTitle( response ) )
                == KMessageBox::Yes ) {
            m_bbscgi = Kita::BoardManager::boardRoot( m_datURL ) + "/test/subbbs.cgi";
            slotPostMessage();
        }

        break;

    default:
        break;
    }
}


/* update preview screen */ /* private slot */
void KitaWriteView::slotCurrentChanged ( QWidget * w )
{
    if ( w == bodyText ) {
        w->setFocus();
        return ;
    }
    if ( m_preview == NULL ) return ;

    int resnum;

    if ( m_mode == WRITEMODE_NORMAL ) resnum = Kita::DatManager::getReadNum( m_datURL ) + 1 ;
    else resnum = 1;

    QDateTime now = QDateTime::currentDateTime();
    QString bodystr = body();
    bodystr.replace( "<", "&lt;" ).replace( ">", "&gt;" ).replace( "\n", " <br> " );
    QString namestr = name();
    if ( namestr == NULL || namestr.length() == 0 ) namestr = Kita::BoardManager::getBBSDefaultName( m_datURL );
    QString rawData = namestr + "<>" + mail() + "<>"
                      + now.toString( "yy/MM/dd hh:mm:ss" )
                      + "<> " + bodystr + " <>";

    QString htmlstr = Kita::DatToHtml( rawData, resnum );
    m_preview->setInnerHTML( htmlstr );
}


/* private slot */
bool KitaWriteView::slotBodyTextChanged()
{
    int lines = bodyText->lines();
    int maxLines = Kita::BoardManager::getBBSMaxLine( m_datURL );
    int tmpln = bodyText->length();
    int length = m_cp932Codec->fromUnicode( bodyText->text(), tmpln ).length();

    /* replace '\n' -> " <br> ", '>' -> "&lt;", and etc. */
    length += ( ( bodyText->text().contains( '\n' ) ) * 5
                + ( bodyText->text().contains( '>' ) ) * 3
                + ( bodyText->text().contains( '<' ) ) * 3
              );

    int maxLength = Kita::BoardManager::getBBSMsgCount( m_datURL );

    QString str;
    str = QString().setNum( lines ) + "/" + ( maxLines != 0 ? QString().setNum( maxLines ) : QString( "--" ) );
    str += " | " + QString().setNum( length ) + "/" + ( maxLength != 0 ? QString().setNum( maxLength ) : QString( "--" ) );
    lengthLabel->setText( str );

    if ( maxLines != 0 && lines > maxLines ) return FALSE;
    if ( maxLength != 0 && length > maxLength ) return FALSE;

    return TRUE;
}


/* create posting message for 2ch */ /* private */
QString KitaWriteView::setPostStr()
{
    QString ret;

    int mib = m_cp932Codec->mibEnum();
    int serverTime;

    if ( m_mode == WRITEMODE_NORMAL ) serverTime = Kita::DatManager::getServerTime( m_datURL );
    /* TODO: get time from server */
    else serverTime = QDateTime::currentDateTime().toTime_t() - 10 * 60;

    /* submit ( URL encoded ) */
    if ( m_mode == WRITEMODE_NORMAL ) ( ret += "submit=" ) += "%8f%91%82%ab%8d%9e%82%de";  /* kakikomu */
    else ( ret += "submit=" ) += "%90V%8bK%83X%83%8c%83b%83h%8d%ec%90%ac"; /* sinki thread sakusei */

    /* subject */
    if ( m_mode == WRITEMODE_NEWTHREAD ) ( ret += "&subject=" ) += KURL::encode_string( threadName->text(), mib );

    /* from, mail, message, bbs */
    ( ret += "&FROM=" ) += KURL::encode_string( nameLine->text(), mib );
    ( ret += "&mail=" ) += KURL::encode_string( mailLine->text(), mib );
    ( ret += "&MESSAGE=" ) += KURL::encode_string( bodyText->text(), mib );
    ( ret += "&bbs=" ) += Kita::BoardManager::boardID( m_datURL );
    if (cookie_type) {
      ret += "&" ;
      ret += cookie_type;
      ret += "=";
      ret += cookie_value;
    }

    /* key */
    if ( m_mode == WRITEMODE_NORMAL ) ( ret += "&key=" ) += Kita::DatManager::threadID( m_datURL );

    /* time */
    ( ret += "&time=" ) += QString::number( serverTime );

    /* login */
    if ( Kita::DatManager::is2chThread( m_datURL ) && Kita::Account::isLogged() ) {
        ret += "&sid=";
        ret += KURL::encode_string( Kita::Account::getSessionID() );
    }

    return ret;
}


/* create posting message for JBBS */ /* private */ /* private */
QString KitaWriteView::setJBBSPostStr()
{
    QString ret;

    int mib = QTextCodec::codecForName( "eucJP" ) ->mibEnum();
    int serverTime;

    if ( m_mode == WRITEMODE_NORMAL ) serverTime = Kita::DatManager::getServerTime( m_datURL );
    /* TODO: get time from server */
    else serverTime = QDateTime::currentDateTime().toTime_t() - 10 * 60;

    /* submit ( URL encoded ) */
    if ( m_mode == WRITEMODE_NORMAL ) ( ret += "submit=" ) += "%8f%91%82%ab%8d%9e%82%de";  /* kakikomu */
    else ( ret += "submit=" ) += "%90V%8BK%8F%91%82%AB%8D%9E%82%DD"; /* sinki kakikomi */

    /* NAME, MAIL, MESSAGE, BBS */
    ( ret += "&NAME=" ) += KURL::encode_string( nameLine->text(), mib );
    ( ret += "&MAIL=" ) += KURL::encode_string( mailLine->text(), mib );
    ( ret += "&MESSAGE=" ) += KURL::encode_string( bodyText->text(), mib ).replace( ";", "%3B" );
    ( ret += "&BBS=" ) += Kita::BoardManager::boardID( m_datURL ).section( "/", 1, 1 );

    /* DIR */
    ( ret += "&DIR=" ) += Kita::BoardManager::boardID( m_datURL ).section( "/", 0, 0 );

    /* SUBJECT */
    if ( m_mode == WRITEMODE_NEWTHREAD ) ( ret += "&SUBJECT=" ) += KURL::encode_string( threadName->text(), mib );

    /* KEY */
    if ( m_mode == WRITEMODE_NORMAL ) ( ret += "&KEY=" ) += Kita::DatManager::threadID( m_datURL );

    /* TIME */
    ( ret += "&TIME=" ) += QString::number( serverTime );


    return ret;
}


/* create posting message for Flash CGI */ /* private */ /* private */
QString KitaWriteView::setFlashCGIPostStr()
{
    QString ret;

    int mib = m_cp932Codec->mibEnum();

    /* submit ( URL encoded ) */
    if ( m_mode == WRITEMODE_NORMAL ) ( ret += "submit=" ) += "%8f%91%82%ab%8d%9e%82%de";  /* kakikomu */
    else ( ret += "submit=" ) += "%90V%8bK%83X%83%8c%83b%83h%8d%ec%90%ac"; /* sinki thread sakusei */

    /* subject */
    if ( m_mode == WRITEMODE_NEWTHREAD ) ( ret += "&subject=" ) += KURL::encode_string( threadName->text(), mib );

    /* from, mail, message, bbs */
    ( ret += "&FROM=" ) += KURL::encode_string( nameLine->text(), mib );
    ( ret += "&mail=" ) += KURL::encode_string( mailLine->text(), mib );
    ( ret += "&MESSAGE=" ) += KURL::encode_string( bodyText->text(), mib );
    ( ret += "&bbs=" ) += Kita::BoardManager::boardID( m_datURL );

    /* key */
    if ( m_mode == WRITEMODE_NORMAL ) ( ret += "&key=" ) += Kita::DatManager::threadID( m_datURL );

    return ret;
}


/* save post log */ /* private */
void KitaWriteView::logPostMessage()
{
    QString boardName = Kita::BoardManager::boardName( m_datURL );
    QString threadName = Kita::DatManager::threadName( m_datURL );
    QString threadURL = Kita::DatManager::threadURL( m_datURL );

    QDateTime now = QDateTime::currentDateTime();
    QString logPath = locateLocal( "appdata", "log.txt" );

    QFile file( logPath );
    if ( file.open( IO_WriteOnly | IO_Append ) ) {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        stream << "Date   : " << now.toString( "yyyy/MM/dd hh:mm:ss" ) << endl; // current time
        stream << "Board  : " << boardName << endl;
        stream << "Thread : " << threadName << endl;
        stream << "URL    : " << threadURL << endl;
        stream << "Name   : " << nameLine->text() << endl;
        stream << "Mail   : " << mailLine->text() << endl << endl;
        stream << bodyText->text() << endl;
        stream << "----------------------------------------" << endl;

        file.close();
    }
}


/* get result code from 2ch tag or title. */  /* private */
int KitaWriteView::resultCode( const QString& response ) const
{
    QRegExp regexp( "<!-- 2ch_X:(.*) -->" );
    int pos = regexp.search( response );

    if ( pos != -1 ) {
        /* get code from 2ch tag */
        QString k2ch_X = regexp.cap( 1 );

        if ( k2ch_X == "true" ) return K2ch_True;
        if ( k2ch_X == "false" ) return K2ch_False;
        if ( k2ch_X == "error" ) return K2ch_Error;
        if ( k2ch_X == "check" ) return K2ch_Check;
        if ( k2ch_X == "cookie" ) return K2ch_Cookie;

        return K2ch_Unknown;
    } else {
        /* get code from title */
        QString title = resultTitle( response );
        if ( title == QString::null ) return K2ch_Unknown;
        
        /* see also libkita/kita-utf8.h */
        QString errstr = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_WRITEERROR );
        QString truestr = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_WRITETRUE );
        QString ckstr = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_WRITECOOKIE );

        if ( title.contains( errstr ) ) return K2ch_Error;
        if ( title.contains( truestr ) ) return K2ch_True;
        if ( title.contains( ckstr ) ) return K2ch_Cookie;

        /* re-post new thread message. adhock... */
        if ( m_mode == WRITEMODE_NEWTHREAD
                    && response.contains( ckstr )
                    && response.contains( "subbbs.cgi" ) ) {
            return K2ch_NewThread;
        }

        /* for Flash CGI */
        if ( m_bbstype == Kita::Board_FlashCGI ) {
            if ( title.contains( "ERROR!!" ) ) {
                return K2ch_Error;
            } else {
                return K2ch_True;
            }
        }

        /* for JBBS.  adhoc... */
        if ( m_bbstype == Kita::Board_JBBS ) {

            // x-euc-jp & euc-jp
            if ( response.contains( "euc-jp" ) ) {
                return K2ch_True;
            } else {
                return K2ch_Error;
            }
        }
        
        return K2ch_Unknown;
    }
}


/* private */
QString KitaWriteView::resultMessage( const QString& response ) const
{
    QRegExp tags( "(<[^<]+>|</[^<]+>)" );
    
    if ( m_bbstype == Kita::Board_FlashCGI ) {

        QRegExp regexp( "<blockquote>(.*)</blockquote>" );
        int pos = regexp.search( response );
        if ( pos != -1 ) {
            return regexp.cap( 1 ).replace( "<br>", "\n" );
        }

        return QString::null;
    }

    {
        QRegExp bold_start( "<b>" );
        QRegExp bold_end( "</b>" );
        int startPos = bold_start.search( response );
        if ( startPos != -1 ) {
            startPos += 3;
            int endPos = bold_end.search( response );
            if ( endPos != -1 && startPos < endPos ) {
                QString message = response.mid( startPos, endPos - startPos );
                message.remove( tags );
                return message;
            }
        }
    }

    QRegExp body_regexp( "<body>([^<]*)</body>" );
    if ( body_regexp.search( response ) != -1 ) {
        return body_regexp.cap( 1 );
    }

    return QString::null;
}


/* private */
QString KitaWriteView::resultTitle( const QString& response ) const
{
    QRegExp regexp( "<title>(.*)</title>" );
    regexp.setCaseSensitive( FALSE );
    int pos = regexp.search( response );
    if ( pos != -1 ) {
        return regexp.cap( 1 );
    } else {
        return QString::null;
    }
}

/* create cookie header for 2ch.net */ /* private */
void KitaWriteView::set2chCookieStr( KIO::TransferJob* job )
{
    QString cookie = "Cookie: ";

    /* be */
    QRegExp host_2ch( ".+\\.2ch\\.net" );
    if ( host_2ch.search( m_bbscgi.host() ) != -1 && beBox->isChecked() ) {
        QString BeMailAddress = KitaConfig::BeMailAddress();
        QString BeAuthCode = KitaConfig::BeAuthCode();
        if ( BeMailAddress.length() > 0 && BeAuthCode.length() > 0 ) {
            cookie += "DMDM=" + BeMailAddress + "; ";
            cookie += "MDMD=" + BeAuthCode + "; ";
        }
    }

    job->addMetaData( "customHTTPHeader", cookie );
}
