/*
 * nicks.c
 *
 * this file contains code to implement nick name related stuff.
 *
 * written by Joshua J. Drake
 * copyright (c) 1998
 */

#include "irc.h"
#include "ircaux.h"
#include "nicks.h"
#include "dma.h"
#include "list.h"
#include "server.h"
#include "output.h"
#include "channels.h"
#include "numbers.h"

/* islegal: true if c is a legal nickname char anywhere but first char */
#define islegal(c) ((((c) >= 'A') && ((c) <= '}')) || \
                    (((c) >= '0') && ((c) <= '9')) || \
                     ((c) == '-') || ((c) == '_'))


u_char *old_nick = UNULL;

static	u_char	*randcrap _((int));

/*
 * checkNick()
 * checks to see if a nick name is legal.  it will check the first character first.
 * if it is legal we return from here to the first illegal character.  if the first
 * character is invalid, we return NULL.
 *
 **** rewritten, with help from do_nick_name() from the server code (2.8.5),
 **** phone, april 1993.
 * 
 * moved to nicks.c 11/13/98
 */
u_char *
check_nick(nick)
   u_char *nick;
{
   u_char *s;

   /* check the first character. */
   if (!nick || *nick == '-' || isdigit(*nick))
      return NULL;

   /* skip to the first illegal character. */
   for (s = nick; *s; s++)
      if (!islegal(*s) || isspace(*s))
         break;
   *s = '\0';

   /* if we have something, return it, otherwise forget it. */
   return *nick ? nick : NULL;
}

void
free_nick(nick)
    Nick **nick;
{
   Nick *tmp = *nick;
   
   dma_Free(&(tmp->nick));
   dma_Free(&(tmp->user));
   dma_Free(&(tmp->host));
   dma_Free(&(tmp->realname));
   dma_Free(&(tmp->server));
   dma_Free(&tmp);
}

/*
 * find a nick on a channel on a server
 * 
 * this will find a nick ptr via the channel or tmpchan variable..
 * the tmpchan is preferred as it doesn't require lookup_channel..
 * 
 */
Nick *
find_nick(u_char *nick, u_char *channel, int server, Channel *tmpchan)
{
   Channel *chan;

   if (!nick || (!tmpchan && !channel) || server < 0 || server >= number_of_servers)
     return NULL;
   
   /* tmpchan is preferred */
   if (tmpchan)
     {
	if (tmpchan->server != server)
	  {
	     put_error("findnick() received bad Channel pointer server!=%d", server);
	     return NULL;
	  }
	return (Nick *) find_in_list((List **)&tmpchan->nicks, nick, !USE_WILDCARDS);
     }
   
   /* okay fine, we'll look it up */
   chan = lookup_channel(channel, server, 0);
   if (!chan) /* we're not on the channel! */
     return NULL;
   
   /* return it! */
   return (Nick *) find_in_list((List **)&chan->nicks, nick, !USE_WILDCARDS);
}


/*
 * this will rotate your nickname characters and send it to the server
 */
#define MAX_AUTO_NICK_ATTEMPTS 6
void
rotate_nick(nick)
   u_char *nick;
{
   static int rot_att_cnt = 1;	/* count of how many rotations attempts in a row we've done */
   int nicklen = 0, i;

   if (!attempting_to_connect)
     return;
   
   /* put_info("passed in: %s (%s #%u)", nick, old_nick, rot_att_cnt); */

   /* same as we just passed out? */
   if (!my_stricmp(old_nick, nick))
     {
	rot_att_cnt++;
	if (rot_att_cnt == (MAX_AUTO_NICK_ATTEMPTS - 1))
	  {
	     /* one last shot!  lets try random */
	     dma_strcpy(&old_nick, randcrap(12));
	     send_to_server("NICK %s", old_nick);
	     return;
	  }
	else if (rot_att_cnt >= MAX_AUTO_NICK_ATTEMPTS)
	  {
	     /* absolute last resort: manual intervention */
	     reset_nickname();
	     rot_att_cnt = 1;
	     return;
	  }
     }
   else		/* if we're not looping, reset the attempt counter */
     rot_att_cnt = 1;

   /* rotate the nickname */
   nicklen = my_strlen(nick);
   dma_strcpy(&old_nick, nick);
   if (!old_nick)
     {
	reset_nickname(); /* bleh, must be out of memory */
	return;
     }

   for (i = 0; i < nicklen-1; i++)
     old_nick[i] = old_nick[i+1];
   old_nick[nicklen-1] = *nick;
   /* send it to the server */
   /* put_info("sending to server: %s", old_nick); */
   send_to_server("NICK %s", old_nick);
}


/*
 * return len of random isalpha()s
 */
static u_char *
randcrap(len)
   int len;
{
   static u_char rstr[128];
   u_char all[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ\\[]{}-_|^`";
   int i, t;
   long int r;
   float alen = my_strlen(all);

   srandom(time(NULL)/rand());
   r = random();
#ifndef _Windows
   srand(r*getpid());
#endif

   for (i = 0; i < len && i < sizeof(rstr)-1; i++)
     {
       t = (int)(alen * rand() / (RAND_MAX+1.0));
       // printf("RAND: %d %c\n", t, all[t]);
       rstr[i] = all[t];
     }
   rstr[i] = '\0';
   return rstr;
}

