/*
 * bans.c
 *
 * This source code implements methods and algorithms used to deal with channel bans.
 *
 * written by Joshua J. Drake
 * copyright (c) 1998
 * 
 * -= SuckSoft =-
 */

#include "irc.h"

#include "bans.h"
#include "channels.h"
#include "dma.h"
#include "server.h"
#include "output.h"
#include "ircaux.h"
#include "list.h"
#include "how_many.h"
#include "ninja.h"

/*
 * adds a ban to the cached ban list for a channel
 * accessed via Channel.bans
 */
void
add_ban(Channel *chan, u_char *ban, u_char *owner, u_char *bantime)
{
   Ban *new;

   /* if we have no channel or ban, report so and return */
   if (!chan || !ban)
     {
	put_error("add_ban() called with no channel or ban.");
	return;
     }
   
   /* try to remove the ban specified, if it was found, free the memory used. */
   if ((new = (Ban *) remove_from_list((List **)&(chan->bans), ban)) != NULL)
     {
	dma_Free(&(new->ban));
	dma_Free(&(new->owner));
	dma_Free(&new);
     }
   
   /* create a new ban entry */
   new = (Ban *) dma_Malloc(sizeof(Ban));
   /* pointless.. handled by dma_Malloc()
   new->ban = NULL;
   new->owner = NULL;
   new->time = 0;
    */
   if (bantime)
     new->time = my_atol(bantime);
   if (owner)
     dma_strcpy(&(new->owner), owner);
   dma_strcpy(&(new->ban), ban);
   
   /* add this new entry to the list. */
   add_to_list((List **)&(chan->bans), (List *)new);
}

/*
 * this is done after a special whois is done..
 * it bans a user
 */
void
ban_queue(WhoisStuff *stuff, u_char *nick, u_char *args)
{
   u_char *foo;

   /* check out the whois reply, make sure it returned something base a ban on. */
   if (!stuff || !stuff->nick || !nick || strcmp(stuff->user, "<UNKNOWN>") == 0 ||
       my_stricmp(stuff->nick, nick) || !args)
     {
        put_info("%s is not on irc.", nick);
        return;
     }

   /* skip over any bullshit characters before the username. */
   foo = stuff->user;
   if (foo && *foo && *foo == '~')
     foo++;
   if (foo && *foo)
     foo++;

   /* ban the mask */
   send_to_server("MODE %s +b *!*%s@%s", args, foo ? foo : empty_string, cluster(stuff->host));
}

/*
 * this is done after a special whois is done..
 * it unbans all bans on a channel matching the nick!user@host
 */
void
unban_queue(WhoisStuff *stuff, u_char *nick, u_char *args)
{
   /* check out the whois reply... */
   if (!stuff || !stuff->nick || !nick || strcmp(stuff->user, "<UNKNOWN>") == 0 ||
       my_stricmp(stuff->nick, nick) || !args)
     {
        put_info("%s is not on irc.", nick);
        return;
     }

    /* remove them bans. */
   remove_matching_bans(stuff->nick, stuff->user, stuff->host, args, 0);
}

/*
 * remove all bans matching nick!user@host on channel
 * if quiet, don't talk about the fact that we're doing it..
 */
void
remove_matching_bans(u_char *nick, u_char *user, u_char *host, u_char *channel, int quiet)
{
   Channel *ctmp;
   Ban *btmp;
   int found = 0, count = 0, mtch = 0, rmtch = 0;
   /* int nuhl, nml; */
   u_char nuh[512], tmpbuf[1024];

   /* if called there are invalid arguments, get out */
   if (!nick || !channel)
      return;

   /* try to find the specified channel, if it's not in the list or the user doesn't have
    * ops on it, get out.
    */
   if ((ctmp = lookup_channel(channel, from_server, 0)) == NULL || !(ctmp->status & CHAN_CHOP))
     {
        put_info("You don't have access to remove bans from %s", channel);
        return;
     }
   
   /* if a mask was specified, remove bans matching it. */
   if (my_index(nick, '@') && my_index(nick, '!'))
     my_strncpy(nuh, nick, sizeof(nuh)-1);
   else
     {
	/* if the user and host aren't there, get out */
	if (!user || !host)
	  {
	     if (!quiet)
	       put_error("invalid mask: %s", nick);
	     return;
	  }
	/* otherwise, we will remove bans matching nick!user@host */
	snprintf(nuh, sizeof(nuh)-1, "%s!%s@%s", nick, user, host);
     }
   nuh[sizeof(nuh)-1] = '\0';

   tmpbuf[0] = '\0';
   /* nuhl = my_strlen(nuh); */
   /* run through the bans and remove all bans matching the nuh */
   for (btmp = ctmp->bans; btmp; btmp = btmp->next)
     {
	/* if the nuh matches the ban or the ban matches the nuh.. */
	
	/* nml = my_strlen(btmp->ban); */
	mtch = match(btmp->ban, nuh);
	rmtch = match(nuh, btmp->ban);
	/* put_info("mtch: %d, rmtch: %d", mtch, rmtch); */
        if (mtch || rmtch)
          {
	     /* add this ban to the list to be removed. */
	     strmcat(tmpbuf, btmp->ban, sizeof(tmpbuf)-1);
	     strmcat(tmpbuf, " ", sizeof(tmpbuf)-1);
             count++;
          }
	/* only four bans can be removed at a time, so if we have four, we remove them and
	 * reset the list.
	 */
        if (count == 4)
          {
             send_to_server("MODE %s -bbbb %s", ctmp->channel, tmpbuf);
             tmpbuf[0] = '\0';
             count = 0;
	     found += 4;
          }
     }

   /* if there are any left over, we remove them too. */
   if (count)
     {
	send_to_server("MODE %s -%s %s", ctmp->channel, strfill('b', count), tmpbuf);
	found += count;
     }

   /* if we aren't supposed to be quiet, tell how many bans were removed. */
   if (!quiet)
     {
	if (!found)
	  put_info("No bans matching %s were found on %s", nuh, ctmp->channel);
     }
}

/*
 * make a nice mask for the hostname passed
 * ex: elite.net = elite.net
 *     1.1.1.1   = 1.1.1.*
 *     boom.zoom.com = *.zoom.com
 *     c123459-a.oak1.il.home.com = *.oak*.il.home.com
 * etc.
 * 
 * if '@' is in "hn" then it will be replaced by '*@' and 
 * anything before it will be removed...
 */
u_char *
cluster(u_char *hn)
{
   u_char *ptr = NULL, *ptr2;
   static u_char cltmp[512];
   int dots_in_host, ip = 0, clidx = 0;

   /* default to returning '*' */
   memset(cltmp, 0, sizeof(cltmp));
   
   /* if there is a hostname, try to mask it */
   if (hn)
     {
	/* got a user name? */
	ptr2 = strrchr(hn, '@');
	if (ptr2)
	  {
	     cltmp[clidx++] = '*';
	     *ptr2++ = '\0';
	     if (*hn)
	       {
		  hn++;
		  my_strmcat(cltmp, hn, sizeof(cltmp)-1);
		  clidx += my_strlen(hn);
	       }
	     cltmp[clidx++] = '@';
	     hn = ptr2;
	  }
	
	/* no more stuff? */
	if (!hn)
	  my_strmcat(cltmp, "ninja!irc@rulez", sizeof(cltmp)-1);
	else
	  {
	     dots_in_host = how_many(hn, '.');
	     /* its a hostname! */
	     if (dots_in_host <= 1)
	       my_strmcat(cltmp, hn, sizeof(cltmp)-1);
	     else
	       {
		  dma_strcpy(&ptr, hn);
		  switch (dots_in_host)
		    {
		     case 3:
		       /* if we have an ip address... */
		       if (isdigit(ptr[my_strlen(ptr) - 1]))
			 {
			    /* truncate after the last . to ban class c */
			    ptr2 = my_rindex(ptr, '.');
			    *(ptr2+1) = '\0';
			    my_strmcat(cltmp, ptr, sizeof(cltmp)-1);
			    my_strmcat(cltmp, "*", sizeof(cltmp)-1);
			    ip = 1;
			    break;
			 }
		     default:
		       /* hostname with 3 or more parts */
		       ptr2 = my_index(ptr, '.');
		       my_strmcat(cltmp, "*", sizeof(cltmp)-1);
		       my_strmcat(cltmp, ptr2, sizeof(cltmp)-1);
		       break;
		    }
		  dma_Free(&ptr);
		  
		  /* now replace digit strings in non-ips with '*' */
		  if (!ip)
		    {
		       int last_star = 0;
		       u_char *lptr;
		       
		       for (ptr2 = cltmp; ptr2 && *ptr2; ptr2++)
			 {
			    /* when we find a digit... */
			    if (isdigit(*ptr2))
			      {
				 /* ..skip to the end of the digits */
				 for (lptr = ptr2; lptr && *lptr && isdigit(*lptr); lptr++);
				 
				 /* did we run out of input? */
				 if (!lptr || !*lptr)
				   break;
				 
				 /* if there's no '*' on either side, 
				  * put a '*' in the place of the first digit
				  */
				 if (!last_star && *lptr != '*')
				   *ptr2++ = '*';
				 
				 /* we have a '*' in place, move the rest back */
				 memcpy(ptr2, lptr, MIN(my_strlen(lptr)+1, sizeof(cltmp)));
			      }
			    else if (*ptr2 == '*')
			      last_star = 1;
			    else
			      last_star = 0;
			 }
		    }
	       }
	  }
     }
   return cltmp;
}
