/*
 * ctcp.c: handles the client-to-client protocol(ctcp). 
 *
 * Written By Michael Sandrof 
 *
 * Copyright (c) 1990 Michael Sandrof.
 * Copyright (c) 1991, 1992 Troy Rollo.
 * Copyright (c) 1992-2001 Matthew R. Green.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "irc.h"
IRCII_RCSID("@(#)$eterna: ctcp.c,v 1.66 2001/08/12 16:24:59 mrg Exp $");

#ifndef _Windows
#include <pwd.h>
#endif /* _Windows */

#ifdef HAVE_UNAME
# include <sys/utsname.h>
#endif /* HAVE_UNAME */

#include "ircaux.h"
#include "hook.h"
#include "irccrypt.h"
#include "ctcp.h"
#include "vars.h"
#include "server.h"
#include "status.h"
#include "lastlog.h"
#include "ignore.h"
#include "output.h"
#include "window.h"
#include "dcc.h"
#include "channels.h"
#include "parse.h"

/* ninja includes */
#include "info.h"
#include "friends.h"
#include "dma.h"
#include "bans.h"
#include "mileston.h"

#ifdef HAVE_CRYPT
# ifdef HAVE_CRYPT_H
#  include <crypt.h>
# endif
#endif

static u_char FAR CTCP_Reply_Buffer[BIG_BUFFER_SIZE + 1] = "";

static void do_new_notice_ctcp _((u_char *, u_char *, u_char **, u_char *));

/* forward declarations for the built in CTCP functions */
static u_char *do_crypto _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_version _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_clientinfo _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_echo _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_userinfo _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_finger _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_time _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_atmosphere _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_dcc _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_utc _((CtcpEntry *, u_char *, u_char *, u_char *));

/* ninja friends list extensions */
static u_char *do_help _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_ident _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_pass _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_op _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_invite _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_chops _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_unban _((CtcpEntry *, u_char *, u_char *, u_char *));
static u_char *do_whoami _((CtcpEntry *, u_char *, u_char *, u_char *));

/* ninja misc extensions */
static u_char *do_nver _((CtcpEntry *, u_char *, u_char *, u_char *));

/* simple stuff to tell people (just makes do_* friends list stuff look
 * a bit cleaner, and to be called from multiple places
 */
static void tell_only_private _((u_char *));
static void tell_friend_usage _((u_char *, u_char *));
static void tell_no_access _((u_char *));
static void tell_not_this_chan _((u_char *, u_char *, u_char *));
static void tell_not_there _((u_char *, u_char *));
static void tell_already_there _((u_char *, u_char *));
static void tell_not_opped _((u_char *, u_char *));
static void tell_im_not_there _((u_char *, u_char *));
static void tell_wrong_password _((u_char *, u_char *));
static void tell_chops _((u_char *, u_char *));
static Friend *friend_ctcp_check _((CtcpEntry *, u_char *, u_char *, u_char **, u_char **, u_char *));

static u_char *tmpaway = UNULL;

static CtcpEntry ctcp_cmd[] = {
   {UP("VERSION"), UP("shows client type, version and environment"),
      CTCP_VERBOSE, do_version},
   {UP("CLIENTINFO"), UP("gives information about available CTCP commands"),
      CTCP_VERBOSE, do_clientinfo},
   {UP("USERINFO"), UP("returns user settable information"),
      CTCP_VERBOSE, do_userinfo},
#define CTCP_ERRMSG	3
   {UP("ERRMSG"), UP("returns error messages"),
      CTCP_VERBOSE, do_echo},
   {UP("FINGER"), UP("shows real name, login name and idle time of user"),
      CTCP_VERBOSE, do_finger},
   {UP("TIME"), UP("tells you the time on the user's host"),
      CTCP_VERBOSE, do_time},
   {UP("ACTION"), UP("contains action descriptions for atmosphere"),
      CTCP_SHUTUP, do_atmosphere},
   {UP("DCC"), UP("requests a direct_client_connection"),
      CTCP_SHUTUP | CTCP_NOREPLY, do_dcc},
   {UP("UTC"), UP("substitutes the local timezone"),
      CTCP_SHUTUP | CTCP_NOREPLY, do_utc},
   {UP("PING"), UP("returns the arguments it receives"),
      CTCP_VERBOSE, do_echo},
   {UP("ECHO"), UP("returns the arguments it receives"),
      CTCP_VERBOSE, do_echo},
     { CAST_STRING, UP("contains CAST-128 strongly encrypted data, CBC mode"),
	CTCP_SHUTUP | CTCP_NOREPLY, do_crypto },
#if 0
     { RIJNDAEL_STRING, UP("contains rijndael (AES) strongly encrypted data, CBC mode"),
	CTCP_SHUTUP | CTCP_NOREPLY, do_crypto },
#endif
     { SED_STRING, UP("contains simple weekly encrypted data"),
	CTCP_SHUTUP | CTCP_NOREPLY, do_crypto },
/* ninja extensions */
   {UP("HELP"), UP("returns ctcp help information"), 
      CTCP_VERBOSE, do_help},
   {UP("IDENT"), UP("adds host to friend's masks"), 
      CTCP_QUIET, do_ident},
   {UP("PASS"), UP("change friend's password"),
      CTCP_QUIET, do_pass},
   {UP("INVITE"), UP("invite friend to channel"),
      CTCP_QUIET, do_invite},
   {UP("OP"), UP("ops friend in a channel"),
      CTCP_QUIET, do_op},
   {UP("WHOAMI"), UP("show friend their info"),
      CTCP_QUIET, do_whoami},
   {UP("CHOPS"), UP("show friend channel ops/mode"), 
      CTCP_QUIET, do_chops},
   {UP("UNBAN"), UP("unbans friend from channel"), 
      CTCP_QUIET, do_unban},
   {UP("NVER"), UP("shows the ninja version"), 
      CTCP_SHUTUP, do_nver},
};
#define	NUMBER_OF_CTCPS (sizeof(ctcp_cmd) / sizeof(CtcpEntry))	/* XXX */

u_char *ctcp_type[] = {
   UP("PRIVMSG"),
   UP("NOTICE")
};

static u_char FAR ctcp_buffer[BIG_BUFFER_SIZE + 1] = "";

/* This is set to one if we parsed an SED */
int sed = 0;

/*
 * in_ctcp_flag is set to true when IRCII is handling a CTCP request.  This
 * is used by the ctcp() sending function to force NOTICEs to be used in any
 * CTCP REPLY 
 */
int in_ctcp_flag = 0;

/*
 * quote_it: This quotes the given string making it sendable via irc.  A
 * pointer to the length of the data is required and the data need not be
 * null terminated (it can contain nulls).  Returned is a malloced, null
 * terminated string.   
 */
u_char *
ctcp_quote_it(str, len)
   u_char *str;
   size_t len;
{
   u_char lbuf[BIG_BUFFER_SIZE + 1];
   u_char *ptr;
   int i;

   ptr = lbuf;
   for (i = 0; i < len; i++)
     {
	switch (str[i])
	  {
	     case CTCP_DELIM_CHAR:
		*(ptr++) = CTCP_QUOTE_CHAR;
		*(ptr++) = 'a';
		break;
	     case '\n':
		*(ptr++) = CTCP_QUOTE_CHAR;
		*(ptr++) = 'n';
		break;
	     case '\r':
		*(ptr++) = CTCP_QUOTE_CHAR;
		*(ptr++) = 'r';
		break;
	     case CTCP_QUOTE_CHAR:
		*(ptr++) = CTCP_QUOTE_CHAR;
		*(ptr++) = CTCP_QUOTE_CHAR;
		break;
	     case '\0':
		*(ptr++) = CTCP_QUOTE_CHAR;
		*(ptr++) = '0';
		break;
	     case ':':
		*(ptr++) = CTCP_QUOTE_CHAR;
	     default:
		*(ptr++) = str[i];
		break;
	  }
     }
   *ptr = '\0';
   str = (u_char *) 0;
   malloc_strcpy(&str, lbuf);
   return (str);
}

/*
 * ctcp_unquote_it: This takes a null terminated string that had previously
 * been quoted using ctcp_quote_it and unquotes it.  Returned is a malloced
 * space pointing to the unquoted string.  The len is modified to contain
 * the size of the data returned. 
 */
u_char *
ctcp_unquote_it(str, len)
   u_char *str;
   size_t *len;
{
   u_char *lbuf;
   u_char *ptr;
   u_char c;
   int i, new_size = 0;

   lbuf = (u_char *) new_malloc(sizeof(u_char) * *len);
   ptr = lbuf;
   i = 0;
   while (i < *len)
     {
	if ((c = str[i++]) == CTCP_QUOTE_CHAR)
	  {
	     switch (c = str[i++])
	       {
		  case CTCP_QUOTE_CHAR:
		     *(ptr++) = CTCP_QUOTE_CHAR;
		     break;
		  case 'a':
		     *(ptr++) = CTCP_DELIM_CHAR;
		     break;
		  case 'n':
		     *(ptr++) = '\n';
		     break;
		  case 'r':
		     *(ptr++) = '\r';
		     break;
		  case '0':
		     *(ptr++) = '\0';
		     break;
		  default:
		     *(ptr++) = c;
		     break;
	       }
	  }
	else
	   *(ptr++) = c;
	new_size++;
     }
   *len = new_size;
   return (lbuf);
}

/*
 * do_crypto: performs the ecrypted data trasfer for ctcp.  Returns in a
 * malloc string the decryped message (if a key is set for that user) or the
 * text "[ENCRYPTED MESSAGE]" 
 */
static u_char *
do_crypto(ctcp, from, to, args)
   CtcpEntry *ctcp;
   u_char *from, *to, *args;
{
   crypt_key *key;
   u_char *crypt_who, *msg;
   u_char *ret = NULL;

   if (is_channel(to))
      crypt_who = to;
   else
      crypt_who = from;
   if ((key = is_crypted(crypt_who)) && (msg = crypt_msg(args, key, 0)))
     {
	/* this doesn't work ...
	   ... when it does, set this to 0 ... */
	static int the_youth_of_america_on_elle_esse_dee = 1;

	malloc_strcpy(&ret, msg);
	/*
	 * since we are decrypting, run it through do_ctcp() again
	 * to detect embeded CTCP messages, in an encrypted message.
	 * we avoid recusing here more than once.
	 */
	if (the_youth_of_america_on_elle_esse_dee++ == 0)
	   ret = do_ctcp(from, to, ret);
	the_youth_of_america_on_elle_esse_dee--;
	sed = 1;
     }
   else
      malloc_strcpy(&ret, UP("[ENCRYPTED MESSAGE]"));
   return (ret);
}

/*
 * do_clientinfo: performs the CLIENTINFO CTCP.  If cmd is empty, returns the
 * list of all CTCPs currently recognized by IRCII.  If an arg is supplied,
 * it returns specific information on that CTCP.  If a matching CTCP is not
 * found, an ERRMSG ctcp is returned 
 */
static u_char *
do_clientinfo(ctcp, from, to, cmd)
   CtcpEntry *ctcp;
   u_char *from, *to, *cmd;
{
   int i;
   u_char *ucmd = (u_char *) 0;
   u_char buffer[BIG_BUFFER_SIZE];

   if (get_int_var(CLOAK_VAR))
     return NULL;
   if (cmd && *cmd)
     {
	malloc_strcpy(&ucmd, cmd);
	upper(ucmd);
	for (i = 0; i < NUMBER_OF_CTCPS; i++)
	  {
	     if (my_strcmp(ucmd, ctcp_cmd[i].name) == 0)
	       {
		  send_ctcp_reply(from, ctcp->name, "%s %s", ctcp_cmd[i].name, ctcp_cmd[i].desc);
		  return NULL;
	       }
	  }
	send_ctcp_reply(from, ctcp_cmd[CTCP_ERRMSG].name, "%s: %s is not a valid function", ctcp->name, cmd);
     }
   else
     {
	*buffer = '\0';
	for (i = 0; i < NUMBER_OF_CTCPS; i++)
	  {
	     my_strmcat(buffer, ctcp_cmd[i].name, BIG_BUFFER_SIZE);
	     my_strmcat(buffer, " ", BIG_BUFFER_SIZE);
	  }
	send_ctcp_reply(from, ctcp->name, "%s :Use CLIENTINFO <COMMAND> to get more specific information", buffer);
     }
   return NULL;
}

/* do_version: does the CTCP VERSION command */
static u_char *
do_version(ctcp, from, to, cmd)
   CtcpEntry *ctcp;
   u_char *from, *to, *cmd;
{
   if (get_int_var(CLOAK_VAR))
     return NULL;
   else
     {
#if defined(PARANOID)
	send_ctcp_reply(from, ctcp->name, "Ninja IRC user");
#else
	u_char *tmp;
# if defined(HAVE_UNAME)
	struct utsname un;
	u_char *the_unix, *the_version;
	
	if (uname(&un) < 0)
	  {
	     the_version = empty_string;
	     the_unix = UP("unknown");
	  }
	else
	  {
	     the_version = UP(un.release);
	     the_unix = UP(un.sysname);
	  }
	send_ctcp_reply(from, ctcp->name, "Ninja IRC v%s for %s %s [%s], %s", irc_version, 
			the_unix, the_version,
			ninja_release,
# else
#  ifdef _Windows
        send_ctcp_reply(from, ctcp->name, "Ninja IRC v%s MS-Windows [%s], %s", irc_version, ninja_release,
#  else
        send_ctcp_reply(from, ctcp->name, "Ninja IRC v%s *NIX [%s], %s", irc_version, ninja_release,
#  endif /* _Windows */
# endif /* HAVE_UNAME */
			(tmp = get_string_var(CLIENTINFO_VAR)) ? tmp : UP(IRCII_COMMENT));
#endif /* PARANOID */
        return NULL;
     }
}

#if 0 // make jed happy	
))}
#endif	

/* do_time: does the CTCP TIME command --- done by Veggen */
static u_char *
do_time(ctcp, from, to, cmd)
   CtcpEntry *ctcp;
   u_char *from, *to, *cmd;
{
   time_t tm = time((time_t *) 0);
   u_char *s, *t = (u_char *) ctime(&tm);

   if (get_int_var(CLOAK_VAR))
     return NULL;
   if ((u_char *) 0 != (s = my_index(t, '\n')))
     *s = '\0';
   send_ctcp_reply(from, ctcp->name, "%s", t);
   return NULL;
}

/* do_userinfo: does the CTCP USERINFO command */
static u_char *
do_userinfo(ctcp, from, to, cmd)
   CtcpEntry *ctcp;
   u_char *from, *to, *cmd;
{
   if (get_int_var(CLOAK_VAR))
     return NULL;
   send_ctcp_reply(from, ctcp->name, "%s", get_string_var(USER_INFO_VAR));
   return NULL;
}

/*
 * do_echo: does the CTCP ECHO, CTCP ERRMSG and CTCP PING commands. Does
 * not send an error for ERRMSG and if the CTCP was sent to a channel.
 */
static u_char *
do_echo(ctcp, from, to, cmd)
   CtcpEntry *ctcp;
   u_char *from, *to, *cmd;
{
   if (get_int_var(CLOAK_VAR))
     return NULL;
   if (!is_channel(to) || my_strncmp(cmd, "ERRMSG", 6))
     send_ctcp_reply(from, ctcp->name, "%s", cmd);
   return NULL;
}

static u_char *
do_finger(ctcp, from, to, cmd)
   CtcpEntry *ctcp;
   u_char *from, *to, *cmd;
{
   if (get_int_var(CLOAK_VAR))
     return NULL;
   else
     {
#if defined(PARANOID)
	send_ctcp_reply(from, ctcp->name, "Ninja IRC user");
#else
	struct passwd *pwd;
	time_t diff;
	unsigned uid;
	u_char c;

	/*
	 * sojge complained that ircII says 'idle 1 seconds'
	 * well, now he won't ever get the chance to see that message again
	 *   *grin*  ;-)    -lynx
	 *
	 * Made this better by saying 'idle 1 second'  -phone
	 */
	
	diff = time(0) - idle_time;
	c = (diff == 1) ? ' ' : 's';
	/* XXX - fix me */
# ifdef _Windows
	send_ctcp_reply(from, ctcp->name, "Ninja IRC For MS-Windows User Idle %d second%c", (int) diff, c);
# else
	uid = getuid();
#  ifdef DAEMON_UID
	if (uid != DAEMON_UID)
	  {
#  endif /* DAEMON_UID */
	     if ((pwd = getpwuid(uid)) != NULL)
	       {
		  u_char *tmp;

#  ifndef GECOS_DELIMITER
#   define GECOS_DELIMITER ','
#  endif /* GECOS_DELIMITER */
		  if ((tmp = my_index(pwd->pw_gecos, GECOS_DELIMITER)) != NULL)
		    *tmp = '\0';
		  send_ctcp_reply(from, ctcp->name, "%s (%s@%s) Idle %d second%c", pwd->pw_gecos, pwd->pw_name, hostname, (int) diff, c);
	       }
#  ifdef DAEMON_UID
	  }
	else
	  send_ctcp_reply(from, ctcp->name, "Ninja IRC Telnet User (%s) Idle %d second%c", realname, (int) diff, c);
#  endif /* DAEMON_UID */
# endif	/* _Windows */
#endif /* PARANOID */
	return NULL;
     }
}

/*
 * do_atmosphere: does the CTCP ACTION command --- done by lynX
 * Changed this to make the default look less offensive to people
 * who don't like it and added a /on ACTION. This is more in keeping
 * with the design philosophy behind IRCII
 */
static u_char *
do_atmosphere(ctcp, from, to, cmd)
   CtcpEntry *ctcp;
   u_char *from, *to, *cmd;
{
   if (cmd && *cmd)
     {
	int old;

	save_message_from();
	old = set_lastlog_msg_level(LOG_ACTION);
	if (is_channel(to))
	  {
	     message_from(to, LOG_ACTION);
	     if (do_hook(ACTION_LIST, "%s %s %s", from, to, cmd))
	       {
		  if (is_current_channel(to, parsing_server_index, 0))
		     put_it("* %s %s", from, cmd);
		  else
		     put_it("* %s:%s %s", from, to, cmd);
	       }
	  }
	else
	  {
	     if ('=' == *from)
	       {
		  set_lastlog_msg_level(LOG_DCC);
		  message_from(from + 1, LOG_DCC);
	       }
	     else
		message_from(from, LOG_ACTION);
	     if (do_hook(ACTION_LIST, "%s %s %s", from, to, cmd))
		put_it("*> %s %s", from, cmd);
	  }
	set_lastlog_msg_level(old);
	restore_message_from();
     }
   return NULL;
}

/*
 * do_dcc: Records data on an incoming DCC offer. Makes sure it's a
 *	user->user CTCP, as channel DCCs don't make any sense whatsoever
 */
static u_char *
do_dcc(ctcp, from, to, args)
   CtcpEntry *ctcp;
   u_char *from, *to, *args;
{
   u_char *type;
   u_char *description;
   u_char *inetaddr;
   u_char *port;
   u_char *size;
   /* u_char *chksum; */
   
   type = next_arg(args, &args);
   if (my_stricmp(to, get_server_nickname(parsing_server_index)))
     {
	put_error("DCC: Invalid request (%s) from %s to %s.", type ? type : q_mark, from, to);
	return NULL;
     }
   if (!type
       || !(description = next_arg(args, &args)) 
       || !(inetaddr = next_arg(args, &args)) 
       || !(port = next_arg(args, &args)))
     return NULL;
   size = next_arg(args, &args);
   /* chksum = next_arg(args, &args); */
   register_dcc_offer(from, type, description, inetaddr, port, size);
   return NULL;
}

/*
 * do_utc: converts `UTC <number>' into a date string, for
 # inclusion into the normal chat stream.
 */
static u_char *
do_utc(ctcp, from, to, args)
   CtcpEntry *ctcp;
   u_char *from, *to, *args;
{
   time_t tm;
   u_char *date = NULL;

   if (get_int_var(CLOAK_VAR))
     return NULL;
   if (!args || !*args)
      return NULL;
   tm = my_atol(args);
   malloc_strcpy(&date, UP(ctime(&tm)));
   date[my_strlen(date) - 1] = '\0';
   return date;
}

/*
 * do_ctcp: handles the client to client protocol embedded in PRIVMSGs.  Any
 * such messages are removed from the original str, so after do_ctcp()
 * returns, str will be changed
 */
u_char *
do_ctcp(from, to, str)
   u_char *from, *to, *str;
{
   int i = 0, ctcp_flag = 1;
   u_char *end, *cmd, *args, *ptr;
   u_char *arg_copy = NULL;
   int flag, ch_flag;
   int messages = 0;
   time_t curtime = time(NULL);

   flag = double_ignore(from, FromUserHost, IGNORE_CTCPS);
   ch_flag = double_ignore(to, NULL, IGNORE_CTCPS);

   if (!in_ctcp_flag)
      in_ctcp_flag = 1;
   *ctcp_buffer = '\0';
   while ((cmd = my_index(str, CTCP_DELIM_CHAR)) != NULL)
     {
	if (messages > 3)
	   break;
	*(cmd++) = '\0';
	my_strcat(ctcp_buffer, str);
	if ((end = my_index(cmd, CTCP_DELIM_CHAR)) != NULL)
	  {
	     messages++;
	     if (flag == IGNORED)
		continue;
	     if (ch_flag == IGNORED)
		continue;
	     *(end++) = '\0';
	     if ((args = my_index(cmd, ' ')) != NULL)
		*(args++) = '\0';
	     else
		args = empty_string;
	     /* Skip leading : for arguments */
	     if (*args == ':')
		++args;
	     malloc_strcpy(&arg_copy, args);
	     for (i = 0; i < NUMBER_OF_CTCPS; i++)
	       {
		  if (my_strcmp(cmd, ctcp_cmd[i].name) == 0)
		    {
		       /* protect against global (oper) messages */
		       if (*to != '$' && !(*to == '#' && !lookup_channel(to, parsing_server_index, CHAN_NOUNLINK)))
			 {
			    ptr = ctcp_cmd[i].func(&ctcp_cmd[i], from, to, arg_copy);
			    if (ptr)
			      {
				 my_strcat(ctcp_buffer, ptr);
				 new_free(&ptr);
			      }
			 }
		       ctcp_flag = ctcp_cmd[i].flag;
		       cmd = ctcp_cmd[i].name;
		       break;
		    }
	       }
	     new_free(&arg_copy);
	     if (in_ctcp_flag == 1 && do_hook(CTCP_LIST, "%s %s %s %s", from, to, cmd, args) && get_int_var(VERBOSE_CTCP_VAR))
	       {
		  int lastlog_level;

		  save_message_from();
		  lastlog_level = set_lastlog_msg_level(LOG_CTCP);
		  if (is_channel(to))
		    message_from(to, LOG_CTCP);
		  else
		    message_from((u_char *) 0, LOG_CTCP);
		  if (i == NUMBER_OF_CTCPS)
		    {
		       if (*to == '#' || *to == '&')
			 put_info("Unknown CTCP %s from %s(%s@%s) to %s%s%s", cmd, from,
				  FromUser, FromHost, to,
				  *args ? UP(": ") : empty_string, args);
		       else
			 put_info("Unknown CTCP %s from %s(%s@%s)%s%s", cmd, from,
				  FromUser, FromHost,
				  *args ? UP(": ") : empty_string, args);
		    }
		  else if (ctcp_flag & CTCP_VERBOSE)
		    {
		       if (my_stricmp(to, get_server_nickname(parsing_server_index)))
			 put_info("CTCP %s from %s(%s@%s) to %s%s%s%s", cmd, from, 
				  FromUser, FromHost, to,
				  *args ? UP(": ") : empty_string, args,
				  tmpaway ? tmpaway : empty_string);
		       else
			 put_info("CTCP %s from %s(%s@%s)%s%s%s", cmd, from,
				  FromUser, FromHost, 
				  *args ? UP(": ") : empty_string, args,
				  tmpaway ? tmpaway : empty_string);
		       dma_Free(&tmpaway);
		    }
		  else if (ctcp_flag & CTCP_QUIET)
		    {
		       if (my_stricmp(to, get_server_nickname(parsing_server_index)))
			 put_info("CTCP %s from %s(%s@%s) to %s%s", cmd, from,
				  FromUser, FromHost, to,
				  tmpaway ? tmpaway : empty_string);
		       else
			 put_info("CTCP %s from %s(%s@%s)%s", cmd, from,
				  FromUser, FromHost, 
				  tmpaway ? tmpaway : empty_string);
		       dma_Free(&tmpaway);
		    }
		  set_lastlog_msg_level(lastlog_level);
		  restore_message_from();
	       }
	     str = end;
	  }
	else
	  {
	     my_strcat(ctcp_buffer, CTCP_DELIM_STR);
	     str = cmd;
	  }
     }
   if (in_ctcp_flag == 1)
      in_ctcp_flag = 0;
   if (CTCP_Reply_Buffer && *CTCP_Reply_Buffer)
     {
#ifdef PARANOID
	/*
	 * paranoid users don't want to send ctcp replies to
	 * requests send to channels, probably...
	 */
	if (is_channel(to))
	   goto clear_ctcp_reply_buffer;
#endif
	/*
	 * Newave ctcp flood protection : each time you are requested to send
	 * more than CTCP_REPLY_FLOOD_SIZE bytes in CTCP_REPLY_BACKLOG_SECONDS
	 * no ctcp replies will be done for CTCP_REPLY_IGNORE_SECONDS.
	 * Current default is 256 bytes/ 5s/ 10s
	 * This is a sliding window, i.e. you can't get caught sending too much
	 * because of a 5s boundary, and the checking is still active even if
	 * you don't reply anymore.
	 */

	if (*from == '=')
	   send_ctcp(ctcp_type[CTCP_NOTICE], from, NULL, "%s", CTCP_Reply_Buffer);
	else
	  {
	     Server *cur_serv = &server_list[parsing_server_index];
	     int no_reply,
		no_flood = get_int_var(NO_CTCP_FLOOD_VAR),
		delta = cur_serv->ctcp_last_reply_time ? curtime - cur_serv->ctcp_last_reply_time : 0,
		size = 0,
		was_ignoring = cur_serv->ctcp_flood_time != 0,
		crbs = get_int_var(CTCP_REPLY_BACKLOG_SECONDS_VAR),
		crfs = get_int_var(CTCP_REPLY_FLOOD_SIZE_VAR), cris = get_int_var(CTCP_REPLY_IGNORE_SECONDS_VAR);

	     cur_serv->ctcp_last_reply_time = curtime;

	     if (delta)
	       {
		  for (i = crbs - 1; i >= delta; i--)
		     cur_serv->ctcp_send_size[i] = cur_serv->ctcp_send_size[i - delta];
		  for (i = 0; i < delta && i < crbs; i++)
		     cur_serv->ctcp_send_size[i] = 0;
	       }

	     cur_serv->ctcp_send_size[0] += my_strlen(CTCP_Reply_Buffer);

	     for (i = 0; i < crbs; i++)
		size += cur_serv->ctcp_send_size[i];
	     if (size >= crfs)
		cur_serv->ctcp_flood_time = curtime;

	     no_reply = cur_serv->ctcp_flood_time && (curtime <= cur_serv->ctcp_flood_time + cris);

	     if (no_flood && get_int_var(VERBOSE_CTCP_VAR))
	       {
		  save_message_from();
		  message_from((u_char *) 0, LOG_CTCP);
		  if (no_reply && was_ignoring == 0)
		    say("CTCP flood from %s(%s@%s) detected - suspending replies",
			from, FromUser, FromHost);
		  else if (no_reply == 0 && was_ignoring)
		    say("CTCP reply suspending time elapsed - replying normally");
		  restore_message_from();
	       }
	     if (no_flood == 0 || no_reply == 0)
	       {
		  cur_serv->ctcp_flood_time = 0;
		  send_ctcp(ctcp_type[CTCP_NOTICE], from, NULL, "%s", CTCP_Reply_Buffer);
	       }
	  }
#ifdef PARANOID
      clear_ctcp_reply_buffer:
#endif
	*CTCP_Reply_Buffer = '\0';
     }
   if (*str)
      my_strcat(ctcp_buffer, str);
   return (ctcp_buffer);
}

u_char *
do_notice_ctcp(from, to, str)
   u_char *from, *to, *str;
{
   u_char *cmd;

   in_ctcp_flag = -1;
   *ctcp_buffer = '\0';
   /*
    * The following used to say "While". It now says "if" because people
    * Started using CTCP ERRMSG replies to CTCP bomb. The effect of this
    * is that IRCII users can only send one CTCP/message if they expect a
    * reply. This shouldn't be a problem as that is the way IRCII operates
    *
    * Changed this behavouir to follow NO_CTCP_FLOOD
    */

   if (get_int_var(NO_CTCP_FLOOD_VAR))
     {
	if ((cmd = my_index(str, CTCP_DELIM_CHAR)) != NULL)
	   do_new_notice_ctcp(from, to, &str, cmd);
     }
   else
      while ((cmd = my_index(str, CTCP_DELIM_CHAR)) != NULL)
	 do_new_notice_ctcp(from, to, &str, cmd);
   in_ctcp_flag = 0;
   my_strcat(ctcp_buffer, str);
   return (ctcp_buffer);
}

static void
do_new_notice_ctcp(from, to, str, cmd)
   u_char *from, *to, **str, *cmd;
{
   u_char *end, *args, *ptr, *arg_copy = NULL;
   int flags, i, lastlog_level;

   flags = 0;
   *(cmd++) = '\0';
   my_strcat(ctcp_buffer, *str);
   if ((end = my_index(cmd, CTCP_DELIM_CHAR)) != NULL)
     {
	*(end++) = '\0';
	if ((args = my_index(cmd, ' ')) != NULL)
	   *(args++) = '\0';
	malloc_strcpy(&arg_copy, args);
	for (i = 0; i < NUMBER_OF_CTCPS; i++)
	  {
	     if ((my_strcmp(cmd, ctcp_cmd[i].name) == 0) && ctcp_cmd[i].flag & CTCP_NOREPLY)
	       {
		  if ((ptr = ctcp_cmd[i].func(&(ctcp_cmd[i]), from, to, arg_copy)) != NULL)
		    {
		       my_strcat(ctcp_buffer, ptr);
		       new_free(&ptr);
		       flags = ctcp_cmd[i].flag;
		    }
		  break;
	       }
	  }
	new_free(&arg_copy);
	if (!args)
	   args = empty_string;
	if (do_hook(CTCP_REPLY_LIST, "%s %s %s", from, cmd, args) && !(flags & CTCP_NOREPLY))
	  {
	     if (!my_strcmp(cmd, "PING"))
	       {
		  u_char buf[64], *arg0;
		  struct timeval tod;
		  time_t rtime;

		  gettimeofday(&tod, NULL);
		  if (args && (arg0 = next_arg(args, &args)))
		    {
		       u_char *arg1;
		       
		       rtime = my_atol(arg0);
		       arg1 = next_arg(args, &args);
		       if (!arg1)
			 {
			    rtime = tod.tv_sec - rtime;
			    snprintf(CP(buf), sizeof(buf)-1, "%ld second%s", (long) rtime, PLURAL(rtime));
			 }
		       else
			 {
			    time_t rt2 = my_atol(arg1);
			    double ftv;
			    
			    tod.tv_sec -= rtime;
			    if (tod.tv_usec >= rt2)
			      tod.tv_usec -= rt2;
			    else
			      {
				 tod.tv_usec = tod.tv_usec - rt2 + 1000000;
				 tod.tv_sec --;
			      }
			    ftv = (double)tod.tv_sec + ((double)tod.tv_usec / 1000000.0);
			    snprintf(CP(buf), sizeof(buf)-1, "%.6g second%s", ftv, PLURAL(ftv));
			 }
		       buf[sizeof(buf)-1] = '\0';
		    }
		  else
		    my_strcpy(buf, "0 seconds"); /* safe! */
		  args = buf;
	       }
	     save_message_from();
	     lastlog_level = set_lastlog_msg_level(LOG_CTCP);
	     message_from((u_char *) 0, LOG_CTCP);
	     say("CTCP %s reply from %s: %s", cmd, from, args);
	     set_lastlog_msg_level(lastlog_level);
	     restore_message_from();
	  }
	*str = end;
     }
   else
     {
	my_strcat(ctcp_buffer, CTCP_DELIM_STR);
	*str = cmd;
     }
}

/* in_ctcp: simply returns the value of the ctcp flag */
int
in_ctcp()
{
   return (in_ctcp_flag);
}

/* These moved here because they belong here - phone */

/*
 * send_ctcp: A simply way to send CTCP queries.   if the datatag
 * is NULL, we must have already formatted the ctcp reply (it has the
 * ctcp delimiters), so don't add them again, etc.
 */
void
#ifdef HAVE_STDARG_H
send_ctcp(u_char * type, u_char * to, u_char * datatag, char *format, ...)
{
   va_list vl;
#else
send_ctcp(type, to, datatag, format, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
   u_char *type, *to, *datatag;
   char *format;
   char *arg0, *arg1, *arg2, *arg3, *arg4, *arg5, *arg6, *arg7, *arg8, *arg9;
{
#endif /* HAVE_STDARG_H */
   u_char putbuf[BIG_BUFFER_SIZE + 1], sendbuf[BIG_BUFFER_SIZE + 1];
   u_char *sendp;

   if (in_on_who)
      return;			/* Silently drop it on the floor */
   if (format)
     {
#ifdef HAVE_STDARG_H
	va_start(vl, format);
	vsnprintf(CP(putbuf), sizeof(putbuf)-1, format, vl);
	va_end(vl);
#else
	snprintf(CP(putbuf), sizeof(putbuf)-1, format, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
#endif /* HAVE_STDARG_H */
	putbuf[sizeof(putbuf)-1] = '\0';

	if (datatag)
	  {
	     snprintf(CP(sendbuf), sizeof(sendbuf)-1, "%c%s %s%c", CTCP_DELIM_CHAR, datatag, putbuf, CTCP_DELIM_CHAR);
	     sendp = sendbuf;
	  }
	else
	   sendp = putbuf;
     }
   else
     {
	snprintf(CP(sendbuf), sizeof(sendbuf)-1, "%c%s%c", CTCP_DELIM_CHAR, datatag, CTCP_DELIM_CHAR);
	sendp = sendbuf;
     }
   sendbuf[sizeof(sendbuf)-1] = '\0';

   /*
    * ugh, special case dcc because we don't want to go through
    * send_text in it's current state.  XXX - fix send_text to
    * deal with ctcp's as well.
    */
   if (*to == '=')
      dcc_message_transmit(to + 1, sendp, DCC_CHAT, 0);
   else
      send_to_server("%s %s :%s", type, to, sendp);
}
   
/*
 * send_ctcp_notice: A simply way to send CTCP replies.   I put this here
 * rather than in ctcp.c to keep my compiler quiet 
 */
void
#ifdef HAVE_STDARG_H
send_ctcp_reply(u_char * to, u_char * datatag, char *format, ...)
{
   va_list vl;
#else
send_ctcp_reply(to, datatag, format, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
   u_char *to, *datatag;
   char *format;
   char *arg0, *arg1, *arg2, *arg3, *arg4, *arg5, *arg6, *arg7, *arg8, *arg9;
{
#endif /* HAVE_STDARG_H */
   u_char putbuf[BIG_BUFFER_SIZE + 1];

   if (in_on_who)
      return;			/* Silently drop it on the floor */
   if (to && (*to == '='))
      return;			/* don't allow dcc replies */
   my_strmcat(CTCP_Reply_Buffer, "\001", BIG_BUFFER_SIZE);
   my_strmcat(CTCP_Reply_Buffer, datatag, BIG_BUFFER_SIZE);
   my_strmcat(CTCP_Reply_Buffer, " ", BIG_BUFFER_SIZE);
   if (format)
     {
#ifdef HAVE_STDARG_H
	va_start(vl, format);
	vsnprintf(CP(putbuf), sizeof(putbuf)-1, format, vl);
	va_end(vl);
#else
	snprintf(CP(putbuf), sizeof(putbuf)-1, format, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
#endif /* HAVE_STDARG_H */
	putbuf[sizeof(putbuf)-1] = '\0';
	my_strmcat(CTCP_Reply_Buffer, putbuf, BIG_BUFFER_SIZE);
     }
   else
      my_strmcat(CTCP_Reply_Buffer, putbuf, BIG_BUFFER_SIZE);
   my_strmcat(CTCP_Reply_Buffer, "\001", BIG_BUFFER_SIZE);
}

#if 0 /* make jed happy again */
))}
#endif

static void
tell_only_private(u_char *to)
{
   dma_strcpy(&tmpaway, ", insecure attempt.");
   send_to_server("NOTICE %s :This feature is only available via direct CTCP.  Did you just give your password out?", to);
}

static void
tell_friend_usage(u_char *to, u_char *command)
{
   send_to_server("NOTICE %s :usage: /ctcp %s %s [#/&]channel [<password>]", to, get_server_nickname(from_server), command);
   dma_strcpy(&tmpaway, ", incorrect arguments.");
}

static void
tell_no_access(u_char *to)
{
   u_char tbuf[128];
   
   snprintf(tbuf, sizeof(tbuf)-1, ", %s is not your friend.", to);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   if (!get_int_var(CLOAK_VAR))
     send_to_server("NOTICE %s :Access denied.", to);
}

static void
tell_not_this_chan(u_char *to, u_char *command, u_char *channel)
{
   u_char tbuf[256];
   
   snprintf(tbuf, sizeof(tbuf)-1, ", no access on %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   send_to_server("NOTICE %s :Access denied for %s on %s.", to, command, channel);
}

static void
tell_not_there(u_char *to, u_char *channel)
{
   u_char tbuf[256];
   
   snprintf(tbuf, sizeof(tbuf)-1, ", they're not on %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   send_to_server("NOTICE %s :You're not on %s", to, channel);
}

static void
tell_already_there(u_char *to, u_char *channel)
{
   u_char tbuf[256];
   
   snprintf(tbuf, sizeof(tbuf)-1, ", already on %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   send_to_server("NOTICE %s :You're already on %s.", to, channel);
}

static void
tell_not_opped(u_char *to, u_char *channel)
{
   u_char tbuf[256];
   
   snprintf(tbuf, sizeof(tbuf)-1, ", you're not opped on %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   send_to_server("NOTICE %s :I'm not a channel operator on %s.", to, channel);
}

static void
tell_im_not_there(u_char *to, u_char *channel)
{
   u_char tbuf[256];
   
   snprintf(tbuf, sizeof(tbuf)-1, ", you're not on %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   send_to_server("NOTICE %s :I'm not on %s.", to, channel);
}

static void
tell_wrong_password(u_char *to, u_char *fnick)
{
   u_char tbuf[128];
   
   snprintf(tbuf, sizeof(tbuf)-1, ", wrong password for %s.", fnick);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   send_to_server("NOTICE %s :Incorrect password.", to);
}

static void
tell_chops(u_char *to, u_char *channel)
{
   Channel *cchan;
   Nick *nicks;
   int count = 0;
   u_char tmpbuf[1024], *cmode;
   
   tmpbuf[0] = '\0';
   if ((cchan = lookup_channel(channel, from_server, 0)) != NULL)
     {
	int old_hide_key = get_int_var(HIDE_CHANNEL_KEYS_VAR);
	
	for (nicks = cchan->nicks; nicks; nicks = nicks->next)
	  {
	     if (nicks->status & NICK_CHOP)
	       {
		  my_strmcat(tmpbuf, " ", sizeof(tmpbuf) - 1);
		  my_strmcat(tmpbuf, nicks->nick, sizeof(tmpbuf) - 1);
		  count++;
	       }
	  }
	
	if (old_hide_key)
	  {
	     set_int_var(HIDE_CHANNEL_KEYS_VAR, 0);
	     cmode = get_channel_mode(cchan->channel, from_server); /* could use optimization */
	  }
	send_to_server("NOTICE %s :There %s %d op%s on %s(+%s)%s%s",
		       to,
		       count == 1 ? UP("is") : UP("are"),
		       count, PLURAL(count),
		       cchan->channel,
		       cmode ? cmode : empty_string,
		       *tmpbuf == '\0' ? UP(":") : empty_string,
		       tmpbuf);
	if (old_hide_key)
	  {
	     set_int_var(HIDE_CHANNEL_KEYS_VAR, old_hide_key);
	     cmode = get_channel_mode(cchan->channel, from_server); /* could use optimization */
	  }
     }
}

/*
 * do_help: replies the ctcp user help information crap
 */
static u_char *
do_help(ctcp, from, to, args)
   CtcpEntry *ctcp;
   u_char *from, *to, *args;
{
   Friend *tmp;

   if ((tmp = get_friend_by_nuh(from, FromUser, FromHost)) == NULL)
     {
	tell_no_access(from);
	return UNULL;
     }
   dma_strcpy(&tmpaway, ", providing help.");
   send_to_server("NOTICE %s :Usage /CTCP %s <command> <channel> [<password>] where command is: INVITE CHOPS OP UNBAN WHOAMI HELP, others: IDENT PASS",
		  from, get_server_nickname(from_server));
   return UNULL;
}

/*
 * do_ident:
 */
static u_char *
do_ident(CtcpEntry * ctcp, u_char *from, u_char *to, u_char *args)
{
   Friend *tmp;
   u_char *pass = UNULL, *nick = UNULL, *cpass = UNULL;
   u_char tbuf[1024], tmpbuf1[1024];
   
   /* check arguments */
   pass = next_arg(args, &args);
   nick = next_arg(args, &args);
   if (!nick)
     nick = from;
   if (!pass && !get_int_var(CLOAK_VAR))
     {
	send_to_server("NOTICE %s :Usage /CTCP %s IDENT <password> <nickname>", from, get_server_nickname(from_server));
	dma_strcpy(&tmpaway, ", wrong usage");
	return UNULL;
     }
   
   /* try to get the friend list entry */
   tmp = get_friend(nick);
   if (!tmp)
     {
	tell_no_access(from);
	return UNULL;
     }

   /* they they don't have a password make them set one.. */
   if (!tmp->password)
     {
	send_to_server("NOTICE %s :You do not have a password set, you must set one to IDENT.", from);
	snprintf(tbuf, sizeof(tbuf)-1, ", %s has no password.", from);
	tbuf[sizeof(tbuf)-1] = '\0';
	dma_strcpy(&tmpaway, tbuf);
	return UNULL;
     }
   
   /* check the password */
   cpass = UP(crypt(pass, tmp->password));
   if (my_strcmp(tmp->password, cpass) == 0)
     {
	u_char *foo = FromUser;
	
	/* strip the leading characters */
	if (foo && *foo && *foo == '~')
	  foo++;
	if (!foo || !*foo)
	  foo = FromUser;
	
	/* update them */
	send_to_server("NOTICE %s :Added host *!%s@%s", from, foo, cluster(FromHost));
	
	/* tell ourself */
	snprintf(tbuf, sizeof(tbuf)-1, ", added host *!%s@%s to user %s", foo, cluster(FromHost), tmp->nick);
	tbuf[sizeof(tbuf)-1] = '\0';
	dma_strcpy(&tmpaway, tbuf);
	
	/* call the ninja friend add host command with the proper args */
	snprintf(tmpbuf1, sizeof(tmpbuf1)-1, "%s *!%s@%s", tmp->nick, foo, cluster(FromHost));
	tmpbuf1[sizeof(tmpbuf1)-1] = '\0';
	fcmd_add_host(tmpbuf1);
     }
   else
     tell_wrong_password(from, tmp->nick);
   return UNULL;
}

/*
 * do_pass: set userlist password via ctcp
 */
static u_char *
do_pass(CtcpEntry * ctcp, u_char *from, u_char *to, u_char *args)
{
   Friend *tmp;
   u_char *password = UNULL, *oldpass = UNULL, *cpass;
   int set = 0;
   
   if ((tmp = get_friend_by_nuh(from, FromUser, FromHost)) == NULL)
     {
	tell_no_access(from);
	return UNULL;
     }
   password = next_arg(args, &args);
   if (!password)
     {
	send_to_server("NOTICE %s :Usage /CTCP %s PASS <password> [<old password>]", from, get_server_nickname(from_server));
	dma_strcpy(&tmpaway, ", wrong usage.");
	return UNULL;
     }
   oldpass = next_arg(args, &args);
   if (tmp->password && !oldpass)
     {
	send_to_server("NOTICE %s :Your password has already been set, to change it use: /CTCP %s PASS <password> <old password>",
		       from, from, get_server_nickname(from_server));
	dma_strcpy(&tmpaway, ", password is already set.");
	return UNULL;
     }
   if (tmp->password)
     {
	cpass = UP(crypt(oldpass, tmp->password));
	if (my_strcmp(tmp->password, cpass) != 0)
	  {
	     tell_wrong_password(from, tmp->nick);
	     return UNULL;
	  }
     }
   else
     set = 1;
   
   /* update the users's password */
   cpass = UP(crypt(password, make_salt(tmp->nick)));
   dma_strcpy(&tmp->password, cpass);
   
   /* tell them what happened */
   send_to_server("NOTICE %s :Your password has been %s.", from, set ? "set" : "changed");
   dma_strcpy(&tmpaway, ", password has been ");
   if (set)
     dma_strcat(&tmpaway, "set.");
   else
     dma_strcat(&tmpaway, "changed.");
   
   (void)save_friends(1);	
   			/* this is ok here..
			 * perhaps make something like update_friend_password() in friends.c 
			 */
   return UNULL;
}

/*
 * does initial checking for friends list commands using
 * the syntax: /ctcp nick command <channel> <password>
 */

static Friend *
friend_ctcp_check(CtcpEntry *ctcp, u_char *from, u_char *to, u_char **channel, u_char **args, u_char *modes)
{
   Friend *tmp;
   FChan *fchan;
   u_char *pass, *cpass;
   
   /* try to look them up by nick!user@host mask */
   tmp = get_friend_by_nuh(from, FromUser, FromHost);
   if (!tmp && !is_channel(to))
     {
	tell_no_access(from);
	return NULL;
     }
   
   /* check to make sure that the user/friend isn't giving away their password
    * and isn't trying to ask everyone.
    */
   if (is_channel(to))
     {
	tell_only_private(from);
	return NULL;
     }
   
   /* check to see if there are any arguments.  if not, error out. */
   if (!args || !*args)
     {
	tell_friend_usage(from, ctcp->name);
	return NULL;
     }

   /* turn the first agument into a channel and the second into the password */
   *channel = make_chan(next_arg(*args, args));
   *channel = make_chan(*channel);
   pass = next_arg(*args, args);

   /* check to see if this friend has access to this type of operation on the
      specified channel. */
   fchan = get_fchan(tmp, *channel, 0);
   if (!fchan)
     {
	/* tell them that they have no access. */
	tell_not_this_chan(from, ctcp->name, *channel);
	return NULL;
     }
   *modes = fchan->modes;
   
   /* check password authentication */
   if (!tmp->password) /* password is not required */
     return tmp;
   
   if (!pass)
     {
	tell_wrong_password(from, tmp->nick);
	return NULL;
     }
   
   cpass = UP(crypt(pass, tmp->password));
   if (my_strcmp(tmp->password, cpass) != 0)
     {
	tell_wrong_password(from, tmp->nick);
	return NULL;
     }
   return tmp;
}

/*
 * re-written by Senor Pato on 9/28/97
 * do_op: basic algorithm:
 * 1. check to see if the ctcper is a friend of ours.
 * 2. check to see that they specified correct arguments and it's
 *      possible to fulfill the request.
 * 3. fulfill the request.
 * 4. return the correct message.
 * 
 * took away massive overloading on 5/9/99
 * modularized some stuff on 9/26/01
 */
static u_char *
do_op(CtcpEntry * ctcp, u_char *from, u_char *to, u_char *args)
{
   Friend *tmp;
   u_char *channel;
   u_char modes;
   
   /* do inital checks */
   tmp = friend_ctcp_check(ctcp, from, to, &channel, &args, &modes);
   if (!tmp)
     return UNULL;
   
   /* make sure they have access to do what they want. */
   if (!(modes & FL_CMODE_OPS))
     {
	tell_not_this_chan(from, ctcp->name, channel);
	return UNULL;
     }
   
   /* make sure that you're on the channel they want something done on */
   if (!is_chanop(channel, get_server_nickname(from_server)))
     {
	tell_not_opped(from, channel);
	return UNULL;
     }

   /* if the person is not on the channel, forget it */
   if (!is_on_channel(channel, from_server, from))
     {
	tell_not_there(from, channel);
	return UNULL;
     }
   
   send_to_server("MODE %s +o %s", channel, from);
   dma_strcpy(&tmpaway, ", giving ops.");
   return UNULL;
}

/*
 * invite a friend if they are indeed a friend and have access..
 */
static u_char *
do_invite(CtcpEntry * ctcp, u_char *from, u_char *to, u_char *args)
{
   Friend *tmp;
   u_char *channel;
   u_char modes;
   u_char tbuf[128];

   /* do inital checks */
   tmp = friend_ctcp_check(ctcp, from, to, &channel, &args, &modes);
   if (!tmp)
     return UNULL;
   
   /* if the person is on the channel, forget it */
   if (is_on_channel(channel, from_server, from))
     {
	tell_already_there(from, channel);
	return UNULL;
     }
   
   /* make sure they have access to do what they want. */
   if (!(modes & FL_CMODE_INVITE))
     {
	tell_not_this_chan(from, ctcp->name, channel);
	return UNULL;
     }
   
   /* if the request requires ops and you don't have any, tell them too bad. */
   if (!is_chanop(channel, get_server_nickname(from_server)))
     {
	tell_not_opped(from, channel);
	return UNULL;
     }
   
   send_to_server("INVITE %s %s", from, channel);
   snprintf(tbuf, sizeof(tbuf)-1, ", inviting to %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   return UNULL;
}

/*
 * tell a friend the channel ops/mode if they have access to see it
 */
static u_char *
do_chops(CtcpEntry * ctcp, u_char *from, u_char *to, u_char *args)
{
   Friend *tmp;
   u_char *channel;
   u_char tbuf[128];
   u_char modes;

   /* do inital checks */
   tmp = friend_ctcp_check(ctcp, from, to, &channel, &args, &modes);
   if (!tmp)
     return UNULL;
   
   /* if the person is on the channel, forget it */
   if (is_on_channel(channel, from_server, from))
     {
	tell_already_there(from, channel);
	return UNULL;
     }
   
   /* make sure i am on the channel */
   if (!lookup_channel(channel, from_server, 0))
     {
	tell_im_not_there(from, channel);
	return UNULL;
     }
   
   /* make sure they have access to do what they want. */
   if (!(modes & FL_CMODE_CHOPS))
     {
	tell_not_this_chan(from, ctcp->name, channel);
	return UNULL;
     }
   
   /* tell the person the channel operators */
   tell_chops(from, channel);

   snprintf(tbuf, sizeof(tbuf)-1, ", showing channel ops on %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   return UNULL;
}

/*
 * unban a friend if they have access to request such a thing
 */
static u_char *
do_unban(CtcpEntry * ctcp, u_char *from, u_char *to, u_char *args)
{
   Friend *tmp;
   u_char *channel;
   u_char modes;
   u_char tbuf[128];

   /* do inital checks */
   tmp = friend_ctcp_check(ctcp, from, to, &channel, &args, &modes);
   if (!tmp)
     return UNULL;
   
   /* if the request requires ops and you don't have any, tell them too bad. */
   if (!is_chanop(channel, get_server_nickname(from_server)))
     {
	tell_not_opped(from, channel);
	return UNULL;
     }

   /* if the person is on the channel, forget it */
   if (is_on_channel(channel, from_server, from))
     {
	tell_already_there(from, channel);
	return UNULL;
     }
   
   /* make sure they have access to do what they want. */
   if (!(modes & FL_CMODE_UNBAN))
     {
	tell_not_this_chan(from, ctcp->name, channel);
	return UNULL;
     }
   
   /* try to remove any bans that match the friends current user@host */
   remove_matching_bans(from, FromUser, FromHost, channel, 1);
   
   /* update us/them */
   snprintf(tbuf, sizeof(tbuf)-1, ", removing ban(s) on %s.", channel);
   tbuf[sizeof(tbuf)-1] = '\0';
   dma_strcpy(&tmpaway, tbuf);
   send_to_server("NOTICE %s :All bans, if any, matching your nick!user@host were removed from %s.", from, channel);
   return UNULL;
}

/*
 * do_whoami: returns the users userlist information
 */
static u_char *
do_whoami(CtcpEntry * ctcp, u_char *from, u_char *to, u_char *args)
{
   Friend *tmp;
   FHost *host;
   FChan *chan;
   u_char tmpbuf[1024];
   int first = 1;

   /* are they are friend? */
   if ((tmp = get_friend_by_nuh(from, FromUser, FromHost)) == NULL)
     {
	tell_no_access(from);
	return UNULL;
     }
   
   /* friend nick/flags */
   strncpy(tmpbuf, "You are ", sizeof(tmpbuf)-1);
   strmcat(tmpbuf, tmp->nick, sizeof(tmpbuf)-1);
   if (tmp->modes)
     {
	strmcat(tmpbuf, "(+", sizeof(tmpbuf)-1);
	strmcat(tmpbuf, CP(recreate_umode(tmp)), sizeof(tmpbuf)-1);
	strmcat(tmpbuf, ")", sizeof(tmpbuf)-1);
     }
   
   /* channels */
   for (chan = tmp->channels; chan; chan = chan->next)
     {
	if (first)
	  {
	     strmcat(tmpbuf, ", channels(+flags):", sizeof(tmpbuf) - 1);
	     first = 0;
	  }
	strmcat(tmpbuf, " ", sizeof(tmpbuf)-1);
	strmcat(tmpbuf, chan->channel, sizeof(tmpbuf)-1);
	if (chan->modes)
	  {
	     strmcat(tmpbuf, "(+", sizeof(tmpbuf)-1);
	     strmcat(tmpbuf, CP(recreate_cmode(chan)), sizeof(tmpbuf)-1);
	     strmcat(tmpbuf, ")", sizeof(tmpbuf)-1);
	  }
     }

   /* hosts */
   first = 1;
   for (host = tmp->hosts; host; host = host->next)
     {
	if (first)
	  {
	     strmcat(tmpbuf, ", hosts:", sizeof(tmpbuf) - 1);
	     first = 0;
	  }
	strmcat(tmpbuf, " ", sizeof(tmpbuf)-1);
	strmcat(tmpbuf, host->host, sizeof(tmpbuf) - 1);
     }
   
   /* send it */
   send_to_server("NOTICE %s :%s", from, tmpbuf);
   dma_strcpy(&tmpaway, ", telling info.");
   return UNULL;
}

/*
 * ninja version!
 * cloak doesn't protect this..
 */
static u_char *
do_nver(ctcp, from, to, args)
   CtcpEntry *ctcp;
   u_char *from, *to, *args;
{
   send_to_server("NOTICE %s :Ninja IRC v%s, build #%s made by %s@%s on %s [%s]",
		  from, irc_version, 
		  compile_num, compile_user, compile_host, compile_time,
		  ninja_release);
   return UNULL;
}
