/* # skkinput (Simple Kana-Kanji Input)
 * kanji.c --- kanji code
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>

#include "commondef.h"
#include "kanji.h"

#ifdef DEBUG
int dump_myCharset
( struct myChar *string )
{
  while( string->chara != '\0' ){
    switch( string->charset ){
    case CHARSET_ASCII :
      printf( "ASCII:             0x%02X (%c)\n",
	      string->chara, string->chara ) ;
      break ;
#if 0
    case CHARSET_JISX0201_ROMAN :
      printf( "JISX0201-ROMAN:    0x%02X (%c)\n",
	      string->chara, string->chara ) ;
      break ;
    case CHARSET_JISX0201_KATAKANA :
      printf( "JISX0201-KATAKANA: 0x%02X \n",
	      string->chara ) ;
      break ;
#endif
    case CHARSET_JISX0208_1978 :
      printf( "JISX0208-1978:     0x%04X \n",
	      string->chara ) ;
      break ;
    case CHARSET_JISX0208_1983 :
      printf( "JISX0208-1983:     0x%04X \n",
	      string->chara ) ;
      break ;
    case CHARSET_JISX0212_1990 :
      printf( "JISX0212-1990:     0x%04X \n",
	      string->chara ) ;
      break ;
    default :
      break ;
    }
    string ++ ;
  }
  return NOERR ;
}
#endif

/*
 * Хåեؿ
 */
void init_kanjibuffer( struct KanjiBuffer *buf )
{
  buf->buffer  = buf->internal ;
  buf->bufsize = TEXTMAXLEN ;
  buf->usage   = 0 ;
  MYCHAR_SET_END_OF_STRING( buf->buffer[ 0 ] ) ;
  return ;
}

/*
 *
 */
void add_kanjibuffer
( struct KanjiBuffer *buf, struct myChar chara )
{
  if( ( buf->usage + 1 ) >= buf->bufsize ){
    if( buf->buffer == buf->internal ){
      /* TEXTBUFSIZE = TEXTMAXLEN + 1 Ǥ롣ͳϴñǡǸ NUL 
       * ʸ륹ڡɬפ顣*/ 
      buf->bufsize = ( TEXTBUFSIZE + TEXTMAXLEN ) ;
      buf->buffer = malloc
	( sizeof( struct myChar ) * ( buf->bufsize + 1 ) ) ;
      memcpy
	( buf->buffer, buf->internal, 
	  sizeof( struct myChar ) * buf->usage ) ;
    } else {
      buf->bufsize += TEXTBUFSIZE ;
      buf->buffer = realloc
	( buf->buffer, sizeof( struct myChar ) * ( buf->bufsize + 1 ) ) ;
    }
  }
  buf->buffer[ buf->usage ++ ] = chara ;
  MYCHAR_SET_END_OF_STRING( buf->buffer[ buf->usage ] ) ;
  return ;
}

void add_kanjiStringToKanjiBuffer
( struct KanjiBuffer *buf, struct myChar *string, int length )
{
  int buflength = ( ( length / TEXTBUFSIZE ) + 1 ) * TEXTBUFSIZE ;

  if( ( buf->usage + length ) >= buf->bufsize ){
    if( buf->buffer == buf->internal ){
      buf->bufsize = TEXTBUFSIZE + buflength ;
      buf->buffer = malloc
	( sizeof( struct myChar ) * ( buf->bufsize + 1 ) ) ;
      memcpy
	( buf->buffer, buf->internal, 
	  sizeof( struct myChar ) * buf->usage ) ;
    } else {
      buf->bufsize += buflength ;
      buf->buffer = realloc
	( buf->buffer, sizeof( struct myChar ) * ( buf->bufsize + 1 ) ) ;
    }
  }
  memcpy
    ( buf->buffer + buf->usage, string, sizeof( struct myChar ) * length ) ;
  buf->usage += length ;
  MYCHAR_SET_END_OF_STRING( buf->buffer[ buf->usage ] ) ;
  return ;
}

/*
 */
void close_kanjibuffer( struct KanjiBuffer *buf )
{
  if( buf->buffer != buf->internal )
    free( buf->buffer ) ;
  init_kanjibuffer( buf ) ;
  return ;
}

/*
 * ISO2022JP  G0 礬Ѳ뤫Ϥؿ
 */
int iso2022jp_G0charset
( unsigned char **string, int charset )
{
  unsigned char *ptr = *string ;

  if( *( ptr ++ ) != 0x1b )
    return charset ;

  switch( *( ptr ++ ) ){
  case '\0' :
    return ERR ;
  case '(' :
    switch( *( ptr ++ ) ){
    case '\0' :
      return ERR ;
      /* G0  ASCII ؼ롣*/
    case 'B' :
      charset = ISO2022JP_CHARSET_ASCII ;
      break ;
      /* G0  JISX0201 ROMAN ؼ롣*/
    case 'J' :
      charset = ISO2022JP_CHARSET_JISX0201_ROMAN ;
      break ;
    case 'I' :
      charset = ISO2022JP_CHARSET_JISX0201_KATAKANA ;
      break ;
    default :
      /* Τʤץ󥹤դġ*/
      break ;
    }
    break ;
  case '$' :
    switch( *( ptr ++ ) ){
      /* ͽ̥եνλ*/
    case '\0' :
      return ERR ;
      /* G0  JISX0208-1978 ĤʥեȤΤʤĤ
	 롣*/
    case '@' :
      charset = ISO2022JP_CHARSET_JISX0208_1978 ;
      break ;
      /* G0  GB2312-1980 ؼ롣*/
    case 'A' :
      charset = ISO2022JP_CHARSET_GB2312_1980 ;
      break ;
      /* G0  JISX0208-1983 ؼ롣*/
    case 'B' :
      charset = ISO2022JP_CHARSET_JISX0208_1983 ;
      break ;
    case '(' :
      switch( *( ptr ++ ) ){
	/* ͽ̥եνλ*/
      case '\0' :
	return ERR ;
	/* G0  JISX0208-1978 ؼ롣*/
      case '@' :
	charset = ISO2022JP_CHARSET_JISX0208_1978 ;
	break ;
	/* G0  JISX0208-1983 ؼ롣*/
      case 'B':
	charset = ISO2022JP_CHARSET_JISX0208_1983 ;
	break ;
	/* G0  JISX0212 ؼ롣*/
      case 'D' :
	charset = ISO2022JP_CHARSET_JISX0212_1990 ;
	break ;
	/*  KSC5601-1987 ؼ롣*/
      case 'C' :
	charset = ISO2022JP_CHARSET_KSC5601_1987 ;
	break ;
      default :
	/* Τʤץ󥹤դġ*/
	break ;
      }
      break ;
    default :
      /* Τʤץ󥹤դġ*/
      break ;
    }
    break ;
  case '.' :
    switch( *( ptr ++ ) ){
    case '\0' :
      return ERR ;
      /* 96 ʸؼ롣*/
    case 'A' :
      charset = ISO2022JP_CHARSET_ISO8859_1 ;
      break ;
    case 'B' :
      charset = ISO2022JP_CHARSET_ISO8859_2 ;
      break ;
    case 'C' :
      charset = ISO2022JP_CHARSET_ISO8859_3 ;
      break ;
    case 'D' :
      charset = ISO2022JP_CHARSET_ISO8859_4 ;
      break ;
    case 'F' :
      charset = ISO2022JP_CHARSET_ISO8859_7 ;
      break ;
    case 'G' :
      charset = ISO2022JP_CHARSET_ISO8859_6 ;
      break ;
    case 'H' :
      charset = ISO2022JP_CHARSET_ISO8859_8 ;
      break ;
    case 'L' :
      charset = ISO2022JP_CHARSET_ISO8859_5 ;
    case 'M' :
      charset = ISO2022JP_CHARSET_ISO8859_9 ;
      break ;
    default :
      break ;
    }
    break ;
  default :
    /* Τʤץ󥹤դġ*/
    break ;
  }
  *string = ptr ;
  return charset ;
}

/*
 * ISO-2022-JP ɤʸɤʸѴؿ
 *---
 * å󥰥եȤưƤޤSI/SO äƤɤ
 * 褦 ȤߤɡISO-2022-JP εʽ
 * ƲˤΤʡ iso-2022-jp.txt ˤ SI/SO ˤĤƤεҤ̵
 * Ǥɡ
 */
int iso2022jp_stringToMycharset
( unsigned char *string, struct myChar **wstring )
{
  int charset, length, mychar_length ;
  struct myChar *ptr ;

  /* G0 ˳ƤƤʸɤϺǽϥΤȡ*/
  charset      = CHARSET_ASCII ;
  ptr           = NULL ;
  mychar_length = 0 ;
  if( ( length = strlen( string ) ) < 0 ){
    if( wstring != NULL )
      *wstring = NULL ;
    return False ;
  }
  if( wstring != NULL ){
    ptr = ( struct myChar * )malloc
      ( sizeof( struct myChar ) * ( ( length + 1 ) ) ) ;
    *wstring = ptr ;
  }
  while( *string != '\0' ){
    /* ץ󥹤ȯޤcharset פޤ*/
    if( ( charset =
	  iso2022jp_G0charset( &string, charset ) ) == ERR ){
      break ;
    }
    /* ⤷ʸ󤬽äƤġ*/
    if( *string == '\0' )
      break ;
    switch( charset ){
      /* 94 ʸ */
    case ISO2022JP_CHARSET_ASCII :
    case ISO2022JP_CHARSET_JISX0201_ROMAN :
      if( ptr != NULL ){
	/* ASCII  ISO8859-1 κȾ̤Ʊ蘆Ƥ餦줬ɤ
	 * ʬʤ*/
	ptr->charset = CHARSET_ISO8859_1 ;
	ptr->chara   = *string ;
	ptr ++ ;
      }
      mychar_length ++ ;
      string ++ ;
      break ;
    case ISO2022JP_CHARSET_JISX0201_KATAKANA :
      if( ptr != NULL ){
	/* ASCII  ISO8859-1 κȾ̤Ʊ蘆Ƥ餦줬ɤ
	 * ʬʤ*/
	ptr->charset = CHARSET_JISX0201_1976 ;
	ptr->chara   = ( *string ) | 0x80 ;
	ptr ++ ;
      }
      mychar_length ++ ;
      string ++ ;
      break ;
      /* 94^n ʸġǤʸƤʤɡġ*/
    case ISO2022JP_CHARSET_JISX0208_1978 :
    case ISO2022JP_CHARSET_JISX0208_1983 :
    case ISO2022JP_CHARSET_JISX0212_1990 :
    case ISO2022JP_CHARSET_GB2312_1980 :
    case ISO2022JP_CHARSET_KSC5601_1987 :
      if( ptr != NULL ){
	ptr->charset = CHARSET_JISX0208_1978 + 
	  ( charset - ISO2022JP_CHARSET_JISX0208_1978 ) ;
	ptr->chara   = ( unsigned short )
	  ( ( ( *string ) << 8 ) | ( *( string + 1 ) ) ) ; 
	ptr ++ ;
      }
      string ++ ;
      string ++ ;
      mychar_length ++ ;
      break ;
    case ISO2022JP_CHARSET_ISO8859_1 :
    case ISO2022JP_CHARSET_ISO8859_2 :
    case ISO2022JP_CHARSET_ISO8859_3 :
    case ISO2022JP_CHARSET_ISO8859_4 :
    case ISO2022JP_CHARSET_ISO8859_5 :
    case ISO2022JP_CHARSET_ISO8859_6 :
    case ISO2022JP_CHARSET_ISO8859_7 :
    case ISO2022JP_CHARSET_ISO8859_8 :
    case ISO2022JP_CHARSET_ISO8859_9 :
      if( ptr != NULL ){
	if( ( *string ) & 0x80 ){
	  ptr->charset = CHARSET_ISO8859_1 + 
	    ( charset - ISO2022JP_CHARSET_ISO8859_1 ) ;
	} else {
	  ptr->charset = CHARSET_ASCII ;
	}
	ptr->chara = *string ;
	ptr ++ ;
      }
      mychar_length ++ ;
      string ++ ;
      break ;
    default :
      string ++ ;
      break ;
    }
  }
  if( ptr != NULL ){
    ptr->charset = CHARSET_ASCII ;
    ptr->chara   = '\0' ;
    /* 礭·ļ᤮ƤȦ顣­ʤʤ뤳
     * ȤϤꤨʤ*/
    if( mychar_length != length ){
      ptr = ( struct myChar * )realloc
	( *wstring, sizeof( struct myChar ) * ( mychar_length + 1 ) ) ;
      *wstring = ptr ;
    }
  }
  return mychar_length ;
}

/*
 * Extended Unix Code (EUC)ɤѴؿ
 *----
 * ٤Ͽܤ JISX0201-1978  JISX0212-1990 ޤǰ褦ˤ
 * ޤϽʤä
 */
int extendedUnixCode_stringToMycharset
( unsigned char *string, struct myChar **wstring )
{
  int length, mychar_length ;
  int chara ;
  struct myChar *ptr = NULL ;

  /* wstring  NULL ä顢ͻҸˤʤ롣ɤ٤̤ɬפˤ
   * ΤȤġɬ̤ϸΥƥȤʸĹĹʤ뤳Ȥ
   * ʤѤƤ롣ǺǽŬˤȤäƤ顢夫饵
   * Ƥ롣*/
  mychar_length = 0 ;
  length = strlen( string ) ;
  if( wstring != NULL ){
    ptr = *wstring = ( struct myChar * )
      malloc( sizeof( struct myChar ) * ( length + 1 ) ) ;
  }
  while( *string != '\0' ){
    if( *string & 0x80 ){
      chara = *string ++ ;
      if( *string == '\0' )
	goto exit_while ;
      switch( chara ){
	/* SS2 (󥰥륷ե) 褿硣*/
      case 0x8E :
	if( ptr != NULL ){
	  ptr->charset = CHARSET_JISX0201_1976 ;
	  ptr->chara   = ( *string ) | 0x80 ;
	  ptr ++ ;
	}
	mychar_length ++ ;
	string ++ ;
	break ;
	/* SS3 */
      case 0x8F :
	chara = *string ++ ;
	if( *string == '\0' )
	  goto exit_while ;
	if( ptr != NULL ){
	  ptr->charset = CHARSET_JISX0212_1990 ;
	  /* ƥХȤκǾ̥ӥå(MSB) 1 ΩäƤΤǡȤ
	   * Ƥ*/
	  ptr->chara   = ( ( chara << 8 ) | *string ) & 0x7F7F ;
	  ptr ++ ;
	}
	mychar_length ++ ;
	string ++ ;
	break ;
      default :
	if( ptr != NULL ){
	  ptr->charset = CHARSET_JISX0208_1983 ;
	  /* ƥХȤκǾ̥ӥå(MSB) 1 ΩäƤΤǡȤ
	   * Ƥ*/
	  ptr->chara   = ( ( chara << 8 ) | *string ) & 0x7F7F ;
	  ptr ++ ;
	}
	mychar_length ++ ;
	string ++ ;
	break ;
      }
    } else {
      if( ptr != NULL ){
	ptr->charset = CHARSET_ASCII ;
	ptr->chara   = ( *string ) ;
	ptr ++ ;
      }
      mychar_length ++ ;
      string ++ ;
    }
  }
exit_while:
  if( ptr != NULL ){
    /* Ƚüޤ礦ȡ*/
    ptr->charset = CHARSET_ASCII ;
    ptr->chara   = '\0' ;
    /* 礭·ļ᤮ƤȦ顣­ʤʤ뤳
     * ȤϤꤨʤ*/
    if( mychar_length != length ){
      ptr = ( struct myChar * )realloc
	( *wstring, sizeof( struct myChar ) * ( mychar_length + 1 ) ) ;
      *wstring = ptr ;
    }
  }
  return mychar_length ;
}

/*
 * Microsoft Kanji Code ɤѴؿ
 */
int microsoftKanjiCode_stringToMycharset
( unsigned char *string, struct myChar **wstring )
{
  int length, mychar_length ;
  unsigned short chara ;
  struct myChar *ptr = NULL ;

  /* wstring  NULL ä顢ͻҸˤʤ롣ɤ٤̤ɬפˤ
   * ΤȤġɬ̤ϸΥƥȤʸĹĹʤ뤳Ȥ
   * ʤѤƤ롣ǺǽŬˤȤäƤ顢夫饵
   * Ƥ롣*/
  mychar_length = 0 ;
  length = strlen( string ) ;
  if( wstring != NULL ){
    ptr = *wstring = ( struct myChar * )
      malloc( sizeof( struct myChar ) * ( length + 1 ) ) ;
  }
  while( *string != '\0' ){
    if( *string & 0x80 ){
      if( ( *string >= 0x81 && *string <= 0x9F ) ||
	  ( *string >= 0xE0 && *string <= 0xFC ) ){
	chara = *string ++ ;
	/* ͽʸνλ*/
	if( *string == '\0' )
	  break ;
	if( ( *string >= 0x40 && *string <= 0x7E ) ||
	    ( *string >= 0x80 && *string <= 0xFC ) ){
	  if( ptr != NULL ){
	    unsigned short c1, c2 ;

	    c1 = chara ;
	    c2 = *string ;
	    
	    ptr->charset = CHARSET_JISX0208_1983 ;
	    
	    /* SJIS  JISX0208-1983 Ѵ롣*/
	    if( c1 >= 0x81 && c1 <= 0x9F ){
	      c1 -= 0x70 ;
	    } else if( c1 >= 0xE0 && c1 <= 0xFC ){
	      c1 -= 0xB0 ;
	    } else {
	      goto machigatta_sjis ;
	    }
	    if( c2 >= 0x80 )
	      c2 -- ;
	    c1 = c1 << 1 ;
	    c2 -= 0x1F ;
	    if( c2 >= 0x7F ){
	      c2 -= 0x5E ;
	    } else {
	      c1 -- ;
	    }
	    ptr->chara   = ( c1 << 8 ) | c2 ;
	    ptr ++ ;
	  }
	  mychar_length ++ ;
	}
      } else if( *string >= 0xA1 && *string <= 0xDF ){
	if( ptr != NULL ){
	  ptr->charset = CHARSET_JISX0201_1976 ;
	  ptr->chara   = ( *string ) | 0x80 ;
	  ptr ++ ;
	}
	mychar_length ++ ;
      }
    machigatta_sjis:
      string ++ ;
    } else {
      if( ptr != NULL ){
	ptr->charset = CHARSET_JISX0201_1976 ;
	ptr->chara   = ( *string ) & 0x7F ;
	ptr ++ ;
      }
      mychar_length ++ ;
      string ++ ;
    }
  }
  if( ptr != NULL ){
    /* Ƚüޤ礦ȡ*/
    ptr->charset = CHARSET_ASCII ;
    ptr->chara   = '\0' ;
    /* 礭·ļ᤮ƤȦ顣­ʤʤ뤳
     * ȤϤꤨʤ*/
    if( mychar_length != length ){
      ptr = ( struct myChar * )realloc
	( *wstring, sizeof( struct myChar ) * ( mychar_length + 1 ) ) ;
      *wstring = ptr ;
    }
  }
  return mychar_length ;
}

/*
 * ɤȽ̤򤹤ؿ
 *--------
 * EUC 3ХȥɤʤƤΤΤޤ󤫤͡
 */
int check_kanjiCode( unsigned char *text, int length )
{
  unsigned char *ptr ;
  int kanji_code = KANJI_CODE_UNKNOWN ;

  for( ptr = text ; length > 0 ; length --, ptr ++ ){
    if( *ptr == '\x1b' ){
      /* ʸդˤ JIS Ǥǽ˹⤤*/
      if( length < 3 ){
	/* Ĥʸʲ᤮뤫鱳Ǥ礦*/
	continue ;
      }
      if( ( *( ptr + 1 ) == '\x24' || *( ptr + 1 ) == '\x28' ) &&
	  *( ptr + 2 ) == '\x42' )
	return KANJI_CODE_JIS ;
      continue ;
    }
    /* ξˤ EUC  SJIS Τɤ餫ǤȹͤƤޤ*/
    if( ( *ptr ) & 0x80 ){
      kanji_code = KANJI_CODE_EUC_SJIS_UNKNOWN ;
      
      /* Ƚˡϼ̤Ȥ롣
       *----------
       * (1) ¯ȾѲ̾ȸƤФ륳ɤ餷Τä硣
       * ϡ0xA1  0xDF ϰϤˤ륳ɤ1Хܤ褿
       * Хܤˤϡ8 bit ܤΩäƤʤɤ褿
       * 롣 ---> Shift JIS
       *
       * (2) 8 bit ܤΩäƤϤΤΡ0x80 ⤷ϡ0xA0 ⤷
       * ϡ0xFD  0xFD Хܤ褿硣
       * -----> EUC
       *
       * (3) 裱Хܤ 0x81  0x9F δ֤ˤꡢ裲Хܤ
       * 0x80  0x9E ⤷ 0x9F  0xFC δ֤ˤʤäϡȽ
       * ԲǽǤ롣
       * 
       * (4) 裱Хܤ 0x81  0x9F δ֤ˤꡢ裲Хܤ
       * 0x40  0x7E δ֤ˤ SJIS Ǥ롣
       */
      if( length < 2 ){
	/* ¯ȾѲ̾ȯΤǡSJIS ǤȻפ*/
	if( *ptr >= 0xA1 && *ptr <= 0xDF )
	  return KANJI_CODE_SJIS ;
	/* ʤ㡢ǸޤǹԤ*/
	continue ;
      }
      /* Shift JIS ˤƴɤ裱ХȤˤʤʤʸ褿
	 硣*/
      if( *ptr == 0x80 || *ptr == 0xA0 || *ptr >= 0xFD )
	return KANJI_CODE_EUC ;
      /* ¯ȾѲ̾դ硣*/
      if( *ptr >= 0xA1 && *ptr <= 0xDF && !( *( ptr + 1 ) & 0x80 ) )
	return KANJI_CODE_SJIS ;
      /* Shift JIS ˲ǽ裱ХȤäˤϡġ*/
      if( ( *ptr >= 0x81 && *ptr <= 0x9f ) ||
	  ( *ptr >= 0xE0 && *ptr <= 0xFC ) ){
	if( *( ptr + 1 ) >= 0x40 && *( ptr + 1 ) < 0x7F )
	  return KANJI_CODE_SJIS ;
	if( *( ptr + 1 ) > 0xFC )
	  return KANJI_CODE_EUC ;
      }
      /* ϤȽԲǽǤ롣*/
      ptr ++ ;
      length -- ;
      continue ;
    }
  }
  return kanji_code ;
}

int myCharStrlen( struct myChar *string )
{
  int length ;
  if( string == NULL )
    return 0 ;
  for( length = 0 ; string->chara != '\0' ; string ++, length ++ )
    ;
  return length ;
}

int myCharStrncmp
( struct myChar *string1, struct myChar *string2, int count )
{
  while( count > 0 ){
    /* chara ä顢ФƱǤ뤳ȤϤʤ*/
    if( string1->chara != string2->chara )
      return True ;
    /* JISX0201-ROMAN  ASCII 餤Ʊ˰ȻפäƤ뤱
     * ܤǤ礦 */
    if( string1->charset != string2->charset &&
	( !IS_ASCII_CHARA( *string1 ) ||
	  !IS_ASCII_CHARA( *string2 ) ) )
      return True ;
    /* ΰ֤褿ȤȤϡchara Ʊǡ˽üƤ
     * ȤȤˤʤ뤫顢TRUE Ȥʤ롣*/
    if( string1->chara == '\0' )
      return False ;
    string1 ++ ;
    string2 ++ ;
    count -- ;
  }
  return False ;
}

int myCharStrcmp
( struct myChar *string1, struct myChar *string2 )
{
  while( string1->chara != '\0' && string2->chara != '\0' ){
    /* chara ä顢ФƱǤ뤳ȤϤʤ*/
    if( string1->chara != string2->chara )
      return True ;
    /* JISX0201-ROMAN  ASCII 餤Ʊ˰ȻפäƤ뤱
     * ܤǤ礦 */
    if( string1->charset != string2->charset &&
	( !IS_ASCII_CHARA( *string1 ) ||
	  !IS_ASCII_CHARA( *string2 ) ) )
      return True ;
    /* ʸ򸫤롣*/
    string1 ++ ;
    string2 ++ ;
  }
  /* ξԶ '\0' ǤСTrue ˤʤ롣*/
  if( string1->chara == string2->chara )
    return False ;
  return True ;
}

int myCharStrcpy
( struct myChar *string1, struct myChar *string2 )
{
  while( string2->chara != '\0' ){
    *string1 ++ = *string2 ++ ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharKatakanaStrcpy
( struct myChar *string1, struct myChar *string2 )
{
  while( string2->chara != '\0' ){
    if( string2->charset == CHARSET_JISX0208_1983 &&
	string2->chara >= 0x2421 &&
	string2->chara <= 0x2473 ){
      string1->charset = string2->charset ;
      string1->chara   = string2->chara + 0x0100 ;
      string1 ++ ;
      string2 ++ ;
    } else {
      *string1 ++ = *string2 ++ ;
    }
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharStrncpy
( struct myChar *string1, struct myChar *string2, int length )
{
  while( !IS_END_OF_STRING( *string2 ) && length > 0 ){
    *string1 ++ = *string2 ++ ;
    length -- ;
  }
  if( IS_END_OF_STRING( *string2 ) && length > 0 )
    MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

/*
 * Ҳ̾򤸤ʸʿ̾ľƥԡؿ
 */
int myCharHiraganaStrncpy
( struct myChar *string1, struct myChar *string2, int length )
{
  while( !IS_END_OF_STRING( *string2 ) && length > 0 ){
    /* Ҳ̾ä鶯Ūʿ̾ľƥԡ롣*/
    if( string2->charset == CHARSET_JISX0208_1983 &&
	string2->chara >= 0x2521 &&
	string2->chara <= 0x2573 ){
      string1->charset = string2->charset ;
      string1->chara   = string2->chara - 0x0100 ;
      string1 ++ ;
      string2 ++ ;
    } else {
      *string1 ++ = *string2 ++ ;
    }
    length -- ;
  }
  if( IS_END_OF_STRING( *string2 ) && length > 0 )
    MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

/*
 * ʿ̾Ҳ̾򤸤ʸʿ̾Ҳ̾ؤƥԡؿ
 */
int myCharHiraKataStrnncpy
( struct myChar *string1, int length1,
  struct myChar *string2, int length2 )
{
  int length1_backup = length1 ;
  while( !IS_END_OF_STRING( *string2 ) &&
	 length1 > 0 &&
	 length2 > 0 ){
    /* Ҳ̾ä鶯Ūʿ̾ľƥԡ롣*/
    if( string2->charset == CHARSET_JISX0208_1983 ||
	string2->charset == CHARSET_JISX0208_1978 ){
      /* Ҳ̾ä硣*/
      if( string2->chara >= 0x2521 &&
	  string2->chara <= 0x2573 ){
	string1->charset = string2->charset ;
	string1->chara   = string2->chara - 0x0100 ;
	string1 ++ ;
	length1 -- ;
	string2 ++ ;
	length2 -- ;
	continue ;
      }
      /* ֥פä硣*/
      if( string2->chara == 0x2574 ){
	string1->charset = string2->charset ;
	string1->chara   = 0x2426 ; /* ֤ */
	string1 ++ ;
	length1 -- ;
	if( length1 > 0 ){
	  string1->charset = string2->charset ;
	  string1->chara   = 0x212b ;
	  string1 ++ ;
	  length1 -- ;
	}
	string2 ++ ;
	length2 -- ;
	continue ;
      } 
      /* ʿ̾ä硣*/
      if( string2->chara >= 0x2421 &&
	  string2->chara <= 0x2473 ){
	string1->charset = string2->charset ;
	string1->chara   = string2->chara + 0x0100 ;
	string1 ++ ;
	length1 -- ;
	string2 ++ ;
	length2 -- ;
	continue ;
      }
      /* ֤פä硣*/
      if( string2->chara == 0x2474 &&
	  length2 > 1 &&
	  ( string2[ 1 ].charset == CHARSET_JISX0208_1983 ||
	    string2[ 1 ].charset == CHARSET_JISX0208_1978 ) &&
	  string2[ 1 ].chara == 0x212b ){
	string1->charset = string2->charset ;
	string1->chara   = 0x2574 ;
	string1 ++ ;
	length1 -- ;
	string2 ++ ;
	string2 ++ ;
	length2 -- ;
	length2 -- ;
	continue ;
      }
    }
    /* ʳʸξˤϤΤޤޥԡ롣*/
    *string1 ++ = *string2 ++ ;
    length1 -- ;
    length2 -- ;
  }
  if( IS_END_OF_STRING( *string2 ) && length2 > 0 && length1 > 0 )
    MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return ( length1_backup - length1 ) ;
}

/*
 * δؿ ASCII STRING ʸ˥ԡؿǤ롣
 * äơɤäƤˤ̵ϤǤ롣ɤ
 * ƤˤϡStringToMyChar ϤδؿȤäƲC Υɤǡ
 * "abcdefg" Ȥ񤤤ˤ myChar Ѵ˻ȤǤ
 * (ǽʬȤΥ󥿡եä顢ToMyChar Ȥ
 * ȡ)
 */
int myCharCharStrcpy
( struct myChar *string1, unsigned char *string2 )
{
  while( *string2 != '\0' ){
    MYCHAR_SET_CHARA( *string1, *string2 ) ;
    string1 ++ ;
    string2 ++ ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharCharStrncpy
( struct myChar *string1, unsigned char *string2, int number )
{
  while( *string2 != '\0' && number > 0 ){
    if( ( *string2 ) & 0x80 ){
      string1->charset = CHARSET_JISX0208_1983 ;
      string1->chara   =
	( ( ( string2[ 0 ] ) << 8 ) | ( string2[ 1 ] ) ) & 0x7F7F ;
      string2 ++ ; string2 ++ ;
      number -- ;  number -- ;
    } else {
      MYCHAR_SET_CHARA( *string1, *string2 ) ;
      string2 ++ ;
      number -- ;
    }
    string1 ++ ;
  }
  if( number > 0 && *string2 == '\0' )
    MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharCharStrncmp
( struct myChar *string1, unsigned char *string2, int length )
{
  while( length > 0 ){
    if( !IS_ASCII_CHARA( *string1 ) )
      return True ;
    if( string1->chara != *string2 )
      return True ;
    /* ʸ򸫤롣*/
    string1 ++ ;
    string2 ++ ;
    length -- ;
  }
  return False ;
}

int myCharCharStrcmp
( struct myChar *string1, unsigned char *string2 )
{
  while( *string2 != '\0' && !IS_END_OF_STRING( *string1 ) ){
    if( !IS_ASCII_CHARA( *string1 ) )
      return True ;
    if( string1->chara != *string2 )
      return True ;
    /* ʸ򸫤롣*/
    string1 ++ ;
    string2 ++ ;
  }
  if( *string2 == '\0' && IS_END_OF_STRING( *string1 ) )
    return False ;
  return True ;
}

int myCharStrcat
( struct myChar *string1, struct myChar *string2 )
{
  /* ʸΰֺǸõ*/
  while( !IS_END_OF_STRING( *string1 ) )
    string1 ++ ;
  /* ʸ­ƹԤ*/
  while( !IS_END_OF_STRING( *string2 ) ){
    *string1 ++ = *string2 ++ ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharStrncat
( struct myChar *string1, struct myChar *string2, int length )
{
  /* ʸΰֺǸõ*/
  while( !IS_END_OF_STRING( *string1 ) )
    string1 ++ ;
  /* ʸ­ƹԤ*/
  while( length > 0 && !IS_END_OF_STRING( *string2 ) ){
    *string1 ++ = *string2 ++ ;
    length -- ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharHiraganaStrncat
( struct myChar *string1, struct myChar *string2, int length )
{
  /* ʸΰֺǸõ*/
  while( !IS_END_OF_STRING( *string1 ) )
    string1 ++ ;
  /* ʸ­ƹԤ*/
  while( length > 0 && !IS_END_OF_STRING( *string2 ) ){
    /* Ҳ̾ä鶯Ūʿ̾ľƥԡ롣*/
    if( string2->charset == CHARSET_JISX0208_1983 &&
	string2->chara >= 0x2521 &&
	string2->chara <= 0x2573 ){
      string1->charset = string2->charset ;
      string1->chara   = string2->chara - 0x0100 ;
      string1 ++ ;
      string2 ++ ;
    } else {
      *string1 ++ = *string2 ++ ;
    }
    length -- ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharHiraKataStrnncat
( struct myChar *string1, int length1, struct myChar *string2, int length2 )
{
  int length1_backup = length1 ;
  /* ʸΰֺǸõ*/
  while( !IS_END_OF_STRING( *string1 ) )
    string1 ++ ;
  /* ʸ­ƹԤ*/
  while( !IS_END_OF_STRING( *string2 ) &&
	 length1 > 0 && length2 ){
    /* Ҳ̾ä鶯Ūʿ̾ľƥԡ롣*/
    if( string2->charset == CHARSET_JISX0208_1983 ||
	string2->charset == CHARSET_JISX0208_1978 ){
      /* Ҳ̾ä硣*/
      if( string2->chara >= 0x2521 &&
	  string2->chara <= 0x2573 ){
	string1->charset = string2->charset ;
	string1->chara   = string2->chara - 0x0100 ;
	string1 ++ ;
	length1 -- ;
	string2 ++ ;
	length2 -- ;
	continue ;
      }
      /* ֥פä硣*/
      if( string2->chara == 0x2574 ){
	string1->charset = string2->charset ;
	string1->chara   = 0x2474 ;
	length1 -- ;
	if( length1 > 0 ){
	  string1->charset = string2->charset ;
	  string1->chara   = 0x212b ;
	  length1 -- ;
	}
	length2 -- ;
	continue ;
      } 
      /* ʿ̾ä硣*/
      if( string2->chara >= 0x2421 &&
	  string2->chara <= 0x2473 ){
	string1->charset = string2->charset ;
	string1->chara   = string2->chara + 0x0100 ;
	string1 ++ ;
	length1 -- ;
	string2 ++ ;
	length2 -- ;
	continue ;
      }
      /* ֤פä硣*/
      if( string2->chara == 0x2474 &&
	  length2 > 1 &&
	  ( string2[ 1 ].charset == CHARSET_JISX0208_1983 ||
	    string2[ 1 ].charset == CHARSET_JISX0208_1978 ) &&
	  string2[ 1 ].chara == 0x212b ){
	string1->charset = string2->charset ;
	string1->chara   = 0x2574 ;
	string1 ++ ;
	length1 -- ;
	string2 ++ ;
	string2 ++ ;
	length2 -- ;
	length2 -- ;
	continue ;
      }
    }
    /* ʳʸξˤϤΤޤޥԡ롣*/
    *string1 ++ = *string2 ++ ;
    length1 -- ;
    length2 -- ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return ( length1_backup - length1 ) ;
}

int myCharCharStrncat
( struct myChar *string1, unsigned char *string2, int length )
{
  /* ʸΰֺǸõ*/
  while( !IS_END_OF_STRING( *string1 ) )
    string1 ++ ;
  /* ʸ­ƹԤ*/
  while( length > 0 && *string2 != '\0' ){
    MYCHAR_SET_CHARA( *string1, *string2 ) ;
    string1 ++ ;
    string2 ++ ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharCharStrcat
( struct myChar *string1, unsigned char *string2 )
{
  /* ʸΰֺǸõ*/
  while( !IS_END_OF_STRING( *string1 ) )
    string1 ++ ;
  /* ʸ­ƹԤ*/
  while( *string2 != '\0' ){
    MYCHAR_SET_CHARA( *string1, *string2 ) ;
    string1 ++ ;
    string2 ++ ;
  }
  MYCHAR_SET_END_OF_STRING( *string1 ) ;
  return True ;
}

int myCharItoa( struct myChar *string, int value )
{
  int i, j ;
  for( i = 0, j = value ; j != 0 ; i ++, j = j / 10 )
    ;
  if( i == 0 ){
    MYCHAR_SET_CHARA( string[ 0 ], '0' ) ;
    MYCHAR_SET_END_OF_STRING( string[ 1 ] ) ;
    return True ;
  }
  MYCHAR_SET_END_OF_STRING( string[ i ] ) ;
  i -- ;
  for( j = value ; i >= 0 ; i -- ){
    MYCHAR_SET_CHARA( string[ i ], '0' + ( j % 10 ) ) ;
    j = j / 10 ;
  }
  return True ;
}

/*
 *  EUC ǽ񤭹ळȤͤƤ롣Ūˤϡä¿
 * 褦ˤɬפ뤫⤷ʤ
 */
int myCharFputEUCc( struct myChar chara, FILE *fp )
{
  switch( chara.charset ){
  case CHARSET_ASCII :
    fputc( chara.chara, fp ) ;
    break ;
  case CHARSET_JISX0201_1976 :
    if( chara.chara & 0x80 ){
      /* 󥰥륷ե SS1 롣*/
      fputc( 0x8e, fp ) ;
    }
    fputc( ( chara.chara & 0x00FF ), fp ) ;
    break ;
  case CHARSET_JISX0208_1978 :
  case CHARSET_JISX0208_1983 :
    fputc( ( chara.chara >> 8     ) | 0x80, fp ) ;
    fputc( ( chara.chara & 0x00FF ) | 0x80, fp ) ;
    break ;
    /* JISX0212-1990  EUC 3 Byte Code  */
  case CHARSET_JISX0212_1990 :
    /* 󥰥륷ե SS2 롣*/
    fputc( 0x8f, fp ) ;
    fputc( ( chara.chara >> 8     ) | 0x80, fp ) ;
    fputc( ( chara.chara & 0x00FF ) | 0x80, fp ) ;
    break ;
  default :
    /* ʳ charset  EUC ǤɽǤʤΤǡʤä
     * 夤Ȥʤɤͤġ*/
    break ;
  }
  return True ;
}

/*
 * Хåեؿ
 */
void init_messagebuffer( struct MessageBuffer *buf )
{
  buf->buffer  = buf->internal ;
  buf->bufsize = TEXTMAXLEN ;
  buf->usage   = 0 ;
  buf->buffer[ 0 ] = '\0' ;
  return ;
}

/*
 *
 */
void add_messagebuffer
( struct MessageBuffer *buf, int chara )
{
  if( ( buf->usage + 1 ) >= buf->bufsize ){
    if( buf->buffer == buf->internal ){
      /* TEXTBUFSIZE = TEXTMAXLEN + 1 Ǥ롣ͳϴñǡǸ NUL 
       * ʸ륹ڡɬפ顣*/ 
      buf->bufsize = ( TEXTBUFSIZE + TEXTMAXLEN ) ;
      buf->buffer = malloc
	( sizeof( unsigned char ) * ( buf->bufsize + 1 ) ) ;
      memcpy
	( buf->buffer, buf->internal, 
	  sizeof( unsigned char ) * buf->usage ) ;
    } else {
      buf->bufsize += TEXTBUFSIZE ;
      buf->buffer = realloc
	( buf->buffer, sizeof( unsigned char ) * ( buf->bufsize + 1 ) ) ;
    }
  }
  buf->buffer[ buf->usage ++ ] = chara ;
  buf->buffer[ buf->usage    ] = '\0' ;
  return ;
}

/*
 */
void close_messagebuffer( struct MessageBuffer *buf )
{
  if( buf->buffer != buf->internal )
    free( buf->buffer ) ;
  init_messagebuffer( buf ) ;
  return ;
}

void myCharFputstring( FILE *fp, struct myChar *string )
{
  while( !IS_END_OF_STRING( *string ) ){
    myCharFputEUCc( *string, fp ) ;
    string ++ ;
  }
  return ;
}

void myCharFputstrning( FILE *fp, struct myChar *string, int len )
{
  while( !IS_END_OF_STRING( *string ) && len > 0 ){
    myCharFputEUCc( *string, fp ) ;
    string ++ ;
    len -- ;
  }
  return ;
}

struct myChar *mychar_readOneLine( FILE *fp, int coding_system )
{
  struct MessageBuffer kanji_buf ;
  struct myChar *ptr = NULL ;
  int chara ;

  init_messagebuffer( &kanji_buf ) ;
  while( ( chara = fgetc( fp ) ) != EOF ){
    if( chara == '\n' || chara == '\r' )
      break ;
    add_messagebuffer( &kanji_buf, chara ) ;
  }
  switch( coding_system ){
  case KANJI_CODE_EUC :
    extendedUnixCode_stringToMycharset( kanji_buf.buffer, &ptr ) ;
    break ;
  case KANJI_CODE_JIS :
    iso2022jp_stringToMycharset( kanji_buf.buffer, &ptr ) ;
    break ;
  case KANJI_CODE_SJIS :
    microsoftKanjiCode_stringToMycharset( kanji_buf.buffer, &ptr ) ;
    break ;
  default :
    break ;
  }
  close_messagebuffer( &kanji_buf ) ;
  return ptr ;
}

int myCharAtoi( struct myChar *string )
{
  int value = 0 ;
  while( !IS_END_OF_STRING( *string ) &&
	 IS_ASCII_CHARA( *string ) &&
	 string->chara >= '0' &&
	 string->chara <= '9' ){
    value = value * 10 + ( string->chara - '0' ) ;
    string ++ ;
  }
  return value ;
}

unsigned char *myCharStringToAsciiString( struct myChar *string )
{
  unsigned char *asciiString, *cptr ;
  struct myChar *wptr ;
  int length ;

  length = 0 ;
  for( wptr = string ; !IS_END_OF_STRING( *wptr ) ; wptr ++ ){
    if( IS_ASCII_CHARA( *wptr ) )
      length ++ ;
  }
  asciiString = malloc( sizeof( unsigned char ) * ( length + 1 ) ) ;
  if( asciiString == NULL ){
    fprintf( stderr, "Memory exhaust.\n" ) ;
    exit( 1 ) ;
  }
  for( wptr = string, cptr = asciiString ;
       !IS_END_OF_STRING( *wptr ) ; wptr ++ ){
    if( IS_ASCII_CHARA( *wptr ) )
      *cptr ++ = wptr->chara ;
  }
  *cptr = '\0' ;
  return asciiString ;
}

