/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

/*****************************************************************

    c   o   n   t   e   x   t   .   c   p   p

*****************************************************************/

#include "context.h"
#include "tree.h"
#include "utf8.h"
#include <locale.h>

#include "guard.h"  // GP: clean

/*****************************************************************
    language name aliases
    primarily for different language names in Windows setlocale
    (unix setlocale respects ISO 639 which is also the standard
    for language codes in XML docs)
    SO FAR FOR TESTING PURPOSES

    each entry contains 2 strings with space-separated codes
    - first with the ISO codes (may appear in XML)
    - second with the setlocale-specific codes (needn't be ISO outside unix)
*****************************************************************/

struct LangAlias
{
    const char 
        *namesISO, *namesLocale;
};

LangAlias langAliases[] =
{
    {"en",                      "en_US"},
    {"de",                      "de_DE"},
    {"fr",                      "fr_FR"},
    {"it",                      "it_IT"},
    {"cze ces cs cs_CZ",        "cs_CZ czech"},
    {"sk slk slo sk_SK",        "sk_SK slovak"},
    {NULL, NULL}
};

/*****************************************************************
    C L i s t   methods
*****************************************************************/

CList::CList(Processor *proc_)
:
VertexList(LIST_SIZE_LARGE)
{
    sortDefs = NULL;
    refCount = 1;
    currLevel = 0;
    proc = proc_;
}

// GP: added method to kill values

CList::~CList()
{
    values.freeall(FALSE);
}

void CList::incRefCount()
{
    refCount++;
}

int CList::decRefCount()
{
    return --refCount;
}

//  compareWithoutDocOrd
//  assumes that the 'values' entries have been created
//  compares 2 vertices based on just the values, not their relative document order

int CList::compareWithoutDocOrd(int i, int j)
{
    assert(sortDefs && currLevel < sortDefs -> number());
    assert(i < values.number() && j < values.number());

    SortDef *def = (*sortDefs)[currLevel];
    // *** use case-order and lang from def!!!
    int result;
    if (def -> asText)
        result = strcmp(*values[i], *values[j]);
    else
    {
        Number n1, n2;
        n1 = *values[i];
        n2 = *values[j];
        if (n1 < n2)
            result = -1;
        else 
            result = (n2 < n1) ? 1 : 0;
    }
    return def -> ascend ? result : -result;
}


//  compare assumes that the 'values' entries have been created
int CList::compare(int i, int j)
{
    if (sortDefs)
    {
        int result = compareWithoutDocOrd(i, j);
        if (result)
            return result;
    }
    // sort in document order
    Vertex *v1 = block[i], *v2 = block[j];
    assert(v1 && v2);
    if (v1 -> stamp < v2 -> stamp)
        return -1;
    else if (v1 -> stamp > v2 -> stamp)
        return 1;
    else return 0;
}

Bool hasWord(const char *sentence, const char *w)
{
    const char *p;
    Str currWord;
    int len;
    for (p = sentence; *p; p += len + strspn(p, " "))
    {
        currWord.nset(p, len = strcspn(p, " "));
        if (currWord.eqNoCase(w)) return TRUE;
    }
    return FALSE;
}

char* setLang(const Str& src)
{
    char *result;
    const char *p;
    Str dest;
    int len;
    if (NULL != (result = setlocale(LC_COLLATE, src))) 
        return result;
    
    LangAlias *a = langAliases;
    while(a -> namesISO && !hasWord(a -> namesISO, src)) a++;
    if (a -> namesISO)
    {
        for (p = a -> namesLocale; *p; p += len + strspn(p, " "))
        {
            dest.nset(p, len = strcspn(p, " "));
            if (NULL != (result = setlocale(LC_COLLATE, dest))) 
                return result;
        }
    }
    return FALSE;
}

eFlag CList::sort(XSLElement *caller, Context *ctxt, SortDefList* sortDefs_ /* = NULL */)
{
    // GP: on error, locale is reset
    // checked that all callers destroy the context on error
    assert(caller || !sortDefs_);
    Str theLang;
    sortDefs = sortDefs_;
    if (sortDefs)
    {
        if (!setLang((*sortDefs)[0] -> lang))
        {
            Warn1(proc->situation, W1_LANG, (*sortDefs)[0] -> lang);
            setlocale(LC_COLLATE, "C");
        }
        // this is to reset the locale if there's an error
        E_DO( makeValues(0, number() - 1, 0, caller, ctxt), 
              setlocale(LC_COLLATE, "C") );
    }
    currLevel = 0;
    VertexList::sort(0, number() - 1);
    for (int i = 1; sortDefs && (i < sortDefs -> number()); i++)
    {
        if (!setLang((*sortDefs)[i] -> lang))
        {
            Warn1(proc->situation, W1_LANG, (*sortDefs)[i] -> lang);
            setlocale(LC_COLLATE, "C");
        }
        int j0 = 0;
        for (int j = 1; j <= number(); j++)
            if (j == number() || compareWithoutDocOrd(j0,j))
            {
                if (j > j0 + 1)
                {
                    currLevel = i;
                    E_DO( makeValues(j0, j-1, i, caller, ctxt), 
                          setlocale(LC_COLLATE, "C"));
                    // printf("Sorting %d:%d to %d\n",i,j0,j-1);
                    VertexList::sort(j0, j-1);
                    // reset currLevel for next compareWithoutDocOrd()
                    currLevel = i-1;
                };
                j0 = j;
            }
    }
    if (sortDefs)
    {
        // *** unset the temp locale as necessary
        setlocale(LC_ALL, "C");
        values.freeall(FALSE);
    }
    // reset the current element of parent context
    ctxt -> setPosition(0);
    return OK;
}

void CList::swap(int i, int j)
{
    VertexList::swap(i, j);
    if (sortDefs)
        values.swap(i, j);
}

eFlag CList::makeValues(int from, int to, int level, 
                        XSLElement *caller, Context *ctxt)
{
    assert(ctxt);
    if (!sortDefs) return OK;
    assert(level < sortDefs -> number());
    SortDef *def = (*sortDefs)[level];
    DStr d;
    GP( Str ) s;
    Expression *e, result(caller, caller -> ownerT -> proc, EXF_ATOM);
    for (int i = from; i <= to; i++)
    {
        ctxt -> setPosition(i);
        s = new Str;
        e = def -> sortExpr;
        if (!e)
        {
            E( operator[](i) -> value(d, ctxt) );
            if (def -> asText)
#               if defined(HAVE_WCSXFRM) || defined(_MSC_VER)
                *s = utf8xfrm(d);
#               else
                *s = d;
#               endif
            else
                *s = d;
        }
        else
        {
            E( e -> eval(result, ctxt) );
            if (def -> asText)
#               if defined(HAVE_WCSXFRM) || defined(_MSC_VER)
                *s = utf8xfrm(result.tostring());
#               else
                *s = result.tostring();
#               endif
            else
                *s = result.tostring();
        }
        if (!level)
            values.append(s.keep());
        else
        {
            cdelete(values[i]);
            values[i] = s.keep();
        }
    }
    return OK;
}

/*****************************************************************

    C o n t e x t    methods

******************************************************************/

Context::Context(Processor *proc_)
{
    proc = proc_;
    array = new CList(proc);
    currentNode = NULL;
    position = -1;
    virtualPosition = 0;
    virtualSize = -1;
}

Context::~Context()
{
    assert(array);
    if (!array -> decRefCount())
        delete array;
}

int Context::getPosition() const
{
    return position + virtualPosition;
}

int Context::getSize() const
{
    if (virtualSize != -1)
        return virtualSize;
    else
        return array -> number();
}

Vertex *Context::current() const
{
    if (!isFinished())
        return (*array)[position];
    else
        return NULL;
};

Vertex *Context::getCurrentNode() const
{
    if (currentNode)
        return currentNode;
    else
        return current();
}

void Context::setCurrentNode(Vertex * v)
{
    currentNode = v;
}

void Context::reset()
{
    if (!array -> number())
        position = -1;
    else
        position = 0;
}

Bool Context::isFinished() const
{
    return (Bool) !((position >= 0) && (position < array -> number()));
}

Bool Context::isVoid() const
{
    return (Bool) !(array -> number());
}

Vertex *Context::shift()
{
    if ((position >= 0) && (position < array -> number() - 1))
        return (*this)[++position];
    position = -1;
    return NULL;
}

void Context::set(Vertex *v)
{
    array -> append(v);
    reset();
}

void Context::setVirtual(Vertex *v, int virtualPosition_, int virtualSize_)
{
    assert(!array -> number() && "setVirtual() on nonvoid context");
    array -> append(v);
    virtualSize = virtualSize_;
    virtualPosition = virtualPosition_;
    reset();
}

Vertex *Context::operator[] (int n) const
{
    return (*array)[n];
}

void Context::deppendall()
{
    if (!array -> decRefCount())
        delete array;
    array = new CList(proc);
    position = -1;
}

void Context::append(Vertex *v)
{
    array -> append(v);
    reset();
}

void Context::deppend()
{
    array -> deppend();
    if (position >= array -> number())
        position = array -> number() - 1;
}

Context* Context::copy()
{
    Context *newc = new Context(proc);
    delete NZ(newc -> array);
    newc -> currentNode = currentNode;
    newc -> array = array;
    newc -> virtualPosition = virtualPosition;
    newc -> virtualSize = virtualSize;
    array -> incRefCount();
    newc -> reset();
    return newc;
}

void Context::swap(int i, int j)
{
    array -> swap(i, j);
}

Context *Context::swallow(Context *other)
{
    Context *result = new Context(proc); // GP: OK since no E()
    int i = 0, j = 0,
        iLimit = array -> number(), jLimit = other -> array -> number();
    long vstamp, wstamp;
    Vertex *v, *w;
    while ((i < iLimit) && (j < jLimit))
    {
        vstamp = (v = (*array)[i]) -> stamp;
        wstamp = (w = (*(other -> array))[j]) -> stamp;
        if (v == w) //XXXX wrong! need consistent document ordering
//        if (vstamp == wstamp)
        {
            j++;
            continue;
        }
        if (vstamp < wstamp)
        {
            result -> append(v);
            i++;
        }
        else
        {
            result -> append(w);
            j++;
        }
    }
    while (i < iLimit)
        result -> append((*array)[i++]);
    while (j < jLimit)
        result -> append((*(other -> array))[j++]);
    deppendall();
    other -> deppendall();
    return result;
}

Bool Context::contains(const Vertex *v) const
{
    int i;
    for (i = 0; i < array -> number(); i++)
    {
        if (v == (*array)[i])
            return TRUE;
    }
    return FALSE;
}

void Context::uniquize()
{
    int i;
    for (i = array -> number() - 2; i >= 0; i--)
        if (!array -> compare(i, i+1))
            array -> rm(i);
}

