/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 * Copyright (C) 2004 Markus Brueffer <markus@brueffer.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "mainactions.h"
#include <config.h>

#include "misc.h"
#include "kftpapi.h"
#include "kftpqueue.h"
#include "kftpbookmarks.h"
#include "configdialog.h"
#include "kftpsession.h"

#include "widgets/browser/view.h"
#include "widgets/browser/detailsview.h"
#include "widgets/browser/propsplugin.h"

#include "misc/config.h"
#include "misc/filter.h"

#include <kmainwindow.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kaboutapplication.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <kpropertiesdialog.h>
#include <kio/job.h>
#include <kshred.h>
#include <klineedit.h>

#include <qclipboard.h>

using namespace KFTPGrabberBase;
using namespace KFTPWidgets::Browser;
using namespace KFTPCore::Filter;

MainActions::MainActions(MainWindow *parent)
 : QObject(parent),
   m_configDialog(0),
   m_closeApp(false)
{
  // setup File menu
  m_fileConnectAction = new KAction(i18n("Quick &Connect..."), "connect_creating", KShortcut(), mainWidget(), SLOT(slotQuickConnect()), actionCollection(), "file_quick_connect");
  m_newSessionAction = new KActionMenu(i18n("&New Session"), "filenew", actionCollection(), "file_newsession");

  KAction *leftSide = new KAction(i18n("&Left Side"), KShortcut(), this, SLOT(slotNewSessionLeft()), actionCollection());
  KAction *rightSide = new KAction(i18n("&Right Side"), KShortcut(), this, SLOT(slotNewSessionRight()), actionCollection());
  m_newSessionAction->insert(leftSide);
  m_newSessionAction->insert(rightSide);
  m_newSessionAction->setStickyMenu(true);
  m_newSessionAction->setDelayed(false);
  KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());

  // setup Settings menu
  //KStdAction::configureToolbars(this, SLOT(slotConfigureToolbars()), actionCollection());
  //KStdAction::keyBindings(this, SLOT(slotKeyBindings()), actionCollection());
  KStdAction::saveOptions(this, SLOT(slotSettingsSave()), actionCollection());
  KStdAction::preferences(this, SLOT(slotSettingsConfig()), actionCollection());
  mainWidget()->createStandardStatusBarAction();
  mainWidget()->setStandardToolBarMenuEnabled(true);

  // setup transfermode actions
  m_transModeAction = new KActionMenu(i18n("&Transfer Mode (Auto)"), "binary", actionCollection(), "transfermode");
  m_modeAscii = new KRadioAction(i18n("ASCII"), "ascii", KShortcut(), this, SLOT(slotModeAscii()), actionCollection());
  m_modeBinary = new KRadioAction(i18n("Binary"), "binary", KShortcut(), this, SLOT(slotModeBinary()), actionCollection());
  m_modeAuto = new KRadioAction(i18n("Auto"), KShortcut(), this, SLOT(slotModeAuto()), actionCollection());

  // Set grouping so only one action can be selected
  m_modeAscii->setExclusiveGroup("Transfer mode");
  m_modeBinary->setExclusiveGroup("Transfer mode");
  m_modeAuto->setExclusiveGroup("Transfer mode");

  // Insert the actions into the menu
  m_transModeAction->insert(m_modeAscii);
  m_transModeAction->insert(m_modeBinary);
  m_transModeAction->popupMenu()->insertSeparator();
  m_transModeAction->insert(m_modeAuto);
  m_transModeAction->setStickyMenu(true);
  m_transModeAction->setDelayed(false);
  
  m_renameAction = new KAction(i18n("&Rename"), KShortcut(Qt::Key_F2), this, SLOT(slotRename()), actionCollection(), "edit_rename");
  m_deleteAction = new KAction(i18n("&Delete"), "editdelete", KShortcut(Qt::Key_Delete), this, SLOT(slotDelete()), actionCollection(), "edit_delete");
  m_propsAction = new KAction(i18n("&Properties"), KShortcut(), this, SLOT(slotProps()), actionCollection(), "edit_properties");
  m_shredAction = new KAction(i18n("&Shred"), "editshred", KShortcut(), this, SLOT(slotShred()), actionCollection(), "edit_shred");
  
  m_copyAction = KStdAction::copy(this, SLOT(slotCopy()), actionCollection(), "edit_copy");
  m_pasteAction = KStdAction::paste(this, SLOT(slotPaste()), actionCollection(), "edit_paste");
  
  m_filterActions = new KActionMenu(i18n("&Filter Options"), "", actionCollection(), "edit_filter_options");
  m_alwaysSkipAction = new KAction(i18n("Always &skip this file when queuing"), KShortcut(), this, SLOT(slotAlwaysSkip()), actionCollection());
  m_topPriorityAction = new KAction(i18n("Make this file &top priority"), KShortcut(), this, SLOT(slotTopPriority()), actionCollection());
  m_lowPriorityAction = new KAction(i18n("Make this file &lowest priority"), KShortcut(), this, SLOT(slotLowPriority()), actionCollection());
  
  m_filterActions->insert(m_alwaysSkipAction);
  m_filterActions->insert(m_topPriorityAction);
  m_filterActions->insert(m_lowPriorityAction);

  updateActions();
}

KActionCollection *MainActions::actionCollection()
{
  return KFTPAPI::getInstance()->mainWindow()->actionCollection();
}

KMainWindow *MainActions::mainWidget()
{
  return KFTPAPI::getInstance()->mainWindow();
}

void MainActions::updateActions()
{
  KFTPSession::Session *session = KFTPSession::Manager::self()->getActiveSession();
  
  if (session && session->isRegistred()) {
    m_renameAction->setEnabled(session->isConnected());
    m_deleteAction->setEnabled(session->isConnected());
    m_propsAction->setEnabled(true);
    m_shredAction->setEnabled(!session->isRemote());
    m_copyAction->setEnabled(true);
    m_pasteAction->setEnabled(true);
  } else {
    m_renameAction->setEnabled(false);
    m_deleteAction->setEnabled(false);
    m_propsAction->setEnabled(false);
    m_shredAction->setEnabled(false);
    m_copyAction->setEnabled(false);
    m_pasteAction->setEnabled(false);
  }
}

void MainActions::slotShred()
{
  KFTPWidgets::Browser::View *view = KFTPSession::Manager::self()->getActiveView();
  
  // Shred the file
  if (KMessageBox::warningContinueCancel(0, i18n("Are you sure you want to SHRED this file?"), i18n("Shred File"),KGuiItem(i18n("&Shred"), "editshred")) == KMessageBox::Cancel)
    return;
  
  KShred::shred(view->selectedItems()->getFirst()->url().path());
}

void MainActions::slotRename()
{
  KFTPWidgets::Browser::DetailsView *view = KFTPSession::Manager::self()->getActiveView()->getDetailsView();
  
  // Rename the first file in the current selection
  view->rename(view->KListView::selectedItems().at(0), 0);
  
  // Enhanced rename: Don't highlight the file extension. (from Konqueror)
  KLineEdit *le = view->renameLineEdit();
  
  if (le) {
    const QString txt = le->text();
    QString pattern;
    KMimeType::diagnoseFileName(txt, pattern);
    
    if (!pattern.isEmpty() && pattern.at(0) == '*' && pattern.find('*',1) == -1)
      le->setSelection(0, txt.length()-pattern.stripWhiteSpace().length()+1);
    else {
      int lastDot = txt.findRev('.');
      
      if (lastDot > 0)
        le->setSelection(0, lastDot);
    }
  }
}

void MainActions::slotDelete()
{
  KFTPSession::Session *session = KFTPSession::Manager::self()->getActiveSession();
  KFTPWidgets::Browser::View *view = session->getFileView();
  
  // Delete a file or directory
  KURL::List selection = view->selectedURLs();

  if (KMessageBox::warningContinueCancel(0, i18n("Are you sure you want to delete this file(s)?"), i18n("Delete File"),KGuiItem(i18n("&Delete"),"editdelete")) == KMessageBox::Cancel)
    return;

  // Go trough all files and delete them
  if (!session->isRemote()) {
    KIO::del(selection);
  } else {
    KURL::List::Iterator end(selection.end());
    
    for (KURL::List::Iterator i(selection.begin()); i != end; ++i) {
      if (!(*i).isLocalFile())
        session->getClient()->remove(KURL((*i).url()));
    }
  }
}

void MainActions::slotCopy()
{
  /*KFTPWidgets::Browser::DetailsView *view = KFTPSession::Manager::self()->getActiveView()->getDetailsView();
  
  QClipboard *cb = QApplication::clipboard();
  cb->setData(view->dragObject(), QClipboard::Clipboard);*/
}

void MainActions::slotPaste()
{
  KFTPWidgets::Browser::View *view = KFTPSession::Manager::self()->getActiveView();
  
  // Decode the data and try to init transfer
  KIO::MetaData p_meta;
  KURL::List p_urls;
  
  if (KURLDrag::decode(QApplication::clipboard()->data(), p_urls, p_meta)) {
    // Add destination url and call the QueueManager
    p_meta.insert("DestURL", view->url().url());
    KURLDrag *drag = new KURLDrag(p_urls, p_meta, view, name());
    KFTPQueue::Manager::self()->insertTransfer(drag);
  }
}

void MainActions::slotProps()
{
  KFTPSession::Session *session = KFTPSession::Manager::self()->getActiveSession();
  KFTPWidgets::Browser::View *view = session->getFileView();
  
  // Show file properties
  const KFileItemList *selectedItems = view->selectedItems();
  KFileItem *item = selectedItems->getFirst();
  
  if (selectedItems->count() == 0) {
    if (view->url().isLocalFile())
      item = new KFileItem(view->url(), 0, 0);
    else
      return;
  }

  // Show the dialog
  if (item->isLocalFile()) {
    KPropertiesDialog *propsDialog = new KPropertiesDialog(item);
    propsDialog->exec();
  } else {
    KPropertiesDialog *propsDialog = new KPropertiesDialog(item->name());
    propsDialog->insertPlugin(new KFTPWidgets::Browser::PropsPlugin(propsDialog, item));
    propsDialog->insertPlugin(new KFTPWidgets::Browser::PermissionsPropsPlugin(propsDialog, item, session));
    propsDialog->exec();
  }
}

void MainActions::addPriorityItems(int priority)
{
  KFTPWidgets::Browser::View *view = KFTPSession::Manager::self()->getActiveView();
  
  // Add the files to skiplist
  KURL::List selection = view->selectedURLs();
  KURL::List::Iterator end(selection.end());
    
  for (KURL::List::Iterator i(selection.begin()); i != end; ++i) {
    Rule *rule = new Rule();
    
    if (priority == 0) {
      rule->setName(i18n("Skip '%1'").arg((*i).filename()));
      const_cast<ConditionChain*>(rule->conditions())->append(new Condition(Filename, Condition::Is, (*i).filename()));
      const_cast<ActionChain*>(rule->actions())->append(new Action(Action::Skip, QVariant()));
    } else {
      rule->setName(i18n("Priority '%1'").arg((*i).filename()));
      const_cast<ConditionChain*>(rule->conditions())->append(new Condition(Filename, Condition::Is, (*i).filename()));
      const_cast<ActionChain*>(rule->actions())->append(new Action(Action::Priority, priority));
    }
    
    Filters::self()->append(rule);
  }
}

void MainActions::slotAlwaysSkip()
{
  addPriorityItems(0);
}

void MainActions::slotTopPriority()
{
  addPriorityItems(1);
}

void MainActions::slotLowPriority()
{
  addPriorityItems(-1);
}

void MainActions::slotModeAscii()
{
  m_transModeAction->setIcon("ascii");
  m_transModeAction->setText(i18n("&Transfer Mode (ASCII)"));
  KFTPCore::Config::self()->setGlobalMode('A');
}

void MainActions::slotModeBinary()
{
  m_transModeAction->setIcon("binary");
  m_transModeAction->setText(i18n("&Transfer Mode (Binary)"));
  KFTPCore::Config::self()->setGlobalMode('I');
}

void MainActions::slotModeAuto()
{
  m_transModeAction->setText(i18n("&Transfer Mode (Auto)"));
  KFTPCore::Config::self()->setGlobalMode('X');
}

void MainActions::slotFileQuit()
{
  m_closeApp = true;
  mainWidget()->close();
}

void MainActions::slotSettingsSave()
{
  KFTPCore::Config::self()->saveConfig();
}

void MainActions::slotSettingsConfig()
{
  if (!m_configDialog)
    m_configDialog = new KFTPWidgets::ConfigDialog(mainWidget());

  m_configDialog->prepareDialog();
  m_configDialog->exec();
}

void MainActions::slotNewSessionLeft()
{
  KFTPSession::Manager::self()->spawnLocalSession(KFTPSession::LeftSide, true);
}

void MainActions::slotNewSessionRight()
{
  KFTPSession::Manager::self()->spawnLocalSession(KFTPSession::RightSide, true);
}

void MainActions::slotKeyBindings()
{
  KKeyDialog::configure(actionCollection());
}

void MainActions::slotConfigureToolbars()
{
  KEditToolbar dlg(actionCollection());

  if (dlg.exec())
    mainWidget()->createGUI();
}

#include "mainactions.moc"
