/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "failedtransfers.h"
#include "kftpqueue.h"
#include "listview.h"
#include "misc.h"
#include "kftpapi.h"

#include <klocale.h>
#include <kpopupmenu.h>
#include <kio/job.h>
#include <kmessagebox.h>

#include <qlayout.h>
#include <qheader.h>

using namespace KFTPGrabberBase;

namespace KFTPWidgets {

FailedTransferItem::FailedTransferItem(KFTPQueue::FailedTransfer *transfer, QListView *parent)
  : KFTPWidgets::ListViewItem(parent),
    m_failedTransfer(transfer)
{
  QString desc = "<nobr><b>";
  desc += i18n("Transfer");
  desc += ":</b> ";
  desc += transfer->getTransfer()->getSourceUrl().prettyURL();
  desc += " <b>&gt;&gt;&gt;</b> ";
  desc += transfer->getTransfer()->getDestUrl().prettyURL();
  desc += "<br><b>";
  desc += i18n("Error");
  desc += ":</b> <i>";
  desc += transfer->getError();
  desc += "</i></nobr>";
  
  // Setup columns
  setRichText(0, desc);
  setText(1, KIO::convertSize(transfer->getTransfer()->getActualSize()));
  setText(2, QString::number(transfer->getTransfer()->getId()));
  
  setPixmap(0, loadSmallPixmap("info"));
}

KActionCollection *FailedTransfers::actionCollection()
{
  return KFTPAPI::getInstance()->mainWindow()->actionCollection();
}

FailedTransfers::FailedTransfers(QWidget *parent, const char *name)
 : QWidget(parent, name)
{
  QVBoxLayout *layout = new QVBoxLayout(this);
  
  // Create the list view
  m_list = new KFTPWidgets::ListView(this);
  
  // Create the columns
  m_list->addColumn(i18n("Description"), 500);
  m_list->addColumn(i18n("Size"), 75);
  m_list->header()->setStretchEnabled(true, 0);
  
  // Text when there are no failed transfers
  m_list->setEmptyListText(i18n("There are no failed transfers."));
  m_list->setAllColumnsShowFocus(true);
  
  layout->addWidget(m_list);
  
  connect(KFTPQueue::Manager::self(), SIGNAL(failedTransferNew(KFTPQueue::FailedTransfer*)), this, SLOT(slotFailedTransferNew(KFTPQueue::FailedTransfer*)));
  connect(KFTPQueue::Manager::self(), SIGNAL(failedTransferRemoved(long)), this, SLOT(slotFailedTransferRemoved(long)));
  
  connect(m_list, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)), this, SLOT(contextMenuRequested(KListView*, QListViewItem*, const QPoint&)));
  
  // Initialize the actions
  initActions();
}

FailedTransfers::~FailedTransfers()
{
}

void FailedTransfers::initActions()
{
  m_restartAction = new KAction(i18n("&Restart Transfer"), "launch", KShortcut(), this, SLOT(slotRestart()), actionCollection(), "launch");
  m_addToQueueAction = new KAction(i18n("&Add To Queue"), "queue", KShortcut(), this, SLOT(slotAddToQueue()), actionCollection(), "addtoqueue");
  m_addAllToQueueAction = new KAction(i18n("Add All To Queue"), KShortcut(), this, SLOT(slotAddAllToQueue()), actionCollection(), "addalltoqueue");
  m_removeAction = new KAction(i18n("R&emove"), "editdelete", KShortcut(), this, SLOT(slotRemove()), actionCollection(), "remove");
  m_removeAllAction = new KAction(i18n("Remove All"), KShortcut(), this, SLOT(slotRemoveAll()), actionCollection(), "removeall");
}

void FailedTransfers::updateActions()
{
  m_restartAction->setEnabled(m_list->selectedItem());
  m_addToQueueAction->setEnabled(m_list->selectedItem());
  m_removeAction->setEnabled(m_list->selectedItem());
  
  m_addAllToQueueAction->setEnabled(m_list->childCount());
  m_removeAllAction->setEnabled(m_list->childCount());
}

void FailedTransfers::slotFailedTransferNew(KFTPQueue::FailedTransfer *transfer)
{
  // Create a new failed transfer list item
  new FailedTransferItem(transfer, m_list);
}

void FailedTransfers::slotFailedTransferRemoved(long id)
{
  // Delete the failed transfer item
  QListViewItem *item = m_list->findItem(QString::number(id), 2);
  
  if (item)
    delete item;
}

void FailedTransfers::slotRestart()
{
  // First restore the transfer
  KFTPQueue::TransferFile *transfer = static_cast<FailedTransferItem*>(m_list->selectedItem())->getFailedTransfer()->restore();
  
  // Then execute it
  transfer->delayedExecute();
}

void FailedTransfers::slotAddToQueue()
{
  // Restore the transfer
  static_cast<FailedTransferItem*>(m_list->selectedItem())->getFailedTransfer()->restore();
}

void FailedTransfers::slotAddAllToQueue()
{
  KFTPQueue::FailedTransfer *transfer;
  QPtrList<KFTPQueue::FailedTransfer> *list = KFTPQueue::Manager::self()->getFailedTransferList();
  QPtrListIterator<KFTPQueue::FailedTransfer> i(*list);
  
  while ((transfer = i.current()) != 0) {
    ++i;
    
    // Restore the transfer
    transfer->restore();
  }
}

void FailedTransfers::slotRemove()
{
  if (KMessageBox::questionYesNo(this, i18n("Are you sure you want to remove this failed transfer?")) == KMessageBox::Yes) {
    // Remove the failed transfer
    KFTPQueue::Manager::self()->removeFailedTransfer(static_cast<FailedTransferItem*>(m_list->selectedItem())->getFailedTransfer());
  }
}

void FailedTransfers::slotRemoveAll()
{
  if (KMessageBox::questionYesNo(this, i18n("Are you sure you want to remove ALL failed transfers?")) == KMessageBox::Yes) {
    KFTPQueue::Manager::self()->clearFailedTransferList();
  }
}

void FailedTransfers::contextMenuRequested(KListView*, QListViewItem*, const QPoint &p)
{
  KPopupMenu *contextMenu = new KPopupMenu(m_list);
  
  m_restartAction->plug(contextMenu);
  m_addToQueueAction->plug(contextMenu);
  m_addAllToQueueAction->plug(contextMenu);
  contextMenu->insertSeparator();
  m_removeAction->plug(contextMenu);
  m_removeAllAction->plug(contextMenu);
  
  // Update the actions
  updateActions();
  
  contextMenu->exec(p);
}

}

#include "failedtransfers.moc"

