/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#ifndef KFTPSERVERMANAGEMENTLISTVIEW_H
#define KFTPSERVERMANAGEMENTLISTVIEW_H

#include "../listview.h"
#include "kftpservermanagement.h"

namespace KFTPEngine {
  class Thread;
}

namespace KFTPSession {
  class Session;
}

class QPoint;

class KAction;
class KPopupMenu;
class KListViewItem;
class KActionCollection;

namespace KFTPWidgets {

class Balloon;

namespace ServerManagement {

class Dialog;
class DialogWidget;

/**
 * @author Blacknator
 *
 * @brief represents an item in the list view
 *
 * This Item is used in the class KFTPServerManagementListView to represent a item in the list view.<br>
 * The Item contains all information about the represented command, e.g. description, complete description(whats this),
 * and in future regular expressions to split the output of the command
 */
class ListViewItem : public KListViewItem
{
friend class DialogWidget;
private:
    /** @brief minimal and maximal value */
    QString m_valueMinMax;
    /** @brief minimal value */
    int m_valueMin;
    /** @brief maximum value */
    int m_valueMax;
    /** @brief description data */
    QString m_description;
    /** @brief what's this data */
    QString m_whatThis;

public:
    ListViewItem(QListView *parent, const QString &text);
    ListViewItem(QListViewItem *parent, const QString &text);

    /** set the what's this data
     * @param whatThis contains the what's this data
     */
    void setDescription(QString whatThis);

    /** set the what's this data
     * @param whatThis contains the what's this data
     */
    void setWhatThis(QString whatThis) { m_whatThis = whatThis; }

    QString whatsThis() { return m_whatThis; }

    /**
     * set minimun and maximun value range
     * @param range contains value, e.g. 0-7200
     */
    void setValueRange(QString range);

};

/**
 * @author Blacknator
 *
 * @brief Server Management List View
 *
 * This class represents a common list view for all server layouts.<br>
 * It's generated in the method KFTPServerManagement::Manager::guiPopulateServerManagementTree()
 * and the data is taken from file server_management.xml
 */
class ListView : public KFTPWidgets::ListView
{
Q_OBJECT
public:
    ListView(KFTPServerManagement::Manager *manager, QWidget* parent=0, const char* name=0);
    ~ListView() {};

    /** Needed to debug the Constructor with gdb */
    void init();

    /** init all KActions, KActions are inseret in the Popup Menu */
    void initActions();

    /** shows a balloon at the current clicked item position
     * @param text text to show in the balloon
     */
    void showBalloon(const QString& text);
public slots:
    /** hides the balloon */
    void slotHideBalloon();
protected slots:
    /** send a command to the ftp client
     * Signal is emmited from KFTPServerManagementDialog
     * @param execute true if no error occured, otherwise the dialog is showing again
     * @param command command to send
     */
    void slotSendCommand(bool execute, QString command);

    /** item in the list was double clicked with the mouse<br>
     * show a dialog if the user must input someting
     * @param item item that was double clicked
     */
    void slotDoubleClickedItem(QListViewItem *item, const QPoint &pos, int column);

    /** this method collects the output from the executed ftp command<br>
     * and ends if 500 milli seconds long no ouput from the ftp comes
     * @param text emit this tex
     */
    void slotCollectText(const QString &text);

    /** if the sended command was site stat, all output arrieves at this slot<br>
     * to fill the stat label (removed, maybe added in the future again
     * @param text the returned text from the site stat command before
     */
    void slotSiteStat(const QString &text);
signals:
    /** this signal is connected to KFTPServerManagementSidebar::slotSetStatus(const QString&)
     * @param text text to set in the stat label
     */
    void setStatus(const QString& text);
private slots:
    /** disconnect the signal rawReply(const QString&) */
    void slotDisconnectBalloon();

    /**
     * Context Menu requested ( right mouse button )
     * @param item Item that was pressed
     * @param pos the position of the item
     * @param column the pressed column
     */
    void slotContextMenu(QListViewItem *item, const QPoint &pos, int column);

    /** What's This Entry in the popup clicked */
    void slotWhatsThis();
private:
    /**
     * Check the command @val command for user defined aguments<br>
     * popup a dialog if necessary and send the command
     * @param command command with, maybe user defined settings
     */
    void showDialog();

    /** @brief current selected item */
    ListViewItem* m_item;
    /** @brief current selected item position */
    QPoint m_itemPos;
    /** @brief timer to receive the whole output from the ftp server */
    QTimer *m_timer1;
    /** @brief timer to hide the balloon */
    QTimer *m_timer2;

    /** @brief the whole text from the output (see m_timer1) and slotCollectText() to display in the balloon */
    QString m_balloonText;
    /** @brief balloon to show if data from the sended command arrieves */
    Balloon *m_balloon;
    /** @brief current selected Session Side */
    KFTPSession::Session *m_session;
    /** @brief current KFTPEngine::Thread */
    KFTPEngine::Thread *m_client;
    /** @brief Manager who contains the complete xml data */
    KFTPServerManagement::Manager* m_manager;
    /** @brief user defined dialog */
    Dialog *m_dialog;

    /** @brief popup Menu in the list view */
    KPopupMenu *m_popupMenu;
    /** @brief What's This Action in the popup menu */
    KAction *m_actionWhatsThis;
    /** @brief the KActionCollection for all KAction's */
    KActionCollection *m_actionCollection;
};

}

}

#endif
