#ifndef K3DSDK_PATH_H
#define K3DSDK_PATH_H

// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <boost/filesystem/path.hpp>
#include <vector>

namespace k3d
{

namespace filesystem
{

/// Converts the given absolute path into a relative path, relative to the given reference path
/** \todo Update this to use k3d::filesystem::path */
const boost::filesystem::path make_relative_path(const boost::filesystem::path& AbsolutePath, const boost::filesystem::path& ReferencePath);

/// Portable filesystem path that closely follows the boost::filesystem::path interface
/** \todo Replace all instances of boost::filesystem::path with k3d::filesystem::path */
class path
{
public:
	/// Constructs an empty path
	path();
	/// Constructs a path from a string containing the generic grammar
	friend path generic_path(const std::string& GenericPath);
	/// Constructs a path from a native file string, i.e. any valid Posix or Win32 path (regardless of the platform on which the code executes)
	friend path native_path(const std::string& NativePath);

	/// Path concatenation
	path& operator/=(const path& rhs);
	/// Path concatenation
	path operator/(const path& rhs) const;
	
	// conversion functions:
	std::string string() const;
	std::string native_file_string() const;

/*
	// modification functions:
	path& normalize();

	// decomposition functions:
	path root_path() const;
	std::string root_name() const;
	std::string root_directory() const;
	path relative_path() const;
	std::string leaf() const;
	path branch_path() const;
*/
	
	// query functions: 
	bool empty() const;
/*
	bool is_complete() const;
	bool has_root_path() const;
	bool has_root_name() const;
	bool has_root_directory() const;
	bool has_relative_path() const;
	bool has_leaf() const;
	bool has_branch_path() const;
*/

/*
	// iteration:
	class iterator
	{
	public:
		typedef std::forward_iterator_tag iterator_category;
		typedef std::string value_type;
		typedef std::string& reference;
		typedef std::string* pointer;
		typedef ptrdiff_t difference_type;

		iterator();
		iterator(const std::string& Storage);

		value_type operator*();

		bool operator==(const iterator& RHS);
		bool operator!=(const iterator& RHS);
		iterator& operator++();
		
	private:
		const std::string* storage;
		std::string::size_type index;
	};
	
	iterator begin() const;
	iterator end() const;
*/

	// relational operators:
	bool operator==(const path& that) const;
	bool operator!=(const path& that) const;
	bool operator<(const path& that) const;
	bool operator<=(const path& that) const;
	bool operator>(const path& that) const;
	bool operator>=(const path& that) const;

private:
	path(const std::string& GenericPath);
	std::string storage;
};

/// Constructs a path from a string containing the generic grammar
path generic_path(const std::string& GenericPath);
/// Constructs a path from a native file string, i.e. any valid Posix or Win32 path (regardless of the platform on which the code executes)
path native_path(const std::string& NativePath);

/// Defines a collection of paths
typedef std::vector<path> path_list;

/// Returns true if the operating system reports that the given file exists
bool exists(const path& Path);

/// Splits a string containing delimited paths (such as the PATH environment variable) into a collection of individual paths
/** \note: Splits the string using the filesystem delimiter for the native system (the system on which the code is executing). */
const path_list split_native_paths(const std::string& Paths);

} // namespace filesystem

} // namespace k3d

#endif // !K3DSDK_PATH_H

