
/*
 *  devicelisting.cpp
 *
 *  Copyright (C) 2009 David Hubner <hubnerd@ntlworld.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "devicelisting.h"

DeviceListing::DeviceListing(QWidget *parent, InfoPanel *info) : 
  QTreeWidget(parent), iPanel(info) {
  
    connect(this,SIGNAL(itemClicked(QTreeWidgetItem *,int)),this,SLOT(deviceItemClicked(QTreeWidgetItem *,int)));
    
    setHeaderLabels(QStringList(i18n("Devices")));
    populateListing();
}

DeviceListing::~DeviceListing() {
  
  clear();
}

SolDevice *DeviceListing::createSubListItem(QTreeWidgetItem *parent, QString text) {

  SolDevice *subItem = new SolDevice(parent);
  subItem->setText(0,text);
  
  return subItem;
}

QTreeWidgetItem *DeviceListing::createListItems(Solid::DeviceInterface::Type type) {

    Solid::AudioInterface *auddev; 
  
    QList<Solid::Device> list = Solid::Device::listFromType(type, QString());

    SolDevice *tempRootHolder = new SolDevice();
    tempRootHolder->setText(0,Solid::DeviceInterface::typeToString(type));
    
    // Hack to allow OSS and ALSA in sub-categories, don't like this but will do better later. 
    SolDevice *oss = NULL; SolDevice *alsa = NULL; 
    switch(type) {
      case Solid::DeviceInterface::AudioInterface:
	alsa = createSubListItem(tempRootHolder,i18n("Alsa Devices"));
	oss = createSubListItem(tempRootHolder,i18n("Open Sound System Devices"));
	break;
      default:
	break;
    }
    
    foreach(Solid::Device dev, list) {
      switch(type) {
	case Solid::DeviceInterface::Processor: 
	  new SolProcessorDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::StorageDrive:
	  new SolStorageDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::NetworkInterface:
	  new SolNetworkDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::AudioInterface:
	  
	  auddev = tempRootHolder->getInterface<Solid::AudioInterface>(dev); 
	  switch(auddev->driver()) {
	    case Solid::AudioInterface::Alsa:
	      new SolAudioDevice(alsa,dev);
	      break;
	    case Solid::AudioInterface::OpenSoundSystem:
	      new SolAudioDevice(oss,dev);
	      break;
	    default:
	      new SolAudioDevice(tempRootHolder,dev);
	      break;
	  }
	  
	  break;
	case Solid::DeviceInterface::Camera:
	  new SolCameraDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::PortableMediaPlayer:
	  new SolMediaPlayerDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::Button:
	  new SolButtonDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::Battery:
	  new SolBatteryDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::AcAdapter:
	  new SolAcAdapterDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::DvbInterface:
	  new SolDvbDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::SerialInterface:
	  new SolSerialDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::SmartCardReader:
	  new SolSmartCardDevice(tempRootHolder,dev);
	  break;
	case Solid::DeviceInterface::Video:
	  new SolVideoDevice(tempRootHolder,dev);
	  break;
	default:
	  new SolDevice(tempRootHolder,dev);
	  break;
      }
    }
    return tempRootHolder;
}

void DeviceListing::populateListing() {
 
  Solid::DeviceInterface::Type needHardWare[] = { 
    Solid::DeviceInterface::Processor,
    Solid::DeviceInterface::StorageDrive,
    Solid::DeviceInterface::NetworkInterface,
    Solid::DeviceInterface::AudioInterface,
    Solid::DeviceInterface::Video,
    Solid::DeviceInterface::SerialInterface,
    Solid::DeviceInterface::SmartCardReader,
    Solid::DeviceInterface::DvbInterface,
    Solid::DeviceInterface::Button,
    Solid::DeviceInterface::Battery, 
    Solid::DeviceInterface::AcAdapter,
    Solid::DeviceInterface::PortableMediaPlayer,
    Solid::DeviceInterface::Camera
    //Solid::DeviceInterface::Unknown
  };
  
  clear();
  
  // CHANGE
  for(unsigned int i=0;i<(sizeof(needHardWare)/sizeof(Solid::DeviceInterface::Type));i++) { 
    addTopLevelItem(createListItems(needHardWare[i]));
  }
}

void DeviceListing::deviceItemClicked(QTreeWidgetItem *listItemIn ,int columnIn) {

  if(columnIn) return;
  
  SolDevice *listItem = static_cast<SolDevice *>(listItemIn);
  if(listItem->isDeviceSet()) {
    iPanel->setTopInfo(listItem->getDeviceIcon(),listItem->getDevice());
    iPanel->setBottomInfo(listItem->getInfoPanelLayout());
  }
}