/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-08-25
 * Description : a plugin to simulate Oil Painting
 *
 * Copyright (C) 2004-2009 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2006-2009 by Marcel Wiesweg <marcel dot wiesweg at gmx dot de>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */


#include "oilpainttool.h"
#include "oilpainttool.moc"

// Qt includes

#include <QGridLayout>
#include <QImage>
#include <QLabel>

// KDE includes

#include <kaboutdata.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kconfiggroup.h>
#include <kcursor.h>

#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kstandarddirs.h>

// LibKDcraw includes

#include <libkdcraw/rnuminput.h>

// Local includes

#include "dimg.h"
#include "editortoolsettings.h"
#include "imageiface.h"
#include "imagepanelwidget.h"
#include "oilpaint.h"
#include "version.h"

using namespace KDcrawIface;
using namespace Digikam;

namespace DigikamOilPaintImagesPlugin
{

class OilPaintToolPriv
{
public:

    OilPaintToolPriv()
    {
        brushSizeInput = 0;
        smoothInput    = 0;
        previewWidget  = 0;
        gboxSettings   = 0;
    }

    RIntNumInput*       brushSizeInput;
    RIntNumInput*       smoothInput;

    ImagePanelWidget*   previewWidget;
    EditorToolSettings* gboxSettings;
};

OilPaintTool::OilPaintTool(QObject* parent)
            : EditorToolThreaded(parent),
              d(new OilPaintToolPriv)
{
    setObjectName("oilpaint");
    setToolName(i18n("Oil Paint"));
    setToolIcon(SmallIcon("oilpaint"));

    // -------------------------------------------------------------

    d->gboxSettings = new EditorToolSettings;
    d->gboxSettings->setButtons(EditorToolSettings::Default|
                                EditorToolSettings::Ok|
                                EditorToolSettings::Cancel|
                                EditorToolSettings::Try);

    d->gboxSettings->setTools(EditorToolSettings::PanIcon);

    // -------------------------------------------------------------

    QLabel *label1    = new QLabel(i18n("Brush size:"));
    d->brushSizeInput = new RIntNumInput();
    d->brushSizeInput->setRange(1, 5, 1);
    d->brushSizeInput->setSliderEnabled(true);
    d->brushSizeInput->setDefaultValue(1);
    d->brushSizeInput->setWhatsThis(i18n("Set here the brush size to use for simulating the oil painting."));

    // -------------------------------------------------------------

    QLabel *label2  = new QLabel(i18nc("value of smoothing effect", "Smooth:"));
    d->smoothInput  = new RIntNumInput();
    d->smoothInput->setRange(10, 255, 1);
    d->smoothInput->setSliderEnabled(true);
    d->smoothInput->setDefaultValue(30);
    d->smoothInput->setWhatsThis(i18n("This value controls the smoothing effect of the brush under the canvas."));

    QGridLayout* mainLayout = new QGridLayout();
    mainLayout->addWidget(label1,            0, 0, 1, 2);
    mainLayout->addWidget(d->brushSizeInput, 1, 0, 1, 2);
    mainLayout->addWidget(label2,            2, 0, 1, 2);
    mainLayout->addWidget(d->smoothInput,    3, 0, 1, 2);
    mainLayout->setRowStretch(4, 10);
    mainLayout->setMargin(d->gboxSettings->spacingHint());
    mainLayout->setSpacing(d->gboxSettings->spacingHint());
    d->gboxSettings->plainPage()->setLayout(mainLayout);

    // -------------------------------------------------------------

    setToolSettings(d->gboxSettings);

    // -------------------------------------------------------------

    d->previewWidget = new ImagePanelWidget(470, 350, "oilpaint Tool", d->gboxSettings->panIconView());

    setToolView(d->previewWidget);
    init();
}

OilPaintTool::~OilPaintTool()
{
    delete d;
}

void OilPaintTool::renderingFinished()
{
    d->brushSizeInput->setEnabled(true);
    d->smoothInput->setEnabled(true);
}

void OilPaintTool::readSettings()
{
    KSharedConfig::Ptr config = KGlobal::config();
    KConfigGroup group        = config->group("oilpaint Tool");
    d->brushSizeInput->blockSignals(true);
    d->smoothInput->blockSignals(true);
    d->brushSizeInput->setValue(group.readEntry("BrushSize", d->brushSizeInput->defaultValue()));
    d->smoothInput->setValue(group.readEntry("SmoothAdjustment", d->smoothInput->defaultValue()));
    d->brushSizeInput->blockSignals(false);
    d->smoothInput->blockSignals(false);
}

void OilPaintTool::writeSettings()
{
    KSharedConfig::Ptr config = KGlobal::config();
    KConfigGroup group        = config->group("oilpaint Tool");
    group.writeEntry("BrushSize", d->brushSizeInput->value());
    group.writeEntry("SmoothAdjustment", d->smoothInput->value());
    d->previewWidget->writeSettings();
    group.sync();
}

void OilPaintTool::slotResetSettings()
{
    d->brushSizeInput->blockSignals(true);
    d->smoothInput->blockSignals(true);

    d->brushSizeInput->slotReset();
    d->smoothInput->slotReset();

    d->brushSizeInput->blockSignals(false);
    d->smoothInput->blockSignals(false);

    slotEffect();
}

void OilPaintTool::prepareEffect()
{
    d->brushSizeInput->setEnabled(false);
    d->smoothInput->setEnabled(false);

    DImg image = d->previewWidget->getOriginalRegionImage();
    int b      = d->brushSizeInput->value();
    int s      = d->smoothInput->value();

    setFilter(dynamic_cast<DImgThreadedFilter*>(new OilPaint(&image, this, b, s)));
}

void OilPaintTool::prepareFinal()
{
    d->brushSizeInput->setEnabled(false);
    d->smoothInput->setEnabled(false);

    int b = d->brushSizeInput->value();
    int s = d->smoothInput->value();

    ImageIface iface(0, 0);
    setFilter(dynamic_cast<DImgThreadedFilter*>(new OilPaint(iface.getOriginalImg(), this, b, s)));
}

void OilPaintTool::putPreviewData()
{
    d->previewWidget->setPreviewImage(filter()->getTargetImage());
}

void OilPaintTool::putFinalData()
{
    ImageIface iface(0, 0);

    iface.putOriginalImage(i18n("Oil Paint"), filter()->getTargetImage().bits());
}

}  // namespace DigikamOilPaintImagesPlugin
