#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#define G_LOG_DOMAIN "dialog"
#include "gvn-commit.h"

#include <gnubversion.h>
#include <assert.h>
#include <ctype.h>

#define MAIN_WINDOW gnubversion_get_widget("commitApp")

/* Flags for keeping track of *why* we disabled the Commit button. */
#define OKFLAGS_BAD_TICKS  ( 1 << 0 )
#define OKFLAGS_BAD_LOGMSG ( 1 << 1 )

/* bitmask of OKFLAGS_BAD_xxx of why the commit button is disabled */
static int okbutton_badflags = 0;

static void
set_status ( const char *status )
{
    GnomeAppBar *appbar = GNOME_APPBAR ( gnubversion_get_widget ( "mainAppBar" ) );
    gnome_appbar_clear_stack ( appbar );
    if ( status != NULL )
        gnome_appbar_push ( appbar, status );
}

/*
 * Enables/disables the OK button in the main window.
 */
static void 
update_okbutton ( int mask, gboolean flagset, const char *message )
{
    if ( flagset )
        okbutton_badflags |= mask;
    else
        okbutton_badflags &= ~mask;

    gtk_widget_set_sensitive ( gnubversion_get_widget ( "commitButton"), 
                               ( okbutton_badflags == 0 ) );

    if ( flagset )
        set_status ( message );
    else 
        set_status ( NULL );
}

static GtkTextBuffer *
get_logbuffer ()
{
    GtkTextBuffer *buf;

    GtkTextView *msgView = GTK_TEXT_VIEW (gnubversion_get_widget ("logMessage"));
    buf =  gtk_text_view_get_buffer (msgView);
    assert (buf != NULL);
    return buf;
}

static gchar *
get_logmessage ()
{
    GtkTextBuffer *buf = get_logbuffer ();
    GtkTextIter start, end;

    gtk_text_buffer_get_start_iter (buf, &start);
    gtk_text_buffer_get_end_iter (buf, &end);

    return gtk_text_buffer_get_text (buf, &start, &end, FALSE);
}

static gboolean 
has_letters ( const gchar *s)
{
    for ( ; (*s) != 0; s++)
    {
        if (isalpha(*s))
            return TRUE;
    }
    return FALSE;
}

static void
validate_logmessage ()
{
    gchar *log_message = get_logmessage ();
    gchar *status_message = _("Invalid log message");
    gboolean msg_bad = FALSE;

    if ( (log_message == NULL) || (strlen (log_message) == 0) )
    {
        msg_bad = TRUE;
        status_message = _("Log message is empty");
    }
    else if ( ! has_letters (log_message))
    {
        msg_bad = TRUE;
        status_message = _("Log message is invalid: no letters at all");
    }

    update_okbutton ( OKFLAGS_BAD_LOGMSG, 
                      msg_bad,
                      status_message);
}

static void
validate_commitlist (GnubVersionFileSelectorStore *store)
{
    int ticks = gnubversion_fileselectorstore_count_ticks (store);

    update_okbutton ( OKFLAGS_BAD_TICKS,
                      ticks == 0,
                      _("No files are selected") );
}

/* This only gets used in non-interactive mode */
static void task_finished_cb(GnubVersionProgressWindow *progress, gboolean successfull, gpointer data)
{
    gtk_main_quit();
}

static void progressWindow_closed_cb(GnubVersionProgressWindow *progress, gpointer data)
{
    if (gnubversion_progresswindow_was_task_successfull(progress))
    {
        gtk_main_quit();
    }
    else
    {
        gtk_widget_show (MAIN_WINDOW);
        set_status ( NULL );
    }
}

GLADE_CALLBACK void
commitButton_clicked_cb ( GtkWidget *widget, gpointer userdata )
{
    struct worker_thread_baton *baton = g_malloc(sizeof(struct worker_thread_baton));

    GnubVersionFileSelector *selector = GNUBVERSION_FILESELECTOR(gnubversion_get_widget ("fileSelector"));

    baton->store = gnubversion_fileselector_get_model (selector);
    assert(baton->store != NULL);

    baton->commitMessage = get_logmessage();

    gtk_widget_hide (MAIN_WINDOW);
    baton->progressWindow = gnubversion_progresswindow_new ( _("Preparing Commit"));
    assert(baton->progressWindow != NULL);
    gnubversion_auth_baton->progressWindow = baton->progressWindow;

    gtk_window_set_title ( GTK_WINDOW (baton->progressWindow), _("Commit Progress"));
    gtk_widget_show ( GTK_WIDGET(baton->progressWindow));
    
    gnubversion_progresswindow_connect__window_closed(baton->progressWindow, 
                                                      progressWindow_closed_cb, 
                                                      baton);

    if (gnubversion_get_options()->non_interactive)
        gnubversion_progresswindow_connect__task_finished(baton->progressWindow, task_finished_cb, NULL);

    GThread *worker = NULL;
    worker = g_thread_create ( commit_thread_main, baton, TRUE, NULL );
    assert ( worker != NULL );

    gvn_debug ("Main thread");
}

void 
dialog_init (gboolean non_recursive, const gchar **args, GString *log_message)
{
    GnubVersionFileSelectorStore *store;
    
    store = commit_list_init (non_recursive, args);

    GtkTextBuffer *buf = get_logbuffer ();
    assert(buf != NULL);
    gtk_text_buffer_set_text(buf, log_message->str, -1);

    GnubVersionFileSelector *selector;
    selector = GNUBVERSION_FILESELECTOR (gnubversion_get_widget ("fileSelector"));
    gnubversion_fileselector_set_label_text( selector, _("Files to Commit:"));
    gnubversion_fileselector_set_tick_column_title( selector, _("Commit"));

    gtk_widget_show (MAIN_WINDOW);

    validate_logmessage ();
    validate_commitlist (store);
    g_signal_connect ( get_logbuffer (), "changed", G_CALLBACK (validate_logmessage), NULL);

    /* Since we will never add/remove rows once initialized, we do not need
     * to connect to the row-inserted and row-deleted signals */
    g_signal_connect ( store, "row-changed", G_CALLBACK (validate_commitlist), NULL);
}

void 
dialog_run ()
{
    if (gnubversion_get_options()->non_interactive)
        commitButton_clicked_cb ( gnubversion_get_widget("commitButton"), NULL);

    gtk_main ();
}

void
dialog_shutdown ()
{
}
