#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define G_LOG_DOMAIN "main"

#include <gnome.h>
#include <glade/glade.h>
#include <gnubversion.h>
#include <assert.h>
#include "gvn-update.h"

#define APP_NAME "gvn-update"

#define MAIN_WINDOW gnubversion_get_widget ("updateApp")
#define FILESELECTOR GNUBVERSION_FILESELECTOR (gnubversion_get_widget ("fileselector"))
#define OPTIONS GNUBVERSION_COMMONOPTIONS (gnubversion_get_widget ("options"))
#define REVISION GNUBVERSION_REVISIONCHOOSER (gnubversion_get_widget ("revisionchooser"))

static void fileselector_ticks_changed(
        GnubVersionFileSelector *selector, 
        GnubVersionFileSelectorStore *store, 
        gpointer data);

static void
update_updateButton(GtkButton *widget, GnubVersionFileSelectorStore *store);

GLADE_CALLBACK void
updateButton_clicked_cb ( GtkWidget *widget, gpointer userdata );

static void 
on_progresswindow_task_finished (GnubVersionProgressWindow *window, gboolean successfull, gpointer data);

static gboolean include_func (const gchar *path, 
                              svn_wc_status2_t *status, 
                              gpointer user_data);

static void 
on_progresswindow_closed (GnubVersionProgressWindow *window, gpointer data);

int 
main (
    int argc, 
    char **argv, 
    char **env
    )
{
    int exitcode = 0;
    const gchar *blank_args[] = { "", NULL };
    GString *revision;
    const gchar **args;
    gboolean non_recursive = FALSE;
    gboolean ignore_externals = FALSE;

    #ifdef ENABLE_NLS
        bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
        textdomain (PACKAGE);
    #endif

    args = blank_args;

    revision = g_string_new("");

    GOptionEntry option_entries[] = 
    {
        /* ... your application's command line options go here ... */
        {
            "non-recursive", 'N',
            0, 
            G_OPTION_ARG_NONE,
            &non_recursive,
            _("operate on single directory only"),
            NULL
        },
        {
            "ignore-externals", 0,
            0, 
            G_OPTION_ARG_NONE,
            &ignore_externals,
            _("ignore externals definitions"),
            NULL
        },
        {
            "revision", 'r',
            0, 
            G_OPTION_ARG_STRING,
            revision,
            _("Revision number"),
            _("ARG")
        },
        { G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_FILENAME_ARRAY,
          &args,
          "Special option that collects any remaining arguments for us" },
        { NULL }
    };

    if (args == NULL)
        gvn_debug ("blank args");
    else
    {
        gvn_debug ("args[0] = %p = \"%s\"", args[0], args[0]);
        gvn_debug ("args[1] = %p = \"%s\"", args[1], args[1]);
    }

    GOptionContext *option_context;
    GnomeProgram *my_app;

    option_context = g_option_context_new ( "[ filename ... ]" );

    g_option_context_add_main_entries (option_context, option_entries, GETTEXT_PACKAGE);

    g_option_context_add_group ( option_context, gnubversion_get_option_group() );

    my_app = gnome_program_init( APP_NAME, VERSION,
                                 LIBGNOMEUI_MODULE, argc, argv,
                                 GNOME_PARAM_GOPTION_CONTEXT, option_context,
                                 GNOME_PARAM_NONE);
    gnubversion_init_with_glade (APP_NAME);

    /* Initialize the options */
    GnubVersionCommonOptions *options;
    options = OPTIONS;
    gnubversion_commonoptions_set_recursive (options, !non_recursive);
    gnubversion_commonoptions_set_ignore_externals (options, ignore_externals);

    /* Initialize revision display */
    GnubVersionRevisionChooser *revisionchooser;
    revisionchooser = REVISION;
    if (strlen (revision->str) > 0)
    {
        GError * err = gnubversion_revisionchooser_set_revision (revisionchooser, revision->str);
        if (err)
        {
            g_critical("%s", err->message);
            exit(2);
        }
    }

    /* Initialize the list of paths to update */
    GnubVersionFileSelectorStore *model;

    model = gnubversion_fileselectorstore_new ();

    gnubversion_fileselectorstore_populate_from_array (
        model, 
        args,
        NULL,
        FALSE, /* do_update */
        FALSE, /* ignore_externals */
        FALSE, /* recursive */
        FALSE, /* no ignore */
        TRUE,  /* get_all */
        include_func,  /* include */
        NULL   /* user_data */
        );

    GtkButton *updateButton ;
    updateButton = GTK_BUTTON (gnubversion_get_widget ("updateButton"));

    update_updateButton (updateButton, model);

    GnubVersionFileSelector *fileselector;
    fileselector = FILESELECTOR;

    gnubversion_fileselector_set_model (fileselector, model);
    gnubversion_fileselector_set_label_text (fileselector, _("Paths to Update:"));
    gnubversion_fileselector_set_tick_column_title (fileselector, _("Update"));
    gnubversion_fileselector_connect__ticks_changed (fileselector, fileselector_ticks_changed, updateButton);

    gtk_widget_show_all (MAIN_WINDOW);

    if (gnubversion_get_options()->non_interactive)
        updateButton_clicked_cb ( gnubversion_get_widget("updateButton"), NULL);

    gvn_debug("Entering gtkmain");
    gtk_main();
    gvn_debug("gtkmain done");

    gnubversion_shutdown ();
    g_object_unref ( my_app );
    return exitcode;
}

/* Decides which files should be included in the FileSelector */
static gboolean include_func (const gchar *path, 
                              svn_wc_status2_t *status, 
                              gpointer user_data)
{
    switch (status->text_status)
    {
        case svn_wc_status_none: /** does not exist */
        case svn_wc_status_normal: /** exists, but uninteresting */
        case svn_wc_status_added: /** is scheduled for addition */
        case svn_wc_status_missing: /** under v.c., but is missing */
        case svn_wc_status_deleted: /** scheduled for deletion */
        case svn_wc_status_replaced: /** was deleted and then re-added */
        case svn_wc_status_modified: /** text or props have been modified */
        case svn_wc_status_merged: /** local mods received repos mods */
        case svn_wc_status_conflicted: /** local mods received conflicting repos mods */
        case svn_wc_status_obstructed: /** an unversioned resource is in the way of the versioned resource */
        case svn_wc_status_external: /** an unversioned path populated by an svn:externals property */
        case svn_wc_status_incomplete: /** a directory doesn't contain a complete entries list */
            return TRUE;

        case svn_wc_status_unversioned: /** is not a versioned thing in this wc */
        case svn_wc_status_ignored: /** is unversioned but configured to be ignored */
            return FALSE;

        default:
            g_warning("Unknown text status %d - ignoring file %p", status->text_status, path);
            return FALSE;
    }
}

static void fileselector_ticks_changed(
        GnubVersionFileSelector *selector, 
        GnubVersionFileSelectorStore *store, 
        gpointer data)
{
    update_updateButton (GTK_BUTTON (data), store);
}

static void
update_updateButton(GtkButton *button, GnubVersionFileSelectorStore *store)
{
    gtk_widget_set_sensitive (GTK_WIDGET (button), gnubversion_fileselectorstore_count_ticks (store) > 0);
}

GLADE_CALLBACK void
updateButton_clicked_cb ( GtkWidget *widget, gpointer userdata )
{
    gvn_debug("Update button clicked");

    struct update_thread_baton *baton = g_malloc(sizeof(struct update_thread_baton));

    baton->store = gnubversion_fileselector_get_model (FILESELECTOR);

    {
        const char **files;
        files = gnubversion_fileselectorstore_get_ticked_fullnames (baton->store);

        const char **f;
        for (f = files; *f != NULL; f++)
        {
            gvn_debug("filename: %s", *f);
        }
    }


    GnubVersionCommonOptions *options;
    options = OPTIONS;
    baton->ignore_externals = gnubversion_commonoptions_get_ignore_externals(options);
    baton->non_recursive = !gnubversion_commonoptions_get_recursive(options);

    GnubVersionRevisionChooser *revisionchooser;
    revisionchooser = REVISION;
    gnubversion_revisionchooser_get_revision(revisionchooser, &baton->revision);

    gtk_widget_hide (MAIN_WINDOW);

    baton->progressWindow = gnubversion_progresswindow_new(_("Preparing Update..."));
    gnubversion_auth_baton->progressWindow = baton->progressWindow;

    gnubversion_progresswindow_connect__window_closed(baton->progressWindow, on_progresswindow_closed, NULL);
    gnubversion_progresswindow_connect__task_finished(baton->progressWindow, on_progresswindow_task_finished, NULL);

    gtk_widget_show ( GTK_WIDGET (baton->progressWindow));

    GError *err = NULL;
    GThread *thread;
    thread = g_thread_create(update_thread_main, baton, TRUE, &err);

    if (thread == NULL)
    {
        g_object_unref(baton->progressWindow);
        g_free(baton);

        gnubversion_error(NULL, _("Cannot create worker thread: %s"), err->message);
        gtk_widget_show (MAIN_WINDOW);
        g_error_free(err);
    }
}

static void 
on_progresswindow_closed (GnubVersionProgressWindow *window, gpointer data)
{
    gtk_main_quit();
}

/* This only gets used in non-interactive mode */
static void 
on_progresswindow_task_finished (GnubVersionProgressWindow *window, gboolean successfull, gpointer data)
{
    gtk_main_quit();
}
