#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnubversion.h>
#include <gnome.h>
#include <assert.h>

#define APP_NAME "test-revisionchooser"

static void revision_changed(GnubVersionRevisionChooser *options, gpointer data);
static void dump_revision(GnubVersionRevisionChooser *options);

int 
main (
    int argc, 
    char **argv, 
    char **env
    )
{
    GString *revision;
    gchar **remaining_args = NULL;

    #ifdef ENABLE_NLS
        bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
        textdomain (PACKAGE);
    #endif

    revision = g_string_new("HEAD");

    GOptionEntry option_entries[] = 
    {
        { 
            "revision", 'r',  
            0 /* flags */, 
            G_OPTION_ARG_STRING /* arg */, 
            revision /* arg_data */, 
            _("Revision number to use.\n"
              "\t'arg' can be one of:\n"
              "\t  NUMBER       revision number\n"
              "\t  '{' DATE '}' revision at start of the date\n"
              "\t  'HEAD'       latest in repository (default)"
            ), 
            _("arg" )
        },
        /* last but not least a special option that collects filenames */
        { G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_FILENAME_ARRAY,
          &remaining_args,
          "Special option that collects any remaining arguments for us" },
        { NULL }
    };

    GOptionContext *option_context;
    GnomeProgram *my_app;

    option_context = g_option_context_new ( "" );
    g_option_context_add_main_entries (option_context, option_entries, GETTEXT_PACKAGE);
    g_option_context_add_group( option_context, gnubversion_get_option_group() );

    my_app = gnome_program_init(APP_NAME, VERSION,
                                LIBGNOMEUI_MODULE, argc, argv,
                                GNOME_PARAM_GOPTION_CONTEXT, option_context,
                                GNOME_PARAM_NONE);
    if (remaining_args != NULL)
    {
        g_error("bad parameters");
        gvn_debug("bye bye");
        exit(2);
    }
    gnubversion_init_with_glade(APP_NAME);

    GnubVersionRevisionChooser *rev;
    rev = GNUBVERSION_REVISIONCHOOSER( gnubversion_get_widget("revision"));

    gtk_widget_show (gnubversion_get_widget("window"));

    gnubversion_revisionchooser_set_revision(rev, revision->str);

    gnubversion_revisionchooser_connect__revision_changed(rev, revision_changed, NULL);
    gtk_main();

    gvn_debug("End of dialog:");
    dump_revision(rev);

    gnubversion_shutdown();
    g_object_unref (my_app);
    return 0;
}

static void revision_changed(GnubVersionRevisionChooser *rev, gpointer data)
{
    gvn_debug("Revision changed signal received:");
    dump_revision(rev);
}

static void dump_revision(GnubVersionRevisionChooser *rev)
{
    svn_opt_revision_t svnrev;

    gnubversion_revisionchooser_get_revision( rev, &svnrev);
    switch (svnrev.kind)
    {
        /** No revision information given. */
        case svn_opt_revision_unspecified:
            gvn_debug("No revision information given.");
            break;

        /** revision given as number */
        case svn_opt_revision_number:
            gvn_debug("revision given as number %"SVN_REVNUM_T_FMT, svnrev.value.number);
            break;

        /** revision given as date */
        case svn_opt_revision_date:
            {
                char thedate[APR_RFC822_DATE_LEN];
                apr_rfc822_date(thedate, svnrev.value.date);
                gvn_debug("revision given as date: %s", thedate);
            }
            break;

        /** rev of most recent change */
        case svn_opt_revision_committed:
            gvn_debug("rev of most recent change");
            break;

        /** (rev of most recent change) - 1 */
        case svn_opt_revision_previous:
            gvn_debug("(rev of most recent change) - 1");
            break;

        /** .svn/entries current revision */
        case svn_opt_revision_base:
            gvn_debug(".svn/entries current revision");
            break;

        /** current, plus local mods */
        case svn_opt_revision_working:
            gvn_debug("current, plus local mods");
            break;

        /** repository youngest */
        case svn_opt_revision_head:
            gvn_debug("repository youngest");
            break;
    }
}
