//
// GenericPrincipalTest.cs - NUnit Test Cases for GenericPrincipal
//
// Author:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// (C) 2003 Motus Technologies Inc. (http://www.motus.com)
// Copyright (C) 2006 Novell, Inc (http://www.novell.com)
//

using NUnit.Framework;
using System;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Security.Principal;

namespace MonoTests.System.Security.Principal {

	[TestFixture]
	public class GenericPrincipalTest {

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void NullIdentity () 
		{
			GenericPrincipal gp = new GenericPrincipal (null, new string [5]);
		}

		[Test]
		public void NullRoles () 
		{
			GenericIdentity gi = new GenericIdentity ("user");
			GenericPrincipal gp = new GenericPrincipal (gi, null);
			Assert.AreEqual ("user", gp.Identity.Name, "Identity");
			Assert.IsFalse (gp.IsInRole ("role 1"), "NoRole.IsInRole(x)");
		}

		[Test]
		public void IsInRole () 
		{
			GenericIdentity gi = new GenericIdentity ("user");
			string[] roles = new string [5];
			roles [0] = "role 1";
			GenericPrincipal gp = new GenericPrincipal (gi, roles);
			roles [1] = "role 2";
			Assert.IsTrue (gp.IsInRole ("role 1"), "IsInRole (role added before constructor)");
			Assert.IsFalse (gp.IsInRole ("role 2"), "IsInRole (role added after constructor)");
		}

		[Test]
		public void IsInRole_CaseInsensitive ()
		{
			GenericIdentity gi = new GenericIdentity ("user");
			GenericPrincipal gp = new GenericPrincipal (gi, new string[2] { "mono", "hackers" });
			Assert.IsTrue (gp.IsInRole ("MoNo"), "MoNo");
			Assert.IsTrue (gp.IsInRole ("hAcKeRs"), "hAcKeRs");
		}

		[Test]
		public void SerializationRoundtrip ()
		{
			GenericIdentity gi = new GenericIdentity ("mono", "dna");
			GenericPrincipal gp = new GenericPrincipal (gi, new string[2] { "monkey", "hackers" });
			BinaryFormatter bf = new BinaryFormatter ();
			MemoryStream ms = new MemoryStream ();
			bf.Serialize (ms, gp);

			Console.WriteLine (BitConverter.ToString (ms.ToArray ()));

			ms.Position = 0;
			GenericPrincipal clone = (GenericPrincipal) bf.Deserialize (ms);
			Assert.AreEqual (gp.Identity.Name, clone.Identity.Name, "Name");
			Assert.AreEqual (gp.Identity.AuthenticationType, clone.Identity.AuthenticationType, "AuthenticationType");
			Assert.AreEqual (gp.Identity.IsAuthenticated, clone.Identity.IsAuthenticated, "IsAuthenticated");
			Assert.IsTrue (gp.IsInRole ("monkey"), "IsInRole-monkey");
			Assert.IsTrue (gp.IsInRole ("hackers"), "IsInRole-hackers");
			Assert.IsFalse (gp.IsInRole ("donkey"), "IsInRole-donkey");
		}

#if NET_2_0
		static byte[] principal = { 0x00, 0x01, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x00, 0x2A, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x53, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x2E, 0x47, 0x65, 0x6E, 0x65, 0x72, 0x69, 0x63, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x02, 0x00, 0x00, 0x00, 0x0A, 0x6D, 0x5F, 0x69, 0x64, 0x65, 0x6E, 0x74, 0x69, 0x74, 0x79, 0x07, 0x6D, 0x5F, 0x72, 0x6F, 0x6C, 0x65, 0x73, 0x03, 0x06, 0x29, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x53, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x2E, 0x47, 0x65, 0x6E, 0x65, 0x72, 0x69, 0x63, 0x49, 0x64, 0x65, 0x6E, 0x74, 0x69, 0x74, 0x79, 0x09, 0x02, 0x00, 0x00, 0x00, 0x09, 0x03, 0x00, 0x00, 0x00, 0x04, 0x02, 0x00, 0x00, 0x00, 0x29, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x53, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x2E, 0x47, 0x65, 0x6E, 0x65, 0x72, 0x69, 0x63, 0x49, 0x64, 0x65, 0x6E, 0x74, 0x69, 0x74, 0x79, 0x02, 0x00, 0x00, 0x00, 0x06, 0x6D, 0x5F, 0x6E, 0x61, 0x6D, 0x65, 0x06, 0x6D, 0x5F, 0x74, 0x79, 0x70, 0x65, 0x01, 0x01, 0x06, 0x04, 0x00, 0x00, 0x00, 0x04, 0x6D, 0x6F, 0x6E, 0x6F, 0x06, 0x05, 0x00, 0x00, 0x00, 0x03, 0x64, 0x6E, 0x61, 0x11, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x00, 0x00, 0x06, 0x6D, 0x6F, 0x6E, 0x6B, 0x65, 0x79, 0x06, 0x07, 0x00, 0x00, 0x00, 0x07, 0x68, 0x61, 0x63, 0x6B, 0x65, 0x72, 0x73, 0x0B };
#else
		static byte[] principal = { 0x00, 0x01, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x00, 0x2A, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x53, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x2E, 0x47, 0x65, 0x6E, 0x65, 0x72, 0x69, 0x63, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x02, 0x00, 0x00, 0x00, 0x0A, 0x6D, 0x5F, 0x69, 0x64, 0x65, 0x6E, 0x74, 0x69, 0x74, 0x79, 0x07, 0x6D, 0x5F, 0x72, 0x6F, 0x6C, 0x65, 0x73, 0x03, 0x06, 0x29, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x53, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x2E, 0x47, 0x65, 0x6E, 0x65, 0x72, 0x69, 0x63, 0x49, 0x64, 0x65, 0x6E, 0x74, 0x69, 0x74, 0x79, 0x09, 0x02, 0x00, 0x00, 0x00, 0x09, 0x03, 0x00, 0x00, 0x00, 0x04, 0x02, 0x00, 0x00, 0x00, 0x29, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x53, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x50, 0x72, 0x69, 0x6E, 0x63, 0x69, 0x70, 0x61, 0x6C, 0x2E, 0x47, 0x65, 0x6E, 0x65, 0x72, 0x69, 0x63, 0x49, 0x64, 0x65, 0x6E, 0x74, 0x69, 0x74, 0x79, 0x02, 0x00, 0x00, 0x00, 0x06, 0x6D, 0x5F, 0x6E, 0x61, 0x6D, 0x65, 0x06, 0x6D, 0x5F, 0x74, 0x79, 0x70, 0x65, 0x01, 0x01, 0x06, 0x04, 0x00, 0x00, 0x00, 0x04, 0x6D, 0x6F, 0x6E, 0x6F, 0x06, 0x05, 0x00, 0x00, 0x00, 0x03, 0x64, 0x6E, 0x61, 0x11, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x00, 0x00, 0x06, 0x6D, 0x6F, 0x6E, 0x6B, 0x65, 0x79, 0x06, 0x07, 0x00, 0x00, 0x00, 0x07, 0x68, 0x61, 0x63, 0x6B, 0x65, 0x72, 0x73, 0x0B };
#endif

		[Test]
		public void Deserialize ()
		{
			BinaryFormatter bf = new BinaryFormatter ();
			MemoryStream ms = new MemoryStream (principal);
			GenericPrincipal gp = (GenericPrincipal) bf.Deserialize (ms);
			Assert.AreEqual ("mono", gp.Identity.Name, "Name");
			Assert.AreEqual ("dna", gp.Identity.AuthenticationType, "AuthenticationType");
			Assert.IsTrue (gp.Identity.IsAuthenticated, "IsAuthenticated");
			Assert.IsTrue (gp.IsInRole ("monkey"), "IsInRole-monkey");
			Assert.IsTrue (gp.IsInRole ("hackers"), "IsInRole-hackers");
			Assert.IsFalse (gp.IsInRole ("donkey"), "IsInRole-donkey");
		}
	}
}
