/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * NumberUtil.java
 * ------------
 */

package org.pentaho.reporting.libraries.formula.util;

import java.math.BigDecimal;

import org.pentaho.reporting.libraries.formula.LibFormulaBoot;

public class NumberUtil
{
  public static final BigDecimal DELTA =          new BigDecimal("0.000000000000000000000000000000000000005");
  public static final BigDecimal INT_TEST_DELTA = new BigDecimal("0.00000000000000000000000000000000005");
  private static final int ROUND_SCALE = LibFormulaBoot.GLOBAL_SCALE - 6;

  private NumberUtil()
  {
  }

  public static BigDecimal getAsBigDecimal(final Number number)
  {
    if (number == null)
    {
      throw new NullPointerException();
    }

    if(number instanceof BigDecimal)
    {
      return (BigDecimal)number;
    }
    else
    {
      return new BigDecimal(number.toString());
    }
  }

  /**
   * Performs a rounding to get a more reliable (int) cast. This makes sure that nearly exact values like
   * 0.9999999..9999 are correctly interpreted as 1 while exact values like 0.99 are interpreted as 0.
   *
   * @param n
   * @return
   */
  public static BigDecimal performIntRounding(BigDecimal n)
  {
    try
    {
      // no need to go further if the value is already an integer
      return n.setScale(0);
    }
    catch(ArithmeticException e)
    {
      //ignore and continue
    }

    final BigDecimal round;
    if(n.signum()<0)
    {
      n = n.subtract(DELTA);
      round = n.setScale(0, BigDecimal.ROUND_UP);
    }
    else
    {
      n = n.add(DELTA);
      round = n.setScale(1, BigDecimal.ROUND_DOWN);
    }
    return new BigDecimal(round.intValue());
  }

  public static BigDecimal performTuneRounding(BigDecimal n)
  {
    try
    {
      // no need to go further if the value is already an integer
      n.setScale(ROUND_SCALE);
      return n;
    }
    catch(ArithmeticException e)
    {
      //ignore and continue
    }

    final BigDecimal round;
    if(n.signum()<0)
    {
      n = n.subtract(INT_TEST_DELTA);
      round = n.setScale(ROUND_SCALE, BigDecimal.ROUND_UP);
    }
    else
    {
      n = n.add(INT_TEST_DELTA);
      round = n.setScale(ROUND_SCALE, BigDecimal.ROUND_DOWN);
    }
    if (n.compareTo(round) == 0)
    {
      return n;
    }
    return NumberUtil.removeTrailingZeros(round);
  }


  public static BigDecimal removeTrailingZeros(final BigDecimal bd)
  {
    if(bd.signum() == 0)
    {
      return bd.setScale(0);
    }

    final String text = bd.toString();
    int scale = bd.scale();
    for (int i = text.length() - 1; i >= 0; i--)
    {
      final char c = text.charAt(i);
      if (c != '0')
      {
        break;
      }
      scale -= 1;
    }
    return bd.setScale(scale);
  }
}
