#------------------------------------------------------------------------------
# File:         Charset.pm
#
# Description:  ExifTool character set definitions
#
# Revisions:    2009/08/28 - P. Harvey created
#------------------------------------------------------------------------------

package Image::ExifTool::Charset;

use strict;
use vars qw($VERSION);

$VERSION = '1.01';

# this is a special table used to define character sets
%Image::ExifTool::Charset::Main = (
# (Latin is pre-defined in Writer.pl)
#  Latin => { # cp1252 (Latin 1)
#    0x20ac => 0x80,  0x02c6 => 0x88,  0x2019 => 0x92,  0x2122 => 0x99,
#    0x201a => 0x82,  0x2030 => 0x89,  0x201c => 0x93,  0x0161 => 0x9a,
#    0x0192 => 0x83,  0x0160 => 0x8a,  0x201d => 0x94,  0x203a => 0x9b,
#    0x201e => 0x84,  0x2039 => 0x8b,  0x2022 => 0x95,  0x0153 => 0x9c,
#    0x2026 => 0x85,  0x0152 => 0x8c,  0x2013 => 0x96,  0x017e => 0x9e,
#    0x2020 => 0x86,  0x017d => 0x8e,  0x2014 => 0x97,  0x0178 => 0x9f,
#    0x2021 => 0x87,  0x2018 => 0x91,  0x02dc => 0x98,
#  },
  Latin2 => { # cp1250 (Windows Central Europe)
    0x20ac => 0x80,  0x2014 => 0x97,  0x02dd => 0xbd,  0x0155 => 0xe0,
    0x201a => 0x82,  0x2122 => 0x99,  0x013e => 0xbe,  0x0103 => 0xe3,
    0x201e => 0x84,  0x0161 => 0x9a,  0x017c => 0xbf,  0x013a => 0xe5,
    0x2026 => 0x85,  0x203a => 0x9b,  0x0154 => 0xc0,  0x0107 => 0xe6,
    0x2020 => 0x86,  0x015b => 0x9c,  0x0102 => 0xc3,  0x010d => 0xe8,
    0x2021 => 0x87,  0x0165 => 0x9d,  0x0139 => 0xc5,  0x0119 => 0xea,
    0x2030 => 0x89,  0x017e => 0x9e,  0x0106 => 0xc6,  0x011b => 0xec,
    0x0160 => 0x8a,  0x017a => 0x9f,  0x010c => 0xc8,  0x010f => 0xef,
    0x2039 => 0x8b,  0x02c7 => 0xa1,  0x0118 => 0xca,  0x0111 => 0xf0,
    0x015a => 0x8c,  0x02d8 => 0xa2,  0x011a => 0xcc,  0x0144 => 0xf1,
    0x0164 => 0x8d,  0x0141 => 0xa3,  0x010e => 0xcf,  0x0148 => 0xf2,
    0x017d => 0x8e,  0x0104 => 0xa5,  0x0110 => 0xd0,  0x0151 => 0xf5,
    0x0179 => 0x8f,  0x015e => 0xaa,  0x0143 => 0xd1,  0x0159 => 0xf8,
    0x2018 => 0x91,  0x017b => 0xaf,  0x0147 => 0xd2,  0x016f => 0xf9,
    0x2019 => 0x92,  0x02db => 0xb2,  0x0150 => 0xd5,  0x0171 => 0xfb,
    0x201c => 0x93,  0x0142 => 0xb3,  0x0158 => 0xd8,  0x0163 => 0xfe,
    0x201d => 0x94,  0x0105 => 0xb9,  0x016e => 0xd9,  0x02d9 => 0xff,
    0x2022 => 0x95,  0x015f => 0xba,  0x0170 => 0xdb,
    0x2013 => 0x96,  0x013d => 0xbc,  0x0162 => 0xde,
  },
  Cyrillic => { # cp1251 (Windows Cyrillic)
    0x0402 => 0x80,  0x045c => 0x9d,  0x0418 => 0xc8,  0x0434 => 0xe4,
    0x0403 => 0x81,  0x045b => 0x9e,  0x0419 => 0xc9,  0x0435 => 0xe5,
    0x201a => 0x82,  0x045f => 0x9f,  0x041a => 0xca,  0x0436 => 0xe6,
    0x0453 => 0x83,  0x040e => 0xa1,  0x041b => 0xcb,  0x0437 => 0xe7,
    0x201e => 0x84,  0x045e => 0xa2,  0x041c => 0xcc,  0x0438 => 0xe8,
    0x2026 => 0x85,  0x0408 => 0xa3,  0x041d => 0xcd,  0x0439 => 0xe9,
    0x2020 => 0x86,  0x0490 => 0xa5,  0x041e => 0xce,  0x043a => 0xea,
    0x2021 => 0x87,  0x0401 => 0xa8,  0x041f => 0xcf,  0x043b => 0xeb,
    0x20ac => 0x88,  0x0404 => 0xaa,  0x0420 => 0xd0,  0x043c => 0xec,
    0x2030 => 0x89,  0x0407 => 0xaf,  0x0421 => 0xd1,  0x043d => 0xed,
    0x0409 => 0x8a,  0x0406 => 0xb2,  0x0422 => 0xd2,  0x043e => 0xee,
    0x2039 => 0x8b,  0x0456 => 0xb3,  0x0423 => 0xd3,  0x043f => 0xef,
    0x040a => 0x8c,  0x0491 => 0xb4,  0x0424 => 0xd4,  0x0440 => 0xf0,
    0x040c => 0x8d,  0x0451 => 0xb8,  0x0425 => 0xd5,  0x0441 => 0xf1,
    0x040b => 0x8e,  0x2116 => 0xb9,  0x0426 => 0xd6,  0x0442 => 0xf2,
    0x040f => 0x8f,  0x0454 => 0xba,  0x0427 => 0xd7,  0x0443 => 0xf3,
    0x0452 => 0x90,  0x0458 => 0xbc,  0x0428 => 0xd8,  0x0444 => 0xf4,
    0x2018 => 0x91,  0x0405 => 0xbd,  0x0429 => 0xd9,  0x0445 => 0xf5,
    0x2019 => 0x92,  0x0455 => 0xbe,  0x042a => 0xda,  0x0446 => 0xf6,
    0x201c => 0x93,  0x0457 => 0xbf,  0x042b => 0xdb,  0x0447 => 0xf7,
    0x201d => 0x94,  0x0410 => 0xc0,  0x042c => 0xdc,  0x0448 => 0xf8,
    0x2022 => 0x95,  0x0411 => 0xc1,  0x042d => 0xdd,  0x0449 => 0xf9,
    0x2013 => 0x96,  0x0412 => 0xc2,  0x042e => 0xde,  0x044a => 0xfa,
    0x2014 => 0x97,  0x0413 => 0xc3,  0x042f => 0xdf,  0x044b => 0xfb,
    0x2122 => 0x99,  0x0414 => 0xc4,  0x0430 => 0xe0,  0x044c => 0xfc,
    0x0459 => 0x9a,  0x0415 => 0xc5,  0x0431 => 0xe1,  0x044d => 0xfd,
    0x203a => 0x9b,  0x0416 => 0xc6,  0x0432 => 0xe2,  0x044e => 0xfe,
    0x045a => 0x9c,  0x0417 => 0xc7,  0x0433 => 0xe3,  0x044f => 0xff,
  },
  Greek => { # cp1253 (Windows Greek)
    0x20ac => 0x80,  0x0389 => 0xb9,  0x03a3 => 0xd3,  0x03ba => 0xea,
    0x201a => 0x82,  0x038a => 0xba,  0x03a4 => 0xd4,  0x03bb => 0xeb,
    0x0192 => 0x83,  0x038c => 0xbc,  0x03a5 => 0xd5,  0x03bc => 0xec,
    0x201e => 0x84,  0x038e => 0xbe,  0x03a6 => 0xd6,  0x03bd => 0xed,
    0x2026 => 0x85,  0x038f => 0xbf,  0x03a7 => 0xd7,  0x03be => 0xee,
    0x2020 => 0x86,  0x0390 => 0xc0,  0x03a8 => 0xd8,  0x03bf => 0xef,
    0x2021 => 0x87,  0x0391 => 0xc1,  0x03a9 => 0xd9,  0x03c0 => 0xf0,
    0x2030 => 0x89,  0x0392 => 0xc2,  0x03aa => 0xda,  0x03c1 => 0xf1,
    0x2039 => 0x8b,  0x0393 => 0xc3,  0x03ab => 0xdb,  0x03c2 => 0xf2,
    0x2018 => 0x91,  0x0394 => 0xc4,  0x03ac => 0xdc,  0x03c3 => 0xf3,
    0x2019 => 0x92,  0x0395 => 0xc5,  0x03ad => 0xdd,  0x03c4 => 0xf4,
    0x201c => 0x93,  0x0396 => 0xc6,  0x03ae => 0xde,  0x03c5 => 0xf5,
    0x201d => 0x94,  0x0397 => 0xc7,  0x03af => 0xdf,  0x03c6 => 0xf6,
    0x2022 => 0x95,  0x0398 => 0xc8,  0x03b0 => 0xe0,  0x03c7 => 0xf7,
    0x2013 => 0x96,  0x0399 => 0xc9,  0x03b1 => 0xe1,  0x03c8 => 0xf8,
    0x2014 => 0x97,  0x039a => 0xca,  0x03b2 => 0xe2,  0x03c9 => 0xf9,
    0x2122 => 0x99,  0x039b => 0xcb,  0x03b3 => 0xe3,  0x03ca => 0xfa,
    0x203a => 0x9b,  0x039c => 0xcc,  0x03b4 => 0xe4,  0x03cb => 0xfb,
    0x0385 => 0xa1,  0x039d => 0xcd,  0x03b5 => 0xe5,  0x03cc => 0xfc,
    0x0386 => 0xa2,  0x039e => 0xce,  0x03b6 => 0xe6,  0x03cd => 0xfd,
    0x2015 => 0xaf,  0x039f => 0xcf,  0x03b7 => 0xe7,  0x03ce => 0xfe,
    0x0384 => 0xb4,  0x03a0 => 0xd0,  0x03b8 => 0xe8,
    0x0388 => 0xb8,  0x03a1 => 0xd1,  0x03b9 => 0xe9,
  },
  Turkish => { # cp1254 (Windows Turkish)
    0x20ac => 0x80,  0x2030 => 0x89,  0x2022 => 0x95,  0x0178 => 0x9f,
    0x201a => 0x82,  0x0160 => 0x8a,  0x2013 => 0x96,  0x011e => 0xd0,
    0x0192 => 0x83,  0x2039 => 0x8b,  0x2014 => 0x97,  0x0130 => 0xdd,
    0x201e => 0x84,  0x0152 => 0x8c,  0x02dc => 0x98,  0x015e => 0xde,
    0x2026 => 0x85,  0x2018 => 0x91,  0x2122 => 0x99,  0x011f => 0xf0,
    0x2020 => 0x86,  0x2019 => 0x92,  0x0161 => 0x9a,  0x0131 => 0xfd,
    0x2021 => 0x87,  0x201c => 0x93,  0x203a => 0x9b,  0x015f => 0xfe,
    0x02c6 => 0x88,  0x201d => 0x94,  0x0153 => 0x9c,
  },
  Hebrew => { # cp1255 (Windows Hebrew)
    0x20ac => 0x80,  0x203a => 0x9b,  0x05c0 => 0xd0,  0x05da => 0xea,
    0x201a => 0x82,  0x20aa => 0xa4,  0x05c1 => 0xd1,  0x05db => 0xeb,
    0x0192 => 0x83,  0x00d7 => 0xaa,  0x05c2 => 0xd2,  0x05dc => 0xec,
    0x201e => 0x84,  0x00f7 => 0xba,  0x05c3 => 0xd3,  0x05dd => 0xed,
    0x2026 => 0x85,  0x05b0 => 0xc0,  0x05f0 => 0xd4,  0x05de => 0xee,
    0x2020 => 0x86,  0x05b1 => 0xc1,  0x05f1 => 0xd5,  0x05df => 0xef,
    0x2021 => 0x87,  0x05b2 => 0xc2,  0x05f2 => 0xd6,  0x05e0 => 0xf0,
    0x02c6 => 0x88,  0x05b3 => 0xc3,  0x05f3 => 0xd7,  0x05e1 => 0xf1,
    0x2030 => 0x89,  0x05b4 => 0xc4,  0x05f4 => 0xd8,  0x05e2 => 0xf2,
    0x2039 => 0x8b,  0x05b5 => 0xc5,  0x05d0 => 0xe0,  0x05e3 => 0xf3,
    0x2018 => 0x91,  0x05b6 => 0xc6,  0x05d1 => 0xe1,  0x05e4 => 0xf4,
    0x2019 => 0x92,  0x05b7 => 0xc7,  0x05d2 => 0xe2,  0x05e5 => 0xf5,
    0x201c => 0x93,  0x05b8 => 0xc8,  0x05d3 => 0xe3,  0x05e6 => 0xf6,
    0x201d => 0x94,  0x05b9 => 0xc9,  0x05d4 => 0xe4,  0x05e7 => 0xf7,
    0x2022 => 0x95,  0x05bb => 0xcb,  0x05d5 => 0xe5,  0x05e8 => 0xf8,
    0x2013 => 0x96,  0x05bc => 0xcc,  0x05d6 => 0xe6,  0x05e9 => 0xf9,
    0x2014 => 0x97,  0x05bd => 0xcd,  0x05d7 => 0xe7,  0x05ea => 0xfa,
    0x02dc => 0x98,  0x05be => 0xce,  0x05d8 => 0xe8,  0x200e => 0xfd,
    0x2122 => 0x99,  0x05bf => 0xcf,  0x05d9 => 0xe9,  0x200f => 0xfe,
  },
  Arabic => { # cp1256 (Windows Arabic)
    0x20ac => 0x80,  0x2013 => 0x96,  0x0628 => 0xc8,  0x0643 => 0xdf,
    0x067e => 0x81,  0x2014 => 0x97,  0x0629 => 0xc9,  0x0644 => 0xe1,
    0x201a => 0x82,  0x06a9 => 0x98,  0x062a => 0xca,  0x0645 => 0xe3,
    0x0192 => 0x83,  0x2122 => 0x99,  0x062b => 0xcb,  0x0646 => 0xe4,
    0x201e => 0x84,  0x0691 => 0x9a,  0x062c => 0xcc,  0x0647 => 0xe5,
    0x2026 => 0x85,  0x203a => 0x9b,  0x062d => 0xcd,  0x0648 => 0xe6,
    0x2020 => 0x86,  0x0153 => 0x9c,  0x062e => 0xce,  0x0649 => 0xec,
    0x2021 => 0x87,  0x200c => 0x9d,  0x062f => 0xcf,  0x064a => 0xed,
    0x02c6 => 0x88,  0x200d => 0x9e,  0x0630 => 0xd0,  0x064b => 0xf0,
    0x2030 => 0x89,  0x06ba => 0x9f,  0x0631 => 0xd1,  0x064c => 0xf1,
    0x0679 => 0x8a,  0x060c => 0xa1,  0x0632 => 0xd2,  0x064d => 0xf2,
    0x2039 => 0x8b,  0x06be => 0xaa,  0x0633 => 0xd3,  0x064e => 0xf3,
    0x0152 => 0x8c,  0x061b => 0xba,  0x0634 => 0xd4,  0x064f => 0xf5,
    0x0686 => 0x8d,  0x061f => 0xbf,  0x0635 => 0xd5,  0x0650 => 0xf6,
    0x0698 => 0x8e,  0x06c1 => 0xc0,  0x0636 => 0xd6,  0x0651 => 0xf8,
    0x0688 => 0x8f,  0x0621 => 0xc1,  0x0637 => 0xd8,  0x0652 => 0xfa,
    0x06af => 0x90,  0x0622 => 0xc2,  0x0638 => 0xd9,  0x200e => 0xfd,
    0x2018 => 0x91,  0x0623 => 0xc3,  0x0639 => 0xda,  0x200f => 0xfe,
    0x2019 => 0x92,  0x0624 => 0xc4,  0x063a => 0xdb,  0x06d2 => 0xff,
    0x201c => 0x93,  0x0625 => 0xc5,  0x0640 => 0xdc,
    0x201d => 0x94,  0x0626 => 0xc6,  0x0641 => 0xdd,
    0x2022 => 0x95,  0x0627 => 0xc7,  0x0642 => 0xde,
  },
  Baltic => { # cp1257 (Windows Baltic)
    0x20ac => 0x80,  0x203a => 0x9b,  0x0136 => 0xcd,  0x010d => 0xe8,
    0x201a => 0x82,  0x00af => 0x9d,  0x012a => 0xce,  0x017a => 0xea,
    0x201e => 0x84,  0x02db => 0x9e,  0x013b => 0xcf,  0x0117 => 0xeb,
    0x2026 => 0x85,  0x00d8 => 0xa8,  0x0160 => 0xd0,  0x0123 => 0xec,
    0x2020 => 0x86,  0x0156 => 0xaa,  0x0143 => 0xd1,  0x0137 => 0xed,
    0x2021 => 0x87,  0x00c6 => 0xaf,  0x0145 => 0xd2,  0x012b => 0xee,
    0x2030 => 0x89,  0x00f8 => 0xb8,  0x014c => 0xd4,  0x013c => 0xef,
    0x2039 => 0x8b,  0x0157 => 0xba,  0x0172 => 0xd8,  0x0161 => 0xf0,
    0x00a8 => 0x8d,  0x00e6 => 0xbf,  0x0141 => 0xd9,  0x0144 => 0xf1,
    0x02c7 => 0x8e,  0x0104 => 0xc0,  0x015a => 0xda,  0x0146 => 0xf2,
    0x00b8 => 0x8f,  0x012e => 0xc1,  0x016a => 0xdb,  0x014d => 0xf4,
    0x2018 => 0x91,  0x0100 => 0xc2,  0x017b => 0xdd,  0x0173 => 0xf8,
    0x2019 => 0x92,  0x0106 => 0xc3,  0x017d => 0xde,  0x0142 => 0xf9,
    0x201c => 0x93,  0x0118 => 0xc6,  0x0105 => 0xe0,  0x015b => 0xfa,
    0x201d => 0x94,  0x0112 => 0xc7,  0x012f => 0xe1,  0x016b => 0xfb,
    0x2022 => 0x95,  0x010c => 0xc8,  0x0101 => 0xe2,  0x017c => 0xfd,
    0x2013 => 0x96,  0x0179 => 0xca,  0x0107 => 0xe3,  0x017e => 0xfe,
    0x2014 => 0x97,  0x0116 => 0xcb,  0x0119 => 0xe6,  0x02d9 => 0xff,
    0x2122 => 0x99,  0x0122 => 0xcc,  0x0113 => 0xe7,
  },
  Vietnam => { # cp1258 (Windows Vietnam)
    0x20ac => 0x80,  0x0152 => 0x8c,  0x203a => 0x9b,  0x0103 => 0xe3,
    0x201a => 0x82,  0x2018 => 0x91,  0x0153 => 0x9c,  0x0301 => 0xec,
    0x0192 => 0x83,  0x2019 => 0x92,  0x0178 => 0x9f,  0x0111 => 0xf0,
    0x201e => 0x84,  0x201c => 0x93,  0x0102 => 0xc3,  0x0323 => 0xf2,
    0x2026 => 0x85,  0x201d => 0x94,  0x0300 => 0xcc,  0x01a1 => 0xf5,
    0x2020 => 0x86,  0x2022 => 0x95,  0x0110 => 0xd0,  0x01b0 => 0xfd,
    0x2021 => 0x87,  0x2013 => 0x96,  0x0309 => 0xd2,  0x20ab => 0xfe,
    0x02c6 => 0x88,  0x2014 => 0x97,  0x01a0 => 0xd5,
    0x2030 => 0x89,  0x02dc => 0x98,  0x01af => 0xdd,
    0x2039 => 0x8b,  0x2122 => 0x99,  0x0303 => 0xde,
  },
  Thai => { # cp874 (Windows Thai)
    0x20ac => 0x80,  0x0e10 => 0xb0,  0x0e28 => 0xc8,  0x0e44 => 0xe4,
    0x2026 => 0x85,  0x0e11 => 0xb1,  0x0e29 => 0xc9,  0x0e45 => 0xe5,
    0x2018 => 0x91,  0x0e12 => 0xb2,  0x0e2a => 0xca,  0x0e46 => 0xe6,
    0x2019 => 0x92,  0x0e13 => 0xb3,  0x0e2b => 0xcb,  0x0e47 => 0xe7,
    0x201c => 0x93,  0x0e14 => 0xb4,  0x0e2c => 0xcc,  0x0e48 => 0xe8,
    0x201d => 0x94,  0x0e15 => 0xb5,  0x0e2d => 0xcd,  0x0e49 => 0xe9,
    0x2022 => 0x95,  0x0e16 => 0xb6,  0x0e2e => 0xce,  0x0e4a => 0xea,
    0x2013 => 0x96,  0x0e17 => 0xb7,  0x0e2f => 0xcf,  0x0e4b => 0xeb,
    0x2014 => 0x97,  0x0e18 => 0xb8,  0x0e30 => 0xd0,  0x0e4c => 0xec,
    0x0e01 => 0xa1,  0x0e19 => 0xb9,  0x0e31 => 0xd1,  0x0e4d => 0xed,
    0x0e02 => 0xa2,  0x0e1a => 0xba,  0x0e32 => 0xd2,  0x0e4e => 0xee,
    0x0e03 => 0xa3,  0x0e1b => 0xbb,  0x0e33 => 0xd3,  0x0e4f => 0xef,
    0x0e04 => 0xa4,  0x0e1c => 0xbc,  0x0e34 => 0xd4,  0x0e50 => 0xf0,
    0x0e05 => 0xa5,  0x0e1d => 0xbd,  0x0e35 => 0xd5,  0x0e51 => 0xf1,
    0x0e06 => 0xa6,  0x0e1e => 0xbe,  0x0e36 => 0xd6,  0x0e52 => 0xf2,
    0x0e07 => 0xa7,  0x0e1f => 0xbf,  0x0e37 => 0xd7,  0x0e53 => 0xf3,
    0x0e08 => 0xa8,  0x0e20 => 0xc0,  0x0e38 => 0xd8,  0x0e54 => 0xf4,
    0x0e09 => 0xa9,  0x0e21 => 0xc1,  0x0e39 => 0xd9,  0x0e55 => 0xf5,
    0x0e0a => 0xaa,  0x0e22 => 0xc2,  0x0e3a => 0xda,  0x0e56 => 0xf6,
    0x0e0b => 0xab,  0x0e23 => 0xc3,  0x0e3f => 0xdf,  0x0e57 => 0xf7,
    0x0e0c => 0xac,  0x0e24 => 0xc4,  0x0e40 => 0xe0,  0x0e58 => 0xf8,
    0x0e0d => 0xad,  0x0e25 => 0xc5,  0x0e41 => 0xe1,  0x0e59 => 0xf9,
    0x0e0e => 0xae,  0x0e26 => 0xc6,  0x0e42 => 0xe2,  0x0e5a => 0xfa,
    0x0e0f => 0xaf,  0x0e27 => 0xc7,  0x0e43 => 0xe3,  0x0e5b => 0xfb,
  },
  MacRoman => { # cp10000 (Mac Roman)
    0x00c4 => 0x80,  0x00fc => 0x9f,  0x221a => 0xc3,  0x201a => 0xe2,
    0x00c5 => 0x81,  0x2020 => 0xa0,  0x0192 => 0xc4,  0x201e => 0xe3,
    0x00c7 => 0x82,  0x00b0 => 0xa1,  0x2248 => 0xc5,  0x2030 => 0xe4,
    0x00c9 => 0x83,  0x00a7 => 0xa4,  0x2206 => 0xc6,  0x00c2 => 0xe5,
    0x00d1 => 0x84,  0x2022 => 0xa5,  0x00ab => 0xc7,  0x00ca => 0xe6,
    0x00d6 => 0x85,  0x00b6 => 0xa6,  0x00bb => 0xc8,  0x00c1 => 0xe7,
    0x00dc => 0x86,  0x00df => 0xa7,  0x2026 => 0xc9,  0x00cb => 0xe8,
    0x00e1 => 0x87,  0x00ae => 0xa8,  0x00a0 => 0xca,  0x00c8 => 0xe9,
    0x00e0 => 0x88,  0x2122 => 0xaa,  0x00c0 => 0xcb,  0x00cd => 0xea,
    0x00e2 => 0x89,  0x00b4 => 0xab,  0x00c3 => 0xcc,  0x00ce => 0xeb,
    0x00e4 => 0x8a,  0x00a8 => 0xac,  0x00d5 => 0xcd,  0x00cf => 0xec,
    0x00e3 => 0x8b,  0x2260 => 0xad,  0x0152 => 0xce,  0x00cc => 0xed,
    0x00e5 => 0x8c,  0x00c6 => 0xae,  0x0153 => 0xcf,  0x00d3 => 0xee,
    0x00e7 => 0x8d,  0x00d8 => 0xaf,  0x2013 => 0xd0,  0x00d4 => 0xef,
    0x00e9 => 0x8e,  0x221e => 0xb0,  0x2014 => 0xd1,  0xf8ff => 0xf0,
    0x00e8 => 0x8f,  0x2264 => 0xb2,  0x201c => 0xd2,  0x00d2 => 0xf1,
    0x00ea => 0x90,  0x2265 => 0xb3,  0x201d => 0xd3,  0x00da => 0xf2,
    0x00eb => 0x91,  0x00a5 => 0xb4,  0x2018 => 0xd4,  0x00db => 0xf3,
    0x00ed => 0x92,  0x2202 => 0xb6,  0x2019 => 0xd5,  0x00d9 => 0xf4,
    0x00ec => 0x93,  0x2211 => 0xb7,  0x00f7 => 0xd6,  0x0131 => 0xf5,
    0x00ee => 0x94,  0x220f => 0xb8,  0x25ca => 0xd7,  0x02c6 => 0xf6,
    0x00ef => 0x95,  0x03c0 => 0xb9,  0x00ff => 0xd8,  0x02dc => 0xf7,
    0x00f1 => 0x96,  0x222b => 0xba,  0x0178 => 0xd9,  0x00af => 0xf8,
    0x00f3 => 0x97,  0x00aa => 0xbb,  0x2044 => 0xda,  0x02d8 => 0xf9,
    0x00f2 => 0x98,  0x00ba => 0xbc,  0x20ac => 0xdb,  0x02d9 => 0xfa,
    0x00f4 => 0x99,  0x03a9 => 0xbd,  0x2039 => 0xdc,  0x02da => 0xfb,
    0x00f6 => 0x9a,  0x00e6 => 0xbe,  0x203a => 0xdd,  0x00b8 => 0xfc,
    0x00f5 => 0x9b,  0x00f8 => 0xbf,  0xfb01 => 0xde,  0x02dd => 0xfd,
    0x00fa => 0x9c,  0x00bf => 0xc0,  0xfb02 => 0xdf,  0x02db => 0xfe,
    0x00f9 => 0x9d,  0x00a1 => 0xc1,  0x2021 => 0xe0,  0x02c7 => 0xff,
    0x00fb => 0x9e,  0x00ac => 0xc2,  0x00b7 => 0xe1,
  },
  MacLatin2 => { # cp10029 (Mac Latin 2, Central European)
    0x00c4 => 0x80,  0x2020 => 0xa0,  0x00ac => 0xc2,  0x201a => 0xe2,
    0x0100 => 0x81,  0x00b0 => 0xa1,  0x221a => 0xc3,  0x201e => 0xe3,
    0x0101 => 0x82,  0x0118 => 0xa2,  0x0144 => 0xc4,  0x0161 => 0xe4,
    0x00c9 => 0x83,  0x00a7 => 0xa4,  0x0147 => 0xc5,  0x015a => 0xe5,
    0x0104 => 0x84,  0x2022 => 0xa5,  0x2206 => 0xc6,  0x015b => 0xe6,
    0x00d6 => 0x85,  0x00b6 => 0xa6,  0x00ab => 0xc7,  0x00c1 => 0xe7,
    0x00dc => 0x86,  0x00df => 0xa7,  0x00bb => 0xc8,  0x0164 => 0xe8,
    0x00e1 => 0x87,  0x00ae => 0xa8,  0x2026 => 0xc9,  0x0165 => 0xe9,
    0x0105 => 0x88,  0x2122 => 0xaa,  0x00a0 => 0xca,  0x00cd => 0xea,
    0x010c => 0x89,  0x0119 => 0xab,  0x0148 => 0xcb,  0x017d => 0xeb,
    0x00e4 => 0x8a,  0x00a8 => 0xac,  0x0150 => 0xcc,  0x017e => 0xec,
    0x010d => 0x8b,  0x2260 => 0xad,  0x00d5 => 0xcd,  0x016a => 0xed,
    0x0106 => 0x8c,  0x0123 => 0xae,  0x0151 => 0xce,  0x00d3 => 0xee,
    0x0107 => 0x8d,  0x012e => 0xaf,  0x014c => 0xcf,  0x00d4 => 0xef,
    0x00e9 => 0x8e,  0x012f => 0xb0,  0x2013 => 0xd0,  0x016b => 0xf0,
    0x0179 => 0x8f,  0x012a => 0xb1,  0x2014 => 0xd1,  0x016e => 0xf1,
    0x017a => 0x90,  0x2264 => 0xb2,  0x201c => 0xd2,  0x00da => 0xf2,
    0x010e => 0x91,  0x2265 => 0xb3,  0x201d => 0xd3,  0x016f => 0xf3,
    0x00ed => 0x92,  0x012b => 0xb4,  0x2018 => 0xd4,  0x0170 => 0xf4,
    0x010f => 0x93,  0x0136 => 0xb5,  0x2019 => 0xd5,  0x0171 => 0xf5,
    0x0112 => 0x94,  0x2202 => 0xb6,  0x00f7 => 0xd6,  0x0172 => 0xf6,
    0x0113 => 0x95,  0x2211 => 0xb7,  0x25ca => 0xd7,  0x0173 => 0xf7,
    0x0116 => 0x96,  0x0142 => 0xb8,  0x014d => 0xd8,  0x00dd => 0xf8,
    0x00f3 => 0x97,  0x013b => 0xb9,  0x0154 => 0xd9,  0x00fd => 0xf9,
    0x0117 => 0x98,  0x013c => 0xba,  0x0155 => 0xda,  0x0137 => 0xfa,
    0x00f4 => 0x99,  0x013d => 0xbb,  0x0158 => 0xdb,  0x017b => 0xfb,
    0x00f6 => 0x9a,  0x013e => 0xbc,  0x2039 => 0xdc,  0x0141 => 0xfc,
    0x00f5 => 0x9b,  0x0139 => 0xbd,  0x203a => 0xdd,  0x017c => 0xfd,
    0x00fa => 0x9c,  0x013a => 0xbe,  0x0159 => 0xde,  0x0122 => 0xfe,
    0x011a => 0x9d,  0x0145 => 0xbf,  0x0156 => 0xdf,  0x02c7 => 0xff,
    0x011b => 0x9e,  0x0146 => 0xc0,  0x0157 => 0xe0,
    0x00fc => 0x9f,  0x0143 => 0xc1,  0x0160 => 0xe1,
  },
  MacGreek => { # cp10006
    0x00c4 => 0x80,  0x2020 => 0xa0,  0x00ac => 0xc2,  0x03b2 => 0xe2,
    0x00b9 => 0x81,  0x0393 => 0xa1,  0x039f => 0xc3,  0x03c8 => 0xe3,
    0x00b2 => 0x82,  0x0394 => 0xa2,  0x03a1 => 0xc4,  0x03b4 => 0xe4,
    0x00c9 => 0x83,  0x0398 => 0xa3,  0x2248 => 0xc5,  0x03b5 => 0xe5,
    0x00b3 => 0x84,  0x039b => 0xa4,  0x03a4 => 0xc6,  0x03c6 => 0xe6,
    0x00d6 => 0x85,  0x039e => 0xa5,  0x00ab => 0xc7,  0x03b3 => 0xe7,
    0x00dc => 0x86,  0x03a0 => 0xa6,  0x00bb => 0xc8,  0x03b7 => 0xe8,
    0x0385 => 0x87,  0x00df => 0xa7,  0x2026 => 0xc9,  0x03b9 => 0xe9,
    0x00e0 => 0x88,  0x00ae => 0xa8,  0x00a0 => 0xca,  0x03be => 0xea,
    0x00e2 => 0x89,  0x03a3 => 0xaa,  0x03a5 => 0xcb,  0x03ba => 0xeb,
    0x00e4 => 0x8a,  0x03aa => 0xab,  0x03a7 => 0xcc,  0x03bb => 0xec,
    0x0384 => 0x8b,  0x00a7 => 0xac,  0x0386 => 0xcd,  0x03bc => 0xed,
    0x00a8 => 0x8c,  0x2260 => 0xad,  0x0388 => 0xce,  0x03bd => 0xee,
    0x00e7 => 0x8d,  0x00b0 => 0xae,  0x0153 => 0xcf,  0x03bf => 0xef,
    0x00e9 => 0x8e,  0x00b7 => 0xaf,  0x2013 => 0xd0,  0x03c0 => 0xf0,
    0x00e8 => 0x8f,  0x0391 => 0xb0,  0x2015 => 0xd1,  0x03ce => 0xf1,
    0x00ea => 0x90,  0x2264 => 0xb2,  0x201c => 0xd2,  0x03c1 => 0xf2,
    0x00eb => 0x91,  0x2265 => 0xb3,  0x201d => 0xd3,  0x03c3 => 0xf3,
    0x00a3 => 0x92,  0x00a5 => 0xb4,  0x2018 => 0xd4,  0x03c4 => 0xf4,
    0x2122 => 0x93,  0x0392 => 0xb5,  0x2019 => 0xd5,  0x03b8 => 0xf5,
    0x00ee => 0x94,  0x0395 => 0xb6,  0x00f7 => 0xd6,  0x03c9 => 0xf6,
    0x00ef => 0x95,  0x0396 => 0xb7,  0x0389 => 0xd7,  0x03c2 => 0xf7,
    0x2022 => 0x96,  0x0397 => 0xb8,  0x038a => 0xd8,  0x03c7 => 0xf8,
    0x00bd => 0x97,  0x0399 => 0xb9,  0x038c => 0xd9,  0x03c5 => 0xf9,
    0x2030 => 0x98,  0x039a => 0xba,  0x038e => 0xda,  0x03b6 => 0xfa,
    0x00f4 => 0x99,  0x039c => 0xbb,  0x03ad => 0xdb,  0x03ca => 0xfb,
    0x00f6 => 0x9a,  0x03a6 => 0xbc,  0x03ae => 0xdc,  0x03cb => 0xfc,
    0x00a6 => 0x9b,  0x03ab => 0xbd,  0x03af => 0xdd,  0x0390 => 0xfd,
    0x20ac => 0x9c,  0x03a8 => 0xbe,  0x03cc => 0xde,  0x03b0 => 0xfe,
    0x00f9 => 0x9d,  0x03a9 => 0xbf,  0x038f => 0xdf,  0x00ad => 0xff,
    0x00fb => 0x9e,  0x03ac => 0xc0,  0x03cd => 0xe0,
    0x00fc => 0x9f,  0x039d => 0xc1,  0x03b1 => 0xe1,
  },
  MacCyrillic => { # cp10007
    0x0410 => 0x80,  0x042f => 0x9f,  0x00ac => 0xc2,  0x0431 => 0xe1,
    0x0411 => 0x81,  0x2020 => 0xa0,  0x221a => 0xc3,  0x0432 => 0xe2,
    0x0412 => 0x82,  0x00b0 => 0xa1,  0x0192 => 0xc4,  0x0433 => 0xe3,
    0x0413 => 0x83,  0x0490 => 0xa2,  0x2248 => 0xc5,  0x0434 => 0xe4,
    0x0414 => 0x84,  0x00a7 => 0xa4,  0x2206 => 0xc6,  0x0435 => 0xe5,
    0x0415 => 0x85,  0x2022 => 0xa5,  0x00ab => 0xc7,  0x0436 => 0xe6,
    0x0416 => 0x86,  0x00b6 => 0xa6,  0x00bb => 0xc8,  0x0437 => 0xe7,
    0x0417 => 0x87,  0x0406 => 0xa7,  0x2026 => 0xc9,  0x0438 => 0xe8,
    0x0418 => 0x88,  0x00ae => 0xa8,  0x00a0 => 0xca,  0x0439 => 0xe9,
    0x0419 => 0x89,  0x2122 => 0xaa,  0x040b => 0xcb,  0x043a => 0xea,
    0x041a => 0x8a,  0x0402 => 0xab,  0x045b => 0xcc,  0x043b => 0xeb,
    0x041b => 0x8b,  0x0452 => 0xac,  0x040c => 0xcd,  0x043c => 0xec,
    0x041c => 0x8c,  0x2260 => 0xad,  0x045c => 0xce,  0x043d => 0xed,
    0x041d => 0x8d,  0x0403 => 0xae,  0x0455 => 0xcf,  0x043e => 0xee,
    0x041e => 0x8e,  0x0453 => 0xaf,  0x2013 => 0xd0,  0x043f => 0xef,
    0x041f => 0x8f,  0x221e => 0xb0,  0x2014 => 0xd1,  0x0440 => 0xf0,
    0x0420 => 0x90,  0x2264 => 0xb2,  0x201c => 0xd2,  0x0441 => 0xf1,
    0x0421 => 0x91,  0x2265 => 0xb3,  0x201d => 0xd3,  0x0442 => 0xf2,
    0x0422 => 0x92,  0x0456 => 0xb4,  0x2018 => 0xd4,  0x0443 => 0xf3,
    0x0423 => 0x93,  0x0491 => 0xb6,  0x2019 => 0xd5,  0x0444 => 0xf4,
    0x0424 => 0x94,  0x0408 => 0xb7,  0x00f7 => 0xd6,  0x0445 => 0xf5,
    0x0425 => 0x95,  0x0404 => 0xb8,  0x201e => 0xd7,  0x0446 => 0xf6,
    0x0426 => 0x96,  0x0454 => 0xb9,  0x040e => 0xd8,  0x0447 => 0xf7,
    0x0427 => 0x97,  0x0407 => 0xba,  0x045e => 0xd9,  0x0448 => 0xf8,
    0x0428 => 0x98,  0x0457 => 0xbb,  0x040f => 0xda,  0x0449 => 0xf9,
    0x0429 => 0x99,  0x0409 => 0xbc,  0x045f => 0xdb,  0x044a => 0xfa,
    0x042a => 0x9a,  0x0459 => 0xbd,  0x2116 => 0xdc,  0x044b => 0xfb,
    0x042b => 0x9b,  0x040a => 0xbe,  0x0401 => 0xdd,  0x044c => 0xfc,
    0x042c => 0x9c,  0x045a => 0xbf,  0x0451 => 0xde,  0x044d => 0xfd,
    0x042d => 0x9d,  0x0458 => 0xc0,  0x044f => 0xdf,  0x044e => 0xfe,
    0x042e => 0x9e,  0x0405 => 0xc1,  0x0430 => 0xe0,  0x20ac => 0xff,
  },
  MacRomanian => { # cp10010
    0x00c4 => 0x80,  0x00fc => 0x9f,  0x221a => 0xc3,  0x201a => 0xe2,
    0x00c5 => 0x81,  0x2020 => 0xa0,  0x0192 => 0xc4,  0x201e => 0xe3,
    0x00c7 => 0x82,  0x00b0 => 0xa1,  0x2248 => 0xc5,  0x2030 => 0xe4,
    0x00c9 => 0x83,  0x00a7 => 0xa4,  0x2206 => 0xc6,  0x00c2 => 0xe5,
    0x00d1 => 0x84,  0x2022 => 0xa5,  0x00ab => 0xc7,  0x00ca => 0xe6,
    0x00d6 => 0x85,  0x00b6 => 0xa6,  0x00bb => 0xc8,  0x00c1 => 0xe7,
    0x00dc => 0x86,  0x00df => 0xa7,  0x2026 => 0xc9,  0x00cb => 0xe8,
    0x00e1 => 0x87,  0x00ae => 0xa8,  0x00a0 => 0xca,  0x00c8 => 0xe9,
    0x00e0 => 0x88,  0x2122 => 0xaa,  0x00c0 => 0xcb,  0x00cd => 0xea,
    0x00e2 => 0x89,  0x00b4 => 0xab,  0x00c3 => 0xcc,  0x00ce => 0xeb,
    0x00e4 => 0x8a,  0x00a8 => 0xac,  0x00d5 => 0xcd,  0x00cf => 0xec,
    0x00e3 => 0x8b,  0x2260 => 0xad,  0x0152 => 0xce,  0x00cc => 0xed,
    0x00e5 => 0x8c,  0x0102 => 0xae,  0x0153 => 0xcf,  0x00d3 => 0xee,
    0x00e7 => 0x8d,  0x0218 => 0xaf,  0x2013 => 0xd0,  0x00d4 => 0xef,
    0x00e9 => 0x8e,  0x221e => 0xb0,  0x2014 => 0xd1,  0xf8ff => 0xf0,
    0x00e8 => 0x8f,  0x2264 => 0xb2,  0x201c => 0xd2,  0x00d2 => 0xf1,
    0x00ea => 0x90,  0x2265 => 0xb3,  0x201d => 0xd3,  0x00da => 0xf2,
    0x00eb => 0x91,  0x00a5 => 0xb4,  0x2018 => 0xd4,  0x00db => 0xf3,
    0x00ed => 0x92,  0x2202 => 0xb6,  0x2019 => 0xd5,  0x00d9 => 0xf4,
    0x00ec => 0x93,  0x2211 => 0xb7,  0x00f7 => 0xd6,  0x0131 => 0xf5,
    0x00ee => 0x94,  0x220f => 0xb8,  0x25ca => 0xd7,  0x02c6 => 0xf6,
    0x00ef => 0x95,  0x03c0 => 0xb9,  0x00ff => 0xd8,  0x02dc => 0xf7,
    0x00f1 => 0x96,  0x222b => 0xba,  0x0178 => 0xd9,  0x00af => 0xf8,
    0x00f3 => 0x97,  0x00aa => 0xbb,  0x2044 => 0xda,  0x02d8 => 0xf9,
    0x00f2 => 0x98,  0x00ba => 0xbc,  0x20ac => 0xdb,  0x02d9 => 0xfa,
    0x00f4 => 0x99,  0x03a9 => 0xbd,  0x2039 => 0xdc,  0x02da => 0xfb,
    0x00f6 => 0x9a,  0x0103 => 0xbe,  0x203a => 0xdd,  0x00b8 => 0xfc,
    0x00f5 => 0x9b,  0x0219 => 0xbf,  0x021a => 0xde,  0x02dd => 0xfd,
    0x00fa => 0x9c,  0x00bf => 0xc0,  0x021b => 0xdf,  0x02db => 0xfe,
    0x00f9 => 0x9d,  0x00a1 => 0xc1,  0x2021 => 0xe0,  0x02c7 => 0xff,
    0x00fb => 0x9e,  0x00ac => 0xc2,  0x00b7 => 0xe1,
  },
  MacIceland => { # cp10079
    0x00c4 => 0x80,  0x00fc => 0x9f,  0x221a => 0xc3,  0x201e => 0xe3,
    0x00c5 => 0x81,  0x00dd => 0xa0,  0x0192 => 0xc4,  0x2030 => 0xe4,
    0x00c7 => 0x82,  0x00b0 => 0xa1,  0x2248 => 0xc5,  0x00c2 => 0xe5,
    0x00c9 => 0x83,  0x00a7 => 0xa4,  0x2206 => 0xc6,  0x00ca => 0xe6,
    0x00d1 => 0x84,  0x2022 => 0xa5,  0x00ab => 0xc7,  0x00c1 => 0xe7,
    0x00d6 => 0x85,  0x00b6 => 0xa6,  0x00bb => 0xc8,  0x00cb => 0xe8,
    0x00dc => 0x86,  0x00df => 0xa7,  0x2026 => 0xc9,  0x00c8 => 0xe9,
    0x00e1 => 0x87,  0x00ae => 0xa8,  0x00a0 => 0xca,  0x00cd => 0xea,
    0x00e0 => 0x88,  0x2122 => 0xaa,  0x00c0 => 0xcb,  0x00ce => 0xeb,
    0x00e2 => 0x89,  0x00b4 => 0xab,  0x00c3 => 0xcc,  0x00cf => 0xec,
    0x00e4 => 0x8a,  0x00a8 => 0xac,  0x00d5 => 0xcd,  0x00cc => 0xed,
    0x00e3 => 0x8b,  0x2260 => 0xad,  0x0152 => 0xce,  0x00d3 => 0xee,
    0x00e5 => 0x8c,  0x00c6 => 0xae,  0x0153 => 0xcf,  0x00d4 => 0xef,
    0x00e7 => 0x8d,  0x00d8 => 0xaf,  0x2013 => 0xd0,  0xf8ff => 0xf0,
    0x00e9 => 0x8e,  0x221e => 0xb0,  0x2014 => 0xd1,  0x00d2 => 0xf1,
    0x00e8 => 0x8f,  0x2264 => 0xb2,  0x201c => 0xd2,  0x00da => 0xf2,
    0x00ea => 0x90,  0x2265 => 0xb3,  0x201d => 0xd3,  0x00db => 0xf3,
    0x00eb => 0x91,  0x00a5 => 0xb4,  0x2018 => 0xd4,  0x00d9 => 0xf4,
    0x00ed => 0x92,  0x2202 => 0xb6,  0x2019 => 0xd5,  0x0131 => 0xf5,
    0x00ec => 0x93,  0x2211 => 0xb7,  0x00f7 => 0xd6,  0x02c6 => 0xf6,
    0x00ee => 0x94,  0x220f => 0xb8,  0x25ca => 0xd7,  0x02dc => 0xf7,
    0x00ef => 0x95,  0x03c0 => 0xb9,  0x00ff => 0xd8,  0x00af => 0xf8,
    0x00f1 => 0x96,  0x222b => 0xba,  0x0178 => 0xd9,  0x02d8 => 0xf9,
    0x00f3 => 0x97,  0x00aa => 0xbb,  0x2044 => 0xda,  0x02d9 => 0xfa,
    0x00f2 => 0x98,  0x00ba => 0xbc,  0x20ac => 0xdb,  0x02da => 0xfb,
    0x00f4 => 0x99,  0x03a9 => 0xbd,  0x00d0 => 0xdc,  0x00b8 => 0xfc,
    0x00f6 => 0x9a,  0x00e6 => 0xbe,  0x00f0 => 0xdd,  0x02dd => 0xfd,
    0x00f5 => 0x9b,  0x00f8 => 0xbf,  0x00fe => 0xdf,  0x02db => 0xfe,
    0x00fa => 0x9c,  0x00bf => 0xc0,  0x00fd => 0xe0,  0x02c7 => 0xff,
    0x00f9 => 0x9d,  0x00a1 => 0xc1,  0x00b7 => 0xe1,
    0x00fb => 0x9e,  0x00ac => 0xc2,  0x201a => 0xe2,
  },
  MacTurkish => { # cp10081
    0x00c4 => 0x80,  0x00fc => 0x9f,  0x221a => 0xc3,  0x201a => 0xe2,
    0x00c5 => 0x81,  0x2020 => 0xa0,  0x0192 => 0xc4,  0x201e => 0xe3,
    0x00c7 => 0x82,  0x00b0 => 0xa1,  0x2248 => 0xc5,  0x2030 => 0xe4,
    0x00c9 => 0x83,  0x00a7 => 0xa4,  0x2206 => 0xc6,  0x00c2 => 0xe5,
    0x00d1 => 0x84,  0x2022 => 0xa5,  0x00ab => 0xc7,  0x00ca => 0xe6,
    0x00d6 => 0x85,  0x00b6 => 0xa6,  0x00bb => 0xc8,  0x00c1 => 0xe7,
    0x00dc => 0x86,  0x00df => 0xa7,  0x2026 => 0xc9,  0x00cb => 0xe8,
    0x00e1 => 0x87,  0x00ae => 0xa8,  0x00a0 => 0xca,  0x00c8 => 0xe9,
    0x00e0 => 0x88,  0x2122 => 0xaa,  0x00c0 => 0xcb,  0x00cd => 0xea,
    0x00e2 => 0x89,  0x00b4 => 0xab,  0x00c3 => 0xcc,  0x00ce => 0xeb,
    0x00e4 => 0x8a,  0x00a8 => 0xac,  0x00d5 => 0xcd,  0x00cf => 0xec,
    0x00e3 => 0x8b,  0x2260 => 0xad,  0x0152 => 0xce,  0x00cc => 0xed,
    0x00e5 => 0x8c,  0x00c6 => 0xae,  0x0153 => 0xcf,  0x00d3 => 0xee,
    0x00e7 => 0x8d,  0x00d8 => 0xaf,  0x2013 => 0xd0,  0x00d4 => 0xef,
    0x00e9 => 0x8e,  0x221e => 0xb0,  0x2014 => 0xd1,  0xf8ff => 0xf0,
    0x00e8 => 0x8f,  0x2264 => 0xb2,  0x201c => 0xd2,  0x00d2 => 0xf1,
    0x00ea => 0x90,  0x2265 => 0xb3,  0x201d => 0xd3,  0x00da => 0xf2,
    0x00eb => 0x91,  0x00a5 => 0xb4,  0x2018 => 0xd4,  0x00db => 0xf3,
    0x00ed => 0x92,  0x2202 => 0xb6,  0x2019 => 0xd5,  0x00d9 => 0xf4,
    0x00ec => 0x93,  0x2211 => 0xb7,  0x00f7 => 0xd6,  0xf8a0 => 0xf5,
    0x00ee => 0x94,  0x220f => 0xb8,  0x25ca => 0xd7,  0x02c6 => 0xf6,
    0x00ef => 0x95,  0x03c0 => 0xb9,  0x00ff => 0xd8,  0x02dc => 0xf7,
    0x00f1 => 0x96,  0x222b => 0xba,  0x0178 => 0xd9,  0x00af => 0xf8,
    0x00f3 => 0x97,  0x00aa => 0xbb,  0x011e => 0xda,  0x02d8 => 0xf9,
    0x00f2 => 0x98,  0x00ba => 0xbc,  0x011f => 0xdb,  0x02d9 => 0xfa,
    0x00f4 => 0x99,  0x03a9 => 0xbd,  0x0130 => 0xdc,  0x02da => 0xfb,
    0x00f6 => 0x9a,  0x00e6 => 0xbe,  0x0131 => 0xdd,  0x00b8 => 0xfc,
    0x00f5 => 0x9b,  0x00f8 => 0xbf,  0x015e => 0xde,  0x02dd => 0xfd,
    0x00fa => 0x9c,  0x00bf => 0xc0,  0x015f => 0xdf,  0x02db => 0xfe,
    0x00f9 => 0x9d,  0x00a1 => 0xc1,  0x2021 => 0xe0,  0x02c7 => 0xff,
    0x00fb => 0x9e,  0x00ac => 0xc2,  0x00b7 => 0xe1,
  },
  MacCroatian => { # cp10082
    0x00c4 => 0x80,  0x00fc => 0x9f,  0x00ac => 0xc2,  0x00b7 => 0xe1,
    0x00c5 => 0x81,  0x2020 => 0xa0,  0x221a => 0xc3,  0x201a => 0xe2,
    0x00c7 => 0x82,  0x00b0 => 0xa1,  0x0192 => 0xc4,  0x201e => 0xe3,
    0x00c9 => 0x83,  0x00a7 => 0xa4,  0x2248 => 0xc5,  0x2030 => 0xe4,
    0x00d1 => 0x84,  0x2022 => 0xa5,  0x0106 => 0xc6,  0x00c2 => 0xe5,
    0x00d6 => 0x85,  0x00b6 => 0xa6,  0x00ab => 0xc7,  0x0107 => 0xe6,
    0x00dc => 0x86,  0x00df => 0xa7,  0x010c => 0xc8,  0x00c1 => 0xe7,
    0x00e1 => 0x87,  0x00ae => 0xa8,  0x2026 => 0xc9,  0x010d => 0xe8,
    0x00e0 => 0x88,  0x0160 => 0xa9,  0x00a0 => 0xca,  0x00c8 => 0xe9,
    0x00e2 => 0x89,  0x2122 => 0xaa,  0x00c0 => 0xcb,  0x00cd => 0xea,
    0x00e4 => 0x8a,  0x00b4 => 0xab,  0x00c3 => 0xcc,  0x00ce => 0xeb,
    0x00e3 => 0x8b,  0x00a8 => 0xac,  0x00d5 => 0xcd,  0x00cf => 0xec,
    0x00e5 => 0x8c,  0x2260 => 0xad,  0x0152 => 0xce,  0x00cc => 0xed,
    0x00e7 => 0x8d,  0x017d => 0xae,  0x0153 => 0xcf,  0x00d3 => 0xee,
    0x00e9 => 0x8e,  0x00d8 => 0xaf,  0x0110 => 0xd0,  0x00d4 => 0xef,
    0x00e8 => 0x8f,  0x221e => 0xb0,  0x2014 => 0xd1,  0x0111 => 0xf0,
    0x00ea => 0x90,  0x2264 => 0xb2,  0x201c => 0xd2,  0x00d2 => 0xf1,
    0x00eb => 0x91,  0x2265 => 0xb3,  0x201d => 0xd3,  0x00da => 0xf2,
    0x00ed => 0x92,  0x2206 => 0xb4,  0x2018 => 0xd4,  0x00db => 0xf3,
    0x00ec => 0x93,  0x2202 => 0xb6,  0x2019 => 0xd5,  0x00d9 => 0xf4,
    0x00ee => 0x94,  0x2211 => 0xb7,  0x00f7 => 0xd6,  0x0131 => 0xf5,
    0x00ef => 0x95,  0x220f => 0xb8,  0x25ca => 0xd7,  0x02c6 => 0xf6,
    0x00f1 => 0x96,  0x0161 => 0xb9,  0xf8ff => 0xd8,  0x02dc => 0xf7,
    0x00f3 => 0x97,  0x222b => 0xba,  0x00a9 => 0xd9,  0x00af => 0xf8,
    0x00f2 => 0x98,  0x00aa => 0xbb,  0x2044 => 0xda,  0x03c0 => 0xf9,
    0x00f4 => 0x99,  0x00ba => 0xbc,  0x20ac => 0xdb,  0x00cb => 0xfa,
    0x00f6 => 0x9a,  0x03a9 => 0xbd,  0x2039 => 0xdc,  0x02da => 0xfb,
    0x00f5 => 0x9b,  0x017e => 0xbe,  0x203a => 0xdd,  0x00b8 => 0xfc,
    0x00fa => 0x9c,  0x00f8 => 0xbf,  0x00c6 => 0xde,  0x00ca => 0xfd,
    0x00f9 => 0x9d,  0x00bf => 0xc0,  0x00bb => 0xdf,  0x00e6 => 0xfe,
    0x00fb => 0x9e,  0x00a1 => 0xc1,  0x2013 => 0xe0,  0x02c7 => 0xff,
  },
);


1; # end

__END__

=head1 NAME

Image::ExifTool::Charset - ExifTool character set definitions

=head1 SYNOPSIS

This module is required by Image::ExifTool.

=head1 DESCRIPTION

This module contains definitions for the following character sets:

Windows:  Latin2, Cyrillic, Greek, Turkish, Hebrew, Arabic, Baltic, Vietnam
and Thai.

Macintosh:  MacRoman, MacLatin2, MacCyrillic, MacGreek, MacTurkish,
MacRomanian, MacIceland and MacCroatian.

=head1 AUTHOR

Copyright 2003-2009, Phil Harvey (phil at owl.phy.queensu.ca)

This library is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=head1 SEE ALSO

L<Image::ExifTool(3pm)|Image::ExifTool>

=cut
