/* Copyright (c) 2002-2009 Dovecot Sieve authors, see the included COPYING file
 */

#include "lib.h"
#include "str.h"

#include "managesieve-capability.h"

char *managesieve_capability_sieve = NULL;
char *managesieve_capability_notify = NULL;

typedef	enum { CAP_SIEVE, CAP_NOTIFY } capability_type_t;

static void _capability_assign(capability_type_t cap_type, const char *value)
{
	const char *duplicate = NULL;
	
	switch ( cap_type ) {
	case CAP_SIEVE:
		if ( managesieve_capability_sieve == NULL )
			managesieve_capability_sieve = i_strdup(value);
		else
			duplicate = "SIEVE";
		break;
	case CAP_NOTIFY:
		if ( managesieve_capability_notify == NULL )
			managesieve_capability_notify = i_strdup(value);
		else
			duplicate = "NOTIFY";
		break;
	}
	
	if ( duplicate != NULL ) 
		i_warning("managesieve-login: duplicate capability '%s' listed in "
			"capability string (ignored).", duplicate);
}

void managesieve_capabilities_init(const char *cap_string)
{
	capability_type_t cap_type = CAP_SIEVE;
	const char *p = cap_string;
	string_t *part = t_str_new(256);

	if ( cap_string == NULL || *cap_string == '\0' ) {
		i_warning("managesieve-login: capability string is empty.");
		return;
	}
	
	while ( *p != '\0' ) {
		if ( *p == '\\' ) {
			p++;
			if ( *p != '\0' ) {
				str_append_c(part, *p);
				p++;
			} else break;
		} else if ( *p == ':' ) {
			if ( strcasecmp(str_c(part), "SIEVE") == 0 )
				cap_type = CAP_SIEVE;
			else if ( strcasecmp(str_c(part), "NOTIFY") == 0 )
				cap_type = CAP_NOTIFY;
			else
				i_warning("managesieve-login: unknown capability '%s' listed in "
					"capability string (ignored).", str_c(part));
			str_truncate(part, 0); 
		} else if ( *p == ',' ) {
			_capability_assign(cap_type, str_c(part));
			str_truncate(part, 0);
		} else {
			/* Append character, but omit leading spaces */
			if ( str_len(part) > 0 || *p != ' ' )
				str_append_c(part, *p);
		}
		p++;
	}
	
	if ( str_len(part) > 0 ) {
		_capability_assign(cap_type, str_c(part));
	}
}

void managesieve_capabilities_deinit(void)
{
	if ( managesieve_capability_sieve != NULL ) 
		i_free(managesieve_capability_sieve);
	
	if ( managesieve_capability_notify != NULL )
		i_free(managesieve_capability_notify);	
}

