/* Copyright (c) 2002-2009 Dovecot Sieve authors, see the included COPYING file
 */

#include <string.h>
#include "common.h"
#include "ioloop.h"
#include "istream.h"
#include "ostream.h"
#include "str.h"
#include "str-sanitize.h"
#include "safe-memset.h"
#include "buffer.h"
#include "base64.h"

#include "client.h"
#include "client-authenticate.h"

#include "managesieve-quote.h"
#include "managesieve-proxy.h"
#include "managesieve-parser.h"

#define PROXY_FAILURE_MSG \
	"Account is temporarily unavailable. Try again later."

enum {
	PROXY_STATE_INITIAL,
	PROXY_STATE_TLS_START,
	PROXY_STATE_TLS_READY,
	PROXY_STATE_AUTHENTICATE,
};

typedef enum {
	MANAGESIEVE_RESPONSE_NONE,
	MANAGESIEVE_RESPONSE_OK,
	MANAGESIEVE_RESPONSE_NO,
	MANAGESIEVE_RESPONSE_BYE
} managesieve_response_t;

static void proxy_free_password(struct managesieve_client *client)
{   
	if (client->proxy_password == NULL)
		return;

	safe_memset(client->proxy_password, 0, strlen(client->proxy_password));
	i_free_and_null(client->proxy_password);
}

static void proxy_failed(struct managesieve_client *client, bool send_tagline)
{   
	if (send_tagline)
		client_send_noresp(client, "TRYLATER", PROXY_FAILURE_MSG);

	login_proxy_free(&client->proxy);
	proxy_free_password(client);
	i_free_and_null(client->proxy_user);
	i_free_and_null(client->proxy_master_user);

	/* call this last - it may destroy the client */
	client_auth_failed(client, TRUE);
}

static void get_plain_auth(struct managesieve_client *client, string_t *dest)
{   
	string_t *str, *base64;

	str = t_str_new(128);
	if ( client->proxy_master_user == NULL ) {
		str_append_c(str, '\0');
		str_append(str, client->proxy_user);
	} else {
		str_append(str, client->proxy_user);
		str_append_c(str, '\0');
		str_append(str, client->proxy_master_user);
	}
	str_append_c(str, '\0');
	str_append(str, client->proxy_password);

	base64 = t_str_new(128);
	base64_encode(str_data(str), str_len(str), base64);

	managesieve_quote_append_string(dest, str_c(base64), FALSE);
}

static int proxy_write_login(struct managesieve_client *client, string_t *str)
{   
	if ( !client->proxy_sasl_plain ) {
		client_syslog_err(&client->common, "proxy: "
			"Server does not support required PLAIN SASL mechanism");
		return -1;
	}

	/*   Send command */
	str_append(str, "AUTHENTICATE \"PLAIN\" ");
	get_plain_auth(client, str);
	proxy_free_password(client);
	str_append(str, "\r\n");

	return 1;
}

static managesieve_response_t proxy_read_response
(struct managesieve_arg *args)
{
	if ( args[0].type == MANAGESIEVE_ARG_ATOM ) {
		const char *response = MANAGESIEVE_ARG_STR(&(args[0]));

		if ( strcasecmp(response, "OK") == 0 ) {
			/* Received OK response; greeting is finished */
			return MANAGESIEVE_RESPONSE_OK; 

        } else if ( strcasecmp(response, "NO") == 0 ) {
			/* Received OK response; greeting is finished */
			return MANAGESIEVE_RESPONSE_NO; 

        } else if ( strcasecmp(response, "BYE") == 0 ) {
			/* Received OK response; greeting is finished */
			return MANAGESIEVE_RESPONSE_BYE;
 
		}
	}

	return MANAGESIEVE_RESPONSE_NONE;
}

static int proxy_input_capability
(struct managesieve_client *client, const char *line, managesieve_response_t *resp_r)
{   
	struct istream *input;
	struct managesieve_parser *parser;
 	struct managesieve_arg *args;
	int ret;
	bool fatal = FALSE;

	*resp_r = MANAGESIEVE_RESPONSE_NONE;

	/* Build an input stream for the managesieve parser 
	 *  FIXME: It would be nice if the line-wise parsing could be
	 *    substituded by something similar to the command line interpreter.
	 *    However, the current login_proxy structure does not make streams
	 *    known until inside proxy_input handler.
	 */
	line = t_strconcat(line, "\r\n", NULL);
	input = i_stream_create_from_data(line, strlen(line));
	parser = managesieve_parser_create(input, NULL, MAX_MANAGESIEVE_LINE);
	managesieve_parser_reset(parser);

	/* Parse input 
	 *  FIXME: Theoretically the OK response could include a 
	 *   response code which could be rejected by the parser. 
	 */ 
	(void)i_stream_read(input);
	ret = managesieve_parser_read_args(parser, 2, 0, &args);
		
	if ( ret >= 1 ) {
		if ( args[0].type == MANAGESIEVE_ARG_ATOM ) {
			*resp_r = proxy_read_response(args);

			if ( *resp_r == MANAGESIEVE_RESPONSE_NONE ) {
				client_syslog_err(&client->common, t_strdup_printf("proxy: "
					"Remote sent invalid response: %s",
					str_sanitize(line,160)));
		
				fatal = TRUE;
			}
      	} else if ( args[0].type == MANAGESIEVE_ARG_STRING ) {
			const char *capability = MANAGESIEVE_ARG_STR(&(args[0]));

        	if ( strcasecmp(capability, "SASL") == 0 ) {
				/* Check whether the server supports the SASL mechanism 
		    	 * we are going to use (currently only PLAIN supported). 
				 */
				if ( ret == 2 && args[1].type == MANAGESIEVE_ARG_STRING ) {
					const char *const *mechs = 
						t_strsplit(MANAGESIEVE_ARG_STR(&(args[1])), " "); 

					if ( str_array_icase_find(mechs, "PLAIN") )
						client->proxy_sasl_plain = TRUE;
					else
						client->proxy_sasl_plain = FALSE;

				} else {
					client_syslog_err(&client->common, "proxy: "
		         		"Server returned erroneous SASL capability");
					fatal = TRUE;
				}

			} else if ( strcasecmp(capability, "STARTTLS") == 0 ) {
				client->proxy_starttls = TRUE;
			}

		} else {
			/* Do not accept faulty server */
			client_syslog_err(&client->common, t_strdup_printf("proxy: "
				"Remote returned with invalid capability/greeting line: %s",
				str_sanitize(line,160)));
			fatal = TRUE;
		}

	} else if ( ret == -2 ) {
		/* Parser needs more data (not possible on mem stream) */
		i_unreached();

    } else if ( ret < 0 ) {
		const char *error_str = managesieve_parser_get_error(parser, &fatal);
		error_str = (error_str != NULL ? error_str : "unknown (bug)" );
	
		/* Do not accept faulty server */
		client_syslog_err(&client->common, t_strdup_printf("proxy: "
			"Protocol parse error(%d) in capability/greeting line: %s (line='%s')",
			ret, error_str, line));
		fatal = TRUE;
	}

	/* Cleanup parser */
    managesieve_parser_destroy(&parser);
	i_stream_destroy(&input);

	/* Time to exit if greeting was not accepted */
	if ( fatal ) return -1;

	/* Wait until greeting is received completely */
	if ( *resp_r == MANAGESIEVE_RESPONSE_NONE ) return 1;

	return 0;
}

static int proxy_input_line
(struct managesieve_client *client, const char *line)
{
	struct ostream *output; 
    enum login_proxy_ssl_flags ssl_flags;
	managesieve_response_t response = MANAGESIEVE_RESPONSE_NONE;
	string_t *command;
	int ret = 0;

	i_assert(!client->destroyed);

    output = login_proxy_get_ostream(client->proxy);
	switch ( client->proxy_state ) {
	case PROXY_STATE_INITIAL:
		if ( (ret=proxy_input_capability(client, line, &response)) < 0 ) {
			proxy_failed(client, TRUE);
			return -1;
		}

		if ( ret == 0 ) {
			if ( response != MANAGESIEVE_RESPONSE_OK ) {
				client_syslog_err(&client->common,
					"proxy: Remote sent unexpected NO/BYE in stead of capability response");
				proxy_failed(client, TRUE);
				return -1;		
			}

			command = t_str_new(128);

    		ssl_flags = login_proxy_get_ssl_flags(client->proxy);
    		if ((ssl_flags & PROXY_SSL_FLAG_STARTTLS) != 0) {
        		if ( !client->proxy_starttls ) {
            		client_syslog_err(&client->common,
                		"proxy: Remote doesn't support STARTTLS");
					proxy_failed(client, TRUE);
					return -1;
        		}
        	
				str_append(command, "STARTTLS\r\n");
				client->proxy_state = PROXY_STATE_TLS_START;
    		} else {
				if ( proxy_write_login(client, command) < 0 ) {
					proxy_failed(client, TRUE);
					return -1;
				}
				client->proxy_state = PROXY_STATE_AUTHENTICATE;
			}

			(void)o_stream_send(output, str_data(command), str_len(command));
		}

		return 0;

	case PROXY_STATE_TLS_START:
		if ( strncasecmp(line, "OK", 2) == 0 && ( strlen(line) == 2 || line[2] == ' ' ) ) {

			/* STARTTLS successful, begin TLS negotiation. */
			if ( login_proxy_starttls(client->proxy) < 0 ) {
				proxy_failed(client, TRUE);
				return -1;
			}
	
			client->proxy_state = PROXY_STATE_TLS_READY;
			return 1;
		}

		client_syslog_err(&client->common,
			"proxy: Remote refused STARTTLS command");
		proxy_failed(client, TRUE);
		return -1;		

	case PROXY_STATE_TLS_READY:
		if ( (ret=proxy_input_capability(client, line, &response)) < 0 ) {
			proxy_failed(client, TRUE);
			return -1;
		}

		if ( ret == 0 ) {
			if ( response != MANAGESIEVE_RESPONSE_OK ) {
				/* STARTTLS failed */
				client_syslog_err(&client->common, t_strdup_printf(
					"proxy: Remote STARTTLS failed: %s",
					str_sanitize(line, 160)));
				proxy_failed(client, TRUE);
				return -1;
			}

			command = t_str_new(128);
			if ( proxy_write_login(client, command) < 0 ) {
				proxy_failed(client, TRUE);
				return -1;
			}

			(void)o_stream_send(output, str_data(command), str_len(command));
		
			client->proxy_state = PROXY_STATE_AUTHENTICATE;
		}

		return 0;
	
	case PROXY_STATE_AUTHENTICATE:

		/* Check login status */
		if ( strncasecmp(line, "OK", 2) == 0 && ( strlen(line) == 2 || line[2] == ' ' ) ) {
			string_t *str = t_str_new(128);

			/* Login successful */

			/* FIXME: some SASL mechanisms cause a capability response to be sent */

			/* Send this line to client. */
			o_stream_cork(client->output);
			(void)o_stream_send_str(client->output, line);
			(void)o_stream_send(client->output, "\r\n", 2);
			o_stream_uncork(client->output);

			str_printfa(str, "proxy(%s): Started proxying to %s:%u",
				client->common.virtual_user,
				login_proxy_get_host(client->proxy),
				login_proxy_get_port(client->proxy));
			if ( strcmp(client->common.virtual_user, client->proxy_user) != 0 ) {
				/* remote username is different, log it */
				str_append_c(str, '/');
				str_append(str, client->proxy_user);
        	}
			if (client->proxy_master_user != NULL) {
				str_printfa(str, " (master %s)", client->proxy_master_user);
			}

			(void)client_skip_line(client);
			login_proxy_detach(client->proxy, client->common.input,
				client->output);

			client->proxy = NULL;
			client->common.input = NULL;
			client->output = NULL;
			client->common.fd = -1;
			client_destroy_success(client, str_c(str));
			
			return 1;
		} 
		
		/* Login failed */

		if (verbose_auth) {
			string_t *str = t_str_new(128);
				
			str_printfa(str, "proxy(%s): Login failed to %s:%u",
				client->common.virtual_user,
 				login_proxy_get_host(client->proxy),
				login_proxy_get_port(client->proxy));
			if (strcmp(client->common.virtual_user, client->proxy_user) != 0) {
				/* remote username is different, log it */
				str_append_c(str, '/');
				str_append(str, client->proxy_user);
			}
			if (client->proxy_master_user != NULL) {
				str_printfa(str, " (master %s)", client->proxy_master_user);
			}
			str_append(str, ": ");

			if (strncasecmp(line, "NO ", 3) == 0)
				str_append(str, line + 3);
			else
				str_append(str, line);
	    
			i_info("%s", str_c(str));
		}

		/* Login failed. Send our own failure reply so client can't
		 * figure out if user exists or not just by looking at the
		 * reply string.
		 */
		client_send_no(client, AUTH_FAILED_MSG);

		proxy_failed(client, FALSE);
		return -1;

	default:
		/* Not supposed to happen */
		break;
	}

	i_unreached();
	return -1;
}


static void proxy_input(struct managesieve_client *client)
{   
	struct istream *input;
	const char *line;

	if (client->proxy == NULL) {
		/* we're just freeing the proxy */
		return;
	}

	input = login_proxy_get_istream(client->proxy);
	if (input == NULL) {
		if (client->destroyed) {
			/* we came here from client_destroy() */
			return;
 		}

		/* failed for some reason, probably server disconnected */
		proxy_failed(client, TRUE);
		return;
	}

	i_assert(!client->destroyed);

	switch (i_stream_read(input)) {
	case -2:
		client_syslog_err(&client->common, 
			"proxy: Remote input buffer full");
 		proxy_failed(client, TRUE);
		return;
	case -1:
		client_syslog_err(&client->common,
			"proxy: Remote disconnected");
		proxy_failed(client, TRUE);
		return;
	}

	while ((line = i_stream_next_line(input)) != NULL) {
		if (proxy_input_line(client, line) != 0)
			break;
	}
}

int managesieve_proxy_new
(struct managesieve_client *client, const char *host,
	unsigned int port, const char *user, const char *master_user,
	const char *password, enum login_proxy_ssl_flags ssl_flags,
	unsigned int connect_timeout_msecs)
{
	i_assert(user != NULL);
	i_assert(!client->destroyed);

	if (password == NULL) {
		client_syslog(&client->common, "proxy: Password not given");
		client_send_noresp(client, "TRYLATER", PROXY_FAILURE_MSG);
		return -1;
	}

	if ( (ssl_flags & PROXY_SSL_FLAG_YES) > 0 
		&& (ssl_flags & PROXY_SSL_FLAG_STARTTLS) == 0 ) {
		client_syslog(&client->common, 
			"proxy: SSL is enabled, but STARTTLS is not; "
			"not using SSL for managesieve");
		ssl_flags = 0;
	}

	i_assert(client->refcount > 1);
	connection_queue_add(1);

	if (client->destroyed) {
		/* connection_queue_add() decided that we were the oldest
		   connection and killed us. */
		return -1;
	}

	if ( login_proxy_is_ourself(&client->common, host, port, user )) {
        client_syslog_err(&client->common, "Proxying loops to itself");
		client_send_noresp(client, "TRYLATER", PROXY_FAILURE_MSG);
        return -1;
    }

	client->proxy = login_proxy_new
		(&client->common, host, port, ssl_flags, connect_timeout_msecs,
			proxy_input, client);
    if (client->proxy == NULL) {
		client_send_noresp(client, "TRYLATER", PROXY_FAILURE_MSG);
        return -1;
    }

	client->proxy_starttls = FALSE;
	client->proxy_state = PROXY_STATE_INITIAL;
	client->proxy_user = i_strdup(user);
	client->proxy_master_user = i_strdup(master_user);
	client->proxy_password = i_strdup(password);

	/* disable input until authentication is finished */
	if (client->io != NULL)
		io_remove(&client->io);

	return 0;
}
