/* Copyright (c) 2002-2010 Dovecot Sieve authors, see the included COPYING file
 */

#include "common.h"
#include "base64.h"
#include "buffer.h"
#include "ioloop.h"
#include "istream.h"
#include "ostream.h"
#include "safe-memset.h"
#include "str.h"
#include "str-sanitize.h"

#include "managesieve-parser.h"
#include "managesieve-quote.h"
#include "auth-client.h"
#include "client.h"
#include "client-authenticate.h"
#include "managesieve-proxy.h"

#include <unistd.h>
#include <stdlib.h>

#define AUTH_FAILURE_DELAY_INCREASE_MSECS 5000

#define MANAGESIEVE_SERVICE_NAME "sieve"

/* FIXME: The use of the ANONYMOUS mechanism is currently denied 
 */
static bool _sasl_mechanism_acceptable
	(const struct auth_mech_desc *mech, bool secured) {

	/* a) transport is secured
	   b) auth mechanism isn't plaintext
       c) we allow insecure authentication
	 */

	if ((mech->flags & MECH_SEC_PRIVATE) == 0 &&
		(mech->flags & MECH_SEC_ANONYMOUS) == 0 &&
 		(secured || !disable_plaintext_auth ||
		(mech->flags & MECH_SEC_PLAINTEXT) == 0)) {
    		return 1;     
	}  

	return 0;
}

const char *client_authenticate_get_capabilities(bool secured)
{
	const struct auth_mech_desc *mech;
	unsigned int i, count;
	string_t *str;

	str = t_str_new(128);
	mech = auth_client_get_available_mechs(auth_client, &count);

	if ( count > 0 ) {
		if ( _sasl_mechanism_acceptable(&(mech[0]), secured) ) {
			str_append(str, mech[0].name);
		}
     
		for (i = 1; i < count; i++) {
			if ( _sasl_mechanism_acceptable(&(mech[i]), secured) ) {
				str_append_c(str, ' ');
				str_append(str, mech[i].name);
			}
		}
	}

	return str_c(str);
}

static void client_auth_input(struct managesieve_client *client)
{
	struct managesieve_arg *args;
	const char *msg;
	char *line;
	bool fatal;

	if (!client_read(client))
		return;

	if (client->skip_line) {
		if (i_stream_next_line(client->common.input) == NULL)
			return;

		client->skip_line = FALSE;
	}

	switch (managesieve_parser_read_args(client->parser, 0, 0, &args)) {
	case -1:
		/* error */
		msg = managesieve_parser_get_error(client->parser, &fatal);
		if (fatal) {
			/* FIXME: What to do? */
		}
	  
		sasl_server_auth_failed(&client->common, msg);
		return;
	case -2:
		/* not enough data */
		return;
	}

	client->skip_line = TRUE;

	if (args[0].type != MANAGESIEVE_ARG_STRING || 
		args[1].type != MANAGESIEVE_ARG_EOL) {
		sasl_server_auth_failed(&client->common, 
			"Invalid AUTHENTICATE client response.");
		return;
	}

	line = MANAGESIEVE_ARG_STR(&args[0]);

	if (strcmp(line, "*") == 0)
		sasl_server_auth_abort(&client->common);
	else {
		client_set_auth_waiting(client);
		auth_client_request_continue(client->common.auth_request, line);
		io_remove(&client->io);
	
		/* clear sensitive data */
		safe_memset(line, 0, strlen(line));
	}
}

static void client_authfail_delay_timeout(struct managesieve_client *client)
{
	timeout_remove(&client->to_authfail_delay);

	/* get back to normal client input. */
	i_assert(client->io == NULL);
	client->io = io_add(client->common.fd, IO_READ, client_input, client);
	client_input(client);
}

void client_auth_failed(struct managesieve_client *client, bool nodelay)
{
	unsigned int delay_msecs;

	client->common.auth_command_tag = NULL;

	if ( client->auth_initializing )
		return;

	if ( client->io != NULL )
		io_remove(&client->io);
	if ( nodelay ) {
		client->io = io_add(client->common.fd, IO_READ, client_input, client);
 		client_input(client);
		return;
	}

	/* increase the timeout after each unsuccessful attempt, but don't
		increase it so high that the idle timeout would be triggered */
	delay_msecs = client->common.auth_attempts *
		AUTH_FAILURE_DELAY_INCREASE_MSECS;
	if (delay_msecs > CLIENT_LOGIN_IDLE_TIMEOUT_MSECS)
		delay_msecs = CLIENT_LOGIN_IDLE_TIMEOUT_MSECS - 1000;

	i_assert(client->to_authfail_delay == NULL);
	client->to_authfail_delay =
		timeout_add(delay_msecs, client_authfail_delay_timeout, client);
}

static bool client_handle_args(struct managesieve_client *client,
	const char *const *args, bool success, bool *nodelay_r)
{
	const char *reason = NULL, *host = NULL, *destuser = NULL, *pass = NULL;
	const char *master_user = NULL;
	const char *key, *value, *p;
	enum login_proxy_ssl_flags ssl_flags = 0;
	unsigned int port = 2000;
	unsigned int proxy_timeout_msecs = 0;
	bool proxy = FALSE, temp = FALSE, nologin = !success;
	bool authz_failure = FALSE;

	*nodelay_r = FALSE;
	for (; *args != NULL; args++) {
		p = strchr(*args, '=');
		if (p == NULL) {
			key = *args;
			value = "";
		} else {
			key = t_strdup_until(*args, p);
			value = p + 1;
		}
		if (strcmp(key, "nologin") == 0)
			nologin = TRUE;
		else if (strcmp(key, "nodelay") == 0)
			*nodelay_r = TRUE;
		else if (strcmp(key, "proxy") == 0)
			proxy = TRUE;
		else if (strcmp(key, "temp") == 0)
			temp = TRUE;
		else if (strcmp(key, "authz") == 0)
			authz_failure = TRUE;
		else if (strcmp(key, "reason") == 0)
			reason = value + 7;
		else if (strcmp(key, "host") == 0)
			host = value;
		else if (strcmp(key, "port") == 0)
			port = atoi(value);
		else if (strcmp(key, "destuser") == 0)
			destuser = value;
		else if (strcmp(key, "pass") == 0)
			pass = value;
		else if (strcmp(key, "proxy_timeout") == 0)
			proxy_timeout_msecs = 1000*atoi(value);
		else if (strcmp(key, "master") == 0)
			master_user = value;
		else if (strcmp(key, "ssl") == 0) {
			if (strcmp(value, "yes") == 0)
				ssl_flags |= PROXY_SSL_FLAG_YES;
			else if (strcmp(value, "any-cert") == 0) {
				ssl_flags |= PROXY_SSL_FLAG_YES |
					PROXY_SSL_FLAG_ANY_CERT;
			}
		} else if (strcmp(key, "starttls") == 0) {
			ssl_flags |= PROXY_SSL_FLAG_STARTTLS;
		} else if (strcmp(key, "user") == 0) {
			/* already handled in login-common */
		} else if (auth_debug) {
			i_info("Ignoring unknown passdb extra field: %s", key);
		}
	}

	if (destuser == NULL)
		destuser = client->common.virtual_user;

	if (proxy) {
		/* we want to proxy the connection to another server.
		don't do this unless authentication succeeded. with
		master user proxying we can get FAIL with proxy still set.

		proxy host=.. [port=..] [destuser=..] pass=.. */
		if (!success)
			return FALSE;
		if ( managesieve_proxy_new(client, host, port, destuser, master_user,
			pass, ssl_flags, proxy_timeout_msecs) < 0 )
			client_auth_failed(client, TRUE);
		return TRUE;
	}

	if (host != NULL) {
		string_t *resp_code;

		/* MANAGESIEVE referral

		   [nologin] referral host=.. [port=..] [destuser=..]
		   [reason=..]

		   NO (REFERRAL sieve://user;AUTH=mech@host:port/) Can't login.
		   OK (...) Logged in, but you should use this server instead.
		   .. [REFERRAL ..] (Reason from auth server)
		*/
		resp_code = t_str_new(128);
		str_printfa(resp_code, "REFERRAL sieve://%s;AUTH=%s@%s",
			    destuser, client->common.auth_mech_name, host);
		if (port != 2000)
			str_printfa(resp_code, ":%u", port);

		if (reason == NULL) {
			if (nologin)
				reason = "Try this server instead.";
			else 
				reason = "Logged in, but you should use "
					"this server instead.";
		}

		if (!nologin) {
			client_send_okresp(client, str_c(resp_code), reason);
			client_destroy_success(client, "Login with referral");
			return TRUE;
 		}
		client_send_noresp(client, str_c(resp_code), reason);
	} else if (nologin) {
		/* Authentication went ok, but for some reason user isn't
		   allowed to log in. Shouldn't probably happen. */
		if (reason != NULL)
			client_send_no(client, reason);
		else if (temp)
			client_send_no(client, AUTH_TEMP_FAILED_MSG);
		else if (authz_failure) 
			client_send_no(client, "Authorization failed.");
		else
			client_send_no(client, AUTH_FAILED_MSG);
	} else {
		/* normal login/failure */
		return FALSE;
	}

	i_assert(nologin);

	managesieve_parser_reset(client->parser);

	if (!client->destroyed) 
		client_auth_failed(client, *nodelay_r);
	return TRUE;
}

static void sasl_callback(struct client *_client, enum sasl_server_reply reply,
			  const char *data, const char *const *args)
{
	struct managesieve_client *client = (struct managesieve_client *)_client;
	string_t *str;
	bool nodelay;

	i_assert(!client->destroyed ||
		reply == SASL_SERVER_REPLY_AUTH_ABORTED ||
		reply == SASL_SERVER_REPLY_MASTER_FAILED);

	switch (reply) {
	case SASL_SERVER_REPLY_SUCCESS:
		if ( client->to_auth_waiting != NULL )
			timeout_remove(&client->to_auth_waiting);
		if (args != NULL) {
			if (client_handle_args(client, args, TRUE, &nodelay))
				break;
		}

		client_destroy_success(client, "Login");
		break;

	case SASL_SERVER_REPLY_AUTH_FAILED:
	case SASL_SERVER_REPLY_AUTH_ABORTED:
		if ( client->to_auth_waiting != NULL )
			timeout_remove(&client->to_auth_waiting);
		if (args != NULL) {
			if (client_handle_args(client, args, FALSE, &nodelay))
				break;
		}


		if ( reply == SASL_SERVER_REPLY_AUTH_ABORTED )
			client_send_no(client, "Authentication aborted by client.");
		else
			client_send_no(client, data != NULL ? data : AUTH_FAILED_MSG);

		managesieve_parser_reset(client->parser);

		if (!client->destroyed) 
			client_auth_failed(client, nodelay);
		break;

	case SASL_SERVER_REPLY_MASTER_FAILED:
		if (data == NULL)
			client_destroy_internal_failure(client);
		else {
			client_send_no(client, data);
			client_destroy_success(client, data);
		}
		break;

	case SASL_SERVER_REPLY_CONTINUE:
		T_BEGIN {
			str = t_str_new(256);
			managesieve_quote_append_string(str, data, TRUE);
			str_append(str, "\r\n");
				
			/* don't check return value here. it gets tricky if we try
			   to call client_destroy() in here. */
			(void)o_stream_send(client->output, str_c(str), str_len(str));
		} T_END;

		if (client->to_auth_waiting != NULL)
			timeout_remove(&client->to_auth_waiting);

		managesieve_parser_reset(client->parser);

		i_assert(client->io == NULL);
		client->io = io_add(client->common.fd, IO_READ, client_auth_input, client);
		client_auth_input(client);
		return;
	}

	client_unref(client);
}

int cmd_authenticate(struct managesieve_client *client, struct managesieve_arg *args)
{
	const char *mech_name, *init_resp = NULL;

	/* one mandatory argument: authentication mechanism name */
	if (args[0].type != MANAGESIEVE_ARG_STRING)
		return -1;
	if (args[1].type != MANAGESIEVE_ARG_EOL) {
		/* optional SASL initial response */
		if (args[1].type != MANAGESIEVE_ARG_STRING ||
		    args[2].type != MANAGESIEVE_ARG_EOL)
			return -1;
		init_resp = MANAGESIEVE_ARG_STR(&args[1]);
	}

	mech_name = MANAGESIEVE_ARG_STR(&args[0]);
	if (*mech_name == '\0') 
		return -1;

	/* FIXME: This refuses the ANONYMOUS mechanism. 
	 *   This can be removed once anonymous login is implemented according to the 
	 *   draft RFC. - Stephan
	 */
	if ( strncasecmp(mech_name, "ANONYMOUS", 9) == 0 ) {
		client_send_no(client, "ANONYMOUS mechanism is not implemented.");		
		return 0;
	}

	client_ref(client);
	client->auth_initializing = TRUE;
	sasl_server_auth_begin(&client->common, MANAGESIEVE_SERVICE_NAME, mech_name,
		init_resp, sasl_callback);
	client->auth_initializing = FALSE;
	if (!client->common.authenticating)
		return 1;

	/* don't handle input until we get the initial auth reply */
	if (client->io != NULL)
		io_remove(&client->io);
	client_set_auth_waiting(client);

	managesieve_parser_reset(client->parser);

	return 0;
}

