#!/usr/bin/perl
# Copyright © 2006-2008 Joey Hess <joey@ikiwiki.info>
# Copyright © 2008 Simon McVittie <http://smcv.pseudorandom.co.uk/>
# Licensed under the GNU GPL, version 2, or any later version published by the
# Free Software Foundation
package IkiWiki::Plugin::comments;

use warnings;
use strict;
use IkiWiki 3.00;
use Encode;
use POSIX qw(strftime);

use constant PREVIEW => "Preview";
use constant POST_COMMENT => "Post comment";
use constant CANCEL => "Cancel";

my $postcomment;
my %commentstate;

sub import {
	hook(type => "checkconfig", id => 'comments',  call => \&checkconfig);
	hook(type => "getsetup", id => 'comments',  call => \&getsetup);
	hook(type => "preprocess", id => 'comment', call => \&preprocess);
	# here for backwards compatability with old comments
	hook(type => "preprocess", id => '_comment', call => \&preprocess);
	hook(type => "sessioncgi", id => 'comment', call => \&sessioncgi);
	hook(type => "htmlize", id => "_comment", call => \&htmlize);
	hook(type => "pagetemplate", id => "comments", call => \&pagetemplate);
	hook(type => "formbuilder_setup", id => "comments", call => \&formbuilder_setup);
	# Load goto to fix up user page links for logged-in commenters
	IkiWiki::loadplugin("goto");
	IkiWiki::loadplugin("inline");
}

sub getsetup () {
	return
		plugin => {
			safe => 1,
			rebuild => 1,
			section => "web",
		},
		comments_pagespec => {
			type => 'pagespec',
			example => 'blog/* and !*/Discussion',
			description => 'PageSpec of pages where comments are allowed',
			link => 'ikiwiki/PageSpec',
			safe => 1,
			rebuild => 1,
		},
		comments_closed_pagespec => {
			type => 'pagespec',
			example => 'blog/controversial or blog/flamewar',
			description => 'PageSpec of pages where posting new comments is not allowed',
			link => 'ikiwiki/PageSpec',
			safe => 1,
			rebuild => 1,
		},
		comments_pagename => {
			type => 'string',
			default => 'comment_',
			description => 'Base name for comments, e.g. "comment_" for pages like "sandbox/comment_12"',
			safe => 0, # manual page moving required
			rebuild => undef,
		},
		comments_allowdirectives => {
			type => 'boolean',
			example => 0,
			description => 'Interpret directives in comments?',
			safe => 1,
			rebuild => 0,
		},
		comments_allowauthor => {
			type => 'boolean',
			example => 0,
			description => 'Allow anonymous commenters to set an author name?',
			safe => 1,
			rebuild => 0,
		},
		comments_commit => {
			type => 'boolean',
			example => 1,
			description => 'commit comments to the VCS',
			# old uncommitted comments are likely to cause
			# confusion if this is changed
			safe => 0,
			rebuild => 0,
		},
}

sub checkconfig () {
	$config{comments_commit} = 1
		unless defined $config{comments_commit};
	$config{comments_pagespec} = ''
		unless defined $config{comments_pagespec};
	$config{comments_closed_pagespec} = ''
		unless defined $config{comments_closed_pagespec};
	$config{comments_pagename} = 'comment_'
		unless defined $config{comments_pagename};
}

sub htmlize {
	my %params = @_;
	return $params{content};
}

# FIXME: copied verbatim from meta
sub safeurl ($) {
	my $url=shift;
	if (exists $IkiWiki::Plugin::htmlscrubber::{safe_url_regexp} &&
	    defined $IkiWiki::Plugin::htmlscrubber::safe_url_regexp) {
		return $url=~/$IkiWiki::Plugin::htmlscrubber::safe_url_regexp/;
	}
	else {
		return 1;
	}
}

sub preprocess {
	my %params = @_;
	my $page = $params{page};

	my $format = $params{format};
	if (defined $format && ! exists $IkiWiki::hooks{htmlize}{$format}) {
		error(sprintf(gettext("unsupported page format %s"), $format));
	}

	my $content = $params{content};
	if (! defined $content) {
		error(gettext("comment must have content"));
	}
	$content =~ s/\\"/"/g;

	$content = IkiWiki::filter($page, $params{destpage}, $content);

	if ($config{comments_allowdirectives}) {
		$content = IkiWiki::preprocess($page, $params{destpage},
			$content);
	}

	# no need to bother with htmlize if it's just HTML
	$content = IkiWiki::htmlize($page, $params{destpage}, $format, $content)
		if defined $format;

	IkiWiki::run_hooks(sanitize => sub {
		$content = shift->(
			page => $page,
			destpage => $params{destpage},
			content => $content,
		);
	});

	# set metadata, possibly overriding [[!meta]] directives from the
	# comment itself

	my $commentuser;
	my $commentip;
	my $commentauthor;
	my $commentauthorurl;
	my $commentopenid;
	if (defined $params{username}) {
		$commentuser = $params{username};

		my $oiduser = eval { IkiWiki::openiduser($commentuser) };

		if (defined $oiduser) {
			# looks like an OpenID
			$commentauthorurl = $commentuser;
			$commentauthor = $oiduser;
			$commentopenid = $commentuser;
		}
		else {
			$commentauthorurl = IkiWiki::cgiurl(
				do => 'goto',
				page => IkiWiki::userpage($commentuser)
			);

			$commentauthor = $commentuser;
		}
	}
	else {
		if (defined $params{ip}) {
			$commentip = $params{ip};
		}
		$commentauthor = gettext("Anonymous");
	}

	$commentstate{$page}{commentuser} = $commentuser;
	$commentstate{$page}{commentopenid} = $commentopenid;
	$commentstate{$page}{commentip} = $commentip;
	$commentstate{$page}{commentauthor} = $commentauthor;
	$commentstate{$page}{commentauthorurl} = $commentauthorurl;
	if (! defined $pagestate{$page}{meta}{author}) {
		$pagestate{$page}{meta}{author} = $commentauthor;
	}
	if (! defined $pagestate{$page}{meta}{authorurl}) {
		$pagestate{$page}{meta}{authorurl} = $commentauthorurl;
	}

	if ($config{comments_allowauthor}) {
		if (defined $params{claimedauthor}) {
			$pagestate{$page}{meta}{author} = $params{claimedauthor};
		}

		if (defined $params{url}) {
			my $url=$params{url};

			eval q{use URI::Heuristic}; 
		  	if (! $@) {
				$url=URI::Heuristic::uf_uristr($url);
			}

			if (safeurl($url)) {
				$pagestate{$page}{meta}{authorurl} = $url;
			}
		}
	}
	else {
		$pagestate{$page}{meta}{author} = $commentauthor;
		$pagestate{$page}{meta}{authorurl} = $commentauthorurl;
	}

	if (defined $params{subject}) {
		$pagestate{$page}{meta}{title} = $params{subject};
	}

	if ($params{page} =~ m/\/\Q$config{comments_pagename}\E\d+_/) {
		$pagestate{$page}{meta}{permalink} = urlto(IkiWiki::dirname($params{page}), undef, 1).
			"#".page_to_id($params{page});
	}

	eval q{use Date::Parse};
	if (! $@) {
		my $time = str2time($params{date});
		$IkiWiki::pagectime{$page} = $time if defined $time;
	}

	return $content;
}

sub sessioncgi ($$) {
	my $cgi=shift;
	my $session=shift;

	my $do = $cgi->param('do');
	if ($do eq 'comment') {
		editcomment($cgi, $session);
	}
	elsif ($do eq 'commentmoderation') {
		commentmoderation($cgi, $session);
	}
}

# Mostly cargo-culted from IkiWiki::plugin::editpage
sub editcomment ($$) {
	my $cgi=shift;
	my $session=shift;

	IkiWiki::decode_cgi_utf8($cgi);

	eval q{use CGI::FormBuilder};
	error($@) if $@;

	my @buttons = (POST_COMMENT, PREVIEW, CANCEL);
	my $form = CGI::FormBuilder->new(
		fields => [qw{do sid page subject editcontent type author url}],
		charset => 'utf-8',
		method => 'POST',
		required => [qw{editcontent}],
		javascript => 0,
		params => $cgi,
		action => $config{cgiurl},
		header => 0,
		table => 0,
		template => scalar IkiWiki::template_params('editcomment.tmpl'),
	);

	IkiWiki::decode_form_utf8($form);
	IkiWiki::run_hooks(formbuilder_setup => sub {
			shift->(title => "comment", form => $form, cgi => $cgi,
				session => $session, buttons => \@buttons);
		});
	IkiWiki::decode_form_utf8($form);

	my $type = $form->param('type');
	if (defined $type && length $type && $IkiWiki::hooks{htmlize}{$type}) {
		$type = IkiWiki::possibly_foolish_untaint($type);
	}
	else {
		$type = $config{default_pageext};
	}


	my @page_types;
	if (exists $IkiWiki::hooks{htmlize}) {
		foreach my $key (grep { !/^_/ } keys %{$IkiWiki::hooks{htmlize}}) {
			push @page_types, [$key, $IkiWiki::hooks{htmlize}{$key}{longname} || $key];
		}
	}
	@page_types=sort @page_types;

	$form->field(name => 'do', type => 'hidden');
	$form->field(name => 'sid', type => 'hidden', value => $session->id,
		force => 1);
	$form->field(name => 'page', type => 'hidden');
	$form->field(name => 'subject', type => 'text', size => 72);
	$form->field(name => 'editcontent', type => 'textarea', rows => 10);
	$form->field(name => "type", value => $type, force => 1,
		type => 'select', options => \@page_types);

	$form->tmpl_param(username => $session->param('name'));

	if ($config{comments_allowauthor} and
	    ! defined $session->param('name')) {
		$form->tmpl_param(allowauthor => 1);
		$form->field(name => 'author', type => 'text', size => '40');
		$form->field(name => 'url', type => 'text', size => '40');
	}
	else {
		$form->tmpl_param(allowauthor => 0);
		$form->field(name => 'author', type => 'hidden', value => '',
			force => 1);
		$form->field(name => 'url', type => 'hidden', value => '',
			force => 1);
	}

	if (! defined $session->param('name')) {
		# Make signinurl work and return here.
		$form->tmpl_param(signinurl => IkiWiki::cgiurl(do => 'signin'));
		$session->param(postsignin => $ENV{QUERY_STRING});
		IkiWiki::cgi_savesession($session);
	}

	# The untaint is OK (as in editpage) because we're about to pass
	# it to file_pruned anyway
	my $page = $form->field('page');
	$page = IkiWiki::possibly_foolish_untaint($page);
	if (! defined $page || ! length $page ||
		IkiWiki::file_pruned($page, $config{srcdir})) {
		error(gettext("bad page name"));
	}

	my $baseurl = urlto($page, undef, 1);

	$form->title(sprintf(gettext("commenting on %s"),
			IkiWiki::pagetitle($page)));

	$form->tmpl_param('helponformattinglink',
		htmllink($page, $page, 'ikiwiki/formatting',
			noimageinline => 1,
			linktext => 'FormattingHelp'),
			allowdirectives => $config{allow_directives});

	if ($form->submitted eq CANCEL) {
		# bounce back to the page they wanted to comment on, and exit.
		# CANCEL need not be considered in future
		IkiWiki::redirect($cgi, urlto($page, undef, 1));
		exit;
	}

	if (not exists $pagesources{$page}) {
		error(sprintf(gettext(
			"page '%s' doesn't exist, so you can't comment"),
			$page));
	}

	if (pagespec_match($page, $config{comments_closed_pagespec},
		location => $page)) {
		error(sprintf(gettext(
			"comments on page '%s' are closed"),
			$page));
	}

	# Set a flag to indicate that we're posting a comment,
	# so that postcomment() can tell it should match.
	$postcomment=1;
	IkiWiki::check_canedit($page, $cgi, $session);
	$postcomment=0;

	my $content = "[[!comment format=$type\n";

	# FIXME: handling of double quotes probably wrong?
	if (defined $session->param('name')) {
		my $username = $session->param('name');
		$username =~ s/"/&quot;/g;
		$content .= " username=\"$username\"\n";
	}
	elsif (defined $ENV{REMOTE_ADDR}) {
		my $ip = $ENV{REMOTE_ADDR};
		if ($ip =~ m/^([.0-9]+)$/) {
			$content .= " ip=\"$1\"\n";
		}
	}

	if ($config{comments_allowauthor}) {
		my $author = $form->field('author');
		if (defined $author && length $author) {
			$author =~ s/"/&quot;/g;
			$content .= " claimedauthor=\"$author\"\n";
		}
		my $url = $form->field('url');
		if (defined $url && length $url) {
			$url =~ s/"/&quot;/g;
			$content .= " url=\"$url\"\n";
		}
	}

	my $subject = $form->field('subject');
	if (defined $subject && length $subject) {
		$subject =~ s/"/&quot;/g;
	}
	else {
		$subject = "comment ".(num_comments($page, $config{srcdir}) + 1);
	}
	$content .= " subject=\"$subject\"\n";

	$content .= " date=\"" . decode_utf8(strftime('%Y-%m-%dT%H:%M:%SZ', gmtime)) . "\"\n";

	my $editcontent = $form->field('editcontent') || '';
	$editcontent =~ s/\r\n/\n/g;
	$editcontent =~ s/\r/\n/g;
	$editcontent =~ s/"/\\"/g;
	$content .= " content=\"\"\"\n$editcontent\n\"\"\"]]\n";

	my $location=unique_comment_location($page, $content, $config{srcdir});

	# This is essentially a simplified version of editpage:
	# - the user does not control the page that's created, only the parent
	# - it's always a create operation, never an edit
	# - this means that conflicts should never happen
	# - this means that if they do, rocks fall and everyone dies

	if ($form->submitted eq PREVIEW) {
		my $preview=previewcomment($content, $location, $page, time);
		IkiWiki::run_hooks(format => sub {
			$preview = shift->(page => $page,
				content => $preview);
		});
		$form->tmpl_param(page_preview => $preview);
	}
	else {
		$form->tmpl_param(page_preview => "");
	}

	if ($form->submitted eq POST_COMMENT && $form->validate) {
		IkiWiki::checksessionexpiry($cgi, $session);
		
		$postcomment=1;
		my $ok=IkiWiki::check_content(content => $form->field('editcontent'),
			subject => $form->field('subject'),
			$config{comments_allowauthor} ? (
				author => $form->field('author'),
				url => $form->field('url'),
			) : (),
			page => $location,
			cgi => $cgi,
			session => $session,
			nonfatal => 1,
		);
		$postcomment=0;

		if (! $ok) {
			my $penddir=$config{wikistatedir}."/comments_pending";
			$location=unique_comment_location($page, $content, $penddir);
			writefile("$location._comment", $penddir, $content);
			IkiWiki::printheader($session);
			print IkiWiki::misctemplate(gettext(gettext("comment stored for moderation")),
				"<p>".
				gettext("Your comment will be posted after moderator review").
				"</p>");
			exit;
		}

		# FIXME: could probably do some sort of graceful retry
		# on error? Would require significant unwinding though
		my $file = "$location._comment";
		writefile($file, $config{srcdir}, $content);

		my $conflict;

		if ($config{rcs} and $config{comments_commit}) {
			my $message = gettext("Added a comment");
			if (defined $form->field('subject') &&
				length $form->field('subject')) {
				$message = sprintf(
					gettext("Added a comment: %s"),
					$form->field('subject'));
			}

			IkiWiki::rcs_add($file);
			IkiWiki::disable_commit_hook();
			$conflict = IkiWiki::rcs_commit_staged($message,
				$session->param('name'), $ENV{REMOTE_ADDR});
			IkiWiki::enable_commit_hook();
			IkiWiki::rcs_update();
		}

		# Now we need a refresh
		require IkiWiki::Render;
		IkiWiki::refresh();
		IkiWiki::saveindex();

		# this should never happen, unless a committer deliberately
		# breaks it or something
		error($conflict) if defined $conflict;

		# Jump to the new comment on the page.
		# The trailing question mark tries to avoid broken
		# caches and get the most recent version of the page.
		IkiWiki::redirect($cgi, urlto($page, undef, 1).
			"?updated#".page_to_id($location));

	}
	else {
		IkiWiki::showform ($form, \@buttons, $session, $cgi,
			forcebaseurl => $baseurl);
	}

	exit;
}

sub commentmoderation ($$) {
	my $cgi=shift;
	my $session=shift;

	IkiWiki::needsignin($cgi, $session);
	if (! IkiWiki::is_admin($session->param("name"))) {
		error(gettext("you are not logged in as an admin"));
	}

	IkiWiki::decode_cgi_utf8($cgi);
	
	if (defined $cgi->param('sid')) {
		IkiWiki::checksessionexpiry($cgi, $session);

		my $rejectalldefer=$cgi->param('rejectalldefer');

		my %vars=$cgi->Vars;
		my $added=0;
		foreach my $id (keys %vars) {
			if ($id =~ /(.*)\Q._comment\E$/) {
				my $action=$cgi->param($id);
				next if $action eq 'Defer' && ! $rejectalldefer;

				# Make sure that the id is of a legal
				# pending comment before untainting.
				my ($f)= $id =~ /$config{wiki_file_regexp}/;
				if (! defined $f || ! length $f ||
				    IkiWiki::file_pruned($f, $config{srcdir})) {
					error("illegal file");
				}

				my $page=IkiWiki::possibly_foolish_untaint(IkiWiki::dirname($1));
				my $file="$config{wikistatedir}/comments_pending/".
					IkiWiki::possibly_foolish_untaint($id);

				if ($action eq 'Accept') {
					my $content=eval { readfile($file) };
					next if $@; # file vanished since form was displayed
					my $dest=unique_comment_location($page, $content, $config{srcdir})."._comment";
					writefile($dest, $config{srcdir}, $content);
					if ($config{rcs} and $config{comments_commit}) {
						IkiWiki::rcs_add($dest);
					}
					$added++;
				}

				# This removes empty subdirs, so the
				# .ikiwiki/comments_pending dir will
				# go away when all are moderated.
				require IkiWiki::Render;
				IkiWiki::prune($file);
			}
		}

		if ($added) {
			my $conflict;
			if ($config{rcs} and $config{comments_commit}) {
				my $message = gettext("Comment moderation");
				IkiWiki::disable_commit_hook();
				$conflict=IkiWiki::rcs_commit_staged($message,
					$session->param('name'), $ENV{REMOTE_ADDR});
				IkiWiki::enable_commit_hook();
				IkiWiki::rcs_update();
			}
		
			# Now we need a refresh
			require IkiWiki::Render;
			IkiWiki::refresh();
			IkiWiki::saveindex();
		
			error($conflict) if defined $conflict;
		}
	}

	my @comments=map {
		my ($id, $ctime)=@{$_};
		my $file="$config{wikistatedir}/comments_pending/$id";
		my $content=readfile($file);
		my $preview=previewcomment($content, $id,
			IkiWiki::dirname($_), $ctime);
		{
			id => $id,
			view => $preview,
		} 
	} sort { $b->[1] <=> $a->[1] } comments_pending();

	my $template=template("commentmoderation.tmpl");
	$template->param(
		sid => $session->id,
		comments => \@comments,
	);
	IkiWiki::printheader($session);
	my $out=$template->output;
	IkiWiki::run_hooks(format => sub {
		$out = shift->(page => "", content => $out);
	});
	print IkiWiki::misctemplate(gettext("comment moderation"), $out);
	exit;
}

sub formbuilder_setup (@) {
	my %params=@_;

	my $form=$params{form};
	if ($form->title eq "preferences" &&
	    IkiWiki::is_admin($params{session}->param("name"))) {
		push @{$params{buttons}}, "Comment Moderation";
		if ($form->submitted && $form->submitted eq "Comment Moderation") {
			commentmoderation($params{cgi}, $params{session});
		}
	}
}

sub comments_pending () {
	my $dir="$config{wikistatedir}/comments_pending/";
	return unless -d $dir;

	my @ret;
	eval q{use File::Find};
	error($@) if $@;
	find({
		no_chdir => 1,
		wanted => sub {
			$_=decode_utf8($_);
			if (IkiWiki::file_pruned($_, $dir)) {
				$File::Find::prune=1;
			}
			elsif (! -l $_ && ! -d _) {
				$File::Find::prune=0;
				my ($f)=/$config{wiki_file_regexp}/; # untaint
				if (defined $f && $f =~ /\Q._comment\E$/) {
					my $ctime=(stat($f))[10];
					$f=~s/^\Q$dir\E\/?//;
                                        push @ret, [$f, $ctime];
				}
			}
		}
	}, $dir);

	return @ret;
}

sub previewcomment ($$$) {
	my $content=shift;
	my $location=shift;
	my $page=shift;
	my $time=shift;

	my $preview = IkiWiki::htmlize($location, $page, '_comment',
			IkiWiki::linkify($location, $page,
			IkiWiki::preprocess($location, $page,
			IkiWiki::filter($location, $page, $content), 0, 1)));

	my $template = template("comment.tmpl");
	$template->param(content => $preview);
	$template->param(ctime => displaytime($time));

	IkiWiki::run_hooks(pagetemplate => sub {
		shift->(page => $location,
			destpage => $page,
			template => $template);
	});

	$template->param(have_actions => 0);

	return $template->output;
}

sub commentsshown ($) {
	my $page=shift;

	return ! pagespec_match($page, "internal(*/$config{comments_pagename}*)",
	                        location => $page) &&
	       pagespec_match($page, $config{comments_pagespec},
	                      location => $page);
}

sub commentsopen ($) {
	my $page = shift;

	return length $config{cgiurl} > 0 &&
	       (! length $config{comments_closed_pagespec} ||
	        ! pagespec_match($page, $config{comments_closed_pagespec},
	                         location => $page));
}

sub pagetemplate (@) {
	my %params = @_;

	my $page = $params{page};
	my $template = $params{template};
	my $shown = ($template->query(name => 'commentslink') ||
	             $template->query(name => 'commentsurl') ||
	             $template->query(name => 'atomcommentsurl') ||
	             $template->query(name => 'comments')) &&
	            commentsshown($page);

	if ($template->query(name => 'comments')) {
		my $comments = undef;
		if ($shown) {
			$comments = IkiWiki::preprocess_inline(
				pages => "internal($page/$config{comments_pagename}*)",
				template => 'comment',
				show => 0,
				reverse => 'yes',
				page => $page,
				destpage => $params{destpage},
				feedfile => 'comments',
				emptyfeeds => 'no',
			);
		}

		if (defined $comments && length $comments) {
			$template->param(comments => $comments);
		}

		if ($shown && commentsopen($page)) {
			$template->param(addcommenturl => addcommenturl($page));
		}
	}

	if ($shown) {
		if ($template->query(name => 'commentsurl')) {
			$template->param(commentsurl =>
				urlto($page, undef, 1).'#comments');
		}

		if ($template->query(name => 'atomcommentsurl') && $config{usedirs}) {
			# This will 404 until there are some comments, but I
			# think that's probably OK...
			$template->param(atomcommentsurl =>
				urlto($page, undef, 1).'comments.atom');
		}

		if ($template->query(name => 'commentslink')) {
			my $num=num_comments($page, $config{srcdir});
			my $link;
			if ($num > 0) {
				$link = htmllink($page, $params{destpage}, $page,
					linktext => sprintf(ngettext("%i comment", "%i comments", $num), $num),
					anchor => "comments",
					noimageinline => 1
				);
			}
			elsif (commentsopen($page)) {
				$link = "<a href=\"".addcommenturl($page)."\">".
					#translators: Here "Comment" is a verb;
					#translators: the user clicks on it to
					#translators: post a comment.
					gettext("Comment").
					"</a>";
			}
			$template->param(commentslink => $link)
				if defined $link;
		}
	}

	# everything below this point is only relevant to the comments
	# themselves
	if (!exists $commentstate{$page}) {
		return;
	}
	
	if ($template->query(name => 'commentid')) {
		$template->param(commentid => page_to_id($page));
	}

	if ($template->query(name => 'commentuser')) {
		$template->param(commentuser =>
			$commentstate{$page}{commentuser});
	}

	if ($template->query(name => 'commentopenid')) {
		$template->param(commentopenid =>
			$commentstate{$page}{commentopenid});
	}

	if ($template->query(name => 'commentip')) {
		$template->param(commentip =>
			$commentstate{$page}{commentip});
	}

	if ($template->query(name => 'commentauthor')) {
		$template->param(commentauthor =>
			$commentstate{$page}{commentauthor});
	}

	if ($template->query(name => 'commentauthorurl')) {
		$template->param(commentauthorurl =>
			$commentstate{$page}{commentauthorurl});
	}

	if ($template->query(name => 'removeurl') &&
	    IkiWiki::Plugin::remove->can("check_canremove") &&
	    length $config{cgiurl}) {
		$template->param(removeurl => IkiWiki::cgiurl(do => 'remove',
			page => $page));
		$template->param(have_actions => 1);
	}
}

sub addcommenturl ($) {
	my $page=shift;

	return IkiWiki::cgiurl(do => 'comment', page => $page);
}

sub num_comments ($$) {
	my $page=shift;
	my $dir=shift;

	my @comments=glob("$dir/$page/$config{comments_pagename}*._comment");
	return @comments;
}

sub unique_comment_location ($$$) {
	my $page=shift;

	eval q{use Digest::MD5 'md5_hex'};
	error($@) if $@;
	my $content_md5=md5_hex(Encode::encode_utf8(shift));

	my $dir=shift;

	my $location;
	my $i = num_comments($page, $dir);
	do {
		$i++;
		$location = "$page/$config{comments_pagename}${i}_${content_md5}";
	} while (-e "$dir/$location._comment");

	return $location;
}

sub page_to_id ($) {
	# Converts a comment page name into a unique, legal html id
	# attribute value, that can be used as an anchor to link to the
	# comment.
	my $page=shift;

	eval q{use Digest::MD5 'md5_hex'};
	error($@) if $@;

	return "comment-".md5_hex(Encode::encode_utf8(($page)));
}
	
package IkiWiki::PageSpec;

sub match_postcomment ($$;@) {
	my $page = shift;
	my $glob = shift;

	if (! $postcomment) {
		return IkiWiki::FailReason->new("not posting a comment");
	}
	return match_glob($page, $glob);
}

1
