/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for bookmarks management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbookmarkplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgmainpanel.h"
#include "skgservices.h"
#include "skgnodeobject.h"
#include "skgtransactionmng.h"
#include "skgbookmarkplugindockwidget.h"

#include <QtGui/QWidget>

#include <kactioncollection.h>
#include <kstandarddirs.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY ( SKGBookmarkPluginFactory, registerPlugin<SKGBookmarkPlugin>(); )
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN ( SKGBookmarkPluginFactory ( "skrooge_bookmark", "skrooge_bookmark" ) )

SKGBookmarkPlugin::SKGBookmarkPlugin ( QObject* iParent, const QVariantList& iArg ) :
        SKGInterfacePlugin ( iParent ), dockWidget ( NULL )
{
    Q_UNUSED ( iArg );
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::SKGBookmarkPlugin" );
}

SKGBookmarkPlugin::~SKGBookmarkPlugin()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::~SKGBookmarkPlugin" );
    currentBankDocument=NULL;
    dockWidget=NULL;
    importStdBookmarkAction=NULL;
}

void SKGBookmarkPlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::setupActions" );
    Q_UNUSED ( iArgument );

    currentBankDocument=iDocument;


    setComponentData ( SKGBookmarkPluginFactory::componentData() );
    setXMLFile ( "skrooge_bookmark.rc" );

    dockWidget = new QDockWidget ( SKGMainPanel::getMainPanel() );
    dockWidget->setObjectName ( QString::fromUtf8 ( "skrooge_bookmark_docwidget" ) );
    dockWidget->setAllowedAreas ( Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea );
    dockWidget->setWindowTitle ( title() );
    dockWidget->setWidget ( new SKGBookmarkPluginDockWidget ( currentBankDocument ) );

    // add action to control hide / display of Bookmarks
    dockWidget->toggleViewAction()->setShortcut ( Qt::SHIFT+Qt::Key_F10 );
    actionCollection()->addAction ( "view_bookmarks", dockWidget->toggleViewAction() );

    //Import bookmarks
    QStringList overlaybookmarks;
    overlaybookmarks.push_back ( "rating" );

    importStdBookmarkAction = new KAction ( KIcon ( "document-import", NULL, overlaybookmarks ), i18n ( "Import standard bookmarks" ), this );
    connect ( importStdBookmarkAction, SIGNAL ( triggered ( bool ) ), SLOT ( importStandardBookmarks() ) );
    actionCollection()->addAction ( QLatin1String ( "import_standard_bookmarks" ), importStdBookmarkAction );

    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "import_standard_bookmarks", importStdBookmarkAction );
}

void SKGBookmarkPlugin::refresh()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::refresh" );
    if ( dockWidget )
    {
        SKGBookmarkPluginDockWidget* p= ( SKGBookmarkPluginDockWidget* ) dockWidget->widget();
        if ( p ) p->refresh();
    }

    if ( currentBankDocument )
    {
        bool test= ( currentBankDocument->getDatabase() !=NULL );
        if ( importStdBookmarkAction ) importStdBookmarkAction->setEnabled ( test );

        //Automatic bookmarks creation
        if ( currentBankDocument->getDatabase() !=NULL )
        {
            QString doc_id=currentBankDocument->getUniqueIdentifier();
            if ( docUniqueIdentifier!=doc_id )
            {
                docUniqueIdentifier=doc_id;

                int nbNode=0;
                SKGError err=SKGObjectBase::getNbObjects ( currentBankDocument, "node", "", nbNode );
                if ( err.isSucceeded() && nbNode==0 )
                {
                    importStandardBookmarks();
                }

                //Automatic open of autostart bookmark
                if ( err.isSucceeded() )
                {
                    SKGObjectBase::SKGListSKGObjectBase oNodeList;
                    err = SKGObjectBase::getObjects ( currentBankDocument, "v_node", "t_autostart='Y' ORDER BY f_sortorder, t_name", oNodeList );
                    int nbAutoStartedBookmarks = oNodeList.count();
                    for ( int i=0;i<nbAutoStartedBookmarks;i++ )
                    {
                        SKGNodeObject autostarted_bookmark = oNodeList.at ( i );
                        autostarted_bookmark.load();
                        SKGTRACEIN ( 10, "autostarting bookmark : "+autostarted_bookmark.getName() );
                        SKGBookmarkPluginDockWidget::openBookmark ( autostarted_bookmark, i>0 );

                    }
                }
            }
        }
    }
}

QString SKGBookmarkPlugin::title() const
{
    return i18nc ( "Noun, a bookmark as in a webbrowser bookmark","Bookmark" );
}

QString SKGBookmarkPlugin::icon() const
{
    return "rating";
}

QString SKGBookmarkPlugin::toolTip () const
{
    return i18nc ( "Noun, a bookmark as in a webbrowser bookmark","Bookmark" );
}

QStringList SKGBookmarkPlugin::tips() const
{
    QStringList output;
    QString autostart=i18nc ( "Verb, automatically load when skrooge is started","autostart" );
    output.push_back ( i18n ( "<p>... some bookmarks can be opened automatically when Skrooge is launched." ) );
    output.push_back ( i18n ( "<p>... bookmarks can be reorganized by drag & drop.</p>" ) );
    output.push_back ( i18n ( "<p>... a double click on a folder of bookmarks will open all the bookmarks it contains.</p>" ) );
    output.push_back ( i18n ( "<p>... you can import standard bookmarks.</p>" ) );
    return output;
}

int SKGBookmarkPlugin::getOrder() const
{
    return 3;
}

void SKGBookmarkPlugin::close()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::close" );
}

QDockWidget* SKGBookmarkPlugin::getDockWidget()
{
    return dockWidget;
}

void SKGBookmarkPlugin::importStandardBookmarks()
{
    SKGTRACEIN ( 10, "SKGBookmarkPlugin::importStandardBookmarks" );
    SKGError err;
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGBEGINTRANSACTION ( *currentBankDocument, i18n ( "Import standard bookmarks" ), err );

    QStringList bks;
    bks << i18n ( "Dashboard" ) +"|Y|\"Skrooge dashboard plugin\";\"Dashboard\";\"go-home\";\"<!DOCTYPE SKGML><parameters nbMaxCol=\"\"3\"\" > <ITEM-1 index=\"\"0\"\" name=\"\"Skrooge bank plugin\"\" state=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters menuOther=&quot;Y&quot; menuCurrent=&quot;Y&quot; menuAssets=&quot;Y&quot; menuInvestment=&quot;Y&quot; menuCreditCard=&quot;Y&quot; />&#xa;\"\" /> <ITEM-2 index=\"\"0\"\" name=\"\"Skrooge operation plugin\"\" state=\"\"\"\" /> <ITEM-3 index=\"\"1\"\" name=\"\"Skrooge operation plugin\"\" state=\"\"\"\" /> <ITEM-4 index=\"\"0\"\" name=\"\"Skrooge scheduled plugin\"\" state=\"\"\"\" /> <ITEM-5 index=\"\"0\"\" name=\"\"Skrooge search plugin\"\" state=\"\"\"\" /> <ITEM-6 index=\"\"0\"\" name=\"\"Skrooge unit plugin\"\" state=\"\"\"\" /> <ITEM-7 index=\"\"0\"\" name=\"\"Skrooge calculator plugin\"\" state=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters/>&#xa;\"\" /></parameters>\""
    << i18n ( "Report > Income vs Expenditure on 12 last months" ) +"|N|\"Skrooge report plugin\";\"Rapport\";\"view-statistics\";\"<!DOCTYPE SKGML><parameters tableAndGraphState=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters graphMode=&quot;1&quot; graphicViewState=&quot;&amp;lt;!DOCTYPE SKGML>&amp;#xa;&amp;lt;parameters isToolBarVisible=&amp;quot;Y&amp;quot; />&amp;#xa;&quot; splitterState=&quot;000000ff00000000000000020000025b0000032901000000060100000001&quot; nbLevel=&quot;0&quot; allPositive=&quot;Y&quot; sortColumn=&quot;0&quot; sortOrder=&quot;0&quot; />&#xa;\"\" period=\"\"3\"\" interval=\"\"2\"\" nb_intervals=\"\"12\"\" lines=\"\"t_TYPEEXPENSENLS\"\" columns=\"\"d_DATEMONTH\"\" incomes=\"\"Y\"\" expenses=\"\"Y\"\" transfers=\"\"N\"\" />\""
    << i18n ( "Report > Pie categories in 12 last months" ) +"|N|\"Skrooge report plugin\";\"Rapport\";\"view-statistics\";\"<!DOCTYPE SKGML><parameters tableAndGraphState=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters graphMode=&quot;2&quot; graphicViewState=&quot;&amp;lt;!DOCTYPE SKGML>&amp;#xa;&amp;lt;parameters isToolBarVisible=&amp;quot;Y&amp;quot; />&amp;#xa;&quot; splitterState=&quot;000000ff00000000000000020000025b0000032901000000060100000001&quot; nbLevel=&quot;0&quot; allPositive=&quot;Y&quot; sortColumn=&quot;1&quot; sortOrder=&quot;0&quot; />&#xa;\"\" period=\"\"3\"\" interval=\"\"2\"\" nb_intervals=\"\"12\"\" lines=\"\"t_REALCATEGORY\"\" columns=\"\"\"\" incomes=\"\"Y\"\" expenses=\"\"Y\"\" transfers=\"\"N\"\" />\""
    << i18n ( "Report > History" ) +"|N|\"Skrooge report plugin\";\"Rapport\";\"view-statistics\";\"<!DOCTYPE SKGML><parameters tableAndGraphState=\"\"&lt;!DOCTYPE SKGML>&#xa;&lt;parameters graphMode=&quot;1&quot; graphicViewState=&quot;&amp;lt;!DOCTYPE SKGML>&amp;#xa;&amp;lt;parameters isToolBarVisible=&amp;quot;Y&amp;quot; antialiasing=&amp;quot;Y&amp;quot; />&amp;#xa;&quot; splitterState=&quot;000000ff0000000000000002000002b50000032601000000060100000001&quot; nbLevel=&quot;0&quot; filter=&quot;&quot; allPositive=&quot;N&quot; sortColumn=&quot;68&quot; sortOrder=&quot;0&quot; />&#xa;\"\" mode=\"\"1\"\" period=\"\"0\"\" lines=\"\"t_ACCOUNTTYPE\"\" columns=\"\"d_DATEMONTH\"\" incomes=\"\"Y\"\" expenses=\"\"Y\"\" transfers=\"\"Y\"\" />\""
    ;

    foreach ( const QString& bk, bks )
    {
        QStringList line=SKGServices::splitCSVLine ( bk, '|', false );
        if ( line.count() ==3 )
        {
            SKGNodeObject node;
            err=SKGNodeObject::createPathNode ( currentBankDocument, line.at ( 0 ), node );
            if ( err.isSucceeded() ) err=node.setAutoStart ( line.at ( 1 ) =="Y" );
            if ( err.isSucceeded() ) err=node.setData ( line.at ( 2 ) );
            if ( err.isSucceeded() ) err=node.save();
        }
    }

    QApplication::restoreOverrideCursor();

    //status
    if ( err.isSucceeded() )  err=SKGError ( 0, i18n ( "Standard bookmarks imported." ) );
    else err.addError ( ERR_FAIL, i18n ( "Import standard bookmarks failed" ) );

    //Display error
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}
#include "skgbookmarkplugin.moc"
