/*
 * Photo Image Print System
 * Copyright (C) 2002-2005 EPSON AVASYS Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2002-2005.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, EPSON AVASYS Corporation gives permission to
 * link the code of this program with libraries which are covered by
 * the EPSON AVASYS Public License and distribute their linked
 * combinations.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the libraries which
 * are covered by EPSON AVASYS Public License.
 */

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "config.h"

#include "definitions.h"
#include "ppd_generator.h"


/* PpdInfo class */

typedef struct {
	char			*model;
	ListHandle		paper_list;
	ListHandle		prtopt_list;
	ListHandle		qlty_list;
	FILE			*stream;
} PpdInfo, *PpdHandle;

const char* status_info_arr[STS_NUM] = 
{
	"OK!",
	"Printer not connected!",
	"Printer error!",
	"paper_list.csv not found!",
	"paper_list.csv contains error!",
	"prtopt.csv not found!",
	"prtopt.csv contains error!",
	"Printer option not found!",
	"Media size not recognized",
	"Media type not recognized",
	"PPD file cannot be created",
	"PPD file cannot be written"
};


static PpdInfo ppd_handle;

void create_qlty_name(const QltyNode* qlty_node, char* qlty_code, char* qlty_name)
{
	if (qlty_node->quality_id == QLT_NORMAL)
	{
		sprintf(qlty_code, "%s_%s", qlty_node->opdat_handle->rsc_name, "NORMAL");
		sprintf(qlty_name, "%s - %s", qlty_node->opdat_handle->x_name, "Normal");
	}
	else if (qlty_node->quality_id == QLT_HIGH)
	{
		sprintf(qlty_code, "%s_%s", qlty_node->opdat_handle->rsc_name, "HIGH");
		sprintf(qlty_name, "%s - %s", qlty_node->opdat_handle->x_name, "High");
	}
	else if (qlty_node->quality_id == QLT_DRAFT)
	{
		sprintf(qlty_code, "%s_%s", qlty_node->opdat_handle->rsc_name, "DRAFT");
		sprintf(qlty_name, "%s - %s", qlty_node->opdat_handle->x_name, "Draft");
	}
}

int feed_uiconstraints_line_msize(
	PVOID*	_msize_node_handle,
	PVOID	_qlty_node_handle
)
{
	PrtOptNode**	msize_node_handle = (PrtOptNode**)_msize_node_handle;
	QltyNode**	qlty_node_handle = (QltyNode**)_qlty_node_handle;

	MediaTypeNode	*key_mtnode, **found_mtnode_handle;
	char		qlty_code[50];
	Str100		qlty_name;
	ASSERT(ppd_handle.model && ppd_handle.paper_list && ppd_handle.prtopt_list && ppd_handle.stream)
	ASSERT(qlty_node_handle && *qlty_node_handle && msize_node_handle && *msize_node_handle && 
		(*qlty_node_handle)->opdat_handle)

	key_mtnode = mediatype_init((*qlty_node_handle)->opdat_handle->value);
	if ((found_mtnode_handle = (MediaTypeNode**)list_find_item((*msize_node_handle)->media_type_list, (PVOID*)&key_mtnode)) != NULL)
	{	
		if ((*found_mtnode_handle)->print_mode[FUCHI_NASI][(*qlty_node_handle)->quality_id] == false)
		{
			create_qlty_name(*qlty_node_handle, qlty_code, qlty_name);
			fprintf(ppd_handle.stream, 
				"*UIConstraints: *Quality %s *PageSize T%s\n",
				qlty_code, (*msize_node_handle)->pnode->code);
		}
	}
	else
	{	
		create_qlty_name(*qlty_node_handle, qlty_code, qlty_name);
		fprintf(ppd_handle.stream, 
			"*UIConstraints: *Quality %s *PageSize %s\n",
			qlty_code, (*msize_node_handle)->pnode->code);
		fprintf(ppd_handle.stream, 
			"*UIConstraints: *Quality %s *PageSize T%s\n",
			qlty_code, (*msize_node_handle)->pnode->code);
	}

	mediatype_done((PVOID*)&key_mtnode);
	return 1;
}

int feed_uiconstraints_line(
	PVOID*		_item,
	PVOID		user_data
)
{
	QltyNode**	item = (QltyNode**)_item;
	ASSERT(ppd_handle.model && ppd_handle.paper_list && ppd_handle.prtopt_list && ppd_handle.stream);
	ASSERT(item && *item)

	list_for_each(ppd_handle.prtopt_list, feed_uiconstraints_line_msize, item);
	return 1;
}

int feed_pagesize_line(
	PVOID*	_item,
	PVOID	user_data
)
{
	PrtOptNode**	item = (PrtOptNode**)_item;

	ASSERT(ppd_handle.model && ppd_handle.paper_list && ppd_handle.prtopt_list && ppd_handle.stream);
	ASSERT(item && *item && (*item)->pnode)
	fprintf(ppd_handle.stream, 
		"*PageSize %s/%s:                           \"<</PageSize[%.1f %.1f]/ImagingBBox null>>setpagedevice\"\n",
		(*item)->pnode->code, (*item)->pnode->name, 
		(*item)->pnode->size.width, (*item)->pnode->size.height);
	
	if ((*item)->borderless_supported)
	{
		fprintf(ppd_handle.stream, 
			"*PageSize T%s/%s (no margin):                     \"<</PageSize[%.1f %.1f]/ImagingBBox null>>setpagedevice\"\n",
			(*item)->pnode->code, (*item)->pnode->name, 
			(*item)->pnode->size.width, (*item)->pnode->size.height);
	}
	
	return true;
}

int feed_pageregion_line(
	PVOID*	_item,
	PVOID	user_data
)
{
	PrtOptNode**	item = (PrtOptNode**)_item;

	ASSERT(ppd_handle.model && ppd_handle.paper_list && ppd_handle.prtopt_list && ppd_handle.stream);
	ASSERT(item && *item && (*item)->pnode)
	fprintf(ppd_handle.stream, 
		"*PageRegion %s/%s:                           \"<</PageRegion[%.1f %.1f]/ImagingBBox null>>setpagedevice\"\n",
		(*item)->pnode->code, (*item)->pnode->name, 
		(*item)->pnode->size.width, (*item)->pnode->size.height);
	
	if ((*item)->borderless_supported)
	{
		fprintf(ppd_handle.stream, 
			"*PageRegion T%s/%s (no margin):                     \"<</PageRegion[%.1f %.1f]/ImagingBBox null>>setpagedevice\"\n",
			(*item)->pnode->code, (*item)->pnode->name, 
			(*item)->pnode->size.width, (*item)->pnode->size.height);
	}
	
	return true;
}

int feed_imageablearea_line(
	PVOID*	_item,
	PVOID	user_data
)
{
	PrtOptNode**	item = (PrtOptNode**)_item;

	PaperMargin imageable_area;
	ASSERT(ppd_handle.model && ppd_handle.paper_list && ppd_handle.prtopt_list && ppd_handle.stream);
	ASSERT(item && *item && (*item)->pnode)

	imageable_area.left = (*item)->pnode->margin.left;
	imageable_area.bottom = (*item)->pnode->margin.bottom;
	imageable_area.right = (*item)->pnode->size.width - (*item)->pnode->margin.right;
	imageable_area.top = (*item)->pnode->size.height - (*item)->pnode->margin.top;
	fprintf(ppd_handle.stream, 
		"*ImageableArea %s:          \"%.1f %.1f %.1f %.1f\"\n",
		(*item)->pnode->code, imageable_area.left, imageable_area.bottom, imageable_area.right, imageable_area.top);

	if ((*item)->borderless_supported)
	{
		fprintf(ppd_handle.stream, 
			"*ImageableArea T%s:       \"%.1f %.1f %.1f %.1f\"\n",
			(*item)->pnode->code, 0., 0., (*item)->pnode->size.width, (*item)->pnode->size.height);
	}
	return true;
}

int feed_paperdimension_line(
	PVOID*	_item,
	PVOID	user_data
)
{
	PrtOptNode**	item = (PrtOptNode**)_item;

	ASSERT(ppd_handle.model && ppd_handle.paper_list && ppd_handle.prtopt_list && ppd_handle.stream);
	ASSERT(item && *item)
	fprintf(ppd_handle.stream, 
		"*PaperDimension %s:         \"%.1f %.1f\"\n",
		(*item)->pnode->code, (*item)->pnode->size.width, (*item)->pnode->size.height);
	if ((*item)->borderless_supported)
	{
		fprintf(ppd_handle.stream, 
			"*PaperDimension T%s:     \"%.1f %.1f\"\n",
			(*item)->pnode->code, (*item)->pnode->size.width, (*item)->pnode->size.height);
	}
	return true;
}

int feed_quality_line(
	PVOID*	_item,
	PVOID	user_data
)
{
	QltyNode**	item = (QltyNode**)_item;

	char qlty_code[50];
	Str100 qlty_name;
	ASSERT(ppd_handle.model && ppd_handle.paper_list && ppd_handle.prtopt_list && ppd_handle.stream);
	ASSERT(item && *item && (*item)->opdat_handle)

	create_qlty_name(*item, qlty_code, qlty_name);
	fprintf(ppd_handle.stream, 
		"*Quality %s/ %s:                      \"<</HWResolution[360 360]>>setpagedevice\"\n", qlty_code, qlty_name);

	return true;
}

int find_default_qlty(
	PVOID*	_item,
	PVOID	default_qlty
)
{
	QltyNode**	item = (QltyNode**)_item;
	ASSERT(item && *item && (*item)->opdat_handle)
	
	*(PVOID*)default_qlty = NULL;
	if ((*item)->quality_id == QLT_NORMAL)
	{
		*(PVOID*)default_qlty = (PVOID)*item;
		return false; 
	}

	return true;
}

const char* ppd_translate_error(const STATUS status)
{
	return _(status_info_arr[status]);
}

STATUS ppd_init(
	const char *model,
	const char *paperlist_path, 
	const char *prtoptlist_path
)
{
	STATUS status = STS_SUCCESS;

	ASSERT(paperlist_path && prtoptlist_path);
	ppd_handle.model = (char*)model;

	ppd_handle.paper_list = list_init(NULL, NULL, 0, 
			paper_done, 
			paper_getnext, 
			paper_compare);

	ppd_handle.prtopt_list = list_init(NULL, NULL, 0, 
			prtopt_done, 
			prtopt_getnext, 
			prtopt_compare);

	ppd_handle.qlty_list = list_init(NULL, NULL, 0,
			qlty_done,
			qlty_getnext,
			qlty_compare);

	status = parse_paperlist_file(paperlist_path, ppd_handle.paper_list);

	if (status == STS_SUCCESS)
		status = parse_prtopt_file(prtoptlist_path, ppd_handle.model, ppd_handle.paper_list, 
				ppd_handle.prtopt_list, ppd_handle.qlty_list);
	return status;
}

STATUS ppd_generate(
	const char *ppd_dir)
{
	STATUS	status = STS_SUCCESS;
	char	short_name[100];
	char	pc_filename[MAX_PATH];
	char	ppd_path[MAX_PATH];
	char	qlty_code[50];
	Str100	qlty_name;
	QltyNode* def_qlty = NULL;

	abbreviate(ppd_handle.model, short_name);
	sprintf(pc_filename, "%s%s.ppd", "ek", (char*)strlwr(short_name));
	sprintf(ppd_path, "%s%s", ppd_dir, pc_filename);

	if ((ppd_handle.stream = fopen(ppd_path, "wt")) == NULL)
	{
		status = STS_PPD_FILE_CANNOT_CREATED;
	}
	else
	{
		PPD_FILE_LEADINGPART(ppd_handle.stream, (char*)strupr(pc_filename), ppd_handle.model, (char*)strupr(short_name))

		list_for_each(ppd_handle.qlty_list, feed_uiconstraints_line, NULL);		

		PAPERSIZE_SEC_LEADINGPART(ppd_handle.stream,	
			((PrtOptNode*)ppd_handle.prtopt_list->first_item)->pnode->code)
		list_for_each(ppd_handle.prtopt_list, feed_pagesize_line, NULL);
		PAPERSIZE_SEC_TRAILINGPART(ppd_handle.stream)

		PAPERREGION_SEC_LEADINGPART(ppd_handle.stream, 
			((PrtOptNode*)ppd_handle.prtopt_list->first_item)->pnode->code)
		list_for_each(ppd_handle.prtopt_list, feed_pageregion_line, NULL);
		PAPERREGION_SEC_TRAILINGPART(ppd_handle.stream)

		IMAGEABLEAREA_SEC_LEADINGPART(ppd_handle.stream, 
			((PrtOptNode*)ppd_handle.prtopt_list->first_item)->pnode->code)
		list_for_each(ppd_handle.prtopt_list, feed_imageablearea_line, NULL);
		IMAGEABLEAREA_SEC_TRAILINGPART(ppd_handle.stream)

		PAPERDIMENSION_SEC_LEADINGPART(ppd_handle.stream, 
			((PrtOptNode*)ppd_handle.prtopt_list->first_item)->pnode->code)
		list_for_each(ppd_handle.prtopt_list, feed_paperdimension_line, NULL);
		PAPERDIMENSION_SEC_TRAILINGPART(ppd_handle.stream)

		list_for_each(ppd_handle.qlty_list, find_default_qlty, &def_qlty);
		if (def_qlty)
			create_qlty_name(def_qlty, qlty_code, qlty_name);
		else
			create_qlty_name((QltyNode*)ppd_handle.qlty_list->first_item, qlty_code, qlty_name);

		QUALITY_SEC_LEADINGPART(ppd_handle.stream, qlty_code)
		list_for_each(ppd_handle.qlty_list, feed_quality_line, NULL);
		QUALITY_SEC_TRAILINGPART(ppd_handle.stream)

		PPD_FILE_TRAILINGPART(ppd_handle.stream)

		fclose(ppd_handle.stream);
	}

	return status;
}

STATUS ppd_done()
{
	list_done(&ppd_handle.paper_list);
	list_done(&ppd_handle.prtopt_list);
	list_done(&ppd_handle.qlty_list);
	return STS_SUCCESS;
}
