/*
 * Photo Image Print System
 * Copyright (C) 2000-2005 EPSON AVASYS Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2000-2005.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, EPSON AVASYS Corporation gives permission to
 * link the code of this program with libraries which are covered by
 * the EPSON AVASYS Public License and distribute their linked
 * combinations.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the libraries which
 * are covered by EPSON AVASYS Public License.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <ctype.h>
#include <limits.h>
#include <unistd.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>

#include "pips.h"
#include "pipsError.h"
#include "prtOpt.h"
#include "optBase.h"
#include "getstat.h"
#include "paper.h"
#include "paper.rsc"

#define LINE_BUFFER 1024
#define ARRAY_MAX 64
#define SUB_MAX 3

#define GLOBAL_OPTIONS_NOMBER 2
#define PIPS_UI_MODE 0
#define PIPS_GET_PRINTER_MODE 1

const char *gopts[] = {
	"ui", /* select UI */
	"id", /* get printer ID */
};

const char *opts[] = {
	"sc", "scale",           /* id = 0 */
	"in", "ink",             /* id = 1 */
	"ms", "mediasize",       /* id = 2 */
	"mt", "mediatype",       /* id = 3 */
	"br", "brightness",      /* id = 4 */
	"co", "contrast",        /* id = 5 */
	"sa", "saturation",      /* id = 6 */
	"ql", "qualitylevel",    /* id = 8 */
	"mg", "margin",	         /* id = 9 */
	"ft", "fit2page",        /* id = 10 */
	"mp", "multipage",       /* id = 11 -not use option */
	"bl", "borderlessprint", /* id = 12 */
	NULL
};

/* A revision of notation mistake. */
const char *opts_strings_mistake[] =
{
	"mediatipe", "mediatype",
	NULL
};

extern void *opt_address[EX_N_OPT];
extern int Lang;

static int globalInitOpts(char*, int, PIPS_INIT_PARAM*);
static int getRscError(SEP_INIT_PARAM*, int, int, FILE*);
static int scanRscFile(FILE*, const char*);
static char* existsp(char*);
static int checkConf(SEP_INIT_PARAM*);
static void defSet(SEP_INIT_PARAM *);
static int setConf(int id, SEP_INIT_PARAM *sepip, char sub[SUB_MAX][ARRAY_MAX]);
static int valueCheck(int id, char sub[SUB_MAX][ARRAY_MAX]);



/*--------------------------------------------------*/
/*          globalInitOpts                          */
/*--------------------------------------------------*/

static int
globalInitOpts (char *strv, int opt_id, PIPS_INIT_PARAM *pip)
{
	int value = 0;
	switch (opt_id)
	{
	case PIPS_UI_MODE:
		/* "Get Printer ID Mode" option is made to give priority. */
		if (pip->action_mode == ACTION_MODE_GET_PRINTER_ID)
			break;

		if (!strncmp (strv, "X", 1))
			value = ACTION_MODE_GUI;
		else if (!strncmp (strv, "C", 1))
			value = ACTION_MODE_CUI;
		else
			return 1;

		pip->action_mode = value;
		break;

	case PIPS_GET_PRINTER_MODE:
		if (!strncmp (strv, "ON", 2))
			value = ACTION_MODE_GET_PRINTER_ID;
		else if (strncmp (strv, "OFF", 1))
			return 1;
		
		pip->action_mode = value;
		break;

	default:
		return 1;
	}

	return 0;
}


/*##################################################*/
/*          xMode                                   */
/*##################################################*/
#if USE_GTK
int
xMode (int argc, char *argv[], char *filename,
       SEP_INIT_PARAM *sepip, PIPS_INIT_PARAM *pip)
{
	char name[256];
	if (printer_id[0] != 0)
		strcpy(name, printer_id);
	else
		strcpy(name, "LITE");
	confWinInit (&argc, &argv, filename, sepip, pip);	
	
	if (confWinShow (sepip, name))
	{
		confWinFinalize ();
		return pipsError ((char *)NULL, UNEXPECTED_ERROR);
	}
	confWinFinalize ();
	return 0;
}
#endif

/*##################################################*/
/*          cMode                                   */
/*##################################################*/

int
cMode (char *infile, SEP_INIT_PARAM *sepip, PIPS_INIT_PARAM *pip)
{
#ifdef CONV_VERSION2
	return graphics_converter_version2 (sepip, infile, NULL);

#else
	FILE *fp = stdout;

	return printLoop (infile, fp, sepip);
#endif
}

/*##################################################*/
/*          databasesCall                           */
/*##################################################*/
/* ͤǡ١ƤӽФ褦ѹ    Mon Aug 28 2000 sk */
OPTION_DATA*
databasesCall (int frag, int id, void *entry)
{
	int i;
	OPTION_DATA *data;

	data = NULL;
	for (i = 0; opt_all[i].id != END_ARRAY; i++)
		if (opt_all[i].id == id)
		{
			data = opt_all[i].data;
			break;
		}
	if (data == NULL) return NULL;
	if (frag == CALL_FOR_RSC_NAME)
	{
		for (i = 0; data[i].value != END_ARRAY; i++)
		{
			if (!strcmp ((char*)entry, data[i].rsc_name)) return (data + i);
		}
	}
	else if (frag == CALL_FOR_X_NAME)
	{
		for (i = 0; data[i].value != END_ARRAY; i++)
		{
			if(!strcmp ((char*)entry, data[i].x_name))
				return (data + i);
		}
	}
	/* add Mon Aug 28 2000 sk */
	else if (frag == CALL_FOR_VALUE)
	{
		for (i = 0; data[i].rsc_name != NULL; i++)
		{
			if (data[i].value == *((int*)entry))
				return (data + i);
		}
	}
  
	return NULL;
}


/*##################################################*/
/*                    getRsc                        */
/*##################################################*/

int
getRsc (SEP_INIT_PARAM *sepip, const char* printer)
{
	FILE *fp;
	char lin[LINE_BUFFER];
	int lin_No;
  
	defSet (sepip);
	if ((fp = fopen (RSC_PATH, "rb")) == NULL)
	{
		return -1;
	}

	lin_No = scanRscFile (fp, printer);
	switch (lin_No)
	{
	case -1:
		getRscError (sepip, lin_No, NO_ERROR, fp);
		return -1;
	case 0:
		fclose (fp);
		return -1;
	}

	lin_No++;
	for (; fgets (lin, LINE_BUFFER - 1, fp) != NULL; lin_No++)
	{
		int p_id, size, i;
		char p_name[ARRAY_MAX], p_sub[ARRAY_MAX + 1];
		char sub_arg[SUB_MAX][ARRAY_MAX];

		existsp (lin);
		switch (*lin)
		{
		case '\0':
			break;
		case '[':
			fclose (fp);
			return 0;

		default:
			if (sscanf (lin, "%[^=]s", p_name) != 1)
			{
				getRscError (sepip, lin_No, RESOURCE_FORM_ERROR, fp);
				return NO_ERROR;
			}

			size = strlen (p_name);
			/* A revision of notation mistake. */
			for (i = 0; opts_strings_mistake[i * 2]; i++)
			{
				if (!strncmp (p_name, opts_strings_mistake[i * 2], size))
				{
					strcpy (p_name, opts_strings_mistake[i * 2 + 1]);
				}
			}
			for (p_id = 0; p_id < N_OPT; p_id++)
			{
				if (!strncmp(p_name, opts[p_id * 2 + 1], size))
				{
					size++;
					if (sscanf (lin + size, "%s", p_sub) != 1)
					{
						getRscError (sepip, lin_No, RESOURCE_FORM_ERROR, fp);
						return NO_ERROR;
					}
					strcat (p_sub, ",");
					size = 0;
					for (i = 0; (sscanf (p_sub + size, "%[^,]s", sub_arg[i]) == 1)
						     && (i < SUB_MAX); i++) size += strlen (sub_arg[i]) + 1;
					if (setConf (p_id, sepip, sub_arg))
					{
						getRscError (sepip, lin_No, RESOURCE_VALUES_ERROR, fp);
						return NO_ERROR;
					}
					break;
				}
			}
			if (p_id == N_OPT)
			{
				getRscError (sepip, lin_No, RESOURCE_VALUES_ERROR, fp);
				return NO_ERROR;
			}
		}
	}
	if (fp != NULL) fclose (fp);
	return 0;
}

/*--------------------------------------------------*/
/*          getRscError                             */
/*--------------------------------------------------*/

static int
getRscError (SEP_INIT_PARAM *sepip, int lin_No, int error_code, FILE *fp)
{
	if (fp != NULL)
		fclose(fp);

	if (lin_No != -1)
	{
		char lin_No_c[128];

		memset (lin_No_c, '\0', 127);
		sprintf (lin_No_c, "%d", lin_No);
		pipsError (lin_No_c, RESOURCE_ERROR);
		pipsError (NULL, error_code);
	}
	fprintf (stderr, _("The default values have been set.\n"));
	defSet (sepip);
	return lin_No;
}

/*##################################################*/
/*          saveRsc                                 */
/*##################################################*/

void
saveRsc (void *save_options[],const char* printer)
{
	FILE *fp;
	char *before_buffer, *back_buffer, lin[LINE_BUFFER];
	int user_id, file_len, first_save, i;
	long insert_pos, protect_pos;
 
	/* No connected printer */
	if (!strcmp(printer,""))
	{
		return;
	}
	before_buffer = back_buffer = 0;
	first_save = 0;


	user_id = geteuid ();

	if ((fp = fopen (RSC_PATH, "a+b")) == NULL)
	{
		fprintf (stderr, _("Resource file not found or resource not set. Do save with root.\n"));
		return;
	}
	fclose (fp);
	
	if ((fp = fopen (RSC_PATH, "rb")) == NULL) return;

	switch (scanRscFile (fp, printer))
	{
	case -1:
		fclose (fp);
		return;

	case 0:
		first_save = 1;

	default:
		break;
	}

	insert_pos = ftell (fp);
	if (insert_pos != 0)
	{
		fseek (fp, 0, SEEK_SET);
		before_buffer = (char *)calloc (insert_pos + 1, sizeof (char));
		fread (before_buffer, sizeof (char), insert_pos, fp);
	}

	for (; fgets (lin, LINE_BUFFER, fp) != NULL;)
	{
		int line_size;

		line_size = strlen (lin);
		existsp (lin);
		if (*lin == '[')
		{
			fseek (fp, line_size * -1, SEEK_CUR);
			break;
		}
	}

	protect_pos = ftell (fp);
	fseek (fp, 0, SEEK_END);
	file_len = ftell (fp);
	if (file_len != 0)
	{
		back_buffer = (char *)calloc (file_len - insert_pos + 1, sizeof (char));
		fseek (fp, protect_pos, SEEK_SET);
		fread (back_buffer, sizeof (char), file_len, fp);
	}

	fclose (fp);
	fp = fopen (RSC_PATH, "wb");

	if (before_buffer != 0)
	{
		fwrite (before_buffer, sizeof (char), strlen (before_buffer), fp);
		free (before_buffer);
	}
	if (first_save)
	{
		fprintf (fp, "[ %s ]\n", printer/* PRINTER_MODEL */);
	}

	for (i = 0; i < N_OPT; i++)
	{
		fprintf (fp, "%s = ", opts[i * 2 + 1]);
		switch (i)
		{
		case P_SCALE:
		case P_BRIGHTNESS:
		case P_CONTRAST:
		case P_SATURATION:
			fprintf (fp, "%d\n", *((int*)save_options[i]));
			break;

		case P_INK:
		case P_MEDIA_SIZE:
		case P_MEDIA_TYPE:
		case P_QUALITY_LEVEL:
		case P_MULTI_PAGE:
			fprintf (fp, "%s\n", (char*)save_options[i]);
			break;

		case P_FIT_PAGE:
		case P_BORDERLESS_PRINT:
			if ((char*)save_options[i] == NULL)
				fprintf (fp, "%s\n", "OFF");
			else
				fprintf (fp, "%s\n", (char*)save_options[i]);
			break;

#ifdef CONV_VERSION2
		case P_MARGIN:
			fprintf (fp, "%d, ", *((int*)save_options[P_MARGIN_X]));
			fprintf (fp, "%d\n", *((int*)save_options[P_MARGIN_Y]));
			break;
#endif /* CONV_VERSION2 */
		default: break;
		}
	}
	fprintf (fp, "\n");
	if (back_buffer != 0)
	{
		fwrite (back_buffer, sizeof (char), strlen(back_buffer), fp);
		free (back_buffer);
	}
	fclose(fp);
	if(!user_id) chmod(RSC_PATH, 438);
}

/*--------------------------------------------------*/
/*          scanRscFile                             */
/*--------------------------------------------------*/

static int
scanRscFile (FILE *fp, const char* printer_id)
{
	int lin_No, flag = 0;
	char lin[LINE_BUFFER], ep_name[ARRAY_MAX];
	char printer[256];
	strcpy (printer,printer_id);

	for (lin_No = 1; fgets (lin, LINE_BUFFER - 1, fp) != NULL; lin_No++)
	{
		existsp (lin);
		switch (*lin)
		{
		case '\0': break;
		case '[':
			flag = 1;
			if (sscanf (lin, "[%[^]]s", ep_name) == EOF) return -1;
			if (!strncmp (existsp (printer), ep_name, strlen(ep_name))) return lin_No;
			break;
		default:
			if (flag == 0) return -1;
		}
	}
	return 0;
}


/*--------------------------------------------------*/
/*                    getOpts                       */
/*--------------------------------------------------*/

int
getOpts (int nargc, char **nargv, char **filename,
	 SEP_INIT_PARAM *sepip, PIPS_INIT_PARAM *pip)
{
	int arg_count, opt_id, i, ni, size;
	char error_data[1024];

	for (arg_count = 1; arg_count < nargc; arg_count++)
	{
		if (nargv[arg_count][0] == '-')
		{
			char sub_arg[SUB_MAX][ARRAY_MAX];

			strncpy (error_data, nargv[arg_count], 1023);
			for (opt_id = 0; opt_id < N_OPT; opt_id++)
			{
				size = strlen (opts[opt_id * 2]);
				if (!strncmp (nargv[arg_count] + 1, opts[opt_id * 2], size))
				{
					if (opt_id == P_MARGIN) ni = 2;
					else ni = 1;
		  
					for (i = 0; i < ni; i++)
					{
						/* error check */
						strcat (error_data, " ");
						strcat (error_data, nargv[arg_count + 1]);
						if ((++arg_count <= nargc) && ((nargv[arg_count][0] != '-') || (isdigit(nargv[arg_count][1]))))
						{
							strcpy (sub_arg[i], nargv[arg_count]);
						}
						else
						{
							return pipsError (error_data, OPTION_LESS_ERROR);
						}
					}
					if (setConf (opt_id, sepip, sub_arg))
					{
						return pipsError (error_data, OPTION_VALUES_ERROR);
					}
					goto OPT_SET_END;
				}
			}
	  
			for (opt_id = 0; opt_id < GLOBAL_OPTIONS_NOMBER; opt_id++)
			{
				size = strlen (gopts[opt_id]);
				if (!strncmp (nargv[arg_count] + 1, gopts[opt_id], size))
				{
					/* error check */
					if ((++arg_count <= nargc) && (nargv[arg_count][0] != '-'))
					{
						strcat (error_data, " ");
						strcat (error_data, nargv[arg_count]);
					}
					else
					{
						return pipsError (error_data, OPTION_LESS_ERROR);
					}
					if (globalInitOpts (nargv[arg_count], opt_id, pip))
					{
						return pipsError (error_data, OPTION_VALUES_ERROR);
					}
					goto OPT_SET_END;
				}
			}
			return pipsError (error_data, NO_OPTION_ERROR);
		OPT_SET_END:;
		}
      
		else if (*filename == 0)
		{
			*filename = (char *)malloc (strlen ( nargv[arg_count]) + 1 );
			strcpy (*filename, nargv[arg_count]);
		}
		else if (strcmp (*filename, nargv[arg_count]))
		{
			sprintf (error_data,"\"%s\" \"%s\"", *filename, nargv[arg_count]);
			return pipsError (error_data, DUB_FILENAME_ERROR);
		}
	}

	if (checkConf (sepip))
	{
		return pipsError (NULL , CONNECTION_ERROR);
	}
	return 0;
}


/*##################################################*/
/*          getOptsForX                             */
/*##################################################*/

int
getOptsForX (void *options[], SEP_INIT_PARAM *sepip)
{
	char sub_arg[SUB_MAX][ARRAY_MAX];
	int i, *val;

	for (i = 0; i < N_OPT; i++)
	{
		switch (i)
		{
		case P_SCALE:
		case P_BRIGHTNESS:
		case P_CONTRAST:
		case P_SATURATION:
			val = (int*)options[i];
			sprintf (sub_arg[0], "%d", *val);
			break;

		case P_INK:
		case P_MEDIA_SIZE:
		case P_MEDIA_TYPE:
		case P_QUALITY_LEVEL:
		case P_MULTI_PAGE:
		case P_FIT_PAGE:
		case P_BORDERLESS_PRINT:
			if(options[i] != NULL)
				strcpy (sub_arg[0], (char *)options[i]);
			else
				strcpy (sub_arg[0], "OFF");
			break;

#ifdef CONV_VERSION2
		case P_MARGIN:
			val = (int*)options[P_MARGIN_X];
			sprintf (sub_arg[0], "%d", *val);

			val = (int*)options[P_MARGIN_Y];
			sprintf (sub_arg[1], "%d", *val);
			break;
#endif /* CONV_VERSION2 */

		default: break;
		}
		if (setConf (i, sepip, sub_arg))
		{
			return pipsError ((char *)NULL, OPTION_VALUES_ERROR);
		}
	}
	if (checkConf (sepip))
	{ /* error check */
		return pipsError ((char *)NULL, CONNECTION_ERROR);
	}
	return 0;
}


/*--------------------------------------------------*/
/*          checkConf                               */
/*--------------------------------------------------*/

static int
checkConf (SEP_INIT_PARAM *sepip)
{
	int i, level_count;
	const short *mode;

	mode = pMode;
	level_count = sepip->qlevel;
	for (i = 0; mode[i * MODE_ITEM] >= 0; i++)
	{
		if (!printModeCheck (mode, i, sepip))
		{
			int check = 0; 
 			char **msizeEntry;
			msizeEntry = mediaSizeEntry;
			
			for (i = 0; msizeEntry[i] != NULL; i++)
			{
				if (!strcmp (getRscName (P_MEDIA_SIZE, sepip->paper_id), msizeEntry[i]))
				{
					check = 1;
					break;
				}
			}
			if (!check) return pipsError ((char *)NULL, OPTION_VALUES_ERROR);
			return 0;
		}
	}
	return 1;
}

/*--------------------------------------------------*/
/*          printModeCheck                          */
/*--------------------------------------------------*/

int
printModeCheck (const short *mode, int count, SEP_INIT_PARAM *sepip)
{
	int ofs;

	ofs = count * MODE_ITEM;

	if (sepip->paper_id < 0)			
		return 0;	
	else if (mode[ofs + MOFS_PAPER_SIZE] != sepip->paper_id) 		
		return 1;				
	
	if (sepip->borderless_print < 0 )
		return 0;
	else if ((sepip->borderless_print == 1) && (mode[ofs + MOFS_BORDERLESS_PRINT] == 0))		
		return 1;			

	if (sepip->media_type >= 0			
	    && sepip->media_type != mode[ofs + MOFS_MEDIA]) {		
		return 1;	
	}
	else 
		return 0;
	
	if (sepip->qlevel >= 0			
	    && sepip->qlevel != mode[ofs + MOFS_QUALITY]) {		
		return 1;	
	}		
	
	return 0;
}

/*--------------------------------------------------*/
/*                    existsp                       */
/*--------------------------------------------------*/

static char*
existsp(char *lin)
{
	char tmp_lin[LINE_BUFFER];
	int i, j;
  
	j = 0;
	tmp_lin[0] = '\0';

	for (i = 0; lin[i] != '\0'; i++)
		if (!isspace (lin[i])) tmp_lin[j++] = lin[i];
  
	if (j >= 1)
		if (tmp_lin[j - 1] != '\0')
		{
			tmp_lin[j] = '\0';
		}
	strcpy (lin, tmp_lin);
	return lin;
}


/*--------------------------------------------------*/
/*                    setConf                       */
/*--------------------------------------------------*/

static int
setConf (int id, SEP_INIT_PARAM *sepip, 
	 char sub[SUB_MAX][ARRAY_MAX])
{
  
	if (valueCheck(id, sub)) return 1;
  
	switch (id)
	{
	case P_SCALE:
		sepip->mag = atoi (sub[0]); break;
    
	case P_INK:
		sepip->prt_format = getValueOfDatabases (id, sub[0]); break;
    
	case P_MEDIA_SIZE:
		sepip->paper_id = getValueOfDatabases (id, sub[0]); break;
      
	case P_MEDIA_TYPE:
		sepip->media_type = getValueOfDatabases (id, sub[0]); break;
      
	case P_BRIGHTNESS:
		sepip->brightness = atoi (sub[0]); break;

	case P_CONTRAST:
		sepip->contrast = atoi (sub[0]); break;

	case P_SATURATION:
		sepip->saturation = atoi (sub[0]); break;

	case P_QUALITY_LEVEL:
		sepip->qlevel = getValueOfDatabases (id, sub[0]); break;

	case P_FIT_PAGE:
		sepip->fit_page = getValueOfDatabases (id, sub[0]); break;

	case P_MULTI_PAGE:
		sepip->multi_page = getValueOfDatabases (id, sub[0]); break;
		
#ifdef CONV_VERSION2
	case P_MARGIN:
		sepip->margin.x = atoi (sub[0]);
		sepip->margin.y = atoi (sub[1]);
		break;
#endif /* CONV_VERSION2 */

	case P_BORDERLESS_PRINT:
		sepip->borderless_print = getValueOfDatabases (id, sub[0]); break;

	default: return 1;
	}
	return 0;
}

/*##################################################*/
/*          getValueOfDatabases                     */
/*##################################################*/

int
getValueOfDatabases (int id, char *entry)
{
	OPTION_DATA *data;
	char off[] = "OFF";

	if (entry == NULL) entry = off;
	if ((data = databasesCall (CALL_FOR_RSC_NAME, id, (void*)entry)) == NULL) return -1;
	return data->value;
}

/*--------------------------------------------------*/
/*                    defSet                        */
/*--------------------------------------------------*/

static void
defSet (SEP_INIT_PARAM *sepip)
{	
	int ofs = 0;
	const short *mode;
	
	mode = pMode;
	/* Find the default mode */
	while (1)
	{
		if( mode[ofs*MODE_ITEM + MOFS_QUALITY] != PIPS_LV_DRAFT)
			break;
		ofs ++;
	}	
	sepip->mag = 100;
	sepip->prt_format = PIPS_OUTPUT_CMYKcm;
 	sepip->paper_id = mode [ofs*MODE_ITEM + MOFS_PAPER_SIZE];
	sepip->media_type = mode [ofs*MODE_ITEM + MOFS_MEDIA];
	sepip->brightness = 0;
	sepip->contrast = 0;
	sepip->saturation = 0;
	sepip->qlevel = mode [ofs*MODE_ITEM + MOFS_QUALITY];
	
#ifdef CONV_VERSION2
	sepip->margin.x = 3;
	sepip->margin.y = 3;
#endif /* CONV_VERSION2 */
	sepip->fit_page = PIPS_FP_OFF;
	sepip->multi_page = PIPS_MP_1;
	sepip->borderless_print = PIPS_BL_OFF;
	checkConf(sepip);
}
 
 
/*--------------------------------------------------*/
/*          valueCheck */
/*--------------------------------------------------*/

static int
valueCheck (int id, char sub[SUB_MAX][ARRAY_MAX])
{
	int i, j;
	int value;
	int count = 0;
	char **entry = 0;

	switch (id)
	{
#ifdef CONV_VERSION2
	case P_MARGIN:
#endif /* CONV_VERSION2 */
		count ++;
	case P_SCALE:
	case P_BRIGHTNESS:
	case P_CONTRAST:
	case P_SATURATION: count++;
		for (i = 0; i < count; i++)
		{
			for (j = 0; sub[i][j] != (char)NULL; j++)
			{
				if ((sub[i][j] < '0') || (sub[i][j] > '9'))
				{
					if ((sub[i][j] != '-') && (j != 0)) return 1;
				}
			}
			value = (int)atoi (sub[i]);
			if (id == P_SCALE)
			{
				if ((value < 10) || (value > 400)) return 1;
			}
#ifdef CONV_VERSION2
			else if (id == P_MARGIN)
			{
				if ((value < 0) || (value > 1000)) return 1;
			}
#endif /* CONV_VERSION2 */
			else
			{
				if ((value < -50) || (value > 50)) return 1;
			}
		}
		return 0;
	default: break;
	}

	switch (id)
	{
	case P_INK: entry = inkEntry; break;
	case P_MEDIA_TYPE: entry = mediaTypeEntry; break;
	case P_QUALITY_LEVEL: entry = levelEntry; break;
	case P_MULTI_PAGE: entry = multiPageEntry; break;
	case P_FIT_PAGE:
	case P_BORDERLESS_PRINT:
		if (!strcmp (sub[0], "ON") || !strcmp (sub[0], "OFF")) return 0;
		else return 1;

	default: return 0;
	}

	for (i = 0; entry[i] != (char)NULL; i++)
	{
		if (!strcmp (sub[0], entry[i])) return 0;
	}
	return 1;
}

/*--------------------------------------------------*/
/*          borderless_support */
/*--------------------------------------------------*/
int
borderless_support (char* paper)
{
	int count = 0;
	int paper_id = getValueOfDatabases (P_MEDIA_SIZE, paper);
	while (pMode[count*MODE_ITEM] >= 0)
	{
		if ((paper_id == pMode[count*MODE_ITEM + MOFS_PAPER_SIZE]) && pMode[count*MODE_ITEM + MOFS_BORDERLESS_PRINT])
			return 1;
		count ++;			
	}
	return 0;	
}


int
load_printer_info(char* id)
{
	if (!get_printer_id(id))
	{
		if (!strcmp(id,printer_id))
			return 1;
		
		if(!update_resource(id))
		{			
			free_resource();
			load_printer_option(id);
			printer_connected = 1;
		}	  
		else if (!resource_ready)
		{			
			free_resource();
			strcpy(id,"");
			printer_connected = 0;
			load_dummy_resource();

		}
		else
			/* resource is still OK */
		{
			load_default_printer_info(id);
			printer_connected = 0;
			
		}
	}
	else
	{
		if (!resource_ready)
		{
			free_resource();
			strcpy(id,"");	
			load_dummy_resource();
		}
		else
		{
			load_default_printer_info(id);
		}
		printer_connected = 0;
	}
	strcpy(printer_id, id);
	return printer_connected;
}

int
load_default_printer_info(char* id)
{
	if (!get_default_printer_id(id))
	{
		strcpy (printer_id, id);
		free_resource();
		load_printer_option(printer_id);
		return 0;
	}
	return 1;
}
