/*
    Copyright (C) 2010 Sebastian Pancratz
    Copyright (C) 2010 William Hart
    Copyright (C) 2011 Fredrik Johansson
    Copyright (C) 2012 Lina Kulakova

    This file is part of FLINT.

    FLINT is free software: you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.  See <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <gmp.h>
#include "flint.h"
#include "fmpz_vec.h"
#include "fmpz_mod_poly.h"
#include "ulong_extras.h"

void
_fmpz_mod_poly_powmod_ui_binexp(fmpz * res, const fmpz * poly,
                                ulong e, const fmpz * f, slong lenf, const fmpz_t p)
{
    fmpz * T, * Q;
    fmpz_t invf;
    slong lenT, lenQ;
    int i;

    if (lenf == 2)
    {
        fmpz_powm_ui(res, poly, e, p);
        return;
    }

    lenT = 2 * lenf - 3;
    lenQ = FLINT_MAX(lenT - lenf + 1, 1);

    T = _fmpz_vec_init(lenT + lenQ);
    Q = T + lenT;

    fmpz_init(invf);
    fmpz_invmod(invf, f + lenf - 1, p);

    _fmpz_vec_set(res, poly, lenf - 1);

    for (i = ((int) FLINT_BIT_COUNT(e) - 2); i >= 0; i--)
    {
        _fmpz_mod_poly_sqr(T, res, lenf - 1, p);
        _fmpz_mod_poly_divrem(Q, res, T, 2 * lenf - 3, f, lenf, invf, p);

        if (e & (UWORD(1) << i))
        {
            _fmpz_mod_poly_mul(T, res, lenf - 1, poly, lenf - 1, p);
            _fmpz_mod_poly_divrem(Q, res, T, 2 * lenf - 3, f, lenf, invf, p);
        }
    }

    fmpz_clear(invf);
    _fmpz_vec_clear(T, lenT + lenQ);
}


void
fmpz_mod_poly_powmod_ui_binexp(fmpz_mod_poly_t res,
                         const fmpz_mod_poly_t poly, ulong e,
                         const fmpz_mod_poly_t f)
{
    fmpz * q;
    slong len = poly->length;
    slong lenf = f->length;
    slong trunc = lenf - 1;
    int qcopy = 0;

    if (lenf == 0)
    {
        flint_printf("Exception (fmpz_mod_poly_powmod). Divide by zero\n");
        flint_abort();
    }

    if (lenf == 1)
    {
        fmpz_mod_poly_zero(res);
        return;
    }

    if (len >= lenf)
    {
        fmpz_mod_poly_t t, r;
        fmpz_mod_poly_init(t, &res->p);
        fmpz_mod_poly_init(r, &res->p);
        fmpz_mod_poly_divrem(t, r, poly, f);
        fmpz_mod_poly_powmod_ui_binexp(res, r, e, f);
        fmpz_mod_poly_clear(t);
        fmpz_mod_poly_clear(r);
        return;
    }

    if (e <= 2)
    {
        if (e == UWORD(0))
        {
            fmpz_mod_poly_fit_length(res, 1);
            fmpz_one(res->coeffs);
            _fmpz_mod_poly_set_length(res, 1);
        }
        else if (e == UWORD(1))
        {
            fmpz_mod_poly_set(res, poly);
        }
        else
            fmpz_mod_poly_mulmod(res, poly, poly, f);
        return;
    }

    if (len == 0)
    {
        fmpz_mod_poly_zero(res);
        return;
    }

    if (len < trunc)
    {
        q = _fmpz_vec_init(trunc);
        _fmpz_vec_set(q, poly->coeffs, len);
        _fmpz_vec_zero(q + len, trunc - len);
        qcopy = 1;
    } else
        q = poly->coeffs;

    if ((res == poly && !qcopy) || (res == f))
    {
        fmpz_mod_poly_t t;
        fmpz_mod_poly_init2(t, &poly->p, 2 * lenf - 3);
        _fmpz_mod_poly_powmod_ui_binexp(t->coeffs,
            q, e, f->coeffs, lenf, &poly->p);
        fmpz_mod_poly_swap(res, t);
        fmpz_mod_poly_clear(t);
    }
    else
    {
        fmpz_mod_poly_fit_length(res, 2 * lenf - 3);
        _fmpz_mod_poly_powmod_ui_binexp(res->coeffs,
            q, e, f->coeffs, lenf, &poly->p);
    }

    if (qcopy)
        _fmpz_vec_clear(q, trunc);

    _fmpz_mod_poly_set_length(res, trunc);
    _fmpz_mod_poly_normalise(res);
}
