/*
 *  $Id: good_profile.c 22996 2020-12-03 14:06:04Z yeti-dn $
 *  Copyright (C) 2020 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/level.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libprocess/correct.h>
#include <libprocess/arithmetic.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "preview.h"

#define GOOD_PROF_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

typedef enum {
    GOOD_PROF_SINGLE   = 0,
    GOOD_PROF_MULTIPLE = 1,
    GOOD_PROF_NMODES
} GoodProfMode;

typedef enum {
    GOOD_PROF_IMAGE = 0,
    GOOD_PROF_GRAPH = 1,
    GOOD_PROF_NDISPLAYS
} GoodProfDisplay;

typedef struct {
    GoodProfMode mode;
    GoodProfDisplay display;
    GwyAppDataId other_image;
    GwyAppDataId target_graph;
    gdouble trim_fraction;
    gboolean create_mask;
    gboolean update;
} GoodProfArgs;

typedef struct {
    GoodProfArgs *args;
    GwyContainer *mydata;
    GtkWidget *dialogue;
    GtkWidget *view;
    GtkWidget *graph;
    GSList *mode;
    GSList *display;
    GtkWidget *other_image;
    GtkObject *trim_fraction;
    GtkWidget *target_graph;
    GtkWidget *create_mask;
    GtkWidget *color_button;
    GtkWidget *update;
    GtkWidget *variation;
    GwySIValueFormat *varvf;

    GwyDataField *dfield;
    GwyDataField *dfield2;
    GwyDataField *mfield;
    gboolean in_init;
} GoodProfControls;

static gboolean       module_register      (void);
static void           good_profile         (GwyContainer *data,
                                            GwyRunType run);
static gboolean       dialogue             (GoodProfArgs *args,
                                            GwyContainer *data,
                                            GwyDataField *dfield,
                                            gint id);
static void           update_controls      (GoodProfControls *controls,
                                            GoodProfArgs *args);
static void           display_changed      (GtkToggleButton *button,
                                            GoodProfControls *controls);
static void           mode_changed         (GtkToggleButton *button,
                                            GoodProfControls *controls);
static void           create_mask_changed  (GoodProfControls *controls,
                                            GtkToggleButton *check);
static void           trim_fraction_changed(GoodProfControls *controls,
                                            GtkAdjustment *adj);
static void           other_image_changed  (GwyDataChooser *chooser,
                                            GoodProfControls *controls);
static gboolean       other_image_filter   (GwyContainer *data,
                                            gint id,
                                            gpointer user_data);
static void           update_changed       (GoodProfControls *controls,
                                            GtkToggleButton *check);
static void           update_target_graphs (GoodProfControls *controls);
static void           target_graph_changed (GoodProfControls *controls);
static void           update_sensitivity   (GoodProfControls *controls);
static void           invalidate           (GoodProfControls *controls);
static void           preview              (GoodProfControls *controls);
static GwyGraphModel* good_profile_do      (GwyDataField *dfield,
                                            GwyDataField *dfield2,
                                            GwyDataField *mfield,
                                            const GoodProfArgs *args,
                                            gdouble *variation);
static void           load_args            (GwyContainer *container,
                                            GoodProfArgs *args);
static void           save_args            (GwyContainer *container,
                                            GoodProfArgs *args);
static void           sanitize_args        (GoodProfArgs *args);

static const GoodProfArgs defaults = {
    GOOD_PROF_SINGLE, GOOD_PROF_GRAPH,
    GWY_APP_DATA_ID_NONE, GWY_APP_DATA_ID_NONE,
    0.05,
    TRUE, TRUE,
};

static GwyAppDataId target_id = GWY_APP_DATA_ID_NONE;
static GwyAppDataId other_id = GWY_APP_DATA_ID_NONE;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Calculates good average row from one or multiple images "
       "of repeated scanning of the same feature."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2020",
};

GWY_MODULE_QUERY2(module_info, good_profile)

static gboolean
module_register(void)
{
    gwy_process_func_register("good_profile",
                              (GwyProcessFunc)&good_profile,
                              N_("/_Correct Data/_Good Mean Profile..."),
                              NULL,
                              GOOD_PROF_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Calculate good mean profile"));

    return TRUE;
}

static void
good_profile(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *dfield2 = NULL, *mfield = NULL;
    GwyContainer *data2;
    GwyGraphModel *gmodel;
    GoodProfArgs args;
    GQuark quark2, mquark, quarks[2];
    gint id;
    gboolean ok;

    g_return_if_fail(run & GOOD_PROF_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_MASK_FIELD_KEY, &mquark,
                                     0);
    g_return_if_fail(dfield);
    load_args(gwy_app_settings_get(), &args);

    data2 = gwy_app_data_browser_get(args.other_image.datano);
    if (!other_image_filter(data2, args.other_image.id, dfield)) {
        args.other_image = defaults.other_image;
        args.mode = GOOD_PROF_SINGLE;
    }

    if (run == GWY_RUN_INTERACTIVE) {
        ok = dialogue(&args, data, dfield, id);
        save_args(gwy_app_settings_get(), &args);
        if (!ok)
            return;
    }

    if (args.create_mask) {
        mfield = gwy_data_field_new_alike(dfield, FALSE);
        gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(mfield), NULL);
    }
    if (args.mode == GOOD_PROF_MULTIPLE) {
        quark2 = gwy_app_get_data_key_for_id(args.other_image.id);
        data2 = gwy_app_data_browser_get(args.other_image.datano);
        dfield2 = gwy_container_get_object(data2, quark2);
    }

    gmodel = good_profile_do(dfield, dfield2, mfield, &args, NULL);
    gwy_app_add_graph_or_curves(gmodel, data, &args.target_graph, 1);
    g_object_unref(gmodel);

    if (args.create_mask) {
        /* This is convoluted because dfield1 and dfield2 may come from
         * different files – and we have to create two separate undo
         * checkpoints then, one in each file. */
        if (args.mode == GOOD_PROF_MULTIPLE) {
            quarks[0] = mquark;
            quarks[1] = gwy_app_get_mask_key_for_id(args.other_image.id);
            if (data2 == data)
                gwy_app_undo_qcheckpointv(data, 2, quarks);
            else {
                gwy_app_undo_qcheckpointv(data, 1, quarks);
                gwy_app_undo_qcheckpointv(data2, 1, quarks+1);
            }
            dfield2 = gwy_data_field_duplicate(mfield);
            gwy_container_set_object(data, quarks[0], mfield);
            gwy_container_set_object(data2, quarks[1], dfield2);
            g_object_unref(dfield2);
        }
        else {
            gwy_app_undo_qcheckpointv(data, 1, &mquark);
            gwy_container_set_object(data, mquark, mfield);
        }
        g_object_unref(mfield);
    }
}

static gboolean
dialogue(GoodProfArgs *args, GwyContainer *data, GwyDataField *dfield, gint id)
{
    static const GwyEnum modes[] = {
        { N_("_Single image"),    GOOD_PROF_SINGLE,   },
        { N_("_Multiple images"), GOOD_PROF_MULTIPLE, },
    };
    static const GwyEnum displays[] = {
        { N_("Data"),    GOOD_PROF_IMAGE, },
        { N_("Profile"), GOOD_PROF_GRAPH, },
    };
    GtkWidget *dialogue, *table, *label, *hbox, *vbox;
    GwyGraphModel *gmodel;
    GoodProfControls controls;
    GwyContainer *data2;
    GQuark quark2;
    gint response;
    gint row;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.dfield = dfield;
    controls.mfield = gwy_data_field_new_alike(dfield, TRUE);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(controls.mfield),
                                NULL);
    controls.varvf
        = gwy_data_field_get_value_format_z(dfield,
                                            GWY_SI_UNIT_FORMAT_VFMARKUP, NULL);
    controls.varvf->precision++;
    controls.in_init = TRUE;

    dialogue = gtk_dialog_new_with_buttons(_("Good Mean Profile"), NULL, 0,
                                           NULL);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialogue),
                                 gwy_stock_like_button_new(_("_Update"),
                                                           GTK_STOCK_EXECUTE),
                                 RESPONSE_PREVIEW);
    gtk_dialog_add_button(GTK_DIALOG(dialogue), _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    controls.dialogue = dialogue;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       TRUE, TRUE, 4);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 4);

    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    gwy_container_set_object_by_name(controls.mydata, "/0/mask",
                                     controls.mfield);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, TRUE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.view, FALSE, FALSE, 0);
    if (args->display != GOOD_PROF_IMAGE)
        gtk_widget_set_no_show_all(controls.view, TRUE);

    gmodel = gwy_graph_model_new();
    controls.graph = gwy_graph_new(gmodel);
    gwy_graph_model_set_units_from_data_field(gmodel, dfield, 1, 0, 0, 1);
    gtk_widget_set_size_request(controls.graph, PREVIEW_SIZE, -1);
    gwy_graph_enable_user_input(GWY_GRAPH(controls.graph), FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.graph, TRUE, TRUE, 0);
    if (args->display != GOOD_PROF_GRAPH)
        gtk_widget_set_no_show_all(controls.graph, TRUE);

    table = gtk_table_new(15, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 4);
    row = 0;

    label = gtk_label_new(_("Display:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.display
        = gwy_radio_buttons_create(displays, G_N_ELEMENTS(displays),
                                   G_CALLBACK(display_changed), &controls,
                                   args->display);
    row = gwy_radio_buttons_attach_to_table(controls.display,
                                            GTK_TABLE(table), 2, row);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gtk_label_new(_("Mode:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.mode
        = gwy_radio_buttons_create(modes, G_N_ELEMENTS(modes),
                                   G_CALLBACK(mode_changed), &controls,
                                   args->mode);
    row = gwy_radio_buttons_attach_to_table(controls.mode, GTK_TABLE(table),
                                            2, row);

    controls.other_image = gwy_data_chooser_new_channels();
    gwy_data_chooser_set_filter(GWY_DATA_CHOOSER(controls.other_image),
                                other_image_filter, dfield, NULL);
    gwy_data_chooser_set_active_id(GWY_DATA_CHOOSER(controls.other_image),
                                   &args->other_image);
    if (gwy_data_chooser_get_active_id(GWY_DATA_CHOOSER(controls.other_image),
                                       &args->other_image)) {
        g_signal_connect(controls.other_image, "changed",
                         G_CALLBACK(other_image_changed), &controls);

        data2 = gwy_app_data_browser_get(args->other_image.datano);
        quark2 = gwy_app_get_data_key_for_id(args->other_image.id);
        controls.dfield2 = gwy_container_get_object(data2, quark2);
    }
    else {
        /* The mode should be already single, just make it impossible to select
         * the multi-image mode. */
        gtk_widget_set_sensitive(GTK_WIDGET(controls.mode->next->data), FALSE);
    }
    gwy_table_attach_adjbar(table, row++, _("Second _image:"), NULL,
                            GTK_OBJECT(controls.other_image),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.trim_fraction = gtk_adjustment_new(args->trim_fraction,
                                               0.0, 0.9999, 0.0001, 0.1, 0);
    gwy_table_attach_adjbar(table, row++, _("_Trim fraction:"), NULL,
                            controls.trim_fraction, GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls.trim_fraction, "value-changed",
                             G_CALLBACK(trim_fraction_changed), &controls);

    controls.variation = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(controls.variation), 1.0, 0.5);
    gwy_table_attach_adjbar(table, row++, _("Variation:"), NULL,
                            GTK_OBJECT(controls.variation),
                            GWY_HSCALE_WIDGET);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gwy_label_new_header(_("Output"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.target_graph = create_target_graph(&args->target_graph,
                                                GTK_WIDGET(table), row++,
                                                gmodel);
    g_object_unref(gmodel);
    g_signal_connect_swapped(controls.target_graph, "changed",
                             G_CALLBACK(target_graph_changed), &controls);
    controls.create_mask
        = gtk_check_button_new_with_mnemonic(_("Create _mask"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.create_mask),
                                 args->create_mask);
    gtk_table_attach(GTK_TABLE(table), controls.create_mask,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.create_mask, "toggled",
                             G_CALLBACK(create_mask_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gwy_label_new_header(_("Options"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.color_button = create_mask_color_button(controls.mydata, dialogue,
                                                     0);
    gwy_table_attach_adjbar(table, row++, _("_Mask color:"), NULL,
                            GTK_OBJECT(controls.color_button),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    controls.update = gtk_check_button_new_with_mnemonic(_("I_nstant updates"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.update),
                                 args->update);
    gtk_table_attach(GTK_TABLE(table), controls.update,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.update, "toggled",
                             G_CALLBACK(update_changed), &controls);
    row++;

    update_sensitivity(&controls);
    preview(&controls);
    controls.in_init = FALSE;

    gtk_widget_show_all(dialogue);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialogue));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialogue);
            g_object_unref(controls.mfield);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            controls.in_init = TRUE;
            *args = defaults;
            update_controls(&controls, args);
            controls.in_init = FALSE;
            invalidate(&controls);
            break;

            case RESPONSE_PREVIEW:
            preview(&controls);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialogue);
    g_object_unref(controls.mfield);

    return TRUE;
}

static void
update_controls(GoodProfControls *controls, GoodProfArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->trim_fraction),
                             args->trim_fraction);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->create_mask),
                                 args->create_mask);
    gwy_radio_buttons_set_current(controls->mode, args->mode);
    gwy_radio_buttons_set_current(controls->display, args->display);
}

static void
create_mask_changed(GoodProfControls *controls, GtkToggleButton *check)
{
    controls->args->create_mask = gtk_toggle_button_get_active(check);
    invalidate(controls);
}

static void
trim_fraction_changed(GoodProfControls *controls, GtkAdjustment *adj)
{
    controls->args->trim_fraction = gtk_adjustment_get_value(adj);
    invalidate(controls);
}

static void
display_changed(GtkToggleButton *button, GoodProfControls *controls)
{
    GoodProfDisplay display;

    display = gwy_radio_buttons_get_current(controls->display);
    controls->args->display = display;
    if (display == GOOD_PROF_IMAGE) {
        gtk_widget_set_no_show_all(controls->graph, TRUE);
        gtk_widget_set_no_show_all(controls->view, FALSE);
        gtk_widget_hide(controls->graph);
        gtk_widget_show_all(controls->view);
        update_target_graphs(controls);
    }
    else {
        gtk_widget_set_no_show_all(controls->view, TRUE);
        gtk_widget_set_no_show_all(controls->graph, FALSE);
        gtk_widget_hide(controls->view);
        gtk_widget_show_all(controls->graph);
    }

    if (!gtk_toggle_button_get_active(button))
        return;

    if (!controls->in_init)
        preview(controls);
}

static void
mode_changed(GtkToggleButton *button, GoodProfControls *controls)
{
    GoodProfMode mode;

    mode = gwy_radio_buttons_get_current(controls->mode);
    controls->args->mode = mode;
    /* TODO TODO TODO */
    if (!gtk_toggle_button_get_active(button))
        return;

    update_sensitivity(controls);
    if (!controls->in_init)
        preview(controls);
}

static void
update_changed(GoodProfControls *controls, GtkToggleButton *check)
{
    controls->args->update = gtk_toggle_button_get_active(check);
    update_sensitivity(controls);
    invalidate(controls);
}

static void
other_image_changed(GwyDataChooser *chooser, GoodProfControls *controls)
{
    GoodProfArgs *args = controls->args;
    GwyContainer *data2;
    GQuark quark2;

    data2 = gwy_data_chooser_get_active(chooser, &args->other_image.id);
    args->other_image.datano = gwy_app_data_browser_get_number(data2);
    quark2 = gwy_app_get_data_key_for_id(args->other_image.id);
    controls->dfield2 = gwy_container_get_object(data2, quark2);

    invalidate(controls);
}

static gboolean
other_image_filter(GwyContainer *data,
                   gint id,
                   gpointer user_data)
{
    GwyDataField *dfield = (GwyDataField*)user_data;
    GwyDataField *other_image;
    GQuark quark;

    if (!data || id < 0)
        return FALSE;

    quark = gwy_app_get_data_key_for_id(id);
    if (!gwy_container_gis_object(data, quark, (GObject**)&other_image))
        return FALSE;

    return !gwy_data_field_check_compatibility(other_image, dfield,
                                               GWY_DATA_COMPATIBILITY_ALL);
}

static void
update_target_graphs(GoodProfControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);
    gwy_data_chooser_refilter(chooser);
}

static void
target_graph_changed(GoodProfControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);
    gwy_data_chooser_get_active_id(chooser, &controls->args->target_graph);
}

static void
update_sensitivity(GoodProfControls *controls)
{
    const GoodProfArgs *args = controls->args;
    gboolean is_multiple = (args->mode == GOOD_PROF_MULTIPLE);

    gtk_widget_set_sensitive(controls->other_image, is_multiple);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialogue),
                                      RESPONSE_PREVIEW, !args->update);
}

static void
invalidate(GoodProfControls *controls)
{
    if (controls->in_init || !controls->args->update)
        return;

    preview(controls);
}

static void
preview(GoodProfControls *controls)
{
    GoodProfArgs *args = controls->args;
    GwySIValueFormat *varvf = controls->varvf;
    GwyGraphModel *gmodel;
    gdouble variation;
    gchar *s;

    gmodel = good_profile_do(controls->dfield, controls->dfield2,
                             controls->mfield, args, &variation);
    gwy_data_field_data_changed(controls->mfield);
    set_graph_model_keeping_label_pos(GWY_GRAPH(controls->graph), gmodel);

    s = g_strdup_printf("%.*f%s%s",
                        varvf->precision, variation/varvf->magnitude,
                        *varvf->units ? " " : "", varvf->units);
    gtk_label_set_markup(GTK_LABEL(controls->variation), s);
    g_free(s);

    update_target_graphs(controls);
}

static void
good_profile_do_single(GwyDataField *dfield, GwyDataField *mfield,
                       GwyDataLine *profile,
                       const GoodProfArgs *args)
{
    GwyDataField *ffield, *fmask = NULL;
    const gdouble *ddata, *drow;
    gdouble *data, *mdata = NULL, *row, *mrow, *ldata,
            *low = NULL, *high = NULL;
    gint xres, yres, ntrim, i, j, k[2];
    gdouble v[2];

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    ldata = gwy_data_line_get_data(profile);

    /* Work with a flipped field because we process it by column.  Note
     * that this means xres and yres have swapped meaning! */
    ffield = gwy_data_field_new_alike(dfield, FALSE);
    gwy_data_field_flip_xy(dfield, ffield, FALSE);
    data = gwy_data_field_get_data(ffield);
    if (mfield) {
        fmask = gwy_data_field_new_alike(mfield, FALSE);
        gwy_data_field_flip_xy(mfield, fmask, FALSE);
        mdata = gwy_data_field_get_data(fmask);
        low = g_new(gdouble, xres);
        high = g_new(gdouble, xres);
    }

    ntrim = GWY_ROUND(0.5*args->trim_fraction*yres);
    /* Ensure we are at least using a single value. */
    if (2*ntrim + 1 > yres)
        ntrim = (yres-1)/2;

    k[0] = ntrim;
    k[1] = yres-1-ntrim;
    for (i = 0; i < xres; i++) {
        row = data + i*yres;
        ldata[i] = gwy_math_trimmed_mean(yres, row, ntrim, ntrim);
        if (mdata) {
            mrow = mdata + i*yres;
            gwy_math_kth_ranks(yres, row, 2, k, v);
            low[i] = v[0];
            high[i] = v[1];
        }
    }

    g_object_unref(ffield);
    if (fmask) {
        g_object_unref(fmask);
        ddata = gwy_data_field_get_data_const(dfield);
        mdata = gwy_data_field_get_data(mfield);
        for (i = 0; i < yres; i++) {
            drow = ddata + i*xres;
            mrow = mdata + i*xres;
            for (j = 0; j < xres; j++)
                mrow[j] = (drow[j] < low[j] || drow[j] > high[j]);
        }
        g_free(low);
        g_free(high);
    }
}

static void
good_profile_do_multiple(GwyDataField *dfield, GwyDataField *dfield2,
                         GwyDataField *mfield,
                         GwyDataLine *profile, GwyDataLine *weights,
                         const GoodProfArgs *args)
{
    GwyDataField *buf;
    const gdouble *d1, *d2;
    gdouble *b, *m;
    gdouble p, threshold;
    gint xres, yres, n, i;

    d1 = gwy_data_field_get_data_const(dfield);
    d2 = gwy_data_field_get_data_const(dfield2);
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    n = xres*yres;

    buf = gwy_data_field_new_alike(dfield, FALSE);
    b = gwy_data_field_get_data(buf);

    if (!mfield) {
        mfield = gwy_data_field_new_alike(dfield, FALSE);
        gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(mfield), NULL);
    }
    else
        g_object_ref(mfield);
    m = gwy_data_field_get_data(mfield);

    /* Use m[] temporarily for the difference we pass to gwy_math_percentiles()
     * and they get shuffled. */
    for (i = 0; i < n; i++)
        b[i] = m[i] = fabs(d1[i] - d2[i]);

    p = 100.0*(1.0 - args->trim_fraction);
    gwy_math_percentiles(n, m, GWY_PERCENTILE_INTERPOLATION_MIDPOINT,
                         1, &p, &threshold);

    for (i = 0; i < n; i++) {
        m[i] = (b[i] > threshold);
        b[i] = 0.5*(d1[i] + d2[i]);
    }

    gwy_data_field_get_line_stats_mask(buf, mfield, GWY_MASK_EXCLUDE,
                                       profile, weights,
                                       0, 0, xres, yres,
                                       GWY_LINE_STAT_MEAN,
                                       GWY_ORIENTATION_VERTICAL);

    g_object_unref(mfield);
    g_object_unref(buf);
}

static GwyGraphModel*
good_profile_do(GwyDataField *dfield, GwyDataField *dfield2,
                GwyDataField *mfield,
                const GoodProfArgs *args,
                gdouble *variation)
{
    GwyGraphModel *gmodel;
    GwyGraphCurveModel *gcmodel;
    GwyDataLine *profile, *weights = NULL;
    gdouble dx, xoff;
    gint xres, i, n;
    gdouble *d, *w;
    GwyXY *xy;

    xres = gwy_data_field_get_xres(dfield);
    profile = gwy_data_line_new(xres, gwy_data_field_get_xreal(dfield), TRUE);
    gwy_data_field_copy_units_to_data_line(dfield, profile);

    if (args->mode == GOOD_PROF_SINGLE) {
        good_profile_do_single(dfield, mfield, profile, args);
    }
    else if (args->mode == GOOD_PROF_MULTIPLE) {
        g_assert(dfield2);
        weights = gwy_data_line_new_alike(profile, FALSE);
        good_profile_do_multiple(dfield, dfield2, mfield, profile, weights,
                                 args);
    }
    else {
        g_assert_not_reached();
    }

    gmodel = gwy_graph_model_new();
    gwy_graph_model_set_units_from_data_line(gmodel, profile);
    g_object_set(gmodel,
                 "title", _("Mean profile"),
                 NULL);

    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel,
                 "mode", GWY_GRAPH_CURVE_LINE,
                 "description", _("Mean profile"),
                 NULL);
    if (args->mode == GOOD_PROF_SINGLE)
        gwy_graph_curve_model_set_data_from_dataline(gcmodel, profile, 0, 0);
    else {
        xres = gwy_data_line_get_res(profile);
        dx = gwy_data_line_get_dx(profile);
        xoff = gwy_data_line_get_offset(profile);
        d = gwy_data_line_get_data(profile);
        w = gwy_data_line_get_data(weights);
        xy = g_new(GwyXY, xres);
        for (i = n = 0; i < xres; i++) {
            if (w[i] > 0.0) {
                xy[n].x = dx*i + xoff;
                xy[n].y = d[i];
                n++;
            }
        }
        gwy_graph_curve_model_set_data_interleaved(gcmodel, (gdouble*)xy, n);
        g_free(xy);

        if (variation) {
            for (i = 0; i < xres; i++)
                w[i] = (w[i] <= 0.0);
            gwy_data_line_correct_laplace(profile, weights);
        }
    }
    gwy_graph_model_add_curve(gmodel, gcmodel);
    g_object_unref(gcmodel);

    if (variation)
        *variation = gwy_data_line_get_variation(profile);

    g_object_unref(profile);
    GWY_OBJECT_UNREF(weights);

    return gmodel;
}

static const gchar create_mask_key[]   = "/module/good_profile/create_mask";
static const gchar display_key[]       = "/module/good_profile/display";
static const gchar mode_key[]          = "/module/good_profile/mode";
static const gchar update_key[]        = "/module/good_profile/update";
static const gchar trim_fraction_key[] = "/module/good_profile/trim_fraction";

static void
sanitize_args(GoodProfArgs *args)
{
    args->display = MIN(args->display, GOOD_PROF_NDISPLAYS-1);
    args->mode = MIN(args->mode, GOOD_PROF_NMODES-1);
    args->trim_fraction = CLAMP(args->trim_fraction, 0.0, 0.9999);
    args->create_mask = !!args->create_mask;
    args->update = !!args->update;
    gwy_app_data_id_verify_graph(&args->target_graph);
    gwy_app_data_id_verify_channel(&args->other_image);
}

static void
load_args(GwyContainer *container,
          GoodProfArgs *args)
{
    *args = defaults;

    gwy_container_gis_enum_by_name(container, mode_key, &args->mode);
    gwy_container_gis_enum_by_name(container, display_key, &args->display);
    gwy_container_gis_boolean_by_name(container, create_mask_key,
                                      &args->create_mask);
    gwy_container_gis_boolean_by_name(container, update_key, &args->update);
    gwy_container_gis_double_by_name(container, trim_fraction_key,
                                     &args->trim_fraction);
    args->target_graph = target_id;
    args->other_image = other_id;
    sanitize_args(args);
}

static void
save_args(GwyContainer *container,
          GoodProfArgs *args)
{
    target_id = args->target_graph;
    other_id = args->other_image;
    gwy_container_set_enum_by_name(container, mode_key, args->mode);
    gwy_container_set_enum_by_name(container, display_key, args->display);
    gwy_container_set_boolean_by_name(container, create_mask_key,
                                      args->create_mask);
    gwy_container_set_boolean_by_name(container, update_key, args->update);
    gwy_container_set_double_by_name(container, trim_fraction_key,
                                     args->trim_fraction);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
