// Reconcile DEP-5 debian/copyright to licensecheck
//
// Copyright : 2023-2025 P Blackman
// License   : BSD-2-clause
//
// Test for false positive on GPL or-later licences, or with trailing .0
// Test for gpl-3.0 URL

unit gpl;
{$mode delphi}

interface

function CheckGPL (Fname : AnsiString; Dep5, Actual: String) : Boolean;
Function AdjustGPL (LicenseStr : String) : String;


implementation uses StrUtils, support;

// Extract license short string, expecting a GPL license
function GetFileLicense (Fname : AnsiString) : String;
const MaxLines : Integer = 100;
var lines,
    LposGPL,
    LposGPL3 : Integer;
    L1,L2, line : AnsiString;
    lfile : Text;

begin
    result := '';
    lines := 0;
    if OpenFile (FName, lfile) then
    begin
        while not EOF (lfile) and (lines < MaxLines) do
        begin
            readln (lfile, line);
            LposGPL := NPos ('GPL', line, 1);
            LposGPL3 := NPos ('www.gnu.org/licenses/gpl-3.0', line, 1);

            if LposGPL > 0 then
            begin
                L1 := ExtractSubstr(line, LposGPL, WhiteSpace);
                Result := L1;

                if L1 = 'GPL' then // LposGPL updated
                begin
                    L2 := ExtractSubstr(line, LposGPL, WhiteSpace + ['"']);

                    if (length(L2) = 3) then
                    begin
                        if (L2[3] = '+') then
                            result := L1 + L2
                        else
                        if (L2 = '(>=') then
                            result := L1 + '+';
                    end
                end;

                lines := MaxLines // terminate loop
            end
            else
            if LposGPL3 > 0 then
            begin
                result := 'GPL-3';
                lines := MaxLines // terminate loop
            end
            else
                lines := lines + 1;
        end;
        Close (lfile);
    end;
end;

// Return true if Actual is 2+ or 3+ as shown in d/copyright
// Return true for GPL3 if the URL is found in the file header
function CheckGPL (Fname : AnsiString; Dep5, Actual: String) : Boolean;
var License : String;

begin
    result := false;

    if (Dep5 = 'GPL-3+') and (Actual = 'GPL-3')
    or (Dep5 = 'GPL-2+') and (Actual = 'GPL-2')
    or (Dep5 = 'GPL-3')  and (Actual = 'GPL')
    or (Dep5 = 'GPL-3+') and (Actual = 'GPL') then
    begin
        License := GetFileLicense (FName);
        if (License  <> '') and (license [length(License)] = '+') then
            result := true
        else
        if Dep5 = License then
            result := true;
    end;
end;

// In d/copyright, change GPL3.0 to GPL3 etc,
// so strings match licencecheck
Function AdjustGPL (LicenseStr : String) : String;
begin
    iF StartsStr ('GPL', LicenseStr)
    and (FindPart ('.0', LicenseStr) <> 0)
    and (length (LicenseStr) < 15) then
    begin
        LicenseStr := DelChars (LicenseStr, '.');
        LicenseStr := DelChars (LicenseStr, '0');
    end;

    result := LicenseStr;
end;

end.
