% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reliability.R
\name{AVE}
\alias{AVE}
\title{Calculate average variance extracted}
\usage{
AVE(object, obs.var = TRUE, omit.imps = c("no.conv", "no.se"),
  omit.factors = character(0), dropSingle = TRUE, return.df = TRUE)
}
\arguments{
\item{object}{A \link[lavaan:lavaan-class]{lavaan::lavaan} or \link[lavaan.mi:lavaan.mi-class]{lavaan.mi::lavaan.mi} object,
expected to contain only exogenous common factors (i.e., a CFA model).
Cross-loadings are not allowed and will result in \code{NA} for any factor with
indicator(s) that cross-load.}

\item{obs.var}{\code{logical} indicating whether to compute AVE using
observed variances in the denominator. Setting \code{FALSE} triggers
using model-implied variances in the denominator.}

\item{omit.imps}{\code{character} vector specifying criteria for omitting
imputations from pooled results.  Can include any of
\code{c("no.conv", "no.se", "no.npd")}, the first 2 of which are the
default setting, which excludes any imputations that did not
converge or for which standard errors could not be computed.  The
last option (\code{"no.npd"}) would exclude any imputations which
yielded a nonpositive definite covariance matrix for observed or
latent variables, which would include any "improper solutions" such
as Heywood cases.  NPD solutions are not excluded by default because
they are likely to occur due to sampling error, especially in small
samples.  However, gross model misspecification could also cause
NPD solutions, users can compare pooled results with and without
this setting as a sensitivity analysis to see whether some
imputations warrant further investigation.}

\item{omit.factors}{\code{character} vector naming any common factors
modeled in \code{object} whose indicators' AVE is not of interest.}

\item{dropSingle}{\code{logical} indicating whether to exclude factors
defined by a single indicator from the returned results. If \code{TRUE}
(default), single indicators will still be included in the \code{total}
column when \code{return.total = TRUE}.}

\item{return.df}{\code{logical} indicating whether to return reliability
coefficients in a \code{data.frame} (one row per group/level), which is
possible when every model block includes the same factors (after excluding
those in \code{omit.factors} and applying \code{dropSingle}).}
}
\value{
\code{numeric} vector of average variance extracted from indicators
per factor.  For models with multiple "blocks" (any combination of groups
and levels), vectors may be returned as columns in a \code{data.frame}
with additional columns indicating the group/level (see \verb{return.df=}
argument description for caveat).
}
\description{
Calculate average variance extracted (AVE) per factor from \code{lavaan} object
}
\details{
The average variance extracted (AVE) can be calculated by

\deqn{ AVE = \frac{\bold{1}^\prime
\textrm{diag}\left(\Lambda\Psi\Lambda^\prime\right)\bold{1}}{\bold{1}^\prime
\textrm{diag}\left(\hat{\Sigma}\right) \bold{1}}, }

Note that this formula is modified from Fornell & Larcker (1981) in the case
that factor variances are not 1. The proposed formula from Fornell & Larcker
(1981) assumes that the factor variances are 1. Note that AVE will not be
provided for factors consisting of items with dual loadings. AVE is the
property of items but not the property of factors. AVE is calculated with
polychoric correlations when ordinal indicators are used.
}
\examples{
data(HolzingerSwineford1939)
HS9 <- HolzingerSwineford1939[ , c("x7","x8","x9")]
HSbinary <- as.data.frame( lapply(HS9, cut, 2, labels=FALSE) )
names(HSbinary) <- c("y7","y8","y9")
HS <- cbind(HolzingerSwineford1939, HSbinary)

HS.model <- ' visual  =~ x1 + x2 + x3
              textual =~ x4 + x5 + x6
              speed   =~ y7 + y8 + y9 '

fit <- cfa(HS.model, data = HS, ordered = c("y7","y8","y9"), std.lv = TRUE)

## works for factors with exclusively continuous OR categorical indicators
AVE(fit) # uses observed (or unconstrained polychoric/polyserial) by default
AVE(fit, obs.var = FALSE)


## works for multigroup models and for multilevel models (and both)
data(Demo.twolevel)
## assign clusters to arbitrary groups
Demo.twolevel$g <- ifelse(Demo.twolevel$cluster \%\% 2L, "type1", "type2")
model2 <- ' group: type1
  level: within
    fac =~ y1 + L2*y2 + L3*y3
  level: between
    fac =~ y1 + L2*y2 + L3*y3

group: type2
  level: within
    fac =~ y1 + L2*y2 + L3*y3
  level: between
    fac =~ y1 + L2*y2 + L3*y3
'
fit2 <- sem(model2, data = Demo.twolevel, cluster = "cluster", group = "g")
AVE(fit2)

}
\references{
Fornell, C., & Larcker, D. F. (1981). Evaluating structural equation models
with unobservable variables and measurement errors. \emph{Journal of
Marketing Research, 18}(1), 39--50. \doi{10.2307/3151312}
}
\seealso{
\code{\link[=compRelSEM]{compRelSEM()}} for composite reliability estimates
}
\author{
Terrence D. Jorgensen (University of Amsterdam; \email{TJorgensen314@gmail.com})
}
