use core:lang;

/**
 * The range-based for-loop. Expands to (more or less):
 * var at = c.begin();
 * var end = c.end();
 * for (; at != end; ++i) { name = at.v; ... }
 *
 * Or, for range-style iterators:
 * var i = c.iter();
 * while (name = i.next) { ... }
 */
class RangeFor extends Breakable {
	init(SrcPos pos, Block parent, SStr name, Expr container) {
		init(pos, parent) {
			valName = name.v;
			containerExpr = container;
		}

		createCode(container);
	}

	init(SrcPos pos, Block parent, SStr key, SStr value, Expr container) {
		init(pos, parent) {
			keyName = key.v;
			valName = value.v;
			containerExpr = container;
		}

		createCode(container);
	}

	void body(Expr expr) {
		unless (loopBlock)
			return;

		loopBlock.add(expr);
	}

	// Hide the fact that we're actually a few nested blocks!
	LocalVar? variableHere(SimplePart part) {
		if (loopBlock) {
			Bool lookup = part.name == valName;
			if (!lookup)
				if (keyName)
					lookup = part.name == keyName;

			if (lookup) {
				Block at = loopBlock;
				while (at !is this) {
					if (found = at.variableHere(part))
						return found;

					if (nextLookup = at.lookup.parent as BlockLookup)
						at = nextLookup.block;
					else
						break;
				}
			}
		}

		super:variableHere(part);
	}

	// Generate code.
	void blockCode(CodeGen state, CodeResult to) {
		// We put all initialization logic in its own child block, rather than one for each expression.
		CodeGen child = state.child();
		state.l << core:asm:begin(child.block);
		for (Nat i = 0; i < initExprs.count; i++) {
			initExprs[i].code(child, CodeResult());
		}
		state.l << core:asm:end(child.block);

		if (loop) {
			loop.code(state, to);
		}
	}

	// Support for break and continue.
	void willBreak() {
		if (loop)
			loop.willBreak();
	}
	void willContinue() {
		if (loop)
			loop.willContinue();
	}

	To breakTo() {
		if (loop)
			loop.breakTo();
		else
			throw InternalError("The loop was not initialized properly");
	}
	To continueTo() {
		if (loop)
			loop.continueTo();
		else
			throw InternalError("The loop was not initialized properly");
	}

	void toS(StrBuf to) {
		to << "for (";
		if (keyName)
			to << keyName << ", ";
		to << valName << " in " << containerExpr << ") ";
		if (loopBlock) {
			if (loopBlock.count > 0) {
				to << loopBlock[loopBlock.count - 1];
			} else {
				to << loopBlock;
			}
		} else {
			to << "<empty>";
		}
	}

private:

	// Names of the key and value variables.
	Str? keyName;
	Str valName;

	// Block containing the loop body.
	ExprBlock? loopBlock;

	// Expressions to execute in this block (we're basically an ExprBlock).
	Expr[] initExprs;

	// The loop itself. This is always the last expression in this block.
	Breakable? loop;

	// Copy of the container expression for printing.
	Expr containerExpr;


	void createCode(Expr container) {
		Var cont(this, SStr(" c", pos), container);
		LocalVarAccess getCont(pos, cont.var);
		initExprs.push(cont);

		if (createIterCode(getCont))
			return;

		if (createRangeCode(getCont))
			return;

		if (createCountCode(getCont))
			return;

		// Note: May not use fancy strings, their implementation depend on us!
		throw SyntaxError(pos, container.toS + " is not iterable.");
	}

	Bool createIterCode(LocalVarAccess container) {
		Expr beginExpr = namedExpr(this, SStr("begin", pos), container, Actuals());
		Expr endExpr = namedExpr(this, SStr("end", pos), container, Actuals());

		if (beginExpr as UnresolvedName)
			return false;
		if (endExpr as UnresolvedName)
			return false;

		Var atDecl(this, SStr(" at", pos), beginExpr);
		LocalVarAccess getAt(pos, atDecl.var);
		initExprs.push(atDecl);

		Var endDecl(this, SStr(" end", pos), endExpr);
		LocalVarAccess getEnd(pos, endDecl.var);
		initExprs.push(endDecl);

		For loop(pos, this);
		this.loop = loop;

		ExprBlock body(pos, loop);
		loop.test(namedExpr(loop, SStr("!=", pos), getAt, Actuals(getEnd)));
		loop.update(namedExpr(loop, SStr("++*", pos), getAt, Actuals()));
		loop.body(body);

		if (keyName) {
			Expr keyCall = namedExpr(loop, SStr("k", pos), getAt, Actuals());
			if (keyCall as UnresolvedName) {
				StrBuf message;
				message << "Iterators for " << container.var.result.toS
						<< " do not provide keys for iteration.";
				throw SyntaxError(pos, message.toS);
			}
			body.add(Var(body, SStr(keyName, pos), keyCall));
		}

		body.add(Var(body, SStr(valName), namedExpr(loop, SStr("v", pos), getAt, Actuals())));
		loopBlock = body;

		true;
	}

	Bool createRangeCode(LocalVarAccess container) {
		Expr iterExpr = namedExpr(this, SStr("iter", pos), container, Actuals());
		if (iterExpr as UnresolvedName)
			return false;

		Var iterDecl(this, SStr(" iter", pos), iterExpr);
		LocalVarAccess getIter(pos, iterDecl.var);

		Expr nextExpr = namedExpr(this, SStr("next", pos), getIter, Actuals());
		if (nextExpr as UnresolvedName)
			return false;

		initExprs.push(iterDecl);

		Loop loop(pos, this);
		this.loop = loop;

		if (keyName)
			throw SyntaxError(pos, "Keys are not supported for lazy iterators.");

		WeakMaybeCast cond(nextExpr);
		cond.name(SStr(valName, pos));
		loop.cond(cond);

		ExprBlock eBody(pos, loop.whileBlock);
		loop.whileBody(eBody);

		loopBlock = eBody;

		true;
	}

	Bool createCountCode(LocalVarAccess container) {
		Expr countExpr = namedExpr(this, SStr("count", pos), container, Actuals());
		if (countExpr as UnresolvedName)
			return false;

		SimplePart accessName("[]");
		accessName.params.push(container.result.type);
		accessName.params.push(countExpr.result.type);
		unless (containerType = container.result.type.type)
			return false;
		unless (accessFn = containerType.find(accessName, scope) as Function)
			return false;

		Var countDecl(this, SStr(" count", pos), countExpr);
		LocalVarAccess getCount(pos, countDecl.var);
		initExprs.push(countDecl);

		Str indexName = " id";
		if (keyName)
			indexName = keyName;

		Var indexDecl(this, countDecl.result.type, SStr(indexName, pos), Actuals());
		LocalVarAccess getIndex(pos, indexDecl.var);
		initExprs.push(indexDecl);

		For loop(pos, this);
		this.loop = loop;

		ExprBlock body(pos, loop);
		loop.test(namedExpr(loop, SStr("<", pos), getIndex, Actuals(getCount)));
		loop.update(namedExpr(loop, SStr("++*", pos), getIndex));
		loop.body(body);

		Actuals getValActuals;
		getValActuals.add(container);
		getValActuals.add(getIndex);
		body.add(Var(body, SStr(valName), FnCall(pos, scope, accessFn, getValActuals)));

		loopBlock = body;

		// Don't change the intex variable outside the loop.
		indexDecl.var.constant = true;

		true;
	}

}
