package pusher

import (
	"encoding/json"
	"io/ioutil"
	"os"
	"path/filepath"

	"pkg.deepin.io/daemon/sync/infrastructure/log"

	"github.com/godbus/dbus"
)

// load all handler from dbus
const (
	handlerDir = "/etc/deepin/pusher/handler.d"
)

type notify struct {
	dbusHandlers map[string]*dbusHandler
}

type dbusHandler struct {
	MsgType   string `json:"msg_type"`
	Service   string `json:"service"`
	Path      string `json:"path"`
	Interface string `json:"interface"`
}

func newNotify() *notify {
	n := &notify{
		dbusHandlers: make(map[string]*dbusHandler),
	}
	n.loadConfig()
	return n
}

func (n *notify) Register(msgType, service, path, ifc string) *dbus.Error {
	n.dbusHandlers[msgType] = &dbusHandler{
		MsgType:   msgType,
		Service:   service,
		Path:      path,
		Interface: ifc,
	}
	return nil
}

func (n *notify) Reload() *dbus.Error {
	n.loadConfig()
	return nil
}

func (n *notify) sendMessage(msg message) error {
	handler, ok := n.dbusHandlers[msg.MsgType]
	if !ok {
		log.Infof("receive unknown message: %v", msg)
		return nil
	}

	sessionBus, err := dbus.SessionBus()
	if nil != err {
		log.Infof("get session bus failed: %v", err)
		return err
	}

	caller := sessionBus.Object(handler.Service, dbus.ObjectPath(handler.Path))
	err = caller.Call(handler.Interface+".OnMessage", 0, msg.Playload).Store()
	if nil != err {
		log.Infof("dispatch message failed: %v", err)
		return err
	}
	return err
}

func listDir(root string) ([]string, error) {
	var files []string
	err := filepath.Walk(root, func(path string, info os.FileInfo, err error) error {
		if info != nil && !info.IsDir() {
			files = append(files, path)
		}
		return nil
	})
	return files, err
}

func (n *notify) loadConfig() {
	list, _ := listDir(handlerDir)
	for _, filepath := range list {
		data, _ := ioutil.ReadFile(filepath)
		handler := &dbusHandler{}
		err := json.Unmarshal(data, handler)
		if nil != err {
			log.Infof("load config failed", filepath)
			continue
		}
		n.dbusHandlers[handler.MsgType] = handler
	}
}
