/* NVTV direct backend -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: back_direct.c,v 1.32 2004/03/10 07:18:03 dthierbach Exp $
 *
 * Contents:
 *
 * Backend for direct access of the card (common part for Linux and MS-Windows)
 *
 */

#include "local.h" /* before everything else */

#include <stdio.h>

#include "backend.h"
#include "back_direct.h"
#include "back_nvidia.h"
#include "back_tdfx.h"
#include "back_i810.h"

#ifdef DEBUG_PROBE
#include "tv_i2c.h"
#include "tv_common.h"
#endif

/* -------- Generic find routines -------- */

/*
 *  List all modes (by system)
 */

int 
bdir_listModes (CardType card, DataFunc *data, 
  TVSystem system, TVMode *(list[]))
{
  if (!data) return FALSE;
  return data_listModes (data->modes (), system, list);
}

/*
 *  Find mode by size (and resolution). 
 */

Bool 
bdir_findBySize (CardType card, DataFunc *data, 
  TVSystem system, int xres, int yres, char *size, TVMode *mode)
{
  TVMode *tvm;

  if (!data) return FALSE;
  tvm = data_find (data->modes (), system, xres, yres, size);
  if (!tvm) return FALSE;
  *mode = *tvm;
  return TRUE;
}

/*
 *  Find mode by overscan (and resolution)
 */

Bool 
bdir_findByOverscan (CardType card, DataFunc *data, TVSystem system, 
  int xres, int yres, double hoc, double voc, TVMode *mode)
{
  TVMode *tvm;

  if (!data) return FALSE;
  tvm = data_findNearest (data->modes (), system, xres, yres, hoc, voc);
  if (!tvm) return FALSE;
  *mode = *tvm;
  return TRUE;
}

/* -------- -------- */

/* 
 *  Open card. For now, just pass that on.
 */

void 
bdir_openCard (CardPtr card)
{
  switch (card->type)
  {
    case CARD_NVIDIA:
    case CARD_XBOX:
      back_card = &bnv_func;
      break;
    case CARD_TDFX:
      back_card= &btdfx_func;
      break;
    case CARD_I810:
      back_card= &bi810_func;
      break;
    default:
      RAISE (MSG_ABORT, "Wrong card type in bdir_openCard.");
      exit (1);
  }
  back_card->openCard (card);
}

/*
 * Close card, For now, just pass that on.
 */

void 
bdir_closeCard (void)
{
  if (back_card) back_card->closeCard ();
  back_card = NULL;
}

/* -------- Probe system -------- */

/* This is more or less a hack to get some information about the whole
   system in case something doesn't work. It should be eventually removed
   from the direct backend, but at the moment this is the logical place
   for it. */

#ifdef DEBUG_PROBE
void 
bdir_probe_ch1 (I2CDevPtr dev)
{
  I2CByte res;
  int i;

  for (i = 0x00; i <= 0x3F; i++) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    tvBusOk = TRUE;  
    TVReadBus (dev, i, &res);
    if (!tvBusOk) printf (" --"); else printf (" %02X", res);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
}
#endif

#ifdef DEBUG_PROBE
void 
bdir_probe_ch2 (I2CDevPtr dev)
{
  I2CByte res;
  int i;

  for (i = 0x00; i <= 0x7F; i++) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    tvBusOk = TRUE;  
    TVReadBus (dev, i, &res);
    if (!tvBusOk) printf (" --"); else printf (" %02X", res);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
}
#endif

#ifdef DEBUG_PROBE
void 
bdir_probe_cx (I2CDevPtr dev)
{
  I2CByte res;
  int i;

  tvBusOk = TRUE;  
  TVWriteBus (dev, 0x6c, 0x46); /* enable reg rd */
  for (i = 0x00; i <= 0xFF; i += 1) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    tvBusOk = TRUE;  
    TVReadBus (dev, i, &res);
    if (!tvBusOk) printf (" --"); else printf (" %02X", res);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
  tvBusOk = TRUE;  
  TVWriteBus (dev, 0x6c, 0x06); /* disable reg rd, for bt compatibility */
}
#endif

#ifdef DEBUG_PROBE
void 
bdir_probe_ph (I2CDevPtr dev)
{
  I2CByte res;
  I2CByte rgb[3];
  int i;

  for (i = 0x00; i <= 0xFD; i += 1) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    tvBusOk = TRUE;  
    TVReadBus (dev, i, &res);
    if (!tvBusOk) printf (" --"); else printf (" %02X", res);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
  printf ("\n");
  tvBusOk = TRUE;
  TVWriteBus (dev, 0xfe, 0x00); /* Cursor bit map */
  for (i = 0x00; i < 0x100; i += 1) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    tvBusOk = TRUE;  
    TVStatusBus (dev, &res);
    if (!tvBusOk) printf (" --"); else printf (" %02X", res);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
  TVWriteBus (dev, 0xff, 0x00); /* RGB LUT */
  for (i = 0x00; i < 0x100; i += 1) {
    if ((i & 0x7) == 0x0) printf ("    %02X:", i);
    if ((i & 0x7) == 0x4) printf (" ");
    tvBusOk = TRUE;  
    TVStatusSeqBus (dev, rgb, 3); 
    if (!tvBusOk) 
      printf (" -- -- --"); 
    else 
      printf (" %02X%02X%02X", rgb[0], rgb[1], rgb[2]);
    if ((i & 0x7) == 0x7) printf ("\n");
  }
}
#endif

#ifdef DEBUG_PROBE
void 
bdir_probeSystem (CardPtr card_list)
{
  CardPtr card;
  I2CChainPtr chain, root;

  printf ("NVTV " PACKAGE_VERSION " Probe\n");
  for (card = card_list; card; card = card->next) 
  {
    printf ("%s (%04X) io=0x%08lX\n", card->name, card->pci_id, 
	    card->reg_base);
    bdir_openCard (card);
    back_card->probeCard ();
    printf ("  I2C Devices:");
    root = back_card->probeBus ();
    for (chain = root; chain; chain = chain->next) {
      printf (" %1s:%02X", I2C_ID(chain->dev));
    }
    printf ("\n\n");
    for (chain = root; chain; chain = chain->next) {
      if (chain->type != TV_NO_CHIP) {
	printf ("  Device %1s:%02X = %s\n", I2C_ID(chain->dev), chain->name);
	switch (chain->type & TV_ENCODER) 
	{
	  case TV_CHRONTEL_MODEL1:
	    bdir_probe_ch1 (chain->dev); break;
	  case TV_CHRONTEL_MODEL2:
	    bdir_probe_ch2 (chain->dev); break;
	  case TV_CONEXANT:
	    bdir_probe_cx (chain->dev); break;
	  case TV_PHILIPS_MODEL1:
	  case TV_PHILIPS_MODEL2:
	    bdir_probe_ph (chain->dev); break;
	  default:
	    break;
	}
      }
    }
    bdir_closeCard ();
  }
}
#endif

BackAccessRec bdir_func = {
  openCard:    bdir_openCard,
  closeCard:   bdir_closeCard,
#ifdef DEBUG_PROBE
  probeSystem: bdir_probeSystem,
#endif
};

/* -------- Helper -------- */

void 
bdir_freeChips (CardPtr card)
{
  ChipPtr chip, del;

  chip = card->chips; 
  while (chip) {
    del = chip;
    chip = chip->next;
    /* don't free name */
    free (del);
  }
  card->chips = NULL;
}

ChipPtr 
bdir_copyChips (I2CChainPtr root)
{
  ChipPtr chip, ins, first;
  I2CChainPtr chain;

  first = ins = NULL;
  for (chain = root; chain; chain = chain->next)
  {
    chip = (ChipPtr) xcalloc (1, sizeof(ChipInfo));
    chip->private = chain;
    chip->next = NULL;
    chip->type = chain->type;
    chip->name = chain->name;
    if (chain->dev) {
      chip->bus  = chain->dev->pI2CBus->BusName[2];
      chip->addr = chain->dev->SlaveAddr;
    } else {
      chip->bus = -1;
      chip->addr = 0;
    }
    if (ins) {
      ins->next = chip;
    } else {
      first = chip;
    }
    ins = chip;
  }
  return first;
}

