#!/bin/bash
# A script to configure the module's debug log level.
#
# Note: Please do not modify this script directly,
# as modifying this script will invalidate this script.
set -e

# 设计说明：
# Qt 应用默认开启 debug 和 info 日志。
# - 开启调试时：删除配置文件，使用 Qt 默认行为（debug/info 已启用）
# - 关闭调试时：创建配置文件，显式禁用 debug/info 日志
#
# 本脚本配置两个层级：
# 1. /etc/X11/Xsession.d/ - 用户图形会话中的 Qt 应用（包括用户级服务）
# 2. /etc/systemd/system.conf.d/ - 系统级 systemd 服务

readonly pkg_name="qt"
readonly xsession_file="/etc/X11/Xsession.d/98-deepin-debug-config-qt"
readonly systemd_system_dir="/etc/systemd/system.conf.d"
readonly systemd_system_file="$systemd_system_dir/98-deepin-debug-qt.conf"

debug="no"

is_root() {
    local current_uid
    current_uid=$(id -u)

    if [ "$current_uid" -ne 0 ]; then
        echo "You need to have root privileges to run this script."
        return 1
    fi
    return 0
}

can_exec() {
    if ! is_root; then
        return 1
    fi
    if [ $# -ne 1 ]; then
        echo "Invalid argument"
        return 1
    fi
    for x in "$@"; do
        case $x in
            debug=*)
                debug=${x#debug=}
                ;;
        esac
    done
    case "${debug}" in
        "on" | "off" | "debug" | "warning")
            return 0
            ;;
        *)
            echo "Not support ${debug} parameter: $*"
            return 1
            ;;
    esac
}

update_xsession_config() {
    local level=$1
    local config_file=$2

    case "$level" in
        "on" | "debug")
            # 开启调试时删除文件，使用 Qt 默认行为
            if [[ -f "$config_file" ]]; then
                rm -f "$config_file"
            fi
            ;;
        "off" | "warning")
            # 关闭调试时设置为 false
            echo "export QT_LOGGING_RULES=\"*.debug=false;*.info=false\"" > "$config_file"
            ;;
    esac
}

update_systemd_config() {
    local level=$1
    local config_file=$2

    case "$level" in
        "on" | "debug")
            # 开启调试时删除文件，使用 Qt 默认行为
            if [[ -f "$config_file" ]]; then
                rm -f "$config_file"
            fi
            ;;
        "off" | "warning")
            # 关闭调试时设置为 false
            echo "[Manager]" > "$config_file"
            echo "DefaultEnvironment=\"QT_LOGGING_RULES=*.debug=false;*.info=false\"" >> "$config_file"
            ;;
    esac
}

main() {
    if ! can_exec "$@"; then
        exit 1
    fi

    # 创建必要的目录
    mkdir -p "$(dirname "$xsession_file")"
    mkdir -p "$systemd_system_dir"

    # 更新 X11 会话配置（用户图形会话和用户级服务）
    update_xsession_config "$debug" "$xsession_file"

    # 更新 systemd 系统级配置（系统服务）
    update_systemd_config "$debug" "$systemd_system_file"

    echo "Debug logs are set to '${debug}' for ${pkg_name}."
}

main "$@"
