/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/

#ifndef __keyboard_keyboard

#define __keyboard_keyboard

#include "ui.h"
#include "note.h"
#include "solver.h"
#include "play.h"
#include "multi_curve.h"
#include "curve_editor.h"
#include "listeners.h"
#include "random.h"
#include "curve_library.h"
#include "color.h"

struct keyboard_keyboard;

struct triggered_note {

  note tn; // note info

  float start_hz; // initial frequency of note

  int key; // key on computer keyboard that triggers note

  // solver & player for waveform of keyboard-keyboard
  solver sol;
  play player;

  // state
  // note removed when FINISHED
  //
  enum {ATTACK, DECAY, FINISHED};
  int state;

  //
  // volumes

  float volume_now;
  float volume_max;

  //
  // attack
  //

  float abs_attack, delta_attack;
  solver attack; // solves attack curve from keyboard-keyboard

  //
  // decay
  //

  float decay_time;
  float decay_start_volume;

  float abs_decay, delta_decay;
  solver decay; // solves decay curve from keyboard-keyboard


  // visual
  int x, y; // position
  float r, g, b; // color

  int mx; // mouse x pos (see mouse_bend ())
  int bendx;

  triggered_note (const note& n, int k, multi_curve* wav, multi_curve* atk, multi_curve* dk, float volmax, float xx = 0, float yy = 0, float rr = 1, float gg = 1, float bb = 1, int mox = 0);
  void set_frequency (float f);
  void eval (float* left, float* right, float* wav, float* vol, int n);

};

struct key_info {

  int id; // from X
  char ch; // character

  // visual
  int x, y; // position
  float r, g, b; // color

  int attacked; // attacked?

  key_info (int i = -1, char c = '.', int a = 0) : id (i), ch (c), attacked (a) {
   r = g = b = 1;
  }

};

struct keyboard_keyboard : ui {

  multi_curve wave; // waveform
  curve_editor waved; // waveform shape editor
  wave_listener wavlis; // waveform edit listener
  void update_waveform ();

  std::vector<note> notes; // notes of the scale worth 3 octaves
  void setup_notes (int overwrite = 1);

  std::list <triggered_note> triggered_notes; // currently active notes
  void remove_finished_notes ();
  void attack_note (int ky);
  void decay_note (int ky);
  void render_audio (float* L, float* R);

  std::map <int, std::vector<key_info> > scale2keybd;
  static const int MAX_KEYS = 256; // assume no keycode exceeds 255
  int keybd2notes [MAX_KEYS];
  std::vector<key_info> keys;

  // attack & decay curve
  multi_curve attackcrv, decaycrv;
  curve_editor attacked, decayed;
  attack_listener attacklis;
  decay_listener decaylis;
  curve_library attacklib, decaylib;
  void update_attack ();
  void update_decay ();

  bool handle_input ();

  // visual
  int arm, ymarker, ychars, spacing;
  void calc_visual_params ();

  void draw ();

  keyboard_keyboard ();
  ~keyboard_keyboard ();

  help helptext;

  // MIDI
  //
  static const int MIDI_MAX = 128;
  static const color note_color [];
  static const int color_index [];
  note midi_notes [MIDI_MAX];
  void setup_midi_notes ();
  void note_on (unsigned char id, unsigned char vel);
  void note_off (unsigned char id);
  void pitch_bend (float v);

  // MIDI velocity curve
  multi_curve velcrv;
  velocity_listener vellis;
  curve_editor veled;
  curve_library vellib;
  solver velsol;

  // mouse based pitch bend
  //
  int marker_x;
  void mouse_bend ();

  void enter ();
  void leave ();
  void bg ();

};

#endif
