(* $Id: crypt_blowfish.ml,v 1.2 2001/03/10 16:43:21 gerd Exp $
 * ----------------------------------------------------------------------
 * This module is part of the cryptgps package by Gerd Stolpmann.
 *)

(* This implementation of Blowfish uses 16 bit numbers instead of
 * 32 bits because of Caml limitations.
 *)

open Crypt_aux

module Cryptsystem : Cryptsystem_64.T =
  struct

    type key =
	{ data : string;
	  p_lsb : int array;
	  p_msb : int array;
            (* subkeys: 18 elements of 16 bits *)
	  p_lsb_rev : int array;
	  p_msb_rev : int array;
            (* subkeys in reverse order *)
	  s1_lsb : int array;
	  s1_msb : int array;
	  s2_lsb : int array;
	  s2_msb : int array;
	  s3_lsb : int array; 
	  s3_msb : int array;
	  s4_lsb : int array;
	  s4_msb : int array;
        (* the four s-boxes: sNP, where N=0,1,2,3 denotes the box,
	 * and P=0,1 whether the LSB (0) or MSB (1) of the box
	 * value is stored.
	 *)
	} 


    let encrypt_ecb k (xl_msb,xl_lsb,xr_msb,xr_lsb) =
      (* The 64-bit element (xl, xr) is encrypted using key k with:
       * - xl_lsb the LSB of xl 
       * - xl_msb the MSB of xl 
       * - xr_lsb the LSB of xr 
       * - xr_msb the MSB of xr 
       * Returns (xl_msb', xl_lsb', xr_msb', xr_lsb'). 
       *)
      let p_lsb = k.p_lsb in
      let p_msb = k.p_msb in
      let s1_lsb = k.s1_lsb in
      let s1_msb = k.s1_msb in
      let s2_lsb = k.s2_lsb in
      let s2_msb = k.s2_msb in
      let s3_lsb = k.s3_lsb in
      let s3_msb = k.s3_msb in
      let s4_lsb = k.s4_lsb in
      let s4_msb = k.s4_msb in
      let rec compute_rounds i xl_lsb xl_msb xr_lsb xr_msb =
	if i < 16 then begin
	  let xl_lsb' = xl_lsb lxor p_lsb.(i) in
	  let xl_msb' = xl_msb lxor p_msb.(i) in
	  let a = xl_msb' lsr 8 in
	  let b = xl_msb' land 0xff in
	  let c = xl_lsb' lsr 8 in
	  let d = xl_lsb' land 0xff in
	  let s_1a_plus_s_2b_lsb = 
	    s1_lsb.(a) + s2_lsb.(b) in
	  let s_1a_plus_s_2b_msb = 
	    s1_msb.(a) + s2_msb.(b) + (s_1a_plus_s_2b_lsb lsr 16) in
	  let after_xor_s_3c_lsb =
	    (s_1a_plus_s_2b_lsb land 0xffff) lxor s3_lsb.(c) in
	  let after_xor_s_3c_msb =
	    (s_1a_plus_s_2b_msb land 0xffff) lxor s3_msb.(c) in
	  let y_lsb = 
	    after_xor_s_3c_lsb + s4_lsb.(d) in
	  let y_msb =
	    after_xor_s_3c_msb + s4_msb.(d) + (y_lsb lsr 16) in
	  let xr_lsb' = (y_lsb land 0xffff) lxor xr_lsb in
	  let xr_msb' = (y_msb land 0xffff) lxor xr_msb in
	  compute_rounds (i+1) xr_lsb' xr_msb' xl_lsb' xl_msb' 
	end
	else
	  (xr_msb lxor p_msb.(17),
	   xr_lsb lxor p_lsb.(17),
	   xl_msb lxor p_msb.(16),
	   xl_lsb lxor p_lsb.(16))
	    
      in
      compute_rounds 0 xl_lsb xl_msb xr_lsb xr_msb


    let encrypt_ecb_int32 k xl xr ret_xl ret_xr =
      let x = quadruple_of_int32 xl xr in
      let y = encrypt_ecb k x in
      int32_of_quadruple y ret_xl ret_xr


    let decrypt_ecb k x =
      (* The 64-bit element (xl', xr') is decrypted using key k with:
       * - xl_lsb' the LSB of xl'
       * - xl_msb' the MSB of xl'
       * - xr_lsb' the LSB of xr'
       * - xr_msb' the MSB of xr'
       * Returns (xl_msb, xl_lsb, xr_msb, xr_lsb). 
       *)
      let k' =
	{ k with
	  p_lsb = k.p_lsb_rev;
	  p_msb = k.p_msb_rev
	} in
      encrypt_ecb k' x


    let decrypt_ecb_int32 k xl xr ret_xl ret_xr =
      let x = quadruple_of_int32 xl xr in
      let y = decrypt_ecb k x in
      int32_of_quadruple y ret_xl ret_xr


    let prepare key =
      let l_key = String.length key in
      if l_key = 0 or l_key > 56 then
	failwith "Crypt_blowfish: invalid key length";

      let k =
	{ data = key;
	  p_lsb =
          [| 0x6a88; 0x08d3; 0x8a2e; 0x7344;
	    0x3822; 0x31d0; 0xfa98; 0x6c89;
	    0x21e6; 0x1377; 0x66cf; 0x0c6c;
	    0x29b7; 0x50dd; 0xd5b5; 0x0917;
	    0xd5d9; 0xfb1b |];
	  p_msb =
          [| 0x243f; 0x85a3; 0x1319; 0x0370;
	    0xa409; 0x299f; 0x082e; 0xec4e;
	    0x4528; 0x38d0; 0xbe54; 0x34e9;
	    0xc0ac; 0xc97c; 0x3f84; 0xb547;
	    0x9216; 0x8979 |];
	  p_lsb_rev =
          [| 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; |];
	  p_msb_rev =
          [| 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; |];
	  s1_lsb =
          [| 0x0ba6; 0xb5ac; 0x72db; 0xdfb7;
	    0xafed; 0x7e96; 0x9045; 0x7f99;
	    0x9947; 0x6cf7; 0xf2e2; 0xfc16;
	    0x20d8; 0x4e69; 0xfea3; 0x3d7e;
	    0x748f; 0xb658; 0xcd58; 0x4aee;
	    0xa41d; 0x59b5; 0xd539; 0x6013;
	    0xb023; 0x85f0; 0x7918; 0x38ef;
	    0xdcb0; 0x180e; 0x0e8b; 0x8a3e;
	    0x77c1; 0x4b27; 0x2fda; 0x5c60;
	    0x25f3; 0xab94; 0x9862; 0x1440;
	    0x396a; 0x10b6; 0x5c34; 0xe8ce;
	    0x86af; 0xe993; 0x1411; 0xbc2a;
	    0xc55d; 0x31f6; 0x3e16; 0x931e;
	    0xba33; 0xcf5c; 0x5381; 0x8677;
	    0x4898; 0xb9af; 0xe81b; 0x2193;
	    0x09cc; 0xa991; 0xac60; 0x8032;
	    0x5d5d; 0x75b1; 0x2302; 0x1b88;
	    0x3e81; 0xacc5; 0x6ff3; 0x4239;
	    0x4482; 0x2004; 0xf04a; 0x9b5e;
	    0x6842; 0x6c9a; 0x9c61; 0x88f0;
	    0xa0d2; 0x2f68; 0xa728; 0x33a3;
	    0x0b6c; 0x3be4; 0xf050; 0x2a98;
	    0x651d; 0x0176; 0x593e; 0x0e88;
	    0x8619; 0x9fb4; 0xa5c3; 0x5ebe;
	    0x75d8; 0x2073; 0x449f; 0x6aa6;
	    0xaa62; 0x7706; 0xdf72; 0x023d;
	    0xd724; 0x1248; 0xead3; 0xc09b;
	    0x72c9; 0x1b7b; 0x79d8; 0xdef7;
	    0x501a; 0x4c3b; 0xe0bd; 0x06ba;
	    0x4fb6; 0x60c4; 0x9ec2; 0x2463;
	    0x6faf; 0x53b5; 0xb2eb; 0xec6f;
	    0x511f; 0x952c; 0x4544; 0xbd09;
	    0xd004; 0x4afd; 0x2807; 0x4bb3;
	    0xa857; 0x740f; 0x5f39; 0xfbdb;
	    0xc0bd; 0x320a; 0x00c6; 0x7279;
	    0x25fe; 0xa3cc; 0xe9f8; 0x22f8;
	    0x16df; 0x6b15; 0x1ec8; 0x52ab;
	    0xb5fa; 0x8760; 0x7b48; 0xdf82;
	    0x57bb; 0x8ca0; 0x562e; 0x69db;
	    0xa8f6; 0xffc3; 0x32c6; 0x5573;
	    0x27b0; 0x58c8; 0xa35d; 0x11a0;
	    0x3d98; 0x83b8; 0xb56c; 0xd35b;
	    0xe479; 0x4565; 0x49bc; 0x9790;
	    0xf2da; 0x7e33; 0x1341; 0xc6e8;
	    0xcada; 0x4c01; 0x9efe; 0x1fb4;
	    0xda4d; 0x9198; 0x8e71; 0xd5a0;
	    0xd1d0; 0x25e0; 0x5b2f; 0x94b7;
	    0xe2fb; 0x2b64; 0xb812; 0xf01c;
	    0x5ea0; 0xc31c; 0xf191; 0xc1ad;
	    0x2218; 0x1777; 0x2dfe; 0x1fa1;
	    0xcc0f; 0x74e8; 0xf3d6; 0xe299;
	    0x4fe0; 0xe0b7; 0x3b81; 0xa8d9;
	    0xa266; 0x7705; 0x7314; 0x1477;
	    0x2065; 0xfa86; 0x42f5; 0x35cf;
	    0xaf0c; 0x89a0; 0x1bd3; 0x7e49;
	    0x0e2d; 0xb35e; 0x00bb; 0xe0af;
	    0x369b; 0xb91e; 0x911d; 0xa6aa;
	    0x4389; 0x537f; 0x5ba2; 0xb9c5;
	    0x0376; 0xcfa9; 0x1968; 0x4a41;
	    0x2dca; 0xa94a; 0x0052; 0x2915;
	    0x573f; 0xc6e4; 0xa476; 0x7400;
	    0x6fb5; 0xe91f; 0xec6b; 0xd915;
	    0x6521; 0xf9b6; 0x052e; 0x5664;
	    0x2d5d; 0x8fa1; 0x4799; 0x076a; |];
	  s1_msb =
          [| 0xd131; 0x98df; 0x2ffd; 0xd01a;
	    0xb8e1; 0x6a26; 0xba7c; 0xf12c;
	    0x24a1; 0xb391; 0x0801; 0x858e;
	    0x6369; 0x7157; 0xa458; 0xf493;
	    0x0d95; 0x728e; 0x718b; 0x8215;
	    0x7b54; 0xc25a; 0x9c30; 0x2af2;
	    0xc5d1; 0x2860; 0xca41; 0xb8db;
	    0x8e79; 0x603a; 0x6c9e; 0xb01e;
	    0xd715; 0xbd31; 0x78af; 0x5560;
	    0xe655; 0xaa55; 0x5748; 0x63e8;
	    0x55ca; 0x2aab; 0xb4cc; 0x1141;
	    0xa154; 0x7c72; 0xb3ee; 0x636f;
	    0x2ba9; 0x7418; 0xce5c; 0x9b87;
	    0xafd6; 0x6c24; 0x7a32; 0x2895;
	    0x3b8f; 0x6b4b; 0xc4bf; 0x6628;
	    0x61d8; 0xfb21; 0x487c; 0x5dec;
	    0xef84; 0xe985; 0xdc26; 0xeb65;
	    0x2389; 0xd396; 0x0f6d; 0x83f4;
	    0x2e0b; 0xa484; 0x69c8; 0x9e1f;
	    0x21c6; 0xf6e9; 0x670c; 0xabd3;
	    0x6a51; 0xd854; 0x960f; 0xab51;
	    0x6eef; 0x137a; 0xba3b; 0x7efb;
	    0xa1f1; 0x39af; 0x66ca; 0x8243;
	    0x8cee; 0x456f; 0x7d84; 0x3b8b;
	    0xe06f; 0x85c1; 0x401a; 0x56c1;
	    0x4ed3; 0x363f; 0x1bfe; 0x429b;
	    0x37d0; 0xd00a; 0xdb0f; 0x49f1;
	    0x0753; 0x8099; 0x25d4; 0xf6e8;
	    0xe3fe; 0xb679; 0x976c; 0x04c0;
	    0xc1a9; 0x409f; 0x5e5c; 0x196a;
	    0x68fb; 0x3e6c; 0x1339; 0x3b52;
	    0x6dfc; 0x9b30; 0xcc81; 0xaf5e;
	    0xbee3; 0xde33; 0x660f; 0x192e;
	    0xc0cb; 0x45c8; 0xd20b; 0xb9d3;
	    0x5579; 0x1a60; 0xd6a1; 0x402c;
	    0x679f; 0xfb1f; 0x8ea5; 0xdb32;
	    0x3c75; 0xfd61; 0x2f50; 0xad05;
	    0x323d; 0xfd23; 0x5331; 0x3e00;
	    0x9e5c; 0xca6f; 0x1a87; 0xdf17;
	    0xd542; 0x287e; 0xac67; 0x8c4f;
	    0x695b; 0xbbca; 0xe1ff; 0xb8f0;
	    0x10fa; 0xfd21; 0x4afc; 0x2dd1;
	    0x9a53; 0xb6f8; 0xd28e; 0x4bfb;
	    0xe1dd; 0xa4cb; 0x62fb; 0xcee4;
	    0xef20; 0x3677; 0xd07e; 0x2bf1;
	    0x95db; 0xae90; 0xeaad; 0x6b93;
	    0xd08e; 0xafc7; 0x8e3c; 0x8e75;
	    0x8ff6; 0xf212; 0x8888; 0x900d;
	    0x4fad; 0x688f; 0xd1cf; 0xb3a8;
	    0x2f2f; 0xbe0e; 0xea75; 0x8b02;
	    0xe5a0; 0xb56f; 0x18ac; 0xce89;
	    0xb4a8; 0xfd13; 0x7cc4; 0xd2ad;
	    0x165f; 0x8095; 0x93cc; 0x211a;
	    0xe6ad; 0x77b5; 0xc754; 0xfb9d;
	    0xebcd; 0x7b3e; 0xd641; 0xae1e;
	    0x0025; 0x2071; 0x2268; 0x57b8;
	    0x2464; 0xf009; 0x5563; 0x59df;
	    0x78c1; 0xd95a; 0x207d; 0x02e5;
	    0x8326; 0x6295; 0x11c8; 0x4e73;
	    0xb347; 0x7b14; 0x1b51; 0x9a53;
	    0xd60f; 0xbc9b; 0x2b60; 0x81e6;
	    0x08ba; 0x571b; 0xf296; 0x2a0d;
	    0xb663; 0xe7b9; 0xff34; 0xc585;
	    0x53b0; 0xa99f; 0x08ba; 0x6e85; |];
	  s2_lsb =
          [| 0x70e9; 0x2944; 0x092e; 0x2623;
	    0xa6b0; 0xdf7d; 0x60b8; 0xb266;
	    0x8c71; 0x17ff; 0x526c; 0x9ee1;
	    0x02a5; 0x4c29; 0x1340; 0x3a3e;
	    0x989a; 0x9d65; 0xe4d6; 0x3fd6;
	    0x9c07; 0x30f5; 0x38e6; 0x5dc1;
	    0x2086; 0xeb26; 0xe9c6; 0xcc5e;
	    0x6b3f; 0xefc9; 0x1814; 0x70a1;
	    0x3584; 0xe286; 0x5305; 0x0737;
	    0x841c; 0xae5c; 0x44ec; 0xf2b8;
	    0xda37; 0x0c0d; 0x1f04; 0xb3ff;
	    0xf51a; 0x74b2; 0x7a58; 0x21bd;
	    0x13f9; 0x2ff6; 0x4773; 0x4701;
	    0xe581; 0xdadc; 0x7634; 0xdda7;
	    0x6146; 0x030e; 0xc73e; 0x1e41;
	    0xcd99; 0x0e2f; 0xbba1; 0xb331;
	    0x8b38; 0xb908; 0x0d03; 0x04bf;
	    0x1290; 0x7c79; 0xb072; 0x89af;
	    0x771f; 0x0810; 0xae12; 0x3f2e;
	    0x721f; 0x7124; 0xdde6; 0xcd87;
	    0x4718; 0xda17; 0x9abc; 0x7d8c;
	    0xec3a; 0x1dfa; 0x4366; 0xc3d2;
	    0x1847; 0xd908; 0x3b37; 0xba16;
	    0x4d43; 0xc451; 0x0002; 0xe4dd;
	    0xf89e; 0x4e55; 0x77d6; 0x199b;
	    0x56f1; 0xc76b; 0x183b; 0xa509;
	    0xe6ed; 0xfbfa; 0xbf2c; 0x3c6e;
	    0x4570; 0x6fb1; 0x5e0a; 0x2ab3;
	    0xe71c; 0x06fa; 0xdcb9; 0x1d0f;
	    0x89d6; 0xc825; 0xc978; 0xb36a;
	    0x0eba; 0xea78; 0x3c53; 0x2df4;
	    0x4ea7; 0x2b3d; 0x260f; 0x7960;
	    0xa708; 0x12b6; 0xfe6e; 0x1f66;
	    0x4595; 0xc883; 0x37d1; 0xff28;
	    0xddef; 0x5aa5; 0x2185; 0x9802;
	    0xa50f; 0x953b; 0x7dad; 0x2f84;
	    0xb628; 0x6170; 0x4775; 0x1510;
	    0xa830; 0xbd96; 0xfe1e; 0x63cf;
	    0x5c90; 0xa239; 0x9e0b; 0xde14;
	    0x86bc; 0x2ca7; 0x5cab; 0x846e;
	    0x1eaf; 0xf0ca; 0x69b9; 0xbb50;
	    0x5a32; 0xb4b3; 0xe9d5; 0xb8f7;
	    0x0b19; 0xa099; 0x997e; 0x7da8;
	    0x889a; 0x2d77; 0x935f; 0x1281;
	    0x8829; 0x1fd6; 0xdfa1; 0xba99;
	    0x84a5; 0x7263; 0xc3ff; 0x4696;
	    0x0aeb; 0x3054; 0x48e4; 0x3128;
	    0xf2ef; 0xffea; 0xed61; 0x3c73;
	    0x14d9; 0xb7e3; 0x5d14; 0x13e0;
	    0xe2b6; 0xabea; 0x4f15; 0x4fd0;
	    0xf442; 0xbbb5; 0x3b1d; 0x2105;
	    0x799e; 0x4dc7; 0x476a; 0x6250;
	    0xa1f2; 0x2646; 0x83a0; 0xb6a3;
	    0x24c3; 0x7492; 0x8a0b; 0xb285;
	    0xbf00; 0x489d; 0xb174; 0x0e00;
	    0x8d2a; 0xf5ea; 0xf43e; 0x7061;
	    0xf092; 0x7e41; 0xecf1; 0x3bdb;
	    0x3759; 0x7460; 0xf2a7; 0x326e;
	    0x8084; 0x509e; 0xd855; 0x9735;
	    0xa7aa; 0x06c2; 0xabfc; 0xcadc;
	    0x7a2e; 0x3484; 0x6705; 0x9ec9;
	    0xdbd3; 0x88cd; 0xda79; 0x4340;
	    0x3465; 0x38d8; 0xf89e; 0xff20;
	    0x21e7; 0x3d4a; 0x9f2b; 0xadf7; |];
	  s2_msb = 
          [| 0x4b7a; 0xb5b3; 0xdb75; 0xc419;
	    0xad6e; 0x49a7; 0x9cee; 0x8fed;
	    0xecaa; 0x699a; 0x5664; 0xc2b1;
	    0x1936; 0x7509; 0xa059; 0xe418;
	    0x3f54; 0x5b42; 0x6b8f; 0x99f7;
	    0xa1d2; 0xefe8; 0x4d2d; 0xf025;
	    0x4cdd; 0x8470; 0x6382; 0x021e;
	    0x0968; 0x3eba; 0x3c97; 0x6b6a;
	    0x687f; 0x52a0; 0xb79c; 0xaa50;
	    0x3e07; 0x7fde; 0x8e7d; 0x5716;
	    0xb03a; 0xf050; 0xf01c; 0x0200;
	    0xae0c; 0x3cb5; 0x2583; 0xdc09;
	    0xd191; 0x7ca9; 0x9432; 0x22f5;
	    0x3ae5; 0x37c2; 0xc8b5; 0x9af3;
	    0xa944; 0x0fd0; 0xecc8; 0xa475;
	    0xe238; 0x3bea; 0x3280; 0x183e;
	    0x4e54; 0x4f6d; 0x6f42; 0xf60a;
	    0x2cb8; 0x2497; 0x5679; 0xbcaf;
	    0xde9a; 0xd993; 0xb38b; 0xdccf;
	    0x5512; 0x2e6b; 0x501a; 0x9f84;
	    0x7a58; 0x7408; 0xbc9f; 0xe94b;
	    0xec7a; 0xdb85; 0x6309; 0xc464;
	    0xef1c; 0x3215; 0xdd43; 0x24c2;
	    0x12a1; 0x2a65; 0x5094; 0x133a;
	    0x71df; 0x1031; 0x81ac; 0x5f11;
	    0x0435; 0xd7a3; 0x3c11; 0x5924;
	    0xf28f; 0x97f1; 0x9eba; 0x1e15;
	    0x86e3; 0xeae9; 0x860e; 0x5a3e;
	    0x771f; 0x4e3d; 0x2965; 0x99e7;
	    0x803e; 0x5266; 0x2e4c; 0x9c10;
	    0xc615; 0x94e2; 0xa5fc; 0x1e0a;
	    0xf2f7; 0x361d; 0x1939; 0x19c2;
	    0x5223; 0xf713; 0xebad; 0xeac3;
	    0xe3bc; 0xa67b; 0xb17f; 0x018c;
	    0xc332; 0xbe6c; 0x6558; 0x68ab;
	    0xeece; 0xdb2f; 0x2aef; 0x5b6e;
	    0x1521; 0x2907; 0xecdd; 0x619f;
	    0x13cc; 0xeb61; 0x0334; 0xaa03;
	    0xb573; 0x4c70; 0xd59e; 0xcbaa;
	    0xeecc; 0x6062; 0x9cab; 0xb2f3;
	    0x648b; 0x19bd; 0xa023; 0x655a;
	    0x4068; 0x3c2a; 0x319e; 0xc021;
	    0x9b54; 0x875f; 0x95f7; 0x623d;
	    0xf837; 0x97e3; 0x11ed; 0x1668;
	    0x0e35; 0xc7e6; 0x96de; 0x7858;
	    0x57f5; 0x1b22; 0x9b83; 0x1ac2;
	    0xcdb3; 0x532e; 0x8fd9; 0x6dbc;
	    0x58eb; 0x34c6; 0xfe28; 0xee7c;
	    0x5d4a; 0xe864; 0x4210; 0x203e;
	    0x45ee; 0xa3aa; 0xdb6c; 0xfacb;
	    0xc742; 0xef6a; 0x654f; 0x41cd;
	    0xd81e; 0x8685; 0xe44b; 0x3d81;
	    0xcf62; 0x5b8d; 0xfc88; 0xc1c7;
	    0x7f15; 0x69cb; 0x4784; 0x5692;
	    0x095b; 0xad19; 0x1462; 0x2382;
	    0x5842; 0x0c55; 0x1dad; 0x233f;
	    0x3372; 0x8d93; 0xd65f; 0x6c22;
	    0x7cde; 0xcbee; 0x4085; 0xce77;
	    0xa607; 0x19f8; 0xe8ef; 0x61d9;
	    0xa969; 0xc50c; 0x5a04; 0x800b;
	    0x9e44; 0xc345; 0xfdd5; 0x0e1e;
	    0xdb73; 0x1055; 0x675f; 0xe367;
	    0xc5c4; 0x713e; 0x3d28; 0xf16d;
	    0x153e; 0x8fb0; 0xe6e3; 0xdb83; |];
	  s3_lsb =
          [| 0x5a68; 0x40f7; 0x261c; 0x2934;
	    0x20f7; 0xd4f7; 0x6b2e; 0x0068;
	    0x2471; 0xf46a; 0xd4b7; 0x61af;
	    0xf62e; 0x4546; 0x4f74; 0x8840;
	    0xfc1d; 0x91af; 0xddd3; 0x2f45;
	    0x09ec; 0x9785; 0x6dd0; 0x8504;
	    0x27b3; 0x3941; 0x47e6; 0x0a9a;
	    0x7825; 0x29f4; 0x86da; 0x6dfb;
	    0x1462; 0x6900; 0xc0a4; 0x8dee;
	    0xfea2; 0xad8c; 0xe006; 0xd6b6;
	    0x1e7c; 0x5fec; 0xa399; 0x2a42;
	    0x9e35; 0x85b9; 0xd7ab; 0x4e8b;
	    0xfaf7; 0x1856; 0x6631; 0x97b2;
	    0xfa74; 0x4332; 0xe7f7; 0x20fb;
	    0xf54e; 0xb397; 0x56ac; 0x9527;
	    0x3a3a; 0x8d87; 0xa9b7; 0x954b;
	    0x67bc; 0x9a58; 0x2963; 0xdb33;
	    0x4a56; 0x25f9; 0x7e1c; 0x317c;
	    0xe802; 0x2f70; 0x155c; 0x2ce3;
	    0x1548; 0x6d22; 0x133f; 0x86dc;
	    0xc9ee; 0x1f0f; 0x79a4; 0x6e17;
	    0x51eb; 0xc0d1; 0xc18f; 0x3564;
	    0x7834; 0x9c60; 0xe8a3; 0x6c1b;
	    0xb4c2; 0x329e; 0x4fd1; 0x8115;
	    0x95e0; 0x92e1; 0x0b62; 0xb922;
	    0xa20e; 0x0d99; 0x0c8c; 0xf728;
	    0x7845; 0x94fd; 0x0862; 0xf5f0;
	    0xa36f; 0x48fa; 0xfd27; 0x8d1e;
	    0x6341; 0xff74; 0x6eab; 0xfd37;
	    0xdc60; 0xddf8; 0xe14c; 0x6b0d;
	    0x5510; 0x2c37; 0xd43b; 0xe804;
	    0x0dc7; 0xffa3; 0x0f92; 0xed0b;
	    0x9ffb; 0x7d9c; 0xcf0b; 0x5ea3;
	    0x2f88; 0xad24; 0x79bf; 0xd6eb;
	    0x2eb3; 0x5979; 0xe297; 0x312d;
	    0xada7; 0x2b3b; 0x4ccc; 0xf11c;
	    0x4237; 0x51e7; 0xbbe6; 0x6350;
	    0x1018; 0xedfa; 0xbdd8; 0xc3c9;
	    0x1659; 0x1386; 0xec6e; 0xea2a;
	    0x674e; 0xa85f; 0xe988; 0xc3fe;
	    0x8057; 0xc086; 0x7bf8; 0x604d;
	    0x8346; 0x1fb0; 0xae04; 0xfccc;
	    0x6b33; 0xab71; 0x4187; 0x5e5f;
	    0x57be; 0xae24; 0x4299; 0x2e61;
	    0xf48f; 0xfda2; 0xef38; 0xbdc2;
	    0xf9c3; 0x8e74; 0xf255; 0xd9b9;
	    0x2661; 0xdf84; 0x0e79; 0x95e2;
	    0x598e; 0x5770; 0x5591; 0xde4c;
	    0xace1; 0x05d0; 0x6248; 0xa99e;
	    0x19b6; 0xdc09; 0x09a1; 0x4633;
	    0x1f02; 0xbe8c; 0xa025; 0xfe10;
	    0x3d1d; 0xa4df; 0xf20f; 0xf169;
	    0xda83; 0x06fe; 0xce9b; 0x7f52;
	    0x5e01; 0x83fa; 0xb5c4; 0xd027;
	    0x8c27; 0x8641; 0x4c06; 0x06b5;
	    0x7a28; 0x86e0; 0x58aa; 0x7d62;
	    0x9ed7; 0xea63; 0xdd94; 0x1634;
	    0xee56; 0xb6de; 0x7da1; 0x1d76;
	    0xe409; 0x0188; 0x0a3d; 0x7c24;
	    0x725f; 0x9db9; 0x5bb4; 0xb8fc;
	    0x5578; 0xa5b5; 0x7cd3; 0x0fc4;
	    0xef5e; 0xe6f8; 0x14d9; 0x133c;
	    0xc7e7; 0x4ec4; 0xbfce; 0xc837;
	    0x3234; 0x8212; 0xfa8e; 0x00e0; |];
	  s3_msb =
          [| 0xe93d; 0x9481; 0xf64c; 0x9469;
	    0x4115; 0x7602; 0xbcf4; 0xd4a2;
	    0xd408; 0x3320; 0x43b7; 0x5000;
	    0x1e39; 0x9724; 0x1421; 0xbf8b;
	    0x4d95; 0x96b5; 0x70f4; 0x66a0;
	    0xbfbc; 0x03bd; 0x7fac; 0x31cb;
	    0x96eb; 0x55fd; 0xda25; 0xabca;
	    0x2850; 0x5304; 0x0a2c; 0xe9b6;
	    0x68dc; 0xd748; 0x680e; 0x27a1;
	    0x4f3f; 0xe887; 0xb58c; 0x7af4;
	    0xaace; 0xd337; 0xce78; 0x406b;
	    0x20fe; 0xd9f3; 0xee39; 0x3b12;
	    0x1dc9; 0x4b6d; 0x26a3; 0xeae3;
	    0x3a6e; 0xdd5b; 0x6841; 0xca78;
	    0xfb0a; 0xd8fe; 0x4540; 0xba48;
	    0x5553; 0x2083; 0xfe6b; 0xd096;
	    0x55a8; 0xa115; 0xcca9; 0x99e1;
	    0xa62a; 0x3f31; 0x5ef4; 0x9029;
	    0xfdf8; 0x0427; 0x80bb; 0x0528;
	    0x95c1; 0xe4c6; 0x48c1; 0xc70f;
	    0x07f9; 0x4104; 0x4047; 0x5d88;
	    0x325f; 0xd59b; 0xf2bc; 0x4111;
	    0x257b; 0x602a; 0xdff8; 0x1f63;
	    0x0e12; 0x02e1; 0xaf66; 0xcad1;
	    0x6b23; 0x333e; 0x3b24; 0xeebe;
	    0x85b2; 0xe6ba; 0xde72; 0x2da2;
	    0xd012; 0x95b7; 0x647d; 0xe7cc;
	    0x5449; 0x877d; 0xc39d; 0xf33e;
	    0x0a47; 0x992e; 0x3a6f; 0xf4f8;
	    0xa812; 0xa1eb; 0x991b; 0xdb6e;
	    0xc67b; 0x6d67; 0x2765; 0xdcd0;
	    0xf129; 0xcc00; 0xb539; 0x690f;
	    0x667b; 0xcedb; 0xa091; 0xd915;
	    0xbb13; 0x515b; 0x7b94; 0x763b;
	    0x3739; 0xcc11; 0x8026; 0xf42e;
	    0x6842; 0xc66a; 0x1275; 0x782e;
	    0x6a12; 0xb792; 0x06a1; 0x4bfb;
	    0x1a6b; 0x11ca; 0x3d25; 0xe2e1;
	    0x4442; 0x0a12; 0xd90c; 0xd5ab;
	    0x64af; 0xda86; 0xbebf; 0x64e4;
	    0x9dbc; 0xf0f7; 0x6078; 0x6003;
	    0xd1fd; 0xf638; 0x7745; 0xd736;
	    0x8342; 0xf01e; 0xb080; 0x3c00;
	    0x77a0; 0xbde8; 0x5546; 0xbf58;
	    0x4e58; 0xf2dd; 0xf474; 0x8789;
	    0x5366; 0xc8b3; 0xb475; 0x46fc;
	    0x7aeb; 0x8b1d; 0x846a; 0x915f;
	    0x466e; 0x20b4; 0x8cd5; 0xc902;
	    0xb90b; 0xbb82; 0x11a8; 0x7574;
	    0xb77f; 0xe0a9; 0x662d; 0xc432;
	    0xe85a; 0x09f0; 0x4a99; 0x1d6e;
	    0x1ab9; 0x0ba5; 0xa186; 0x2868;
	    0xdcb7; 0x5739; 0xa1e2; 0x4fcd;
	    0x5011; 0xa706; 0xa002; 0x0de6;
	    0x9af8; 0x773f; 0xc360; 0x61a8;
	    0xf017; 0xc0f5; 0x0060; 0x30dc;
	    0x11e6; 0x2338; 0x53c2; 0xc2c2;
	    0xbbcb; 0x90bc; 0xebfc; 0xce59;
	    0x6f05; 0x4b7c; 0x3972; 0x7c92;
	    0x86e3; 0x724d; 0x1ac1; 0xd39e;
	    0xed54; 0x08fc; 0xd83d; 0x4dad;
	    0x1e50; 0xb161; 0xa285; 0x6c51;
	    0x6fd5; 0x56e1; 0x362a; 0xddc6;
	    0xd79a; 0x9263; 0x670e; 0x4060; |];
	  s4_lsb =
          [| 0xce37; 0xf5cf; 0x7737; 0x2d1b;
	    0x679e; 0x3742; 0x2740; 0x9bbe;
	    0x8e9d; 0x7315; 0x1c7e; 0xc47b;
	    0x1b6b; 0x9045; 0xb1be; 0x6eb4;
	    0xab2f; 0x6e79; 0x76d2; 0xc2c8;
	    0xf8ee; 0xde7d; 0x0a1d; 0x4dc6;
	    0xbbdb; 0x4650; 0x26e8; 0xe304;
	    0xd5f0; 0x519a; 0x8ce2; 0xee22;
	    0xc2b8; 0x2ef6; 0x03aa; 0xd0a4;
	    0x61ba; 0x6a4d; 0x1550; 0x5bd6;
	    0xa2f9; 0x3ae1; 0x9586; 0x62e9;
	    0xefd3; 0xf7da; 0x6f69; 0x0a59;
	    0xa915; 0x8601; 0xe6ad; 0xe593;
	    0xfd5a; 0xd797; 0xb7d9; 0x8b51;
	    0xac3a; 0xa67d; 0x3ed6; 0x2d28;
	    0x25cf; 0xb89b; 0xb472; 0xf54c;
	    0xac71; 0xa5e6; 0xacfd; 0xfa9b;
	    0xc48d; 0x57cc; 0x6629; 0x2e28;
	    0x0191; 0x6055; 0x0e44; 0x5e8c;
	    0x6dd4; 0x6dba; 0x6125; 0xf0bd;
	    0x9e15; 0x57a2; 0x1aec; 0x072a;
	    0x6d9b; 0x21f5; 0x66fb; 0xf319;
	    0xd928; 0xfdf5; 0x3482; 0x3cbb;
	    0x7711; 0xd9f8; 0x5167; 0x925f;
	    0x1751; 0xdc8e; 0x5862; 0xf991;
	    0x90c2; 0x7bce; 0xce64; 0xbe32;
	    0xe37e; 0x3d46; 0x5369; 0xe680;
	    0x0810; 0xb224; 0x2dfd; 0x2166;
	    0x460a; 0xc0dd; 0xdecf; 0xc8ae;
	    0xf7dd; 0x8d40; 0x017f; 0xe3bb;
	    0x6a7e; 0xff45; 0x0a44; 0xcdd5;
	    0xcea8; 0x84bb; 0x12ae; 0x6f47;
	    0xe463; 0x5d9e; 0x771b; 0x6370;
	    0x0d8d; 0x1357; 0x1671; 0x7d5d;
	    0xcb08; 0xe2cc; 0x466a; 0xaf84;
	    0x0428; 0x3a1d; 0x9fb4; 0xa048;
	    0x3b82; 0xab82; 0x1d4b; 0x27f8;
	    0x60b1; 0x3fdc; 0x792b; 0x25bd;
	    0x39e1; 0x794b; 0xc9b7; 0xbac9;
	    0xc87e; 0xd1f6; 0x11c3; 0xaac7;
	    0x8749; 0xbd9a; 0xdecb; 0xda38;
	    0xc32a; 0x3667; 0x317c; 0x2b4f;
	    0x59b7; 0xbb3a; 0x19ff; 0x459c;
	    0x222c; 0xfc2a; 0xfc71; 0x1525;
	    0x9361; 0x9ceb; 0x6459; 0xa8d1;
	    0x075e; 0x6a0c; 0x5065; 0xa442;
	    0x6e0e; 0xdb3b; 0xa0be; 0xe964;
	    0x9532; 0x92df; 0x342b; 0xf21e;
	    0x7441; 0x348c; 0x7120; 0x32d8;
	    0x9f8d; 0x2f2e; 0x6f47; 0xf11d;
	    0xda54; 0xd891; 0x79cf; 0x7e6f;
	    0xb166; 0x1d05; 0xd2c5; 0x2299;
	    0xf357; 0x7623; 0x3531; 0xcd02;
	    0x8162; 0xebb5; 0x3697; 0x73cc;
	    0x6292; 0x49d0; 0x901b; 0x5614;
	    0xc7bd; 0x140a; 0xd006; 0x7b9a;
	    0x53fd; 0x0f00; 0xbfe2; 0xd2f6;
	    0x6905; 0x0222; 0xcf7c; 0x9c2b;
	    0x3ec0; 0xe3d3; 0xbd60; 0xadf0;
	    0x209c; 0xce76; 0xa1c5; 0x6060;
	    0xfe4e; 0x8dd8; 0xf9b0; 0xaa7e;
	    0xc25c; 0x8a8c; 0x6ae4; 0xe1f9;
	    0xf869; 0xdea0; 0x252d; 0xe69f;
	    0x6132; 0xe25b; 0xdfe3; 0x72e6; |];
	  s4_msb = 
          [| 0x3a39; 0xd3fa; 0xabc2; 0x5ac5;
	    0x5cb0; 0x4fa3; 0xd382; 0x99bc;
	    0xd511; 0xbf0f; 0xd62d; 0xc700;
	    0xb78c; 0x21a1; 0xb26e; 0x6a36;
	    0x5748; 0xbc94; 0xc6a3; 0x6549;
	    0x530f; 0x468d; 0xd573; 0x4cd0;
	    0x2939; 0xa9ba; 0xac95; 0xbe5e;
	    0xa1fa; 0x6a2d; 0x63ef; 0x9a86;
	    0xc089; 0x4324; 0xa51e; 0x9cf2;
	    0x83c0; 0x9be9; 0x8fe5; 0xba64;
	    0x2826; 0xa73a; 0x4ba9; 0xef55;
	    0xc72f; 0xf752; 0x3f04; 0x77fa;
	    0x80e4; 0x87b0; 0x9b09; 0x3b3e;
	    0xe990; 0x9e34; 0x2cf0; 0x022b;
	    0x96d5; 0x017d; 0xd1cf; 0x7c7d;
	    0x1f9f; 0xadf2; 0x5ad6; 0x5a88;
	    0xe029; 0xe019; 0x47b0; 0xed93;
	    0xe8d3; 0x283b; 0xf8d5; 0x7913;
	    0x785f; 0xed75; 0xf796; 0xe3d3;
	    0x1505; 0x88f4; 0x03a1; 0x0564;
	    0xc3eb; 0x3c90; 0x9727; 0xa93a;
	    0x1b3f; 0x1e63; 0xf59c; 0x26dc;
	    0x7533; 0xb155; 0x0356; 0x8aba;
	    0x2851; 0xc20a; 0xabcc; 0xccad;
	    0x4de8; 0x3830; 0x379d; 0x9320;
	    0xea7a; 0xfb3e; 0x5121; 0x774f;
	    0xa8b6; 0xc329; 0x48de; 0x6413;
	    0xa2ae; 0xdd6d; 0x6985; 0x0907;
	    0xb39a; 0x6445; 0x586c; 0x1c20;
	    0x5bbe; 0x1b58; 0xccd2; 0x6bb4;
	    0xdda2; 0x3a59; 0x3e35; 0xbcb4;
	    0x72ea; 0xfa64; 0x8d66; 0xbf3c;
	    0xd29b; 0x542f; 0xaec2; 0xf64e;
	    0x740e; 0xe75b; 0xf872; 0xaf53;
	    0x4040; 0x4eb4; 0x34d2; 0x0115;
	    0xe1b0; 0x9598; 0x06b8; 0xce6e;
	    0x6f3f; 0x3520; 0x011a; 0x2772;
	    0x6115; 0xe793; 0xbb3a; 0x3445;
	    0xa088; 0x51ce; 0x2f32; 0xa01f;
	    0xe01c; 0xbcc7; 0xcf01; 0xa1e8;
	    0x1a90; 0xd44f; 0xd0da; 0xd50a;
	    0x0339; 0xc691; 0x8df9; 0xe0b1;
	    0xf79e; 0x43f5; 0xf2d5; 0x27d9;
	    0xbf97; 0x15e6; 0x0f91; 0x9b94;
	    0xfae5; 0xceb6; 0xc2a8; 0x12ba;
	    0xb6c1; 0xe305; 0x10d2; 0xcb03;
	    0xe0ec; 0x1698; 0x4c98; 0x3278;
	    0x9f1f; 0xe0d3; 0xd3a0; 0x8971;
	    0x1b0a; 0x4ba3; 0xc5be; 0xc376;
	    0xdf35; 0x9b99; 0xe60b; 0x0fe3;
	    0xe54c; 0x1eda; 0xce62; 0xcd3e;
	    0x1618; 0xfd2c; 0x848f; 0xf6fb;
	    0xf523; 0xa632; 0x93a8; 0x56cc;
	    0xacf0; 0x5a75; 0x6e16; 0x88d2;
	    0xde96; 0x81b9; 0x4c50; 0x71c6;
	    0xe6c6; 0x327a; 0x45e1; 0xc3f2;
	    0xc9aa; 0x62a8; 0xbb25; 0x35bd;
	    0x7112; 0xb204; 0xb6cb; 0xcd76;
	    0x5311; 0x1640; 0x38ab; 0x2547;
	    0xba38; 0xf746; 0x77af; 0x2075;
	    0x85cb; 0x8ae8; 0x7aaa; 0x4cf9;
	    0x1948; 0x02fb; 0x01c3; 0xd6eb;
	    0x90d4; 0xa65c; 0x3f09; 0xc208;
	    0xb74e; 0xce77; 0x578f; 0x3ac3; |];
	} in
      
      let j = ref 0 in
      for i = 0 to 17 do
	let k0 = Char.code(key.[ !j ]) in
	let k1 = Char.code(key.[ (!j + 1) mod l_key ]) in
	let k2 = Char.code(key.[ (!j + 2) mod l_key ]) in
	let k3 = Char.code(key.[ (!j + 3) mod l_key ]) in
	j := (!j + 4) mod l_key;
	let d_msb = ( k0 lsl 8 ) lor k1 in
	let d_lsb = ( k2 lsl 8 ) lor k3 in
	k.p_lsb.( i ) <- k.p_lsb.( i ) lxor d_lsb;
	k.p_msb.( i ) <- k.p_msb.( i ) lxor d_msb
      done;

      let d = ref (0,0,0,0) in

      for i = 0 to 8 do
	d := encrypt_ecb k !d;
	let (dl_msb, dl_lsb, dr_msb, dr_lsb) = !d in
	k.p_lsb.( 2*i ) <- dl_lsb;
	k.p_msb.( 2*i ) <- dl_msb;
	k.p_lsb.( 2*i+1 ) <- dr_lsb;
	k.p_msb.( 2*i+1 ) <- dr_msb;
      done;

      for i = 0 to 127 do
	d := encrypt_ecb k !d;
	let (dl_msb, dl_lsb, dr_msb, dr_lsb) = !d in
	k.s1_lsb.( 2*i ) <- dl_lsb;
	k.s1_msb.( 2*i ) <- dl_msb;
	k.s1_lsb.( 2*i+1 ) <- dr_lsb;
	k.s1_msb.( 2*i+1 ) <- dr_msb;
      done;

      for i = 0 to 127 do
	d := encrypt_ecb k !d;
	let (dl_msb, dl_lsb, dr_msb, dr_lsb) = !d in
	k.s2_lsb.( 2*i ) <- dl_lsb;
	k.s2_msb.( 2*i ) <- dl_msb;
	k.s2_lsb.( 2*i+1 ) <- dr_lsb;
	k.s2_msb.( 2*i+1 ) <- dr_msb;
      done;
      
      for i = 0 to 127 do
	d := encrypt_ecb k !d;
	let (dl_msb, dl_lsb, dr_msb, dr_lsb) = !d in
	k.s3_lsb.( 2*i ) <- dl_lsb;
	k.s3_msb.( 2*i ) <- dl_msb;
	k.s3_lsb.( 2*i+1 ) <- dr_lsb;
	k.s3_msb.( 2*i+1 ) <- dr_msb;
      done;

      for i = 0 to 127 do
	d := encrypt_ecb k !d;
	let (dl_msb, dl_lsb, dr_msb, dr_lsb) = !d in
	k.s4_lsb.( 2*i ) <- dl_lsb;
	k.s4_msb.( 2*i ) <- dl_msb;
	k.s4_lsb.( 2*i+1 ) <- dr_lsb;
	k.s4_msb.( 2*i+1 ) <- dr_msb;
      done;
      
      for i = 0 to 17 do
	k.p_lsb_rev.( i ) <- k.p_lsb.( 17-i );
	k.p_msb_rev.( i ) <- k.p_msb.( 17-i );
      done;
      
      k


    let textkey k = k.data

    let is_weak k = 
      (* A weak key is one in which two entries for a given S-box are identical
       *)
      (* Time: check takes 129540 loops. *)
      let check s_lsb s_msb =
	for i=0 to 254 do
	  let a_lsb = s_lsb.(i) in
	  let a_msb = s_msb.(i) in
	  for j=i+1 to 255 do
	    if a_lsb = s_lsb.(j) & a_msb = s_msb.(j) then
	      raise Not_found
	  done
	done;
	()
      in
      try
	check k.s1_lsb k.s1_msb;
	check k.s2_lsb k.s2_msb;
	check k.s3_lsb k.s3_msb;
	check k.s4_lsb k.s4_msb;
	false
      with
	Not_found -> true

  end
;;



module Cryptmodes = Cryptmodes_64.Make_modes(Cryptsystem)
;;


(* ======================================================================
 * History:
 * 
 * $Log: crypt_blowfish.ml,v $
 * Revision 1.2  2001/03/10 16:43:21  gerd
 * 	int32 experiments
 *
 * Revision 1.1  1999/06/04 20:42:00  gerd
 * 	Initial revision.
 *
 * 
 *)
