//go:build !gitaly_test_sha256

package conflicts

import (
	"context"
	"crypto/sha1"
	"encoding/json"
	"fmt"
	"strings"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitaly/v16/internal/git"
	"gitlab.com/gitlab-org/gitaly/v16/internal/git/gittest"
	"gitlab.com/gitlab-org/gitaly/v16/internal/git/localrepo"
	"gitlab.com/gitlab-org/gitaly/v16/internal/gitaly/config"
	"gitlab.com/gitlab-org/gitaly/v16/internal/helper/text"
	"gitlab.com/gitlab-org/gitaly/v16/internal/structerr"
	"gitlab.com/gitlab-org/gitaly/v16/internal/testhelper"
	"gitlab.com/gitlab-org/gitaly/v16/internal/testhelper/testcfg"
	"gitlab.com/gitlab-org/gitaly/v16/proto/go/gitalypb"
	"google.golang.org/grpc/metadata"
	"google.golang.org/protobuf/types/known/timestamppb"
)

var (
	user = &gitalypb.User{
		Name:  []byte("John Doe"),
		Email: []byte("johndoe@gitlab.com"),
		GlId:  "user-1",
	}
	conflictResolutionCommitMessage = "Solve conflicts"
)

func TestResolveConflicts(t *testing.T) {
	type setupData struct {
		cfg               config.Cfg
		requestHeader     *gitalypb.ResolveConflictsRequest_Header
		requestsFilesJSON []*gitalypb.ResolveConflictsRequest_FilesJson
		client            gitalypb.ConflictsServiceClient
		repo              *gitalypb.Repository
		repoPath          string
		expectedContent   map[string]map[string][]byte
		expectedResponse  *gitalypb.ResolveConflictsResponse
		expectedError     error
		skipCommitCheck   bool
		additionalChecks  func()
	}

	for _, tc := range []struct {
		desc  string
		setup func(testing.TB, context.Context) setupData
	}{
		{
			"single file conflict, pick ours",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				baseCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithTreeEntries(
					gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apple"},
				))

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON[:50]},
						{FilesJson: filesJSON[50:]},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("apricot"),
						},
					},
				}
			},
		},
		{
			"single file conflict, pick theirs",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				baseCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithTreeEntries(
					gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apple"},
				))

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1): "origin",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON[:50]},
						{FilesJson: filesJSON[50:]},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("acai"),
						},
					},
				}
			},
		},
		{
			"single file conflict without ancestor, pick ours",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON[:50]},
						{FilesJson: filesJSON[50:]},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("apricot"),
						},
					},
				}
			},
		},
		{
			"single file multi conflict",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				baseCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithTreeEntries(
					gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apple\n" + strings.Repeat("filler\n", 10) + "banana"},
				))

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{
						Path: "a", Mode: "100644",
						Content: "apricot\n" + strings.Repeat("filler\n", 10) + "berries",
					}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{
						Path: "a", Mode: "100644",
						Content: "acai\n" + strings.Repeat("filler\n", 10) + "birne",
					}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1):   "head",
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 12, 12): "origin",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON[:50]},
						{FilesJson: filesJSON[50:]},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("apricot\n" + strings.Repeat("filler\n", 10) + "birne"),
						},
					},
				}
			},
		},
		{
			"multi file multi conflict",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				baseCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithTreeEntries(
					gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apple\n" + strings.Repeat("filler\n", 10) + "banana"},
					gittest.TreeEntry{Path: "b", Mode: "100644", Content: "strawberry"},
				))

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{
						Path: "a", Mode: "100644",
						Content: "apricot\n" + strings.Repeat("filler\n", 10) + "berries",
					}, gittest.TreeEntry{Path: "b", Mode: "100644", Content: "blueberry"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommitID), gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{
						Path: "a", Mode: "100644",
						Content: "acai\n" + strings.Repeat("filler\n", 10) + "birne",
					}, gittest.TreeEntry{Path: "b", Mode: "100644", Content: "raspberry"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1):   "head",
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 12, 12): "origin",
						},
					},
					{
						"old_path": "b",
						"new_path": "b",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("b")), 1, 1): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON[:50]},
						{FilesJson: filesJSON[50:]},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("apricot\n" + strings.Repeat("filler\n", 10) + "birne"),
							"b": []byte("blueberry"),
						},
					},
				}
			},
		},
		{
			"single file conflict, remote repo",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				testcfg.BuildGitalySSH(t, cfg)

				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)
				targetRepo, targetRepoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, targetRepoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: targetRepo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON[:50]},
						{FilesJson: filesJSON[50:]},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("apricot"),
						},
					},
				}
			},
		},
		{
			"single file with only newline",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "\n"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "\n"}))

				files := []map[string]interface{}{}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("\n"),
						},
					},
				}
			},
		},
		{
			"conflicting newline with embedded character",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "\nA\n"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "\nB\n"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 2, 2): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("\nA\n"),
						},
					},
				}
			},
		},
		{
			"conflicting carriage-return newlines",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "A\r\nB\r\nC\r\nD\r\nE\r\n"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "A\r\nB\r\nX\r\nD\r\nE\r\n"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 3, 3): "origin",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("A\r\nB\r\nX\r\nD\r\nE\r\n"),
						},
					},
				}
			},
		},
		{
			"conflict with no trailing newline",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "A\nB"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "X\nB"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("A\nB"),
						},
					},
				}
			},
		},
		{
			"conflict with existing conflict markers",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "<<<<<<< HEAD\nA\nB\n======="}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "X\nB"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedError:   structerr.NewInternal(`resolve: parse conflict for "a": unexpected conflict delimiter`),
					skipCommitCheck: true,
				}
			},
		},
		{
			"invalid OID",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     "conflict-resolvable",
							TheirCommitOid:   "conflict-start",
							SourceBranch:     []byte("ours"),
							TargetBranch:     []byte("theirs"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedError:   structerr.NewInternal("Rugged::InvalidError: unable to parse OID - contains invalid characters"),
					skipCommitCheck: true,
				}
			},
		},
		{
			"resolved content is same as the conflict",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"content":  "<<<<<<< a\napricot\n=======\nacai\n>>>>>>> a\n",
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{ResolutionError: "Resolved content has no changes for file a"},
					skipCommitCheck:  true,
				}
			},
		},
		{
			"missing resolution for section",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 4, 4): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedResponse: &gitalypb.ResolveConflictsResponse{
						ResolutionError: fmt.Sprintf("Missing resolution for section ID: %x_%d_%d", sha1.Sum([]byte("a")), 1, 1),
					},
					skipCommitCheck: true,
				}
			},
		},
		{
			"empty User",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TheirCommitOid:   gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError:   structerr.NewInvalidArgument("empty User"),
				}
			},
		},
		{
			"empty Repository",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							User:             user,
							TargetRepository: repo,
							OurCommitOid:     gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TheirCommitOid:   gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError: structerr.NewInvalidArgument(testhelper.GitalyOrPraefect(
						"repository not set",
						"repo scoped: repository not set",
					)),
				}
			},
		},
		{
			"empty TargetRepository",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							User:           user,
							Repository:     repo,
							OurCommitOid:   gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TheirCommitOid: gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TargetBranch:   []byte("theirs"),
							SourceBranch:   []byte("ours"),
							CommitMessage:  []byte(conflictResolutionCommitMessage),
							Timestamp:      &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError:   structerr.NewInvalidArgument("empty TargetRepository"),
				}
			},
		},
		{
			"empty OurCommitID",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							User:             user,
							Repository:       repo,
							TargetRepository: repo,
							TheirCommitOid:   gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError:   structerr.NewInvalidArgument("empty OurCommitOid"),
				}
			},
		},
		{
			"empty TheirCommitID",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							User:             user,
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError:   structerr.NewInvalidArgument("empty TheirCommitOid"),
				}
			},
		},
		{
			"empty CommitMessage",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							User:             user,
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TheirCommitOid:   gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError:   structerr.NewInvalidArgument("empty CommitMessage"),
				}
			},
		},
		{
			"empty SourceBranch",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							User:             user,
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TheirCommitOid:   gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TargetBranch:     []byte("theirs"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError:   structerr.NewInvalidArgument("empty SourceBranch"),
				}
			},
		},
		{
			"empty TargetBranch",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							User:             user,
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     gittest.DefaultObjectHash.EmptyTreeOID.String(),
							TheirCommitOid:   gittest.DefaultObjectHash.EmptyTreeOID.String(),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					skipCommitCheck: true,
					expectedError:   structerr.NewInvalidArgument("empty TargetBranch"),
				}
			},
		},
		{
			"uses quarantine repo",
			func(tb testing.TB, ctx context.Context) setupData {
				cfg, client := setupConflictsService(tb, nil)
				repo, repoPath := gittest.CreateRepository(tb, ctx, cfg)

				testcfg.BuildGitalySSH(t, cfg)
				testcfg.BuildGitalyHooks(t, cfg)

				// We set up a custom "pre-receive" hook which simply prints the commits content to stdout and
				// then exits with an error. Like this, we can both assert that the hook can see the
				// quarantined tag, and it allows us to fail the RPC before we migrate quarantined objects.
				gittest.WriteCustomHook(t, repoPath, "pre-receive", []byte(
					`#!/bin/sh
						read oldval newval ref &&
						git cat-file -p $newval^{commit}:a &&
						exit 1
					`))

				baseCommit := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("ours"),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apple"}))
				ourCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithParents(baseCommit),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "apricot"}))
				theirCommitID := gittest.WriteCommit(tb, cfg, repoPath, gittest.WithBranch("theirs"),
					gittest.WithParents(baseCommit),
					gittest.WithTreeEntries(gittest.TreeEntry{Path: "a", Mode: "100644", Content: "acai"}))

				files := []map[string]interface{}{
					{
						"old_path": "a",
						"new_path": "a",
						"sections": map[string]string{
							fmt.Sprintf("%x_%d_%d", sha1.Sum([]byte("a")), 1, 1): "head",
						},
					},
				}

				filesJSON, err := json.Marshal(files)
				require.NoError(t, err)

				objectsBefore := len(strings.Split(text.ChompBytes(gittest.Exec(t, cfg, "-C", repoPath, "rev-list", "--objects", "--all")), "\n"))

				return setupData{
					cfg:      cfg,
					client:   client,
					repoPath: repoPath,
					repo:     repo,
					requestHeader: &gitalypb.ResolveConflictsRequest_Header{
						Header: &gitalypb.ResolveConflictsRequestHeader{
							Repository:       repo,
							TargetRepository: repo,
							OurCommitOid:     ourCommitID.String(),
							TheirCommitOid:   theirCommitID.String(),
							TargetBranch:     []byte("theirs"),
							SourceBranch:     []byte("ours"),
							CommitMessage:    []byte(conflictResolutionCommitMessage),
							User:             user,
							Timestamp:        &timestamppb.Timestamp{},
						},
					},
					requestsFilesJSON: []*gitalypb.ResolveConflictsRequest_FilesJson{
						{FilesJson: filesJSON},
					},
					expectedContent: map[string]map[string][]byte{
						"refs/heads/ours": {
							"a": []byte("apple"),
						},
					},
					expectedError:   structerr.NewInternal("running pre-receive hooks: apricot"),
					skipCommitCheck: true,
					additionalChecks: func() {
						objectsAfter := len(strings.Split(text.ChompBytes(gittest.Exec(t, cfg, "-C", repoPath, "rev-list", "--objects", "--all")), "\n"))
						require.Equal(t, objectsBefore, objectsAfter, "No new objets should've been added")
					},
				}
			},
		},
	} {
		tc := tc

		t.Run(tc.desc, func(t *testing.T) {
			t.Parallel()

			ctx := testhelper.Context(t)
			setup := tc.setup(t, ctx)

			mdGS := testcfg.GitalyServersMetadataFromCfg(t, setup.cfg)
			mdFF, _ := metadata.FromOutgoingContext(ctx)
			ctx = metadata.NewOutgoingContext(ctx, metadata.Join(mdGS, mdFF))

			stream, err := setup.client.ResolveConflicts(ctx)
			require.NoError(t, err)

			require.NoError(t, stream.Send(&gitalypb.ResolveConflictsRequest{ResolveConflictsRequestPayload: setup.requestHeader}))
			for _, req := range setup.requestsFilesJSON {
				require.NoError(t, stream.Send(&gitalypb.ResolveConflictsRequest{ResolveConflictsRequestPayload: req}))
			}

			r, err := stream.CloseAndRecv()
			testhelper.RequireGrpcError(t, setup.expectedError, err)
			testhelper.ProtoEqual(t, setup.expectedResponse, r)

			for branch, pathAndContent := range setup.expectedContent {
				for path, content := range pathAndContent {
					actual := gittest.Exec(t, setup.cfg, "-C", setup.repoPath, "cat-file", "-p", fmt.Sprintf("%s:%s", branch, path))
					require.Equal(t, content, actual)
				}
			}

			if setup.requestHeader.Header != nil && !setup.skipCommitCheck {
				repo := localrepo.NewTestRepo(t, setup.cfg, setup.repo)
				headCommit, err := repo.ReadCommit(ctx, git.Revision(setup.requestHeader.Header.SourceBranch))
				require.NoError(t, err)
				require.Contains(t, headCommit.ParentIds, setup.requestHeader.Header.OurCommitOid)
				require.Contains(t, headCommit.ParentIds, setup.requestHeader.Header.TheirCommitOid)
				require.Equal(t, headCommit.Author.Email, user.Email)
				require.Equal(t, headCommit.Committer.Email, user.Email)
				require.Equal(t, string(headCommit.Subject), conflictResolutionCommitMessage)
			}

			if setup.additionalChecks != nil {
				setup.additionalChecks()
			}
		})
	}
}
