package assets

/*
	Sliver Implant Framework
	Copyright (C) 2019  Bishop Fox

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/

import (
	"fmt"
	"os"
	"os/user"
	"path"
	"path/filepath"
	"strings"

	"github.com/bishopfox/sliver/server/log"
	ver "github.com/bishopfox/sliver/server/version"
)

const (
	// GoDirName - The directory to store the go compiler/toolchain files in
	GoDirName = "go"

	goPathDirName   = "gopath"
	versionFileName = "version"
	envVarName      = "SLIVER_ROOT_DIR"
)

var (
	setupLog = log.NamedLogger("assets", "setup")
)

// GetRootAppDir - Get the Sliver app dir, default is: ~/.sliver/
func GetRootAppDir() string {
	value := os.Getenv(envVarName)
	var dir string
	if len(value) == 0 {
		user, _ := user.Current()
		dir = filepath.Join(user.HomeDir, ".sliver")
	} else {
		dir = value
	}

	if _, err := os.Stat(dir); os.IsNotExist(err) {
		err = os.MkdirAll(dir, 0700)
		if err != nil {
			setupLog.Fatalf("Cannot write to sliver root dir %s", err)
		}
	}
	return dir
}

// GetZigDir
func GetZigDir() string {
	zigDir, err := filepath.Abs(filepath.Join(GetRootAppDir(), zigDirName))
	if err != nil {
		setupLog.Errorf("Failed to get Zig directory: %s", err)
		return filepath.Join(GetRootAppDir(), zigDirName)
	}
	return zigDir
}

// GetChunkDataDir - Get the Sliver chunk data dir, default is: ~/.sliver/crack/chunks/
func GetChunkDataDir() string {
	chunkDir := filepath.Join(GetRootAppDir(), "crack", "chunks")
	if _, err := os.Stat(chunkDir); os.IsNotExist(err) {
		err = os.MkdirAll(chunkDir, 0700)
		if err != nil {
			setupLog.Errorf("Failed to create chunk data directory: %s", err)
			return ""
		}
	}
	return chunkDir
}

// GetTrafficEncoderDir - Get the Sliver traffic encoder dir, default is: ~/.sliver/traffic-encoders/
func GetTrafficEncoderDir() string {
	trafficDir := filepath.Join(GetRootAppDir(), "traffic-encoders")
	if _, err := os.Stat(trafficDir); os.IsNotExist(err) {
		os.MkdirAll(trafficDir, 0700)
	}
	return trafficDir
}

func assetVersion() string {
	appDir := GetRootAppDir()
	data, err := os.ReadFile(path.Join(appDir, versionFileName))
	if err != nil {
		setupLog.Infof("No version detected %s", err)
		return ""
	}
	return strings.TrimSpace(string(data))
}

func saveAssetVersion(appDir string) {
	versionFilePath := filepath.Join(appDir, versionFileName)
	fVer, _ := os.Create(versionFilePath)
	defer fVer.Close()
	fVer.Write([]byte(ver.GitCommit))
}

// Setup - Extract or create local assets
func Setup(force bool, echo bool) {
	appDir := GetRootAppDir()
	localVer := assetVersion()
	if force || localVer == "" || localVer != ver.GitCommit {
		setupLog.Infof("Version mismatch %v != %v", localVer, ver.GitCommit)
		if echo {
			fmt.Printf(`
Sliver  Copyright (C) 2026  Bishop Fox
This program comes with ABSOLUTELY NO WARRANTY; for details type 'licenses'.
This is free software, and you are welcome to redistribute it
under certain conditions; type 'licenses' for details.`)
			fmt.Printf("\n\nUnpacking assets ...\n")
		}
		setupGo(appDir)
		err := setupZig(appDir)
		if err != nil {
			setupLog.Errorf("Failed to setup Zig: %s", err)
		}
		setupCodenames(appDir)
		saveAssetVersion(appDir)
		unpackDefaultTrafficEncoders(force)
	}
	setupLog.Infof("Initialized english encoder with %d words", len(English()))
}

// English - Extracts the english dictionary for the english encoder
func English() []string {
	rawEnglish, err := assetsFs.ReadFile("fs/english.txt")
	if err != nil {
		return []string{}
	}
	englishWords := strings.Split(string(rawEnglish), "\n")
	return englishWords
}
