// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkSectorSource.h"

#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkLineSource.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPolyData.h"
#include "vtkRotationalExtrusionFilter.h"
#include "vtkStreamingDemandDrivenPipeline.h"

#include "vtkSmartPointer.h"
#define VTK_CREATE(type, name) vtkSmartPointer<type> name = vtkSmartPointer<type>::New()

VTK_ABI_NAMESPACE_BEGIN
vtkStandardNewMacro(vtkSectorSource);

vtkSectorSource::vtkSectorSource()
{
  this->InnerRadius = 1.0;
  this->OuterRadius = 2.0;
  this->ZCoord = 0.0;
  this->StartAngle = 0.0;
  this->EndAngle = 90.0;
  this->RadialResolution = 1;
  this->CircumferentialResolution = 6;

  this->SetNumberOfInputPorts(0);
}

int vtkSectorSource::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** vtkNotUsed(inputVector), vtkInformationVector* outputVector)
{
  // get the info object
  vtkInformation* outInfo = outputVector->GetInformationObject(0);

  // get the output
  vtkPolyData* output = vtkPolyData::SafeDownCast(outInfo->Get(vtkDataObject::DATA_OBJECT()));

  int piece, numPieces;
  piece = outInfo->Get(vtkStreamingDemandDrivenPipeline::UPDATE_PIECE_NUMBER());
  numPieces = outInfo->Get(vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_PIECES());
  outInfo->Get(vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_GHOST_LEVELS());

  //   if( (this->StartAngle == 0. && this->EndAngle == 360.) ||
  //       (this->StartAngle == 360. && this->EndAngle == 0. ) )
  //   {
  //       //use vtkDiskSource
  //     VTK_CREATE(vtkDiskSource, diskSource );
  //     diskSource->SetCircumferentialResolution( this->CircumferentialResolution );
  //     diskSource->SetRadialResolution( this->RadialResolution );
  //     diskSource->SetInnerRadius( this->InnerRadius );
  //     diskSource->SetOuterRadius( this->OuterRadius );

  //     if (output->GetUpdatePiece() == 0 && numPieces > 0)
  //     {
  //       diskSource->Update();
  //       output->ShallowCopy(diskSource->GetOutput());
  //     }
  //     output->SetUpdatePiece(piece);
  //     output->SetUpdateNumberOfPieces(numPieces);
  //     output->SetUpdateGhostLevel(ghostLevel);
  //   }
  //   else
  //   {
  VTK_CREATE(vtkLineSource, lineSource);
  lineSource->SetResolution(this->RadialResolution);

  // set vertex 1, adjust for start angle
  // set vertex 2, adjust for start angle
  double x1[3], x2[3];
  x1[0] = this->InnerRadius * cos(vtkMath::RadiansFromDegrees(this->StartAngle));
  x1[1] = this->InnerRadius * sin(vtkMath::RadiansFromDegrees(this->StartAngle));
  x1[2] = this->ZCoord;

  x2[0] = this->OuterRadius * cos(vtkMath::RadiansFromDegrees(this->StartAngle));
  x2[1] = this->OuterRadius * sin(vtkMath::RadiansFromDegrees(this->StartAngle));
  x2[2] = this->ZCoord;

  lineSource->SetPoint1(x1);
  lineSource->SetPoint2(x2);
  lineSource->SetContainerAlgorithm(this);
  lineSource->Update();

  VTK_CREATE(vtkRotationalExtrusionFilter, rotateFilter);
  rotateFilter->SetResolution(this->CircumferentialResolution);
  rotateFilter->SetInputConnection(lineSource->GetOutputPort());
  rotateFilter->SetAngle(this->EndAngle - this->StartAngle);

  if (piece == 0 && numPieces > 0)
  {
    rotateFilter->SetContainerAlgorithm(this);
    rotateFilter->Update();
    output->ShallowCopy(rotateFilter->GetOutput());
  }
  //  }

  return 1;
}

void vtkSectorSource::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "InnerRadius: " << this->InnerRadius << "\n";
  os << indent << "OuterRadius: " << this->OuterRadius << "\n";
  os << indent << "ZCoord: " << this->ZCoord << "\n";
  os << indent << "StartAngle: " << this->StartAngle << "\n";
  os << indent << "EndAngle: " << this->EndAngle << "\n";
  os << indent << "CircumferentialResolution: " << this->CircumferentialResolution << "\n";
  os << indent << "RadialResolution: " << this->RadialResolution << "\n";
}
VTK_ABI_NAMESPACE_END
