/*
 *  yosys -- Yosys Open SYnthesis Suite
 *
 *  Copyright (C) 2012  Claire Xenia Wolf <claire@yosyshq.com>
 *
 *  Permission to use, copy, modify, and/or distribute this software for any
 *  purpose with or without fee is hereby granted, provided that the above
 *  copyright notice and this permission notice appear in all copies.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 *  WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 *  ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 *  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 *  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 *  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "kernel/register.h"
#include "kernel/log.h"
#include <stdlib.h>
#include <stdio.h>

USING_YOSYS_NAMESPACE
PRIVATE_NAMESPACE_BEGIN

struct ProcPass : public Pass {
	ProcPass() : Pass("proc", "translate processes to netlists") { }
	void help() override
	{
		//   |---v---|---v---|---v---|---v---|---v---|---v---|---v---|---v---|---v---|---v---|
		log("\n");
		log("    proc [options] [selection]\n");
		log("\n");
		log("This pass calls all the other proc_* passes in the most common order.\n");
		log("\n");
		log("    proc_clean\n");
		log("    proc_rmdead\n");
		log("    proc_prune\n");
		log("    proc_init\n");
		log("    proc_arst\n");
		log("    proc_rom\n");
		log("    proc_mux\n");
		log("    proc_dlatch\n");
		log("    proc_dff\n");
		log("    proc_memwr\n");
		log("    proc_clean\n");
		log("    opt_expr -keepdc\n");
		log("\n");
		log("This replaces the processes in the design with multiplexers,\n");
		log("flip-flops and latches.\n");
		log("\n");
		log("The following options are supported:\n");
		log("\n");
		log("    -nomux\n");
		log("        Will omit the proc_mux pass.\n");
		log("\n");
		log("    -norom\n");
		log("        Will omit the proc_rom pass.\n");
		log("\n");
		log("    -global_arst [!]<netname>\n");
		log("        This option is passed through to proc_arst.\n");
		log("\n");
		log("    -ifx\n");
		log("        This option is passed through to proc_mux. proc_rmdead is not\n");
		log("        executed in -ifx mode.\n");
		log("\n");
		log("    -noopt\n");
		log("        Will omit the opt_expr pass.\n");
		log("\n");
	}
	void execute(std::vector<std::string> args, RTLIL::Design *design) override
	{
		std::string global_arst;
		bool ifxmode = false;
		bool nomux = false;
		bool noopt = false;
		bool norom = false;

		log_header(design, "Executing PROC pass (convert processes to netlists).\n");
		log_push();

		size_t argidx;
		for (argidx = 1; argidx < args.size(); argidx++)
		{
			if (args[argidx] == "-nomux") {
				nomux = true;
				continue;
			}
			if (args[argidx] == "-global_arst" && argidx+1 < args.size()) {
				global_arst = args[++argidx];
				continue;
			}
			if (args[argidx] == "-ifx") {
				ifxmode = true;
				continue;
			}
			if (args[argidx] == "-noopt") {
				noopt = true;
				continue;
			}
			if (args[argidx] == "-norom") {
				norom = true;
				continue;
			}
			break;
		}
		extra_args(args, argidx, design);

		Pass::call(design, "proc_clean");
		if (!ifxmode)
			Pass::call(design, "proc_rmdead");
		Pass::call(design, "proc_prune");
		Pass::call(design, "proc_init");
		if (global_arst.empty())
			Pass::call(design, "proc_arst");
		else
			Pass::call(design, "proc_arst -global_arst " + global_arst);
		if (!norom)
			Pass::call(design, "proc_rom");
		if (!nomux)
			Pass::call(design, ifxmode ? "proc_mux -ifx" : "proc_mux");
		Pass::call(design, "proc_dlatch");
		Pass::call(design, "proc_dff");
		Pass::call(design, "proc_memwr");
		Pass::call(design, "proc_clean");
		if (!noopt)
			Pass::call(design, "opt_expr -keepdc");

		log_pop();
	}
} ProcPass;

PRIVATE_NAMESPACE_END
