# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: RateLimiter.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Limit the upload rate.

All the constants below are only for the automatic upload rate adjustment.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type MAX_RATE_PERIOD: C{float}
@var MAX_RATE_PERIOD: maximum amount of time to guess the current rate 
    estimate represents
@type MAX_RATE: C{float}
@var MAX_RATE: the (very large) limit to use to not limit the upload rate
@type PING_BOUNDARY: C{float}
@var PING_BOUNDARY: maximum ping time to consider still working
@type PING_SAMPLES: C{int}
@var PING_SAMPLES: minimum number of pings to require
@type PING_DISCARDS: C{int}
@var PING_DISCARDS: number of initial pings to discard
@type PING_THRESHHOLD: C{int}
@var PING_THRESHHOLD: number of pings that exceed L{PING_BOUNDARY} before the
    connection is assumed to be flooded
@type PING_DELAY: C{int}
@var PING_DELAY: not used
@type PING_DELAY_NEXT: C{int}
@var PING_DELAY_NEXT: not used
@type ADJUST_UP: C{float}
@var ADJUST_UP: fraction to adjust up the upload rate by
@type ADJUST_DOWN: C{float}
@var ADJUST_DOWN: fraction to adjust down the upload rate by
@type UP_DELAY_FIRST: C{int}
@var UP_DELAY_FIRST: number of cycles after an adjust down before adjusting
    up the upload rate
@type UP_DELAY_NEXT: C{int}
@var UP_DELAY_NEXT: number of cycles after an adjust up before adjusting
    up the upload rate again
@type SLOTS_STARTING: C{int}
@var SLOTS_STARTING: the starting number of upload slots to use
@type SLOTS_FACTOR: C{float}
@var SLOTS_FACTOR: factor to use in calculating the number of upload slots

"""

import logging
from binascii import b2a_hex
from clock import clock
from CurrentRateMeasure import Measure
from cStringIO import StringIO
from math import sqrt

logger = logging.getLogger('DebTorrent.RateLimiter')

MAX_RATE_PERIOD = 20.0
MAX_RATE = 10e10
PING_BOUNDARY = 1.2
PING_SAMPLES = 7
PING_DISCARDS = 1
PING_THRESHHOLD = 5
PING_DELAY = 5  # cycles 'til first upward adjustment
PING_DELAY_NEXT = 2  # 'til next
ADJUST_UP = 1.05
ADJUST_DOWN = 0.95
UP_DELAY_FIRST = 5
UP_DELAY_NEXT = 2
SLOTS_STARTING = 6
SLOTS_FACTOR = 1.66/1000

class RateLimiter:
    """Limit the upload rate.
    
    @type sched: C{method}
    @ivar sched: the method to call to schedule a task with the server
    @type last: L{BT1.Connecter.Connection}
    @ivar last: the last connection on the circular queue of connections to send on
    @type unitsize: C{int}
    @ivar unitsize: when limiting upload rate, how many bytes to send at a time
    @type slotsfunc: C{method}
    @ivar slotsfunc: the method to call to set the number of connections limits
    @type measure: L{CurrentRateMeasure.Measure}
    @ivar measure: the measurer to use to help calculate the upload rate
    @type autoadjust: C{boolean}
    @ivar autoadjust: whether the upload limit is being automatically adjusted
    @type upload_rate: C{float}
    @ivar upload_rate: the maximum upload rate to limit to
    @type slots: C{int}
    @ivar slots: the number of upload slots to use (for automatic adjustment)
    @type autoadjustup: C{int}
    @ivar autoadjustup: number of cycles remaining before adjusting up the
        upload rate
    @type pings: C{list} of C{boolean}
    @ivar pings: the latest ping results, True if the threshold was exceeded
    @type lasttime: C{float}
    @ivar lasttime: the last time data was sent
    @type bytes_sent: C{int}
    @ivar bytes_sent: the number of bytes sent on the most recent attempt
    
    """
    
    def __init__(self, sched, unitsize, slotsfunc = lambda x: None):
        """Initialize the instance.
        
        @type sched: C{method}
        @param sched: the method to call to schedule a task with the server
        @type unitsize: C{int}
        @param unitsize: when limiting upload rate, how many bytes to send at a time
        @type slotsfunc: C{method}
        @param slotsfunc: the method to call to set the number of connections limits
            (optional, defaults to not setting anything)
        
        """
        
        self.sched = sched
        self.last = None
        self.unitsize = unitsize
        self.slotsfunc = slotsfunc
        self.measure = Measure(MAX_RATE_PERIOD)
        self.autoadjust = False
        self.upload_rate = MAX_RATE * 1000
        self.slots = SLOTS_STARTING    # garbage if not automatic

    def set_upload_rate(self, rate):
        """Set the upload rate to limit to.
        
        @type rate: C{float}
        @param rate: maximum KB/s to upload at (0 = no limit, -1 = automatic)
        
        """
        
        if rate < 0:
            if self.autoadjust:
                return
            self.autoadjust = True
            self.autoadjustup = 0
            self.pings = []
            rate = MAX_RATE
            self.slots = SLOTS_STARTING
            self.slotsfunc(self.slots)
        else:
            self.autoadjust = False
        if not rate:
            rate = MAX_RATE
        self.upload_rate = rate * 1000
        self.lasttime = clock()
        self.bytes_sent = 0

    def queue(self, conn):
        """Queue the connection for later uploading.
        
        The queue is actually stored in the L{BT1.Connecter.Connection} objects
        using their next_upload instance variable, and is circular.
        
        If the queue is empty, this will start the queue processer.
        
        @type conn: L{BT1.Connecter.Connection}
        @param conn: the connection to add to the queue for uploading to
        
        """
        
        assert conn.next_upload is None
        if self.last is None:
            self.last = conn
            conn.next_upload = conn
            self.try_send(True)
        else:
            conn.next_upload = self.last.next_upload
            self.last.next_upload = conn
            self.last = conn

    def try_send(self, check_time = False):
        """Loop through the circular queue of upload connections, trying to send on each.
        
        @type check_time: C{boolean}
        @param check_time: whether to make sure the bytes sent is not negative
        
        """
        
        t = clock()
        self.bytes_sent -= (t - self.lasttime) * self.upload_rate
        self.lasttime = t
        if check_time:
            self.bytes_sent = max(self.bytes_sent, 0)
            
        # Get the first connection (the next one after the last one)
        cur = self.last.next_upload
        
        # Loop until some data has been sent (or the queue is empty)
        while self.bytes_sent <= 0:
            # Try sending bytes on it
            bytes = cur.send_partial(self.unitsize)
            self.bytes_sent += bytes
            self.measure.update_rate(bytes)
            
            # If nothing could be sent, remove the connection from the queue
            if bytes == 0 or cur.backlogged():
                if self.last is cur:
                    # If this is the only connection in the queue, stop the processer
                    self.last = None
                    cur.next_upload = None
                    break
                else:
                    self.last.next_upload = cur.next_upload
                    cur.next_upload = None
                    cur = self.last.next_upload
            else:
                # Go to the next connection
                self.last = cur
                cur = cur.next_upload
        else:
            # The queue is not empty, so schedule a future attempt to send
            self.sched(self.try_send, self.bytes_sent / self.upload_rate)

    def adjust_sent(self, bytes):
        """Add data from other places to this measure.
        
        @type bytes: C{int}
        @param bytes: the amount of data that was sent/received
        
        """
        
        self.bytes_sent = min(self.bytes_sent+bytes, self.upload_rate*3)
        self.measure.update_rate(bytes)


    def ping(self, delay):
        """Use the new ping time to calculate an automatically adjusted upload limit.
        
        @type delay: C{float}
        @param delay: the elapsed time between unchoking and receiving a request
        
        """
        
        logger.debug('ping delay: '+str(delay))
        if not self.autoadjust:
            return
        self.pings.append(delay > PING_BOUNDARY)
        if len(self.pings) < PING_SAMPLES+PING_DISCARDS:
            return
        logger.debug('cycle')
        pings = sum(self.pings[PING_DISCARDS:])
        del self.pings[:]
        if pings >= PING_THRESHHOLD:   # assume flooded
            if self.upload_rate == MAX_RATE:
                self.upload_rate = self.measure.get_rate()*ADJUST_DOWN
            else:
                self.upload_rate = min(self.upload_rate,
                                       self.measure.get_rate()*1.1)
            self.upload_rate = max(int(self.upload_rate*ADJUST_DOWN),2)
            self.slots = int(sqrt(self.upload_rate*SLOTS_FACTOR))
            self.slotsfunc(self.slots)
            logger.debug('adjust down to '+str(self.upload_rate))
            self.lasttime = clock()
            self.bytes_sent = 0
            self.autoadjustup = UP_DELAY_FIRST
        else:   # not flooded
            if self.upload_rate == MAX_RATE:
                return
            self.autoadjustup -= 1
            if self.autoadjustup:
                return
            self.upload_rate = int(self.upload_rate*ADJUST_UP)
            self.slots = int(sqrt(self.upload_rate*SLOTS_FACTOR))
            self.slotsfunc(self.slots)
            logger.debug('adjust up to '+str(self.upload_rate))
            self.lasttime = clock()
            self.bytes_sent = 0
            self.autoadjustup = UP_DELAY_NEXT
