#!/usr/bin/env python
# SPDX-License-Identifier: BSD-3-Clause
#
# Script to gather statistics from MIDI files about instrument usage.
# Generates stats.py which is used as input for the ultramid.ini
# generator script.
#
# This script requires python-midi; see here:
#   https://github.com/vishnubob/python-midi
#

import midi
import sys

def get_instr_stats(filename):
	"""Get a set of instruments used by the specified MIDI file."""
	result = set()
	midfile = midi.read_midifile(filename)

	for track in midfile:
		for event in track:
			if isinstance(event, midi.ProgramChangeEvent) \
			   and event.channel != 9:
				instr = event.data[0]
				result.add(instr)
			# Percussion:
			if isinstance(event, midi.NoteOnEvent) \
			   and event.channel == 9:
				instr = event.data[0] + 128
				result.add(instr)

	return result

total_stats = [0] * 217

for filename in sys.argv[1:]:
	print "Processing %s" % filename
	stats = get_instr_stats(filename)
	print sorted(stats)
	for instrument in stats:
		total_stats[instrument] += 1

with open("stats.py", "w") as f:
	f.write("# Instrument stats, autogenerated by gather_stats.py\n\n")
	f.write("INSTRUMENT_STATS = [\n\t")

	for index, stat in enumerate(total_stats):
		f.write("% 5i," % stat)
		if (index % 10) == 9:
			f.write("\n\t")

	f.write("\n]\n")


